/*
 *      main.c of piano package -- input sound samples
 *	piano tuning program by Guenther Montag
 *
 *	derived from: unixinput.c from multimon package
 *      Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *	Terminal code from Pawel Jalocha's famous ham radio mt63
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* ---------------------------------------------------------------------- */


#include "piano.h"
#include "term.h"
#include <ncurses.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <sys/wait.h>
#include <stdlib.h>

#ifdef SUN_AUDIO
#include <sys/audioio.h>
#include <stropts.h>
#include <sys/conf.h>
#else /* SUN_AUDIO */
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#endif /* SUN_AUDIO */

/* ---------------------------------------------------------------------- */

const struct demod_param *dem[] = { SMALL_DEMOD };
struct demod_state dem_st[NUMDEMOD];
unsigned int dem_mask[NUMDEMOD];
int verboselevel = 0;
int autotune = 0, noteinput = 0, freqinput = 0, tone = 0;
int user_octave = 0, admin_octave = 0;
int chamberhzbias = 0;
char note[32];
int sample_rate = -1;
unsigned int overlap = 0;
char *input_type = "hw";
float soundcorr = 1.0, bias = 1.0, inputfreq = 0, inputbias;
char params[55];

/* ---------------------------------------------------------------------- */

char usage_str[] = 
"tuning / frequency measuring tool by Gnther Montag Safaridoktor@compuserve.de\n"
"* * *  Options:   * * *\n"
"  -b  bias: numeric <0.9 ... 1.1> or Hz deviation from a (440 Hz) <-44...+44>\n"
"      (which is useful for old pianos which are often tuned lower !)\n"
"  -f  fine tune a given frequency (enter center frequency 55 - 22000 Hz)\n"
"  -n  note <A, A# = Bb, B, C..., a, a# = bb, b, c...> (with -o)\n"
"  -o  octave <-1 ...6> \n"
"  -s  soundcard samplerate correction factor <0.9 ... 1.1> \n"
"  -v  verbose debug output, -h  help\n"
"If another sound-device than /dev/dsp, give it as last argument.\n"
"Default: auto-detect note and auto-switch to fine tune and measure.\n"
"more help: see `man piano`";

/* ---------------------------------------------------------------------- */

void verbprintf(const char *fmt, ...) //unused 
{
        va_list args;
        va_start(args, fmt);
        if (1 <= verboselevel)
                vfprintf(stdout, fmt, args);
        va_end(args);
}

/* ---------------------------------------------------------------------- */

static void process_buffer(float *buf, unsigned int len)
{
    int i;
    i = getch();
    if (i == 27) {
	info("bye! \n");
	Close(0);
	printf("bye\n");
    }
    if ((i == 13) || (i == 32)) {
	if (autotune) { 
    	    info("re-starting autotune mode.\n");
	    dem_mask[0] = 1;
    	    dem_mask[1] = 0;
    	    verbinfo ("automatic tone detection and tuning ...\n");
	    init_demod();
	} else {
	    info ("you selected a frequency or note -> Restart to autotune.\n");
	}
    }	
    for (i = 0; i <  NUMDEMOD; i++) {
    	if ((dem_mask[i] == 1) && dem[i]->demod)
	    dem[i]->demod(dem_st+i, buf, len);
    }

    return;
}

/* ---------------------------------------------------------------------- */
#ifdef SUN_AUDIO

static void input_sound(unsigned int sample_rate, unsigned int overlap,
			const char *ifname)
{
        audio_info_t audioinfo;
        audio_info_t audioinfo2;
	audio_device_t audiodev;
	int fd;
	short buffer[8192];
	float fbuf[16384];
	unsigned int fbuf_cnt = 0;
	int i;
	short *sp;

	if ((fd = open(ifname ? ifname : "/dev/audio", O_RDONLY)) < 0) {
		warninfo("Sorry, cannot open soundcard.\n");
		Close (10);
	}
        if (ioctl(fd, AUDIO_GETDEV, &audiodev) == -1) {
		warninfo("ioctl: AUDIO_GETDEV");
		Close (10);
        }
        AUDIO_INITINFO(&audioinfo);
        audioinfo.record.sample_rate = sample_rate;
        audioinfo.record.channels = 1;
        audioinfo.record.precision = 16;
        audioinfo.record.encoding = AUDIO_ENCODING_LINEAR;
        /*audioinfo.record.gain = 0x20;
	  audioinfo.record.port = AUDIO_LINE_IN;
	  audioinfo.monitor_gain = 0;*/
        if (ioctl(fd, AUDIO_SETINFO, &audioinfo) == -1) {
		warninfo("ioctl: AUDIO_SETINFO");
		Close (10);
        }     
        if (ioctl(fd, I_FLUSH, FLUSHR) == -1) {
		warninfo("ioctl: I_FLUSH");
		Close (10);
        }
        if (ioctl(fd, AUDIO_GETINFO, &audioinfo2) == -1) {
		warninfo("ioctl: AUDIO_GETINFO");
		Close (10);
        }
        info( "Audio device: name %s, ver %s, config %s, "
		"sampling rate %d\n", audiodev.name, audiodev.version,
		audiodev.config, audioinfo.record.sample_rate);
	for (;;) {
		i = read(fd, sp = buffer, sizeof(buffer));
		if (i < 0 && errno != EAGAIN) {
			warninfo("cannot read from soundcard.\n");
			Close(4);
		}
		if (!i)
			break;
		if (i > 0) {
			for (; i >= sizeof(buffer[0]); i -= sizeof(buffer[0]), sp++)
				fbuf[fbuf_cnt++] = (*sp) * (1.0/32768.0);
			if (i)
				warninfo( "warning: noninteger number of samples read\n");
			if (fbuf_cnt > overlap) {
				process_buffer(fbuf, fbuf_cnt-overlap);
				memmove(fbuf, fbuf+fbuf_cnt-overlap, overlap*sizeof(fbuf[0]));
				fbuf_cnt = overlap;
			}
		}
	}
	close(fd);
}

#else /* SUN_AUDIO */
/* ---------------------------------------------------------------------- */

static void input_sound(unsigned int sample_rate, unsigned int overlap,
			const char *ifname)
{
        int sndparam;
	int fd;
	union {
		short s[8192];
		unsigned char b[8192];
	} b;
	float fbuf[16384];
	unsigned int fbuf_cnt = 0;
	int i;
	short *sp;
	unsigned char *bp;
	int fmt = 0;

	if ((fd = open(ifname ? ifname : "/dev/dsp", O_RDONLY)) < 0) {
		warninfo("Cannot open soundcard.\n");
		Close (10);
	}
        sndparam = AFMT_S16_LE; /* we want 16 bits/sample signed */
        /* little endian; works only on little endian systems! */
        if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) == -1) {
		warninfo("ioctl: SNDCTL_DSP_SETFMT");
		Close (10);
	}
        if (sndparam != AFMT_S16_LE) {
		fmt = 1;
		sndparam = AFMT_U8;
		if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) == -1) {
			warninfo("ioctl: SNDCTL_DSP_SETFMT");
			Close (10);
		}
		if (sndparam != AFMT_U8) {
			warninfo("ioctl: SNDCTL_DSP_SETFMT");
			Close (10);
		}
        }
        sndparam = 0;   /* we want only 1 channel */
        if (ioctl(fd, SNDCTL_DSP_STEREO, &sndparam) == -1) {
		warninfo("ioctl: SNDCTL_DSP_STEREO");
		Close (10);
	}
        if (sndparam != 0) {
                info( "soundif: Error, cannot set the channel "
                        "number to 1\n");
                Close (10);
        }
        sndparam = sample_rate; 
        if (ioctl(fd, SNDCTL_DSP_SPEED, &sndparam) == -1) {
		warninfo("ioctl: SNDCTL_DSP_SPEED");
		Close (10);
	}
        if ((10*abs(sndparam-sample_rate)) > sample_rate) {
		warninfo("ioctl: SNDCTL_DSP_SPEED");
		Close (10);
	}
        if (sndparam != sample_rate) {
                info( "Warning: Sampling rate is %u, "
                        "requested %u\n", sndparam, sample_rate);
        }
#if 0
        sndparam = 4;
        if (ioctl(fd, SOUND_PCM_SUBDIVIDE, &sndparam) == -1) {
		warninfo("ioctl: SOUND_PCM_SUBDIVIDE");
        }
        if (sndparam != 4) {
		warninfo("ioctl: SOUND_PCM_SUBDIVIDE");
	}
#endif
	for (;;) {
		if (fmt) {
			i = read(fd, bp = b.b, sizeof(b.b));
			if (i < 0 && errno != EAGAIN) {
				warninfo("Cannot read from soundcard.\n");
				Close(4);
			}
			if (!i) {
				break;
				warninfo( "Soundcard data reading problem\n");
			}
			if (i > 0) {
				for (; i >= sizeof(b.b[0]); i -= sizeof(b.b[0]), sp++)
					fbuf[fbuf_cnt++] = ((int)(*bp)-0x80) * (1.0/128.0);
				if (i)
					warninfo( "warning: noninteger number of samples read\n");
				if (fbuf_cnt > overlap) {
					process_buffer(fbuf, fbuf_cnt-overlap);
					memmove(fbuf, fbuf+fbuf_cnt-overlap, overlap*sizeof(fbuf[0]));
					fbuf_cnt = overlap;
				}
			}
		} else {
			i = read(fd, sp = b.s, sizeof(b.s));
			if (i < 0 && errno != EAGAIN) {
				warninfo("Error in reading soundcard data.\n");
				Close(4);
			}
			if (!i) {
				warninfo( "nothing to read from soundcard\n");
				break;
			}
			if (i > 0) {
				for (; i >= sizeof(b.s[0]); i -= sizeof(b.s[0]), sp++)
					fbuf[fbuf_cnt++] = (*sp) * (1.0/32768.0);
				if (i)
					warninfo( "warning: noninteger number of samples read\n");
				if (fbuf_cnt > overlap) {
					process_buffer(fbuf, fbuf_cnt-overlap);
					memmove(fbuf, fbuf+fbuf_cnt-overlap, overlap*sizeof(fbuf[0]));
					fbuf_cnt = overlap;
				}
			}
		}
	}
	close(fd);
}
#endif /* SUN_AUDIO */

/* ---------------------------------------------------------------------- */

void init_demod()
{
    int demodcount = 0, i;
    //verbinfo("enabled modes:\n");
    for (i = 0; i < NUMDEMOD; i++) {
	if (dem_mask[i] == 1) {
	    demodcount ++;
	    //verbinfo("\t%s\n", dem[i]->name);
	    memset(dem_st+i, 0, sizeof(dem_st[i]));
	    dem_st[i].dem_par = dem[i];
	    if (dem[i]->init) dem[i]->init(dem_st+i);
	    if (sample_rate == -1) sample_rate = dem[i]->samplerate;
	    else if (sample_rate != dem[i]->samplerate) 
		warninfo("samplerate not set right for this mode\n");
	    if (dem[i]->overlap > overlap)
		overlap = dem[i]->overlap;
	}
	//verbinfo("\n");
    }
    if (!demodcount) {
	warninfo("no mode specified, check your options!\n");
	info(usage_str);
	Close(2);
    }
    return;
}

/* ---------------------------------------------------------------------- */

void get_tone(char *note, int admin_octave)
{	
	char lownote[3];

	if (note[0] == '\n') return;
	lownote[0] = tolower(note[0]);
	lownote[1] = tolower(note[1]);
	lownote[2] = '\0';
	tone = (admin_octave + 2) * 12;
	
    	if ( ! strcmp (lownote, "c"))       tone = (tone-9);
	else if ( ! strcmp (lownote, "c#")) tone = (tone-8);    
	else if ( ! strcmp (lownote, "db")) tone = (tone-8);    
	else if ( ! strcmp (lownote, "d"))  tone = (tone-7);
	else if ( ! strcmp (lownote, "d#")) tone = (tone-6);
	else if ( ! strcmp (lownote, "eb")) tone = (tone-6);
	else if ( ! strcmp (lownote, "e"))  tone = (tone-5);    
	else if ( ! strcmp (lownote, "f"))  tone = (tone-4);
	else if ( ! strcmp (lownote, "f#")) tone = (tone-3);
	else if ( ! strcmp (lownote, "gb") ) tone = (tone-3);
	else if ( ! strcmp (lownote, "g"))  tone = (tone-2);
	else if ( ! strcmp (lownote, "g#")) tone = (tone-1);
	else if ( ! strcmp (lownote, "ab")) tone = (tone-1);
	else if ( ! strcmp (lownote, "a"))  tone = (tone+0);
	else if ( ! strcmp (lownote, "a#")) tone = (tone+1);
	else if ( ! strcmp (lownote, "hb"))  tone = (tone+1);
	else if ( ! strcmp (lownote, "bb"))  tone = (tone+1);
	else if ( ! strcmp (lownote, "b"))  tone = (tone+2);
	else if ( ! strcmp (lownote, "h"))  tone = (tone+2);
	else {
	    warninfo("invalid note %s\n", note);
	    Close(2);
	}
	//verbinfo("tone: %d", tone);
	if (tone < 0) {
	    warninfo ("tone % d < 0, too low note");
	    Close(2);
	}
	return;
}

/* ---------------------------------------------------------------------- */

			/*
			user_octave		admin_octave
			-2		-3 ... etc.
			-1		-2
			0, note capital	-1
			0, note small	 0
			1		 1 
			2		 2 ... etc.
			*/	

/* ---------------------------------------------------------------------- */

void user_octave_2_admin_octave (void) {
	if (user_octave < 0) {
	    if (isupper(note[0])) {
		admin_octave = user_octave - 1;
		return;
	    }
	    else {
		warninfo
		    ("invalid note %s for low octave %d", note, user_octave);
		warninfo
		    ("I expect low notes to be civen in capitals.\n");
		close(2);
	    }
	} else if (! user_octave) {
	    if (isupper(note[0])) {
		verbinfo
		    ("You gave capital note %s, I set my admin_octave to -1\n", 
			note);
		admin_octave = -1;
		return;
	    }
	    else if (islower(note[0])) {
		verbinfo
		    ("You gave small note %s, I set my admin_octave to 0\n", 
			note);
		admin_octave = 0;
		return;
	    }
	    else {
		warninfo("invalid note %s\n", note);
		close(2);
	    }
	} else if (user_octave > 0) {
	    if (islower(note[0])) {
		admin_octave = user_octave;
		return;
	    }
	    else {
		warninfo
		    ("invalid note %s for high octave %d\n", note, user_octave);
		warninfo
		    ("I expect high notes to be civen in small letters.\n");
		close(2);
	    }
	}
}

/* ---------------------------------------------------------------------- */

void admin_octave_2_user_octave(void) {
	if (admin_octave < -1 ) {
	    user_octave = admin_octave +1;
	    return; 
	}
	else if (admin_octave <= 0) {
	    user_octave = 0;
	    return;
	}
	else user_octave = admin_octave;
}

/* ---------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
    int c, i, options = 0, errflg = 0; 

    memset (note, 0, sizeof (note));
    Preset(4);
    statusinfo(0, "* * * piano * * *");
    //info("\t\t* * * tuning programme and audio frequency measurer * * * \n");
    verbinfo("available modes:");
    //statusinfo(1, "                                                      ");
    //statusinfo(2, "                                                      ");
    for (i = 0; i < NUMDEMOD; i++) 
	verbinfo("\n\t%s", dem[i]->name);
    //info("\n");
    while ((c = getopt(argc, argv, "b:f:hn:o:s:v")) != EOF) {
	options = 1;
	switch (c) {
	    case 'f':
		inputfreq = strtod(optarg,  0);
		if (50.0 < inputfreq && inputfreq < 25000.0) {
		    freqinput = 1;
		    info("tuning exactly frequency %f ...\n", inputfreq);
		}
		else { 
		    warninfo
			( "invalid frequency %f, can only tune 50 - 25000 !\n", 
			inputfreq);
		    Close (2);
		}
		break;
	    case 'b':
		if (freqinput == 1) {
		    warninfo( "preset tune frequency %f, no sense to accept a bias.\n",
		    inputfreq);
		    Close (2);
		}
		inputbias = strtod(optarg, 0);
		if (inputbias != 1.0 && 0.9 < inputbias && inputbias < 1.1) {
		    bias = inputbias;
		    chamberhzbias = (inputbias * 440) - 440;
		    goto biasisok;
		}
		if (! strstr(optarg, ".") && -45 < inputbias && inputbias < 45) {
		    chamberhzbias = inputbias;
		    bias = (inputbias + 440 ) / 440;
		    goto biasisok;
		}
		else {
		    warninfo( "invalid bias %f !\n"
			"can only take numeric bias from 0.9 - 1.1 or\n" 
			"shift chambertone a (440 Hz) from -44 to +44Hz\n",
		    inputbias);
		    Close (2);
		}
		biasisok:
		info( "bias %2.3f -> chambertone a = 440 Hz %+d Hz)\n",
		     bias, chamberhzbias);
		break;
	    case 's':
		soundcorr = strtod(optarg, 0);
		if ( 0.9 < soundcorr && soundcorr < 1.1) 
		    verbinfo( "soundcard samplerate correction %2.3f \n");
		else {
		    warninfo( "sound correction %2.3f not in range 0.9 - 1.1 !\n", 
			soundcorr);
		    Close (2);
		}
		break;
	    case 'n':
		strcpy(note, optarg);
		note[strlen(note)] = 0;
		verbinfo( "preparing to tune note %s ...\n", note);
		break;
	    case 'o':
		user_octave = atoi(optarg);
		if ((isdigit(optarg[1]) || isdigit(optarg[0])) 
		    && -2 <= user_octave && user_octave <= 6)
			verbinfo( "actual octave is %d ...\n", user_octave);
		else {
		    warninfo
			( "invalid octave %d, can only tune octave -1...6 !\n", 
			user_octave);
		    Close (2);
		    }
	    break;
	    case 'v':
		verboselevel = 1;
		Preset(17);
	        statusinfo(0, "* * * piano debug mode * * *");
    		verbinfo( "Verbose output. Don't frighten...\n");
		break; 
	    case '?':
	    case 'h':
	    case ' ':
		errflg++;
		break;
	    default: // seems not to work
		info("invalid option....");
		errflg++;
		break;
	}
	if (errflg) {
    	    RxStr(usage_str);
	    Close(2);
	}
    }	
    /* if note specified for tuning */
    if ( strcmp (note, "")) {
	/*  if a note is given,
	    calculate my admin_octave,
	    which makes it easier later*/
	user_octave_2_admin_octave();
	get_tone(note, admin_octave);
	if (0 <= tone && tone < 107)
	    verbinfo( "tuning tone %d ...\n", tone);
	else {
	    warninfo 
		("invalid tone %d, can only tune 1 - 106!\n", 
		tone);
	    Close (2);
	}	
	sprintf(foundnote, "%s", note); // for status output	
	dem_mask[0] = 0;
	dem_mask[1] = 1;
	verbinfo ("tuning preset tone...\n");
	noteinput = 1;
    } else if (freqinput) {
        dem_mask[0] = 0;
        dem_mask[1] = 1;
        autotune = 0;
        verbinfo ("tuning preset frequency ...\n");
    } else {
	/* auto-tune */
        dem_mask[0] = 1;
        dem_mask[1] = 0;
        autotune = 1;
        verbinfo ("automatic tone detection and tuning ...\n");
    }

    if (soundcorr != 1.0 && chamberhzbias != 0) {
	sprintf(params, "with bias %1.3f (a = %d), sound correction: %1.5f",
	    bias, 440 + chamberhzbias, soundcorr);
    } else if (chamberhzbias != 0) {
	sprintf(params, "with bias %1.3f (chambertone a = %d)",
	    bias, 440 + chamberhzbias);
    }
    else if (soundcorr != 1.0) {
	sprintf(params, "with soundcard samplerate correction %1.5f",
	    soundcorr);
    } else {
	sprintf(params, " ");
    }
    if (! freqinput && ! noteinput) {
	statusinfo(3, "Quit: <esc>  |  Restart autodetection: <Space>");
    } else {
	statusinfo(3, "Quit: <esc>");
    }
    /* the routine which looks after keyboard inputs
       is in the process_buffer loop
       see above in this file */
    init_demod();

    if (!strcmp(input_type, "hw")) {
	if ((argc - optind) >= 1)
    	    input_sound(sample_rate, overlap, argv[optind]);
	else 
	    input_sound(sample_rate, overlap, NULL);
	Close(0);
	}
    Close (0);
    return 0;
}
/* ---------------------------------------------------------------------- */


