;**********************************************************************;
;*                              V E G A                               *;
;*--------------------------------------------------------------------*;
;*    Task           : Creates elementary functions for accessing the *;
;*                     graphic modes on an EGA/VGA card               *;
;*--------------------------------------------------------------------*;
;*    Author         : MICHAEL TISCHER                                *;
;*    Developed on   :  10/3/1988                                     *;
;*    Last update    :  6/19/1989                                     *;
;*--------------------------------------------------------------------*;
;*    Assembly       : MASM VEGA;                                     *;
;*                     LINK VEGA;                                     *;
;*--------------------------------------------------------------------*;
;*    Call           : VEGA                                           *;
;**********************************************************************;
                                                                        
;== Constants ==========================================================
                                                                         
VIO_SEG     = 0A000h              ;Segment address of video RAM
                                  ;in graphic mode
LINE_LEN    = 80                  ;Every graphi line in EGA/VGA graphic
                                  ;modes require 80 bytes
BITMASK_REG = 8                   ;Bitmask register
MODE_REG    = 5                   ;Mode register
FUNCSEL_REG = 3                   ;Function select register
MAPSEL_REG  = 4                   ;Map-Select register
ENABLE_REG  = 1                   ;Enable Set/Reset register
SETRES_REG  = 0                   ;Set/Reset register
GRAPH_CONT  = 3CEh                ;Port addressd of graphic controller
OP_MODE     = 0                   ;Comparison operator mode:
                                  ;   00h = Replace
                                  ;   08h = AND comparison
                                  ;   10h = OR comparison
                                  ;   18h = EXCLUSIVE OR comparison

GR_640_350  = 10h                 ;BIOS code for 640x350-pixel
                                  ;16-color graphic mode
TX_80_25    = 03h                 ;BIOS code for 80*25-char.
                                  ;text mode

;== Stack ==============================================================

stack     segment para stack      ;Definition of stack segment

          dw 256 dup (?)          ;256-word stack

stack     ends                    ;End of stack segment

;== Data ===============================================================

data      segment para 'DATA'     ;Definition of data segment

;== Data for the demo program ==========================================

initm   db 13,10
	db "VEGA (c) 1988 by Michael Tischer"
     db 13,10,13,10
     db "This demonstration program operates only with an EGA/",13,10
	db "card and a hi-res monitor. If your PC doesn't have this",13,10
	db "configuration, please press the <s> key to abort the",13,10
	db "program.",13,10
     db "Press any other key to start the program.",13,10,"$"
                                                                        
data      ends                    ;End of data segment
                                                                        
;== Code ===============================================================
                                                                        
code      segment para 'CODE'     ;Definition of code segment
                                                                        
          assume cs:code, ds:data, es:data, ss:stack

;== Demo program =======================================================

demo      proc far

          mov  ax,data            ;Get segment addr. from data segment
          mov  ds,ax              ;and load into DS
          mov  es,ax              ;and ES
                                                                        
          ;-- Display opening message and wait for input ---------------
                                                                        
          mov  ah,9               ;Function number for string display
          mov  dx,offset initm    ;Message address
          int  21h                ;Call DOS interrupt
                                                                        
          xor  ah,ah              ;Get function number for key
          int  16h                ;Call BIOS keyboard interrupt
          cmp  al,"s"             ;Was <s> entered?
          je   ende               ;YES --> End program
          cmp  al,"S"             ;Was <S> entered?
          jne  startdemo          ;NO --> Start demo
                                                                        
ende:     mov  ax,4C00h           ;Function no. for end program
          int  21h                ;Call DOS interrupt 21(h)
                                                                        
          ;-- Initialize graphic mode ----------------------------------
                                                                        
startdemo label near
                                                                        
          mov  ax,GR_640_350      ;Initialize 64x350-pixel
          int  10h                ;16-color graphic mode
                                                                        
          mov  ch,000100001b      ;Color: Blue
          mov  ax,350             ;Number of raster lines: 350
          call fillscr            ;Fill screen
                                                                        
          ;-- The program displays two squares on the screens (the    --
          ;-- second is really a copy of the first) until the user    --
          ;-- presses a key to end the program                        --
                                                                        
          xor  ch,ch              ;Set color to 0
d1:       mov  ax,100             ;Starting line of first square
                                                                        
          inc  ch                 ;Increment color
          and  ch,15              ;AND bits 4 and 7
                                                                        
d2:       mov  bx,245             ;Starting column of first square
d3:       call setpix             ;Set pixel
          push cx                 ;Save color
          call getpix             ;Get pixel color
          push ax                 ;Push coordinates onto stack
          push bx
          add  bx,100             ;Compute position of second
          add  ax,100             ;square
          call setpix             ;Set pixel of copy
          pop  bx                 ;Return coordinates of first square
          pop  ax
          pop  cx                 ;Get color
          inc  bx                 ;Increment column
          cmp  bx,295             ;Reached the last column?
          jne  d3                 ;NO --> Set next pixel
                                                                        
          inc  ax                 ;YES, Increment line
          cmp  ax,150             ;Reached the last line?
          jne  d2                 ;NO --> Work with next line
                                                                        
          mov  ah,1               ;Read keyboard
          int  16h                ;Call BIOS keyboard interrupt
          je   d1                 ;No key pressed --> Continue
                                                                        
          mov  ax,TX_80_25        ;80x25 text mode
          int  10h                ;Initialization
          jmp  short ende         ;End programm
                                                                        
demo      endp
                                                                        
;== Functions used in the demo program =================================
                                                                        
;-- PIXPTR: Computes the address of a pixel within video RAM for the   -
;--         new EGA/VGA graphic modes
;-- Input    : AX    = Graphic line
;--            BX    = Graphic column
;-- Output   : ES:BX = Pointer to the byte in video RAM containing pixel
;--            CL    = Number of right shifts for the byte
;--                  = Number of byte shifts in ES:BX needed to isolate
;--                    the pixel
;--            AH    = Bitmask for combining with all other pixels
;-- Registers: ES, AX, BX and CL are changed
                                                                        
pixptr    proc near
                                                                        
          push dx                 ;Push DX onto stack
                                                                        
          mov  cl,bl              ;Save low byte of graphic column
          mov  dx,LINE_LEN        ;Number of bytes per line to DX
          mul  dx                 ;AX = graphic line * LINE_LEN
          shr  bx,1               ;Shift graphic column three places to
          shr  bx,1               ;the right, divide by 8
          shr  bx,1
          add  bx,ax              ;Add line offset
                                                                        
          mov  ax,VIO_SEG         ;Load segment address of video RAM
          mov  es,ax              ;into ES 
                                                                       
          and  cl,7               ;And bits 4 - 7 of graphic column
          xor  cl,7               ;Turn bits 0 - 3 then 
                                  ;subtract 7 - CL
          mov  ah,1               ;After shift, bit 0 should be 
                                  ;left alone
                                                                        
          pop  dx                 ;Pop DX off of stack
          ret                     ;Back to caller
                                                                        
pixptr    endp
                                                                        
;-- SETPIX: Sets a graphic pixel in the new EGA/VGA graphic modes ------
;-- Input    : AX    = graphic line
;--            BX    = graphic column
;--            CH    = pixel color
;-- Output   : none
;-- Registers: ES, DX and CL are changed
                                                                        
setpix    proc near
                                                                        
          push ax                 ;Push coordinates onto
          push bx                 ;the stack
                                                                        
          call pixptr             ;Computer pointer to the pixel
                                                                        
          mov  dx,GRAPH_CONT      ;Load port addr. of graphic controller
                                                                        
          ;-- Set bit position in bitmask register ---------------------
                                                                        
          shl  ah,cl              ;Mask for bit to be changed
          mov  al,BITMASK_REG     ;Move bitmask register from AL
          out  dx,ax              ;Write to register
                                                                        
          ;-- Set read mode 0 and write mode 2 -- ----------------------
                                                                        
          mov  ax,MODE_REG + (2 shl 8) ;Reg. no. and ,mode value
          out  dx,ax              ;Write in the register

          ;-- Define comparison mode between preceding latch -----------
          ;-- contents, and CPU byte                         -----------
                                                                        
          mov  ax,FUNCSEL_REG + (OP_MODE shl 8) ;Write register number 
          out  dx,ax              ;and comparison operator
                                                                        
          ;-- Pixel control --------------------------------------------
                                                                        
          mov  al,es:[bx]         ;Load latches
          mov  es:[bx],ch         ;Move color into bitplanes
                                                                        
          ;-- Set altered registers to their default (BIOS) ------------
          ;-- status                                        ------------
                                                                        
          mov  ax,BITMASK_REG + (0FFh shl 8) ;Set old bitmask
          out  dx,ax              ;Write in the register
          mov  ax,MODE_REG        ;Write old value for for mode register
          out  dx,ax              ;into register
          mov  ah,FUNCSEL_REG     ;Write old value for function select
          out  dx,ax              ;register into register
                                                                        
          pop  bx                 ;Pop coordinates off of stack
          pop  ax                 ;
          ret                     ;Back to caller
                                                                        
setpix    endp
                                                                        
;-- GETPIX: Places a pixel's color in one of the new EGA/VGA -----------
;--         graphic modes
;-- Input    : AX    = graphic line
;--            BX    = graphic column
;-- Output   : CH    = graphic pixel color
;-- Registers: ES, DX , CX and DI are changed

getpix    proc near
                                                                        
          push ax                 ;Push coordinates onto
          push bx                 ;the stack
                                                                        
          call pixptr             ;Computer pointer to pixel
          mov  ch,ah              ;Move bitmask to CH
          shl  ch,cl              ;Shift bitmask by bit positions
                                                                        
          mov  di,bx              ;Move video RAM offset to DI
          xor  bl,bl              ;Color value will be computed in BL
                                                                        
          mov  dx,GRAPH_CONT      ;Load graphic controller port address
          mov  ax,MAPSEL_REG + (3 shl 8) ;Access bitplane #3
                                                                        
          ;-- Go through each of the four bitplanes --------------------

gp1:      out  dx,ax              ;Activate bitplane #AH only
          mov  bh,es:[di]         ;Get byte from the bitplane
          and  bh,ch              ;Omit uninteresting bits
          neg  bh                 ;Bit 7 = 1, when a pixel is set
          rol  bx,1               ;Shift bit 7 from BH to Bit 1 in BL
                                                                        
          dec  ah                 ;Decrement bitplane number
          jge  gp1                ;Not  -1 yet? --> next bitplane
                                                                        
          ;-- The map select register must not be reset, since        --
          ;-- the EGA- and VGA-BIOS default to a value of 0           --
                                                                        
          mov  ch,bl              ;Get color from CH
          pop  bx                 ;Pop coordinates off
          pop  ax                 ;of stack
          ret                     ;Back to caller

getpix    endp
                                                                        
;-- FILLSCR: Sets all screen pixels to one color ------ ----------------
;-- Input    : AX    = number of graphic lines on the screen
;--            CH    = pixel color
;-- Output   : none
;-- Registers: ES, AX, CX, DI, DX and BL are changed
                                                                        
fillscr   proc near
                                                                        
          mov  dx,GRAPH_CONT      ;Load graphic controller port address
          mov  al,SETRES_REG      ;Numbmer of Set-/Reset registers
          mov  ah,ch              ;Move bit combination to AL
          out  dx,ax              ;Write to the register
                                                                        
          mov  ax,ENABLE_REG + (0Fh shl 8) ;Write 0F(h) in the
          out  dx,ax              ;Enable Set-/Reset register
                                                                        
          mov  bx,LINE_LEN / 2    ;Length of a graphic line / 2 into BX
          mul  bx                 ;Multiply by number of graphic lines
          mov  cx,ax              ;Move to CX as repeat counter
          xor  di,di              ;Address first byte in video RAM
          mov  ax,VIO_SEG         ;Segment address of video RAM
          mov  es,ax              ;Load into ES
          cld                     ;Increment on string instructions
          rep  stosw              ;Fill video RAM
                                                                        
          ;-- Return old contents of Enable Set-/Reset register    -----
                                                                        
          mov  dx,GRAPH_CONT      ;Load graphic controller port address
          mov  ax,ENABLE_REG      ;Write 00(h) in Enable Set-/
          out  dx,ax              ;Reset register
                                                                        
          ret                     ;Back to caller
                                                                        
fillscr   endp
                                                                        
;== End ================================================================
                                                                        
code      ends                    ;End of code segment
          end  demo               ;Start program execution with DEMO
                                                                        
