/**********************************************************************/
/*                      F I X P A R T C . C                           */
/*--------------------------------------------------------------------*/
/*    Task           : Displays hard disk partitioning                */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    Developed on   : 04/26/1989                                     */
/*    Last update    : 06/22/1989                                     */
/*--------------------------------------------------------------------*/
/*    Call           : FIXPARTC [ Drive_number ]                      */
/*                       Default is drive 0 (Drive C:)                */
/**********************************************************************/

#include <dos.h>
#include <string.h>
#include <stdlib.h>

/*== Constants =======================================================*/

#define TRUE  ( 1 == 1 )
#define FALSE ( 1== 0 )

/*== Macros ==========================================================*/

#define HI(x) ( *((BYTE *) (&x)+1) )   /* Returns high byte of a word */
#define LO(x) ( *((BYTE *) &x) )        /* Returns low byte of a word */

/*== Type declarations ===============================================*/

typedef unsigned char BYTE;
typedef unsigned int WORD;

typedef struct {                /* Describes the position of a sector */
                BYTE Head;                         /* Read/write head */
                WORD SecCyl;            /* Sector and cylinder number */
               } SECPOS;

typedef struct {                      /* Entry in the partition table */
                BYTE          Status;             /* Partition status */
                SECPOS        StartSec;               /* First sector */
                BYTE          PartTyp;              /* Partition type */
                SECPOS        EndSec;                  /* Last sector */
                unsigned long SecOfs;        /* Offset of boot sector */
                unsigned long SecNum;            /* Number of sectors */
               } PARTENTRY;

typedef struct {                    /* Describes the partition sector */
                BYTE      BootCode[ 0x1BE ];
                PARTENTRY PartTable[ 4 ];
                WORD      IdCode;                           /* 0xAA55 */
               } PARTSEC;

typedef PARTSEC far *PARSPTR; /* Pointer > partition sector in memory */

/**********************************************************************/
/*  ReadPartSec : Reads a partition sector from the hard disk into a  */
/*                buffer                                              */
/*  Input  : - HrdDrive : BIOS code of the drive  (0x80, 0x81 etc.)   */
/*           - Head     : Number of read/write heads                  */
/*           - SecCyl   : Sector and cyinder number in BIOS format    */
/*           - Buf      : Buffer into which sector should be loaded   */
/*  Output : TRUE if sector is read without error, otherwise FALSE    */
/**********************************************************************/

BYTE ReadPartSec( BYTE HrdDrive, BYTE Head, WORD SecCyl, PARSPTR Buf )

{
 union REGS   Regs;              /* Processor regs for interrupt call */
 struct SREGS SRegs;

 Regs.x.ax = 0x0201;                /* Funct.no. for "Read", 1 Sector */
 Regs.h.dl = HrdDrive;                        /* Load parameters into */
 Regs.h.dh = Head;                          /* different registers as */
 Regs.x.cx = SecCyl;                        /* needed                 */
 Regs.x.bx = FP_OFF( Buf );
 SRegs.es  = FP_SEG( Buf );

 int86x( 0x13, &Regs, &Regs, &SRegs );    /* Call hard disk interrupt */
 return !Regs.x.cflag;
}

/**********************************************************************/
/*  GetSecCyl: Determines the combined sector/cylinder coding from    */
/*             BIOS sector/cylinder number                            */
/*  Input  : SecCyl   : Value to be decoded                           */
/*           Sector   : Reference to the sector variable              */
/*           Cylinder : Reference to the cylinder variable            */
/*  Output : none                                                     */
/**********************************************************************/

void GetSecCyl( WORD SecCyl, int *Sector, int *Cylinder )

{
 *Sector   = SecCyl & 63;                     /* Exclude bits 6 and 7 */
 *Cylinder = HI(SecCyl) + ( ( (WORD) LO(SecCyl) & 192 ) << 2 );
}

/**********************************************************************/
/*  ShowPartition: Displays hard disk partitioning on the screen      */
/*  Input:   LW : Number of the hard disk (0, 1, 2, etc.)             */
/*  Output : none                                                     */
/**********************************************************************/

void ShowPartition( BYTE LW )
{
 #define AP ParSec.PartTable[ Entry ]

 BYTE       Head,                       /* Head for current partition */
            Entry,                                    /* Loop counter */
            SecCyl;       /* Sector and cylinder of current partition */
 PARTSEC    ParSec;                       /* Current partition sector */
 int        Sector,                   /* Get sector and cylinder      */
            Cylinder;                 /* number                       */
 union REGS Regs;                /* Processor regs for interrupt call */

 printf("\n");
 LW |= 0x80;                         /* Prepare drive number for BIOS */
 if ( ReadPartSec( LW, 0, 1, &ParSec ) )     /* Read partition sector */
  {                                             /* Sector can be read */
   Regs.h.ah = 8;                                   /* Read disk data */
   Regs.h.dl = LW;
   int86( 0x13, &Regs, &Regs );           /* Call hard disk interrupt */
   GetSecCyl( Regs.x.cx, &Sector, &Cylinder );
   printf( ""
           "ͻ\n");
                  /* Upper left corner can be typed using <Alt><201>  */
                  /* Horizontal line can be typed using <Alt><205>    */
                  /* Upper right corner can be typed using <Alt><187> */
   printf( " Drive    %2d:    %2d heads with        %4d"
           " cylinders,     %3d sectors    \n",
           LW-0x80, Regs.h.dh+1, Cylinder, Sector );
 /* Vertical lines at beginning and end can be typed using <Alt><186> */
   printf( " Partition table in partition sector       "
           "                              \n");
 /* Vertical lines at beginning and end can be typed using <Alt><186> */
  printf( ""
           "͹\n");
                /* Left T can be typed using <Alt><199>               */
                /* Horiz. lines can be typed using <Alt><205>         */
                /* Ts in middle of line can be typed using <Alt><209> */
                /* Right T can be typed using <Alt><185>              */
   printf( "                              Start    "
           "     End      Dis.fr.       \n");
            /* First and last vertical lines in the above line        */
            /* can be typed using <Alt><186>                          */
            /* Remaining vertical lines can be typed using <Alt><179> */
   printf( "#.BootType               Head Cyl. Sec."
           "Head Cyl. Sec.BootSecNumber \n");
            /* First and last vertical lines in the above line        */
            /* can be typed using <Alt><186>                          */
            /* Remaining vertical lines can be typed using <Alt><179> */
            printf( ""
           "͹\n");
                    /* Left T can be typed using <Alt><204>           */
                    /* Horizontal lines can be typed using <Alt><205> */
                    /* Crosses can be typed using <Alt><215>          */
                    /* Right T can be typed using <Alt><185>          */
   /*-- Check partition table ----------------------------------------*/
   for ( Entry=0; Entry < 4; ++Entry )
    {
     printf( " %d", Entry );
                /* First vertical line can be typed using <Alt><186>  */
                /* Second vertical line can be typed using <Alt><179> */
     if ( AP.Status == 0x80 )                    /* Partition active? */
      printf("Yes ");
     else
      printf ("No  ");
     printf("");
                       /* Vertical line can be typed using <Alt><179> */
     switch( AP.PartTyp )                  /* Display partition types */
      {
       case 0x00 : printf( "Not occupied       " );
                   break;
       case 0x01 : printf( "DOS, 12-Bit FAT    " );
                   break;
       case 0x02 : printf( "XENIX              " );
                   break;
       case 0x03 : printf( "XENIX              " );
                   break;
       case 0x04 : printf( "DOS, 16-Bit FAT    " );
                   break;
       case 0x05 : printf( "DOS, extended part." );
                   break;
       case 0xDB : printf( "Concurrent DOS     " );
                   break;
       default   : printf( "Unknown   (%3d)    ",
                           ParSec.PartTable[ Entry ].PartTyp );
      }

     /*-- Display physical and logical parameters --------------------*/
     GetSecCyl( AP.StartSec.SecCyl, &Sector, &Cylinder );
     printf( "%2d %5d  %3d ", AP.StartSec.Head, Cylinder, Sector );
/* Vertical line can be typed using <Alt><179> */
    GetSecCyl( AP.EndSec.SecCyl, &Sector, &Cylinder );
     printf( "%2d %5d  %3d ", AP.EndSec.Head, Cylinder, Sector );
/* Vertical line can be typed using <Alt><179> */
     printf( "%6lu %6lu \n", AP.SecOfs, AP.SecNum );
    }
     /* First and second vertical lines can be typed using <Alt><179> */
     /* Third vertical line can be typed using <Alt><186>             */
   printf( ""
           "ͼ\n" );
                    /* Left angle can be typed using <Alt><200>       */
                    /* Horizontal lines can be typed using <Alt><205> */
                    /* Ts can be typed using <Alt><207>               */
                    /* Right angle can be typed using <Alt><188>      */
  }
 else
  printf("Error during boot sector access!\n");
}

/***********************************************************************
*                          M A I N     P R O G R A M                   *
***********************************************************************/

int main( int argc, char *argv[] )
{
 int HrdDrive;

 printf( "\n-------------------------------- FIXPARTC - (c)"
         " 1989 by MICHAEL TISCHER ---\n" );
 HrdDrive = 0;                          /* Default is first hard disk */
 if ( argc == 2 )                           /* Other drive specified? */
  {                                                            /* YES */
   HrdDrive = atoi ( argv[1] );
   if ( HrdDrive == 0 && *argv[1] != '0' )
    {
     printf("\nIllegal drive specifier!");
     return( 1 );                                      /* End program */
    }
  }
 ShowPartition( HrdDrive );               /* Display partition sector */
 return( 0 );
}
