/*********************************************************************/
/*                            K E Y C                                */
/*-------------------------------------------------------------------*/
/*    Task           : provides a function for reading a             */
/*                     character from the keyboard and to output     */
/*                     the Status of the control keys (INSERT,       */
/*                     CAPS, NUM) on the display.                    */
/*-------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                               */
/*    developed on   : 8/13/87                                       */
/*    last update    : 6/09/89                                       */
/*-------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                  */
/*    Creation       : MSC TASTC;                                    */
/*                     LINK TASTC;                                   */
/*    Call           : TASTC                                         */
/*-------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                              */
/*    Creation      : Make sure that Case-sensitive link is OFF in   */
/*                    the Options menu/Linker option                 */
/*                    Select RUN menu                                */
/*********************************************************************/

#include <dos.h>                             /* include Header-Files */
#include <io.h>
#include <bios.h>

/*== Type definitions ===============================================*/

typedef unsigned char byte;                         /* Create a byte */

/*== Constants ======================================================*/

/*-- Bit layout in BIOS keyboard status -----------------------------*/

#define SCRL 16                                    /* ScrollLock bit */
#define NUML 32     				      /* NumLock bit */
#define CAPL 64 				     /* CapsLock bit */
#define INS 128 				       /* Insert bit */

#define FALSE 0                     /* Constants make reading of the */
#define TRUE 1                     /* Program text easier            */

#define FZ      0       /* Line in which the Flags should be output  */
#define FS      65           /* Column, in which Flags will be output */
#define FlagColour 0x70          /* black characters on white ground */

/*-- Codes of some keys as returned by GETKEY() ------------------*/
#define BEL     7                         /* Code of Bell character*/
#define BS      8                         /* Code of Backspace-key */
#define TAB     9                         /* Code of Tabulator-key */
#define LF      10                        /* Code of Linefeed      */
#define CR      13                        /* Code of Return-key    */
#define ESC     27                        /* Code of Escape-key    */
#define F1      315                       /* Code of F1-key        */
#define F2      316                       /* Code of F2-key        */
#define F3      317                       /* Code of F3-key        */
#define F4      318                       /* Code of F4-key        */
#define F5      319                       /* Code of F5-key        */
#define F6      320                       /* Code of F6-key        */
#define F7      321                       /* Code of F7-key        */
#define F8      322                       /* Code of F8-key        */
#define F9      323                       /* Code of F9-key        */
#define F10     324                       /* Code of F10-key       */
#define CUP     328                       /* Codes of Cursor-Up    */
#define CLEFT   331                       /* Codes of Cursor-Left  */
#define CRIGHT  333                       /* Codes of Cursor-Right */
#define CDOWN   328                       /* Codes of Cursor-Down  */

/*-- global Variables ---------------------------------------------*/

byte Insert,                              /* Status of INSERT-Flags */
     Num,                                    /* Status of NUM-Flags */
     Caps;                                  /* Status of CAPS-Flags */

/********************************************************************/
/* GETPAGE: get the current display page                            */
/* Input : none                                                     */
/* Output : see below                                               */
/********************************************************************/

byte GETPAGE()

{
 union REGS Register;       /* Register-Variable for Interrupt call */

 Register.h.ah = 15;                             /* function number */
 int86(0x10, &Register, &Register);         /* call Interrupt 10(h) */
 return(Register.h.bh);           /* Number of current display page */
}

/********************************************************************/
/* SETPOS: sets the Position of Cursor in current display page      */
/* Input : see below                                                */
/* Output : none                                                    */
/* Info    : the Position of the blinking cursor changes            */
/*           with the call of this function only if                 */
/*           display page indicated is the current display page     */
/********************************************************************/

void SetPos(byte Column, byte Line)

{
 union REGS Register;       /* Register-Variable for Interrupt call */

 Register.h.ah = 2;                              /* function number */
 Register.h.bh = GETPAGE();                         /* Display Page */
 Register.h.dh = Line;                             /* Display Line */
 Register.h.dl = Column;                          /* Display Column */
 int86(0x10, &Register, &Register);         /* call Interrupt 10(h) */
}

/********************************************************************/
/* GETPOS: Gets the Position of Cursor in the current Display Page  */
/* Input : none                                                     */
/* Output : see below                                               */
/********************************************************************/

void GetPos(byte * CurColumn, byte * CurLine)

{
 union REGS Register;      /* Register-Variable for Interrupt call */

 Register.h.ah = 3;                             /* function number */
 Register.h.bh = GETPAGE();                        /* Display page */
  int86(0x10, &Register, &Register);        /* call Interrupt 10(h) */
 *CurColumn = Register.h.dl;                /* Result of the function */
 *CurLine = Register.h.dh;                 /* read from the Register */
}

/*******************************************************************/
/* WRITECHAR: writes a character with an Attribute to              */
/*            the current Cursor position in current Display Page  */
/* Input : see below                                               */
/* Output : none                                                   */
/*******************************************************************/

void WriteChar(char Zcharacter, byte Colour)

{
 union REGS Register;       /* Register-Variable for Interrupt call */

 Register.h.ah = 9;                              /* function number */
 Register.h.bh = GETPAGE();                         /* Display Page */
 Register.h.al = Zcharacter;            /* the character for output */
 Register.h.bl = Colour;         /* Color of character to be output */
 Register.x.cx = 1;                  /* output character only once  */
 int86(0x10, &Register, &Register);         /* call Interrupt 10(h) */
}

/********************************************************************/
/* WRITETEXT: write a character chain with constant color           */
/*            starting at a certain location in the current         */
/*            Display Page                                          */
/* Input : see below                                                */
/* Output : none                                                    */
/* Info    : Text is a Pointer to a character-Vector which          */
/*           contains the Text to be output and is terminated with  */
/*            a '\0' character.                                     */
/********************************************************************/

void WriteText(byte Column, byte Line, char *Text, byte Colour)

{
 union REGS InRegister,
            OutRegister;      /* Register-Variable for Interrupt call */

 SetPos(Column, Line);                              /* set Cursor */
 InRegister.h.ah = 14;                            /* function number */
 InRegister.h.bh = GetPage( );                       /* Display Page */
 while (*Text)                 /* output Text until '\0' character */
  {
   WriteChar(' ', Colour);        /* Indicate color for character  */
   InRegister.h.al = *Text++;            /* the character for output */
   int86(0x10, &InRegister, &OutRegister);            /* call Interrupt */
  }
}

/********************************************************************/
/* CLS: erase current Display Page                                  */
/* Input : none                                                     */
/* Output : none                                                    */
/********************************************************************/

void Cls()

{
 union REGS Register;       /* Register-Variable for Interrupt call */

 Register.h.ah = 6;                /* function number for Scroll-UP */
 Register.h.al = 0;                            /* 0 stand for clear */
 Register.h.bh = 7;            /* white letters on black background */
 Register.x.cx = 0;                   /* upper left display corner  */
 Register.h.dh = 24;                   /* Coordinates of the lower  */
 Register.h.dl = 79;                        /* right display corner */
 int86(0x10, &Register, &Register);    /* call BIOS-Video-Interrupt */
}

/******************************************************************/
/* NEGFLAG: negate Flag and output Text                           */
/* Input : see below                                              */
/* Output : the new Status of Flags (TRUE = on, FALSE = off)      */
/******************************************************************/

byte NegFlag(byte Flag, unsigned int FlagReg,
             byte Column, byte Line, char * Text)


{
byte CurLine,                                     /* current Line */
     CurColumn,                                  /* current Column */
     Colour;                            /* for Output of Flag-Text */

 if (!(Flag == (FlagReg != 0)))                 /* did Flag change? */
  {                                                          /* YES */
   GetPos(&CurColumn, &CurLine);    /* get current Cursor position */
   WriteText(Column, Line, Text, (Flag) ? 0 : FlagColour);
   SetPos(CurColumn, CurLine);         /* set old Cursor position */
   return(Flag ^1);                     /* reverse Bit 1 of Flags  */
  }
 else return(Flag);                 /* everything remains the same */
}

/*******************************************************************/
/* KEYREADY: Tests for a character from the keyboard               */
/* Input:  none                                                    */
/* Output: TRUE if a key is pressed, otherwise FALSE               */
/*******************************************************************/

int KeyReady()

{
#ifdef __TURBOC__

  struct REGPACK Register;

  Register.r_ax = 1 << 8;
  intr(0x16, &Register);
  return(! (Register.r_flags & 64) );

#else

  return( _bios_keybrd( _KEYBRD_READY ) );

#endif
}

/*******************************************************************/
/* GETKEY: Read a character and Output Flag-Status                 */
/* Input : none                                                    */
/* Output : Code of key read  < 256 : normal key                   */
/*                                       >= 256 : extended ke y    */
/********************************************************************/

unsigned int GetKey()

{
 union REGS Register;       /* Register  Variable for Interrupt call */

 do
{
 Register.h.ah = 2;    /* read function number for keyboard status */
 int86(0x16, &Register, &Register); /* call BIOS keyboard interrupt*/
 Insert = NegFlag(Insert, Register.h.al & INS, FS+9, FZ, "INSERT");
 Caps = NegFlag(Caps, Register.h.al & CAPL, FS+3, FZ, " CAPS ");
 Num = NegFlag(Num, Register.h.al & NUML, FS, FZ, "NUM");
}
 while ( !KeyReady() );
 Register.h.ah = 0;              /* read function number for key */
 int86(0x16, &Register, &Register);/*call BIOS-keyboard-Interrupt*/
 return((Register.h.al) ? Register.h.al : Register.h.ah | 256);
}

/*******************************************************************/
/* INIKEY: initialize keyboard-Flags                               */
/* Input : none                                                    */
/* Output : none                                                   */
/* Info    : the keyboard-Flags are reversed compared with the     */
/*           current status. This makes it possible that their     */
/*           current Status is output on the next call of the      */
/*           GETKEY-function.                                      */
/*******************************************************************/

void Inikey()

{
 union REGS Register;       /* Register-Variable for Interrupt call */

 Register.h.ah = 2;      /* read function number for keyboard status */
 int86(0x16, &Register, &Register);   /* call BIOS-keyboard-Interrupt*/
 Insert = (Register.h.al & INS) ? FALSE : TRUE ;  /* reverse the     */
 Caps = (Register.h.al & CAPL) ? FALSE : TRUE ;     /* current content */
 Num = (Register.h.al & NUML) ? FALSE : TRUE ;
}

/*********************************************************************/
/**                           MAIN PROGRAM                          **/
/*********************************************************************/

void main()

{
 unsigned int key;

 Cls();                                          /* Clear Screen */
 SetPos(0,0);             /* Cursor to left upper display corner */
 printf("KEY (c) 1987 by Michael Tischer\n\n");
 printf("You can input some characters and at the same time change ");
 printf("INSERT-, CAPS-\nor NUM-status. Every change ");
 printf("is displayed in the upper right corner of the screen.\n");
 printf("\n<RETURN> or <F1> terminates the Input...\n\n");
 printf("Your Input: ");
 Inikey();                           /* initialize keyboard-Flags */
 do
 {
   if ((key = Getkey()) < 256)                     /* read key */
    printf("%c", (char) key);            /* output (if normal) */
  }
 while (!(key == CR || key == F1)); /* repeat until F1 or CR */
 printf("\n");
}
