/**********************************************************************/
/*                               T S R C                              */
/*--------------------------------------------------------------------*/
/*    Description    : C module which is turned into a TSR program    */
/*                     with the help of an assembly language routine. */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    developed on   : 08/15/1988                                     */
/*    last update    : 08/19/1988                                     */
/*--------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                   */
/*    creation       : CL /AS /c TSRC.C                               */
/*                     LINK TSRC TSRCA;                               */
/*    call           : TSRC                                           */
/*--------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                               */
/*    creation       : Create project file with the following         */
/*                     contents:                                      */
/*                     TSRC                                           */
/*                     TSRCA.OBJ                                      */
/*                     Before compiling, set Options menu / linker    */
/*                     option / Case sensitive link to OFF            */
/**********************************************************************/

/*== Include files ===================================================*/

#include <stdlib.h>
#include <dos.h>

/*== Typedefs ========================================================*/

typedef unsigned char BYTE;                 /* build ourselves a byte */
typedef unsigned int WORD;
typedef BYTE BOOL;                          /* like BOOLEAN in Pascal */
typedef union vel far * VP;      /* VP is a FAR pointer into the VRAM */

/*== Macros ==========================================================*/

#ifndef MK_FP                           /* was MK_FP already defined? */
#define MK_FP(seg, ofs) ((void far *) ((unsigned long) (seg)<<16|(ofs)))
#endif
#define VOFS(x,y) ( 80 * ( y ) + ( x ) )
#define VPOS(x,y) (VP) ( vptr + VOFS( x, y ) )

/*== Structures and unions ===========================================*/

struct velb {             /* describes a screen position as two bytes */
             BYTE character,                        /* the ASCII code */
                  attribute;               /* corresponding attribute */
            };

struct velw {               /* describes a screen position as one word */
             WORD contents;    /* stores ASCII character and attribute */
            };

union vel {                             /* describes a screen position */
           struct velb h;
           struct velw x;
          };

/*== Link the functions from the assembly module ======================*/

extern int is_inst( char * id_string );
extern void uninst( void (*fkt)(void) );
extern int tsr_init(BOOL TC, void (*fkt)(void), unsigned hotkey,
                    unsigned heap, char * id_string);

/*== Constants ========================================================*/

#ifdef __TURBOC__                    /* are we compiling with TURBO-C? */
  #define TC TRUE                                               /* yes */
#else                                      /* we are using Microsoft C */
  #define TC FALSE
#endif

/*-- codes of the individual control keys for building the hotkey mask */

#define RSHIFT       1                      /* right SHIFT key pressed */
#define LSHIFT       2                       /* left SHIFT key pressed */
#define CTRL         4                             /* CTRL key pressed */
#define ALT          8                              /* ALT key pressed */
#define SCRL_AN     16                               /* Scroll Lock ON */
#define NUML_AN     32                                  /* Num Lock ON */
#define CAPL_AN     64                                 /* Caps Lock ON */
#define INS_AN     128                                    /* Insert ON */
#define SCR_LOCK  4096                          /* Scroll Lock pressed */
#define NUM_LOCK  8192                             /* Num Lock pressed */
#define CAP_LOCK 16384                            /* Caps Lock pressed */
#define INSERT   32768                           /* INSERT key pressed */
#define NOF      0x07                                  /* normal color */
#define INV      0x70                                 /* inverse color */
#define HNOF     0x0f                           /* bright normal color */
#define HINV     0xf0                          /* bright inverse color */

#define HEAP_FREE 1024                   /* leave 1K space on the heap */

#define TRUE  1                     /* constants for working with BOOL */
#define FALSE 0

#define NO_END_FTN ((void (*)(void)) -1)     /* don't call an end ftn. */

/*== Global variables =================================================*/

char id_string[] = "MiTi";                    /* identification string */
VP vptr;                /* pointer to the first character in video RAM */
unsigned atimes = 0;       /* number of activations of the TSR program */
union vel * scrbuf;      /* pointer to the buffer with screen contents */
char * blank_line;                          /* pointer to a blank line */

/***********************************************************************
*  Function         : D I S P _ I N I T                                *
**--------------------------------------------------------------------**
*  Description      : Determines the base address of the video RAM.    *
*  Input parameters : none                                             *
*  Return value     : none                                             *
***********************************************************************/

void disp_init(void)
 {
  union REGS regs;            /* processor regs for the interrupt call */

  regs.h.ah = 15;             /* function number: determing video mode */
  int86(0x10, &regs, &regs);          /* call the BIOS video interrupt */

   /* calculate base addr of the video RAM according to the video mode */

  vptr = (VP) MK_FP((regs.h.al == 7) ? 0xb000 : 0xb800, 0);
 }

/***********************************************************************
*  Function         : D I S P _ P R I N T                              *
**--------------------------------------------------------------------**
*  Description      : Output a string to the screen.                   *
*  Input parameters : - COLUMN   = the output column                   *
*                     - LINE     = the output line                     *
*                     - COLOR    = attribute for the characters        *
*                     - STRING   = pointer to the string               *
*  Return value     : none                                             *
***********************************************************************/

void disp_print(BYTE column, BYTE line, BYTE
color, char * string)
 {
  register VP lptr;    /* running pointer for accessing the video RAM */

  lptr = VPOS(column, line);         /* set pointer to the video RAM */
  for ( ; *string ; ++lptr)                  /* run through the string */
   {
    lptr->h.character = *(string++);  /* write char into the video RAM */
    lptr->h.attribute = color;      /* set attribute for the character */
   }
 }

/***********************************************************************
*  Function         : S A V E _ S C R E N                              *
**--------------------------------------------------------------------**
*  Description      : Saves the screen contents in a buffer.           *
*  Input parameters : - SPTR     = pointer to the buffer in which the  *
*                                  screeb will be saved.               *
*  Return value     : none                                             *
*  Info             : It is assumed that the buffer is large enough to *
*                     hold the screen contents.                        *
***********************************************************************/

void save_screen( union vel * sptr )
 {
  register VP lptr;    /* running pointer for accessing the video RAM */
  unsigned i;                                         /* loop counter */
  lptr = VPOS(0, 0);                  /* set pointer in the video RAM */

  for (i=0; i<2000; i++)     /* run through the 2000 screen positions */
   (sptr++)->x.contents = (lptr++)->x.contents; /* save character & attr. */
 }

/***********************************************************************
*  Function         : R E S T O R E _ S C R E E N                      *
**--------------------------------------------------------------------**
*  Description      : Copies the contents of a buffer into the video   *
*                     RAM.                                             *
*  Input parameters : - SPTR     = pointer to the buffer in which the  *
*                                  screen contents are located         *
*  Return value     : none                                             *
***********************************************************************/

void restore_screen( union vel * sptr )
 {
  register VP lptr;            /* pointer for accessing the video RAM */
  unsigned i;                                         /* loop counter */
  lptr = VPOS(0, 0);                  /* set pointer to the video RAM */

  for (i=0; i<2000; i++)     /* run through the 2000 screen positions */
   (lptr++)->x.contents = (sptr++)->x.contents;  /* restore char. & attr. */
 }

/***********************************************************************
*  Function         : E N D F T N                                      *
**--------------------------------------------------------------------**
*  Description      : Called when the TSR program is reinstalled.      *
*  Input parameters : none                                             *
*  Return value     : none                                             *
***********************************************************************/

void endftn( void )
{
 /*-- release the allocated buffers ----------------------------------*/

 free( blank_line );                   /* release the allocated buffer */
 free( (void *) scrbuf );                       /* release the buffer */

 printf("The TSR program was activated %u times.\n", atimes);
}

/***********************************************************************
*  Function         : T S R                                            *
**--------------------------------------------------------------------**
*  Description      : Called by the assembler routine when the hotkey  *
*                     is pressed.                                      *
*  Input parameters : none                                             *
*  Return value     : none                                             *
***********************************************************************/

void tsr( void )
{
 BYTE i;                                              /* loop counter */

 ++atimes;                     /* increment the number of activations */
 disp_init();                   /* determine address of the video RAM */
 save_screen( scrbuf );           /* save the current screen contents */
 for (i=0; i<25; i++)              /* run through the 25 screen lines */
  disp_print(0, i, INV, blank_line);                 /* clear the line */
 disp_print(22, 11, INV, "TSRC  -  (c) 1988 by MICHAEL TISCHER");
 disp_print(28, 13, INV, "Please press a key ...");
 getch();                                           /* wait for a key */
 restore_screen( scrbuf );                /* copy the old screen back */
}

/**********************************************************************/
/**                           MAIN PROGRAM                           **/
/**********************************************************************/

void main()
{
 printf("TSRC  -  (c) 1988 by MICHAEL TISCHER\n\n");
 if ( is_inst( id_string ) )     /* is the program already installed? */
  { /* yes */
   printf("TSRC was already installed--now disabling.\n");
   uninst( endftn );              /* reinstall prg., call ftn. ENDFKT */

   /*-- if no end function is to be called, the call is: -------------*/
   /*-- uninst( NO_END_FTN );                            -------------*/
  }
 else                   /* no, the program has not been installed yet */
  {
   /*-- with MSC the heap buffers must be allocated now --------------*/

   scrbuf = (union vel *) malloc(80 * 25 * sizeof(union vel));
   blank_line = (char *) sbrk( 80 + 1 );            /* allocate buffer */
   *(blank_line + 80 ) = '\0';            /* terminate buffer with NUL */
   memset(blank_line, ' ', 80);         /* fill the buffer with spaces */

   printf("TSRC now enabled - Start: <LSHIFT> + <RSHIFT>\n");
   tsr_init(TC, tsr, RSHIFT | LSHIFT, HEAP_FREE, id_string);
  }
}


