{**********************************************************************}
{*                              E M M P                               *}
{*--------------------------------------------------------------------*}
{*    Task           : Implement certain functions to demonstrat      *}
{*                     access to EMS memory using EMM.                *}
{*--------------------------------------------------------------------*}
{*    Author         : MICHAEL TISCHER                                *}
{*    Developed on   : 08/30/1988                                     *}
{*    Last update    : 06/21/1989                                     *}
{**********************************************************************}

program EMMP;

Uses Dos, CRT;                                 { Add DOS and CRT units }

type  ByteBuf = array[0..1000] of byte;    { One memory range as bytes }
      CharBuf = array[0..1000] of char;    { One memory range as chars }
      BytePtr = ^ByteBuf;                    { Pointer to a byte range }
      CharPtr = ^CharBuf;                    { Pointer to a char range }

const EMS_INT   = $67;                 { Interrupt # for access to EMM }
      EMS_ERR   = -1;                           { Error if this occurs }
      W_EMS_ERR = $FFFF;                     { Error code in WORD form }
      EmmName   : array[0..7] of char = 'EMMXXXX0';      { Name of EMM }

var   EmmEC,                           { Allocation of EMM error codes }
      i        : byte;                                  { Loop counter }
      Handle,                        { Handle for access to EMS memory }
      EmmVer   : integer;                      { Version number of EMM }
      NumPage,                                   { Number of EMS pages }
      PageSeg  : word;                 { Segment address of page frame }
      Keypress    : char;

{**********************************************************************}
{* MK_FP: Creates a byte pointer from the given segment and offset    *}
{*        addresses.                                                  *}
{* Input   : - Seg = Segment to which the pointer should point        *}
{*           - Ofs = Offset addr. to which the pointer should point   *}
{* Output  : Entire pointer                                           *}
{* Info    : The returned pointer can be recast toward any other      *}
{*           pointer.                                                 *}
{**********************************************************************}

{$F+}             { This routine is intended for a FAR model, and      }
                  { should therefore be treated as one UNIT            }

function MK_FP( Seg, Ofs : word ) : BytePtr;

begin
  inline ( $8B / $46 / $08 /  { mov ax,[bp+8] (Get segment address)    }
           $89 / $46 / $FE /  { mov [bp-2],ax (and place in pointer)   }
           $8B / $46 / $06 /  { mov ax,[bp+6] (Get offset address)     }
           $89 / $46 / $FC ); { mov [bp-4],ax (and place in pointer)   }
end;

{$F-}                                        { Re-enable NEAR routines }

{**********************************************************************}
{* EmsInst: Determines the existence of EMS and corresponding EMM     *}
{* Input   : none                                                     *}
{* Output  : TRUE if EMS is available, otherwise FALSE                *}
{**********************************************************************}

function EmsInst : boolean;

var Regs : Registers;      { Processor register for the interrupt call }
    Name : CharPtr;                         { Pointer to the EMM names }
    i    : integer;                                     { Loop counter }

begin
  {*-- Move pointer to name in device driver header ------------------*}

  Regs.ax := $3567;              { Function #: Get interrupt vector $67}
  MsDos( Regs );                              { Call DOS interrupt $21 }
  Name := CharPtr(MK_FP(Regs.es, 10));                  { Move pointer }

  {*-------- Search for EMS driver ---*}
  i := 0;                      { Start comparison with first character }
  while ((i<sizeof(EmmName)) and (Name^[i]=EmmName[i])) do
    Inc( i );                                 { Increment loop counter }
  EmsInst := (i = sizeof(EmmName));            { TRUE if name is found }
end;

{**********************************************************************}
{* EmsNumPage: Determines the total number of EMS pages               *}
{* Input   : none                                                     *}
{* Output  : EMS_ERR if error occurs, otherwise  number of EMS pages  *}
{**********************************************************************}

function EmsNumPage : integer;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $42;              { Function #: Determine number of pages }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  if (Regs.ah <>0 ) then                             { Error occurred? }
    begin                                                        { YES }
      EmmEC := Regs.ah;                               { Get error code }
      EmsNumPage := EMS_ERR;                           { Display error }
    end
  else                                                      { No error }
    EmsNumPage := Regs.dx;              { Return total number of pages }
end;

{**********************************************************************}
{* EmsFreePage: Determines the number of free EMS pages               *}
{* Input   : none                                                     *}
{* Output  : EMS_ERR if error occurs, otherwise the number of un-     *}
{*           used EMS pages                                           *}
{**********************************************************************}

function EmsFreePage : integer;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $42;                 { Function #: Determine no. of pages }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  if (Regs.ah <>0 ) then                             { Error occurred? }
    begin                                                        { YES }
      EmmEC := Regs.ah;                              { Mark error code }
      EmsFreePage := EMS_ERR;                          { Display error }
    end
  else                                                      { No error }
    EmsFreePage := Regs.bx;              { Return number of free pages }
end;

{**********************************************************************}
{* EmsFrameSeg: Determines the segment address of the page frame      *}
{* Input   : none                                                     *}
{* Output  : EMS_ERR if error occurs, otherwise the segment address   *}
{**********************************************************************}

function EmsFrameSeg : word;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $41;           { Function #: Get segment addr. page frame }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  if (Regs.ah <>0 ) then                             { Error occurred? }
    begin                                                        { YES }
      EmmEC := Regs.ah;                              { Mark error code }
      EmsFrameSeg := W_EMS_ERR;                        { Display error }
    end
  else                                                      { No error }
    EmsFrameSeg := Regs.bx;       { Return segment addr. of page frame }
end;

{**********************************************************************}
{* EmsAlloc: Allocates the specified number of pages and returns a    *}
{*           handle for access to these pages                         *}
{* Input   : PAGES: Number of allocated pages                         *}
{* Output  : EMS_ERR returns error, otherwise the handle              *}
{**********************************************************************}

function EmsAlloc( Pages : integer ) : integer;

var Regs : Registers;       { Processor register for the interrupt call}

begin
  Regs.ah := $43;                          { Function #: Alocate pages }
  Regs.bx := Pages;                    { Set number of allocated pages }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  if (Regs.ah <>0 ) then                             { Error occurred? }
    begin                                                        { YES }
      EmmEC := Regs.ah;                              { Mark error code }
      EmsAlloc := EMS_ERR;                             { Display error }
    end
  else                                                      { No error }
    EmsAlloc := Regs.dx;                               { Return handle }
end;

{**********************************************************************}
{* EmsMap  : Creates an allocated logical page from a physical page in*}
{*           the page frame                                           *}
{* Input   : HANDLE: Handle received from EmsAlloc                    *}
{*           LOGP  : Logical page about to be created                 *}
{*           PHYSP : The physical page in page frame                  *}
{* Output  : FALSE if error, otherwise TRUE                           *}
{**********************************************************************}

function EmsMap(Handle, LogP : integer; PhysP : byte) : boolean;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $44;                            { Function #: Set mapping }
  Regs.al := PhysP;                                { Set physical page }
  Regs.bx := LogP;                                  { Set logical page }
  Regs.dx := Handle;                                  { Set EMS handle }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  EmmEC := Regs.ah;                                  { Mark error code }
  EmsMap := (Regs.ah = 0)               { TRUE is returned if no error }
end;

{**********************************************************************}
{* EmsFree : Frees memory when given with an allocated handle         *}
{* Input   : HANDLE: Handle received by AllocEms                      *}
{* Output  : FALSE if an error, otherwise TRUE                        *}
{**********************************************************************}

function EmsFree(Handle : integer) : boolean;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $45;                           { Function #: Release page }
  Regs.dx := handle;                                  { Set EMS handle }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  EmmEC := Regs.ah;                                  { Mark error code }
  EmsFree := (Regs.ah = 0)              { TRUE is returned if no error }
end;

{**********************************************************************}
{* EmsVersion: Determines the version number of EMM                   *}
{* Input   : none                                                     *}
{* Output  : EMS_ERR if error occurs, otherwise the version number    *}
{*           (11=1.1, 40=4.0, etc.)                                   *}
{**********************************************************************}

function EmsVersion : integer;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $46;                  { Function #: Determine EMM version }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  if (Regs.ah <>0 ) then                             { Error occurred? }
    begin                                                        { YES }
      EmmEC := Regs.ah;                              { Mark error code }
      EmsVersion := EMS_ERR;                           { Display error }
    end
  else              { No error, compute version number from BCD number }
   EmsVersion := (Regs.al and 15) + (Regs.al shr 4) * 10;
end;

{**********************************************************************}
{* EmsSaveMap: Saves dispay between logical and physical pages of the *}
{*             given handle                                           *}
{* Input   : HANDLE: Handle assigned by EmsAlloc                      *}
{* Output  : FALSE if error occurs, otherwise TRUE                    *}
{**********************************************************************}

function EmsSaveMap( Handle : integer ) : boolean;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $47;                               { Function #: Map save }
  Regs.dx := handle;                                  { Set EMS handle }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  EmmEC := Regs.ah;                                  { Mark error code }
  EmsSaveMap := (Regs.ah = 0)                { Return TRUE if no error }
end;

{**********************************************************************}
{* EmsRestoreMap: Returns display between logical and physical pages, *}
{*                from the page saved by EmsSaveMap                   *}
{* Input   : HANDLE: Handle assigned by EmsAlloc                      *}
{* Output  : FALSE if an error occurs, otherwise TRUE                 *}
{**********************************************************************}

function EmsRestoreMap( Handle : integer ) : boolean;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $48;                            { Function #: Restore map }
  Regs.dx := handle;                                  { Set EMS handle }
  Intr(EMS_INT, Regs);                                      { Call EMM }
  EmmEC := Regs.ah;                                  { Mark error code }
  EmsRestoreMap := (Regs.ah = 0)           { TRUE returned if no error }
end;

{**********************************************************************}
{* PrintErr: Displays an error message and ends the program           *}
{* Input   : none                                                     *}
{* Output  : none                                                     *}
{* Info    : This function is called only if an error occurs during a *}
{*           function call within this module                         *}
{**********************************************************************}

procedure PrintErr;

begin
  writeln('ATTENTION! Error during EMS memory access');
  write('     ... ');
  if ((EmmEC<$80) or (EmmEc>$8E) or (EmmEc=$82)) then
    writeln('Unidentifiable error')
  else
    case EmmEC of
      $80 : writeln('EMS driver error (EMM trouble)');
      $81 : writeln('EMS hardware error');
      $83 : writeln('Illegal EMM handle');
      $84 : writeln('Called EMS function does not exist');
      $85 : writeln('No more free EMS handles available');
      $86 : writeln('Error while saving or restoring mapping ');
      $87 : writeln('More pages requested than are actually ',
                    'available');
      $88 : writeln('More pages requested than are free');
      $89 : writeln('No pages requested');
      $8A : writeln('Logical page does not belong to handle');
      $8B : writeln('Illegal physical page number');
      $8C : writeln('Mapping memory range is full');
      $8D : writeln('Map save has already been done');
      $8E : writeln('Mapping must be saved before it can',
                    'be restored');
    end;
  Halt;                                                  { Program end }
end;

{**********************************************************************}
{* VrAdr: Returns a pointer to video RAM                              *}
{* Input   : none                                                     *}
{* Output  : Pointer to video RAM                                     *}
{**********************************************************************}

function VrAdr : BytePtr;

var Regs : Registers;      { Processor register for the interrupt call }

begin
  Regs.ah := $0f;                   { Function #: Determine video mode }
  Intr($10, Regs);                         { Call BIOS video interrupt }
  if (Regs.al = 7) then                       { Monochrome video card? }
    VrAdr := MK_FP($B000, 0)             { YES, video RAM at B000:0000 }
  else                                        { Color, EGA or VGA card }
    VrAdr := MK_FP($B800, 0);                 { Video RAM at B800:0000 }
end;

{**********************************************************************}
{* PageAdr: Returns address of a physical page in page frame          *}
{* Input   : PAGE: Physical page number (0-3)                         *}
{* Output  : Pointer to the physical page                             *}
{**********************************************************************}

function PageAdr( Page : integer ) : BytePtr;

begin
  PageAdr := MK_FP( EmsFrameSeg + (Page shl 10), 0 );
end;

{**********************************************************************}
{**                           MAIN  PROGRAM                          **}
{**********************************************************************}

begin
  ClrScr;                                               { Clear screen }
  writeln('EMMP  -  (c) 1988 by MICHAEL TISCHER',#13#10);
  if EmsInst then                           { Is EMS memory installed? }
    begin                                                        { YES }
                    {*-- Display EMS memory information --------------*}

      EmmVer := EmsVersion;             { Determine EMM version number }
      if EmmVer = EMS_ERR then                       { Error occurred? }
        PrintErr;         { YES, Display error message and end program }
      writeln('EMM Version number           : ',EmmVer div 10, '.',
              EmmVer mod 10);

      NumPage := EmsNumPage;         { Determine total number of pages }
      if NumPage = EMS_ERR then                      { Error occurred? }
        PrintErr;         { YES, Display error message and end program }
      writeln('Number of EMS Pages          : ', NumPage, ' (',
               NumPage shl 4, ' KByte)');

      NumPage := EmsFreePage;         { Determine number of free pages }
      if NumPage = EMS_ERR then                      { Error occurred? }
        PrintErr;         { YES, Display error message and end program }
      writeln('... free EMS pages remaining : ', NumPage, ' (',
               NumPage shl 4, ' KByte)');

      PageSeg := EmsFrameSeg;          { Segment address of page frame }
      if PageSeg = W_EMS_ERR then                    { Error occurred? }
        PrintErr;         { YES, Display error message and end program }
      writeln('Segment address of page frame: ', PAgeSeg);

      writeln;
      writeln('Now a page from EMS memory can be allocated,   and the');
      writeln('screen contents can be copied from video RAM into this');
      writeln('page.');
      writeln('                                ... Please press a key');
      Keypress := ReadKey;                       { Wait for a keypress }

      {*-- Page is allocated, and the data is passed to the first-----*}
      {*-- logical page in the page frame                        -----*}

      Handle := EmsAlloc( 1 );                     { Allocate one page }
      if Handle = EMS_ERR then                       { Error occurred? }
        PrintErr;         { YES, Display error message and end program }
      if not(EmsMap(Handle, 0, 0)) then                  { Set mapping }
        PrintErr;       { Error: Display error message and end program }

                {*-- Copy 4000 bytes from video RAM into EMS memory --*}

      Move(VrAdr^, PageAdr(0)^, 4000);

      ClrScr;                                           { Clear screen }
      while KeyPressed do                       { Read keyboard buffer }
        Keypress := ReadKey;
      writeln('Old screen contents are cleared.  However, the  data ');
      writeln('from the screen is in EMS, and can be re-copied onto ');
      writeln('the screen.                                          ');
      writeln('                               ... Please press a key');
      Keypress := ReadKey;                       { Wait for a keypress }

      {*-- Copy contents of video RAM from EMS memory and release   --*}
      {*-- the allocated EMS memory                                 --*}

      Move(PageAdr(0)^, VrAdr^, 4000);           { Copy over video RAM }
      if not(EmsFree(Handle)) then                    { Release memory }
        PrintErr;       { Error: Display error message and end program }
      GotoXY(1, 15);
      writeln('END')
    end
 else                                       { EMS driver not available }
   writeln('ATTENTION! No EMS memory installed.');
end.
