/* MCC.C -- User interface for Mystic v1.0 screen savers using INT 69h API  */

#include <dos.h>                         /* for int86 (), int86x () & MK_FP */
#include <stdio.h>                       /* for printf () */
#include <string.h>                      /* for stricmp () */
#include <stdlib.h>                      /* for atoi () */

#define MC_API_INT   0x69
#define MC_GET_INFO  0x01
#define MC_SET_DELAY 0x02
#define MC_SET_MODES 0x03
#define MC_SUSPEND   0x05
#define MC_ENABLE    0x06

#define TRUE         0xff
#define FALSE        0x00

char *arg_list[] = {                     /* command line options */
                    "info",
                    "setdelay",
                    "suspend",
                    "activate",
                   };

/*--------------------------------------------------------------------------*/
unsigned char Is_Resident (void) {       /* returns TRUE if resident */
    union REGS regs;

    regs.x.ax = MC_GET_INFO;             /* get screen saver information */
    int86 (MC_API_INT, &regs, &regs);    /* use API gateway */

    return (regs.h.ah == 0xab) ? TRUE:FALSE;  /* returns AH = 0xab if resident */
}

/*--------------------------------------------------------------------------*/
void Print_MC_Info (void) {              /* prints screen saver info */
    union REGS regs;
    struct SREGS segregs;

    char far *ss_name;                   /* screen saver name */
    char far *ss_author;                 /* author's name */

    unsigned char ss_status;             /* enabled/disabled status */
    unsigned ss_delay, i;                /* screen saver delay */
    unsigned long ss_modes;              /* video modes supported */

    regs.x.ax = MC_GET_INFO;             /* get screen saver information */
    int86x (MC_API_INT, &regs, &regs, &segregs);

    ss_name = (char far *) MK_FP (segregs.es, regs.x.di);   /* get name */
    ss_author = (char far *) MK_FP (segregs.es, regs.x.si); /* get author */
    ss_delay = regs.x.cx / 18;              /* get time delay & convert */
    ss_status = regs.h.al;                  /* enabled/disabled status */
    ss_modes = (regs.x.dx << 16) | (regs.x.bx);  /* valid video modes */

    printf ("MCC : Information about active screen saver.\n\n");
    printf ("Name    : %Fs\n", ss_name);
    printf ("Author  : %Fs\n", ss_author);
    printf ("Delay   : %u seconds\n", ss_delay);
    printf ("Status  : %s\n", (ss_status == 0) ? "Disabled":"Enabled");
    printf ("Modes   : ");

    for (i = 0; i < 32; ++i) {
        if ((1 << i) & (ss_modes)) {
            printf ("%u ", i);
        }
    }
    printf ("\n");
}

/*--------------------------------------------------------------------------*/
void Set_MC_Delay (unsigned new_delay) { /* sets screen saver delay */
    union REGS regs;

    regs.x.ax = MC_SET_DELAY;            /* load fn-id in AX */
    regs.x.cx = new_delay * 18;          /* delay in * TIMER TICKS * */
    int86 (MC_API_INT, &regs, &regs);    /* set the new delay */
}

/*--------------------------------------------------------------------------*/
void Suspend (void) {
    union REGS regs;

    regs.x.ax = MC_SUSPEND;              /* fn-id in AX */
    int86 (MC_API_INT, &regs, &regs);    /* suspend screen saver */
}

/*--------------------------------------------------------------------------*/
void Activate (void) {
    union REGS regs;

    regs.x.ax = MC_ENABLE;               /* fn-id in AX */
    int86 (MC_API_INT, &regs, &regs);    /* re-activate screen saver */
}

/*--------------------------------------------------------------------------*/
unsigned Get_Arg_Id (char *str) {        /* returns arg no. */
    unsigned arg_id = -1;
    unsigned result = 0;

    do {
        ++arg_id;
        result = stricmp (arg_list[arg_id], str);
    } while ((result != 0) && (arg_id < 4));

    return (result == 0) ? arg_id:-1;
}

/*--------------------------------------------------------------------------*/
void main (int argc, char *argv[]) {
    unsigned arg_id;
    unsigned delay;

    if (~Is_Resident ()) {
        printf ("MCC : Mystic v1.0 not resident in memory.\n");
    }
    else {
        if (argc < 2) {
            printf ("MCC enables you to control/modify screen saver settings.\n\n");
            printf ("MCC {INFO | SETDELAY nnnn | SUSPEND | ACTIVATE}\n\n");
            printf ("Summary of command line options -- \n\n");
            printf ("  info          -- Displays screen saver information.\n");
            printf ("  setdelay nnnn -- Sets screen saver delay to nnnn seconds. (nnnn <= 3640)\n");
            printf ("  suspend       -- Suspends the screen saver.\n");
            printf ("  activate      -- Re-activates the screen saver.\n");
        }
        else {
            arg_id = Get_Arg_Id (argv[1]);

            switch (arg_id) {
                case 0: Print_MC_Info (); break;

                case 1: if (argc == 2) {
                            printf ("MCC : Delay value not known.\n");
                        }
                        else {
                            delay = atoi (argv[2]);
                            Set_MC_Delay (delay);
                            printf ("MCC : Delay set to %u seconds.\n", delay);
                        }
                        break;

                case 2: Suspend ();
                        printf ("MCC : Screen saver suspended.\n");
                        break;

                case 3: Activate ();
                        printf ("MCC : Screen saver activated.\n");
                        break;
            }
        }
    }
}
