/****************************************************************

	gr_tam.c	implementation of Bywater graphics
			standard for AT&T Unix PC using
			tam() and wrastop() libraries

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

	This file specifies the interface for graphics and mouse 
	handling utilized in the Space Flight Simulator.  This file
	may be utilized as the basis for implementations of gr and 
	thus of sfs on varied computers.  If you can implement the
	functions specified in this file, sfs should be implemented.

****************************************************************/

#include "stdio.h"
#include "fcntl.h"
#include "tam.h"
#include "kcodes.h"
#include "message.h"
#include "errno.h"
#include "sys/window.h"
#include "sys/font.h"
#include "gr.h"
#include "bw.h"

static struct fntdef *tam_fsmall; /* ptr to small fntdef struct */
static struct fntdef *tam_fmed;	  /* ptr to medium fntdef struct */
static struct fntdef *tam_flarge; /* ptr to large fntdef struct */
static struct fntdef *tam_fhuge;  /* ptr to huge fntdef struct */
static struct fntdef *tam_fdef;   /* ptr to selected fntdef struct */

#define IMAGES          64
#define	FULLSCREEN	FALSE
#define	CHECK_PARAMS	TRUE
#define SMALLFONT       "/usr/lib/wfont/smallfont"
#define MEDFONT         "/usr/lib/wfont/system.8.ft"
#define LARGEFONT       "/usr/lib/wfont/PLAIN.R.E.18.A"
#define HUGEFONT        "/usr/lib/wfont/PLAIN.R.E.24.A"
#define	ADD_SIZE	1
#define ADD_HIDDEN	2
#define	WRITE_OUTPUT	FALSE

int	gr_colors = 2;
int	gr_pxsize = 287;
int	gr_pysize = 428;
int	gr_ismouse = TRUE;
int     gr_clipping = FALSE;
int     gr_blitting = TRUE;
int     gr_saving = TRUE;
int	gr_screens = 2;

int w, w_id;
static unsigned short tam_pixel[ 2 ] = { 0xffff, 0xffff };
static struct gr_window *grwind;
static unsigned short *tam_selected;
static unsigned short *tam_hidden;
int tam_mready = FALSE;		/* is mouse input ready */
int tam_mx, tam_my;		/* mouse position x, y */
char    *tam_images[ IMAGES ];  /* array of pointers to image buffers */

extern unsigned short patblack[];
extern unsigned short patwhite[];
extern unsigned short patgray[];
extern unsigned short patltgray[];

extern char * malloc();
extern unsigned short tam_words();
extern long tam_imsize();

#ifdef	WRITE_OUTPUT
extern FILE *outfile;
#endif

/****************************************************************

	gr_init()

****************************************************************/

gr_init( grwindow, font_path )
   struct gr_window *grwindow;
   char * font_path;
   {
   static struct uwdata uw;
   static struct umdata um;
   register int i;
   int r;

   grwind = grwindow;

   close( 0 );
   close( 1 );
   close( 2 );
   w_id = open( "/dev/window", O_RDWR );
   dup( w_id );
   dup( w_id );

   winit();
   
#if	FULLSCREEN
   w = wcreate( 1, 0, 24, 80, NBORDER );
#else
   w = wcreate( 2, 2, 19, 70, BORDCANCEL );
#endif
   wuser( w, "Bywater GR Implementation" );
   wlabel( w, "Bywater GR Implementation" );
   wprintf( w, "\033[=1C" );
   clear();

   ioctl( w, WIOCGETD, &uw );

   grwindow->xmax = uw.uw_width - 1;
   grwindow->ymax = uw.uw_height - 1;
   grwindow->initialized = TRUE;
   grwindow->font = F_DEFAULT;

   tam_readfont( SMALLFONT, &tam_fsmall );
   tam_readfont( MEDFONT, &tam_fmed );
   tam_readfont( LARGEFONT, &tam_flarge );
   tam_readfont( HUGEFONT, &tam_fhuge );

   if ( tam_fmed == NULL )
     {
     fprintf( stderr, "ERROR: failed to load system font.\n" );
     wgetc( w );
     wexit( 1 );
     }

   tam_fdef = tam_fmed;
   grwindow->fysize = tam_fdef->ff_vs;
   grwindow->fxsize = tam_fdef->ff_hs;

   if ( ( tam_hidden = (unsigned short *) 
      malloc( tam_imsize( 0, 0,
      grwind->xmax + ADD_HIDDEN, grwind->ymax + ADD_HIDDEN ) )) == NULL )
      {
      fprintf( stderr, "Failed to find memory for gr_hidden\n" );
      return -1;
      }

#ifdef	OLD_DEBUG
      wgoto( w, 0, 0 );
      fprintf( stderr, "test wrastop(): tam_hidden %08lx\n",
         (long) tam_hidden );
      kb_rx();
#endif

#ifdef	OLD_DEBUG
   gr_text( GR_PRIMARY, 0, 0, " !\"#", WHITE, BLACK );
#endif

   /* set image buffer pointers to NULL */

   for ( i = 0; i < IMAGES; ++i )
      {
      tam_images[ i ] = NULL;
      }

   um.um_flags = MSDOWN | MSUP;
   um.um_icon = NULL;
   r = wsetmouse( w, &um );

   if ( r == -1 )
      {
      fprintf( stderr, "Setmouse() returned -1\n" );
      kb_rx();
      }

   return TRUE;
   }

/****************************************************************

   gr_deinit()

****************************************************************/

gr_deinit( screen )
   int screen;
   {
   wdelete( w );
   wexit( 0 );
   }

/****************************************************************

   gr_cls()

****************************************************************/

gr_cls( screen )
   int screen;
   {

   if ( screen == GR_PRIMARY )
      {
      clear();
      }
   else
      {
      gr_rectangle( screen, 0, 0, grwind->xmax, grwind->ymax,
         BLACK, SOLID );
      }
   }


/****************************************************************

   gr_pixel()

****************************************************************/

gr_pixel( screen, x, y, color )
   int screen;
   int x, y;
   int color;
   {
   int r;

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   tam_setscreen( screen );

   r = wrastop( w, tam_pixel, (unsigned short) 2,
      tam_selected, tam_words( grwind->xmax + ADD_HIDDEN ),
      (unsigned short) 0, (unsigned short) 0,
      (unsigned short) x, 
      (unsigned short) ( grwind->ymax - y ),
      (unsigned short) ADD_SIZE, (unsigned short) ADD_SIZE,
      SRCSRC, DSTOR, (unsigned short *) 0 );

#ifdef	DEBUG
   if ( r == -1 )
      {
      fprintf( stderr, "gr_pixel(): wrastop() failed, error %d.\n", errno );
      kb_rx();
      }
#endif
   }
         
/****************************************************************

   gr_line()

****************************************************************/

gr_line( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {

#if	CHECKPARAMS
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   if ( x1 == x2 )
      {
      gr_rectangle( screen, x1, y1, x2, y2, color, style );
      }
   else if ( y1 == y2 )
      {
      gr_rectangle( screen, x1, y1, x2, y2, color, style );
      }
   else
      {
      def_line( screen, x1, y1, x2, y2, color, style );
      }
   }

/****************************************************************

   gr_text()

****************************************************************/

gr_text( screen, x, y, string, foreground, background )
   int screen;
   int x, y;
   int foreground, background;
   char *string;
   {
   char *s;
   unsigned short c;
   char *m;
   register unsigned int x_pos;
#ifdef	DEBUG
   static int do_it = 0;
#endif

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): x value is %d", x );
      kb_rx();
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): y value is %d", y );
      kb_rx();
      return BW_ERROR;
      }
#endif

   tam_setscreen( screen );

   s = string;
   x_pos = x;
   while ( *s != 0 )
      {
      c = *s - 32;
      m = (char *) &(tam_fdef->ff_fc[ c ].fc_mr)
         + ( tam_fdef->ff_fc[ c ].fc_mr );

      if ( ( *s > 31 ) && ( *s < 129 ))
         {
         if ( ( x_pos + tam_fdef->ff_hs ) >= grwind->xmax )
            {
            return BW_ERROR;
            }
         tam_char( screen, x_pos, y, m,
            tam_fdef->ff_fc[ c ].fc_hs,
            tam_fdef->ff_fc[ c ].fc_vs,
	    foreground, background, (int) *s,
            &(tam_fdef->ff_fc[ c ] ) );
         x_pos += tam_fdef->ff_hs;
         }
      ++s;
      }
   }

/****************************************************************

   gr_strlen()

****************************************************************/

unsigned int
gr_strlen( string )
   char *string;
   {
   char *s;
   int size;

   s = string;
   size = 0;
   while ( *s != 0 )
      {
      size += tam_fdef->ff_hs;
      ++s;
      }
   return size;
   }

/****************************************************************

   gr_rectangle()

****************************************************************/

gr_rectangle( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {
   unsigned short *pattern;
   int r;

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   tam_setscreen( screen );

   switch ( style )
      {
      case HOLLOW:
         def_rectangle( screen, x1, y1, x2, y2, color, style );
         return TRUE;
         break;
      case SOLID:
         if ( color == WHITE )
            {
            pattern = patwhite;
            }
         else
            {
            pattern = patblack;
            }
         break;
      case GRID:
         pattern = patgray;
         break;
      case HATCH:
         pattern = patltgray;
         break;
      }

   r = wrastop( w, tam_pixel, (unsigned short) 2,
      tam_selected, tam_words( grwind->xmax + ADD_HIDDEN ),
      (unsigned short) 0, (unsigned short) 0,
      (unsigned short) x1,
      (unsigned short) ( grwind->ymax - y2 ),
      (unsigned short) abs(x2 - x1) + ADD_SIZE,
      (unsigned short) abs(y2 - y1) + ADD_SIZE,
      SRCPAT, DSTSRC, pattern );

#ifdef	DEBUG
   if ( r == -1 )
      {
      wgoto( w, 0, 0 );
      fprintf( stderr, "gr_rectangle(): wrastop() failed, error %d, screen %d.\n",
         errno, screen );
      kb_rx();
      }
#endif
   }

/****************************************************************

   gr_circle()

****************************************************************/

gr_circle( screen, x, y, radius, color, style )
   int screen;
   int x, y, radius;
   int color, style;
   {

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   def_circle( screen, x, y, radius, color, style );
   }

/****************************************************************

   gr_ellipse()

****************************************************************/

gr_ellipse( screen, x, y, x_radius, y_radius, mode, color, style )
   int screen;
   int x, y, x_radius, y_radius;
   int mode, color, style;
   {
   }
         
/****************************************************************

   gr_clip()

****************************************************************/

gr_clip( screen, mode, x1, y1, x2, y2 )
   int screen;
   int mode;
   int x1, y1, x2, y2;
   {
   }

/****************************************************************

   gr_font()

****************************************************************/

gr_font( screen, type, rq_height )
   int screen;
   int type, rq_height;
   {

   if ( rq_height > 21 )
      {
      tam_fdef = tam_fhuge;
      if ( tam_fdef > NULL )
         {
         grwind->fysize = tam_fdef->ff_vs;
         grwind->fxsize = tam_fdef->ff_hs;
         return TRUE;
         }
      }
   else if ( rq_height > 14 )
      {
      tam_fdef = tam_flarge;
      if ( tam_fdef > NULL )
         {
         grwind->fysize = tam_fdef->ff_vs;
         grwind->fxsize = tam_fdef->ff_hs;
         return TRUE;
         }
      }
   else if ( rq_height > 9 )
      {
      tam_fdef = tam_fmed;
      if ( tam_fdef > NULL )
         {
         grwind->fysize = tam_fdef->ff_vs;
         grwind->fxsize = tam_fdef->ff_hs;
         return TRUE;
         }
      }
   else
      {
      tam_fdef = tam_fsmall;
      if ( tam_fdef > NULL )
         {
         grwind->fysize = tam_fdef->ff_vs;
         grwind->fxsize = tam_fdef->ff_hs;
         return TRUE;
         }
      }

   tam_fdef = tam_fmed;		/* default if all failed */
   grwind->fysize = tam_fdef->ff_vs;
   grwind->fxsize = tam_fdef->ff_hs;
   return TRUE;
   }

/****************************************************************

   gr_blit()

****************************************************************/

gr_blit( src, dst, x1, y1, x2, y2 )
   int src, dst;
   int x1, y1, x2, y2;
   {
   unsigned short *srcbase, *dstbase;
   int r;
   register int l;

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( src == dst )
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): src == dst" );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   if ( src == GR_PRIMARY )
      {
      srcbase = (unsigned short *) NULL;
      dstbase = tam_hidden;
      r = wrastop( w, srcbase,
         tam_words( abs( x2 - x1 ) + ADD_SIZE ),
         dstbase, tam_words( grwind->xmax + ADD_HIDDEN ),
         (unsigned short) x1,
         (unsigned short) y1,
         (unsigned short) x1, 
         (unsigned short) y1,
         (unsigned short) abs(x2 - x1),
         (unsigned short) abs(y2 - y1),
         SRCSRC, DSTSRC, (unsigned short *) 0 );

#ifdef	DEBUG
      if ( r == -1 )
         {
         wgoto( w, 0, 0 );
         fprintf( stderr, "gr_blit(): wrastop() failed, error %d.\n", errno );
         kb_rx();
         }
#endif
      }
   else
      {
      dstbase = (unsigned short *) NULL;
      srcbase = tam_hidden;

#ifndef OLD_WAY
      r = wrastop( w,
         srcbase, tam_words( grwind->xmax + ADD_HIDDEN ),
         dstbase, tam_words( grwind->xmax + ADD_HIDDEN ),
         (unsigned short) x1,
         (unsigned short) ( grwind->ymax - y2 ),
         (unsigned short) x1,
         (unsigned short) ( grwind->ymax - y2 ),
         (unsigned short) abs(x2 - x1) + ADD_SIZE,
         (unsigned short) abs(y2 - y1) + ADD_SIZE,
         SRCSRC, DSTSRC, (unsigned short *) 0 );
#else
      for ( l = y1; l < y2; ++l )
         {
         r = wrastop( w,
            srcbase, tam_words( grwind->xmax + ADD_HIDDEN ),
            dstbase, tam_words( grwind->xmax + ADD_HIDDEN ),
            (unsigned short) x1,
            (unsigned short) ( grwind->ymax - l ),
            (unsigned short) x1,
            (unsigned short) ( grwind->ymax - l ),
            (unsigned short) abs(x2 - x1) + ADD_SIZE,
            (unsigned short) 1,
            SRCSRC, DSTSRC, (unsigned short *) 0 );
/*         kb_rx(); */
         }
#endif
      
#ifdef	DEBUG
      if ( r == -1 )
         {
         wgoto( w, 0, 0 );
         fprintf( stderr, "gr_blit(): wrastop() failed, error %d.\n", errno );
         kb_rx();
         }
#endif
      }
   }

/****************************************************************

   tam_imsize()

****************************************************************/

long
tam_imsize( x1, y1, x2, y2 )
   int x1, y1, x2, y2;
   {
   return ( sizeof( unsigned short)
      * ( tam_words(abs(x2 - x1) + ADD_SIZE )
      * ( abs(y2 - y1)) + ADD_SIZE ));
   }

/****************************************************************

   gr_imsave()

****************************************************************/

gr_imsave( screen, mode, x1, y1, x2, y2, image )
   int screen; 
   int mode, x1, y1, x2, y2;
   int *image;
   {
   int r;
   int carry_on;

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_imsave(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_imsave(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_imsave(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_imsave(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   tam_setscreen( screen );

   if ( mode == TRUE )
      {


      /* find an available buffer */

      carry_on = TRUE;
      r = 0;
      while( ( carry_on == TRUE ) && ( r < IMAGES ) )
	 {
	 if ( tam_images[ r ] == NULL )
	    {
	    carry_on = FALSE;
	    }
	 else
	    {
	    ++r;
	    }
	 }

      if ( r >= IMAGES )
	 {
	 bw_error( "No more slots for image storage" );
	 return FALSE;
	 }

      *image = r;

      /* get memory */

      if ( ( tam_images[ *image ] = malloc( (size_t) tam_imsize(
	 x1, grwind->ymax - y2,
	 x2, grwind->ymax - y1 ) ) ) == NULL )
	 {
	 bw_error( "Out of memory to store image" );
	 return FALSE;
	 }

      /* perform the save operation */

      r = wrastop( w, tam_selected,
         tam_words( grwind->xmax + ADD_HIDDEN ),
	 (unsigned short *) tam_images[ *image ],
         tam_words( abs(x2 - x1)),
         (unsigned short) x1, 
         (unsigned short) ( grwind->ymax - y2 ),
         (unsigned short) 0, (unsigned short) 0,
         (unsigned short) abs(x2 - x1) + ADD_SIZE,
         (unsigned short) abs(y2 - y1) + ADD_SIZE,
         SRCSRC, DSTSRC, (unsigned short *) 0 );

#ifdef	DEBUG
      if ( r == -1 )
         {
         wgoto( w, 0, 0 );
         fprintf( stderr, "gr_imsave(): wrastop() failed, error %d.\n", errno );
         kb_rx();
         }
#endif
      }
   else
      {
      r = wrastop( w, (unsigned short *) tam_images[ *image ],
         tam_words( abs(x2 - x1)),
         tam_selected, tam_words( grwind->xmax + ADD_HIDDEN ),
         (unsigned short) 0, (unsigned short) 0,
         (unsigned short) x1,
         (unsigned short) ( grwind->ymax - y2 ),
         (unsigned short) abs(x2 - x1) + ADD_SIZE,
         (unsigned short) abs(y2 - y1) + ADD_SIZE,
         SRCSRC, DSTSRC, (unsigned short *) 0 );

#ifdef	DEBUG
      if ( r == -1 )
         {
         wgoto( w, 0, 0 );
         fprintf( stderr, "gr_imsave(): wrastop() failed, error %d.\n", errno );
         kb_rx();
         }
#endif
      }
   }

/****************************************************************

	gr_imfree()

****************************************************************/

gr_imfree( image )
   int image;
   {

#ifdef DEBUG
   if ( tam_images[ image ] == NULL )
      {
      bw_error( "gr_imfree(): NULL image requested" );
      return FALSE;
      }
#endif

   free( tam_images[ image ] );
   tam_images[ image ] = NULL;

   }

/****************************************************************

   gr_mouse()

****************************************************************/

gr_mouse( mode, x, y, buttons )
   int mode;
   int *x, *y;
   int *buttons;
   {
   static int b, r;
   
   if (( mode == STATUS ) || ( mode == SAMPLE ))
      {

#if WRITE_OUTPUT
   fprintf( outfile, "gr_mouse() STATUS: enter\n" );
   fflush( outfile );
#endif

      if ( tam_mready == FALSE )
         {
         att_rxstat();
         }
      return tam_mready;
      }
   else if ( mode == WAIT )
      {

#if WRITE_OUTPUT
   fprintf( outfile, "gr_mouse() WAIT: enter\n" );
   fflush( outfile );
#endif

      while( tam_mready == FALSE )
         {
         att_rxstat();
         }
      *x = tam_mx;
      *y = grwind->ymax - tam_my;
      tam_mready = FALSE;
      return TRUE;
      }
   }

tam_setscreen( screen )
   int screen;
   {
   if ( screen == GR_HIDDEN )
      {
      tam_selected = tam_hidden;
      }
   else
      {
      tam_selected = (unsigned short *) 0;
      }	
   }

tam_readfont( fontfile, sptr )
   char *fontfile;
   struct fntdef **sptr;
   {
   FILE *ff_fp;
   int size;
   char *m;

   /* open the font file */

   if ( ( ff_fp = fopen( fontfile, "r" )) == NULL )
      {
      fprintf( stderr, "Failed to open font file\n" );
      wgetc( w );
      wexit( 1 );
      }

   /* find the size of the file */

   size = 0;
   while( !feof( ff_fp ) )
      {
      ++size;
      fgetc( ff_fp );
      }

   /* allocate memory for the file */

   if ( ( m = malloc( size )) == NULL )
      {
      fprintf( stderr, "No memory for font file <%s> \n",
         fontfile );
      *sptr = NULL;
      wgetc( w );
      return BW_ERROR;
      }
   *sptr = (struct fntdef *) m;

#ifdef	OLD_DEBUG
   clear();
   wgoto( w, 0, 0 );
   fprintf( stderr, "Loading font <%s> \n", fontfile );
   fprintf( stderr, "Size %d bytes sptr 0x%08lX top 0x%08lX \n",
      size, (long) *sptr, (long) *sptr + size );
   wgetc( w );
   clear();
   wgoto( 0, 0 );
#endif

   /* now read the file into memory */

   rewind( ff_fp );
   while( !feof( ff_fp ) )
      {
      *m = fgetc( ff_fp );
      ++m;
      }

   /* close the font file */

   fclose( ff_fp );

#ifdef	OLD_DEBUG
   fprintf( stderr, "Loaded font <%s>, %d by %d pixels\n",
      fontfile, (*sptr)->ff_vs, (*sptr)->ff_hs );
   wgetc( w );
#endif

   }

tam_char( screen, x, y, icon, x_size, y_size,
   foreground, background, c, cdef )
   int screen, x, y, x_size, y_size, foreground, background, c;
   unsigned short *icon;
   struct fcdef *cdef;
   {
   int r;

   /* paint the background */

   gr_rectangle( screen, x, y,
      x + tam_fdef->ff_hs, y + tam_fdef->ff_vs,
      background, SOLID );

   /* call wrastop() to display the character as an icon */

   r = wrastop( w, icon, tam_words( y_size ),
      tam_selected, tam_words( grwind->xmax + ADD_HIDDEN ),
      (unsigned short) 0, (unsigned short) 0,
      (unsigned short) x + cdef->fc_ha,
      (unsigned short) (grwind->ymax - y) + ( cdef->fc_va - 1 ),
      (unsigned short) x_size,
      (unsigned short) y_size,
      SRCSRC, DSTXOR, (unsigned short *) 0 );

#ifdef	OLD_DEBUG
   wgoto( w, 10, 0 );
   if ( r == -1 )
      {
      fprintf( stderr, "tam_char(): wrastop() failed, error %d, screen %d.\n",
         errno, screen );
      wgetc( w );
      }
#ifdef OLD_STUFF
   fprintf( stderr, "Character 0x%X <%c>:  offset %d (0x%04X\n",
      c, c, cdef->fc_mr, cdef->fc_mr );
   fprintf( stderr, "  x_size %d, y_size %d, words %d \n",
      x_size, y_size, tam_words( y_size ) );
   fprintf( stderr, "  fc_ha %d, fc_va %d, fc_hi %d, fc_vi %d \n",
      cdef->fc_ha, cdef->fc_va, cdef->fc_hi, cdef->fc_vi );
   fprintf( stderr, "  x pos %d, y pos %d\n",
      x, y );
   fprintf( stderr, "  icon %08lX selected %08lX\n",
      (long) icon, (long) tam_selected );
   wgetc( w );
   clear();
#endif
#endif
   }

/* determine the number of 16-bit words necessary
   for horizontal axis */

unsigned short
tam_words( pixels )
   int pixels;
   {
   return ((( pixels + 15) / 16 ) * 2);
   }

#ifdef T_WRASTOP
t_wrastop( tw, srcbase, srcwidth, dstbase, dstwidth,
   srcx, srcy, dstx, dsty, width, height,
   srcop, dstop, pattern )
   int tw;
   unsigned short *srcbase, *dstbase, *pattern;
   unsigned short srcwidth, dstwidth;
   unsigned short srcx, srcy, dstx, dsty;
   char srcop, dstop;
   {


   if ( tam_wtest == TRUE )
      {
/*      gr_rectangle( GR_PRIMARY, 0, grwind->ymax - 40,
         grwind->xmax, grwind->ymax, BLACK, SOLID ); */
      wgoto( w, 0, 0 );
      fprintf( stderr, "test wrastop(): srcwidth %d, dstwidth %d\n",
         srcwidth, dstwidth );
      kb_rx();
/*      gr_rectangle( GR_PRIMARY, 0, grwind->ymax - 40,
         grwind->xmax, grwind->ymax, BLACK, SOLID ); */
      wgoto( w, 0, 0 );
      fprintf( stderr, "test wrastop(): srcbase %08lx, dstbase %08lx\n",
         (long) srcbase, (long) dstbase );
      kb_rx();
/*      gr_rectangle( GR_PRIMARY, 0, grwind->ymax - 40,
         grwind->xmax, grwind->ymax, BLACK, SOLID ); */
      wgoto( w, 0, 0 );
      fprintf( stderr, "test wrastop(): srcx %d, srcy %d, dstx %d dsty %d\n",
         srcx, srcy, dstx, dsty );
      kb_rx();
/*      gr_rectangle( GR_PRIMARY, 0, grwind->ymax - 40,
         grwind->xmax, grwind->ymax, BLACK, SOLID ); */
      wgoto( w, 0, 0 );
      fprintf( stderr, "test wrastop(): width %d, height %d\n",
         width, height );
      kb_rx();
      }

   return wrastop( tw, srcbase, srcwidth, dstbase, dstwidth,
      srcx, srcy, dstx, dsty, width, height,
      srcop, dstop, pattern );

   }
#endif
