/*
   Module:  inout.h
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides the function prototypes for the input/output 
            routines for the ADI converter.  It provides the routines needed 
            by the driver to open input and output files, and read data.  It 
            also provides the output funtion that can be used by the device
            dependant routines to send their output to the file opened by the 
            driver.

            A datatype (PLTFILE) is also provided by this module.  This
            datatype is the basic structure used by all of the functions in
            this module.

   Functions Provided:

            openin
            openout
            closein
            closeout
            getstr
            putarr

*/


  /* BUFFSIZE has been set to get the maximum number of 2048 byte clusters
     and still be less than 32767
  */
#define BUFFSIZE 30720
#define PLTFILE struct _pltfile

PLTFILE {
   char *buf,     /* start of file buffer */
        *buf_ptr; /* current location in buffer */
   int  fd,       /* file descriptor for the file */
        buf_ctr,  /* number of bytes currently in buffer */
        eof;      /* BOOLEAN flag, only used on input files */
};



/*
   Function: openin
   Purpose:  Allocate the input buffer and open the input file for reading.

   Pre: name is a pointer to an ascii string containing the name of the 
        input file.
        pltfpp is a pointer to storage for a pointer to a PLTFILE.

   Post:  The input buffer is allocated.
          The input file is opened for reading.
          If the buffer can't be allocated, *pltfpp is set to NULL and
          NOBUFF is returned.
          If the input file can't be opened, *pltfpp is set to NULL and
          BADOPEN is returned.
          Otherwise, TRUE is returned.
*/

int openin (char *name, PLTFILE **pltfpp);



/*
   Function: openout
   Purpose:  Allocate the output buffer, and open the output file for
             writing.

   Pre: name is a pointer to an ascii string containing the name of the
        output file.
        pltfpp is a pointer to storage for a pointer to a PLTFILE.

   Post:  The output buffer is allocated.
          The output file is opened for writing.
          If the output buffer can't be allocated, *pltfpp is set to NULL and
          NOBUFF is returned.
          If the output file can't be opened, *pltfpp is set to NULL and
          BADOPEN is returned.
          Otherwise, TRUE is returned.
*/

int openout (char *name, PLTFILE **pltfpp);



/*
   Function: closein
   Purpose:  Discard the input buffer, close the input file, and deallocate 
             the input buffer.

   Pre:  pltfpp is a pointer to a pointer to a PLTFILE.
         If *pltfpp does not point to an open PLTFILE, it is NULL.

   Post:  If *pltfpp = NULL, TRUE is returned.
          Otherwise, the contents of the input buffer are discarded, the 
          input file is closed, and the input buffer is deallocated.
          If an error occurs during the above process, FALSE is returned.
*/

int closein(PLTFILE **pltfpp);



/*
   Function: closeout
   Purpose:  Flush the output buffer, close the output file, and deallocate 
             the output buffer.

   Pre:  pltfpp is a pointer to a pointer to a PLTFILE.
         If *pltfpp does not point to an open PLTFILE, it is NULL.

   Post:  If *pltfpp = NULL, TRUE is returned.
          Otherwise, the output buffer is flushed, the output file is closed, 
          and the output buffer is deallocated.
          If an error occurs during the above process, FALSE is returned.
*/

int closeout(PLTFILE **pltfpp);



/*
   Function: getstr
   Purpose:  Read a string from the input buffer, possibly refilling the
             buffer from the input file.  Data is read up to the next
             line feed character.

   Pre:  pltfp is a pointer to a PLTFILE that has been opened for reading.
         max is the maximum number of chars to be stored in string.
         string is a pointer to storage for the input string.
         There is enough storage available in string to hold "max"+1 chars.

   Post: An attempt is made to read a string from the input buffer and store
         it in "string".
         Reading stops at eof or the first line feed.  The line feed,
         although included in the byte count, is replaced with a NULL.
         If the input buffer is emptied during the process, the input file is 
         read to refill the buffer.
         If "max" chars are read without encountering a line feed, the extra
         chars are discarded.
         The number of bytes read from the input buffer is returned.
         If end_of_file has already been reached, 0 is returned.
         If an error occurs, -1 is returned.
*/

int getstr (PLTFILE *pltfp, int max, char *string);



/*
   Function: putarr
   Purpose:  Write a char array to the output buffer, possibly causing the 
             buffer to be written to the output file.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.
        num is the number of chars to be written to the output buffer.
        chrarr is a pointer to the char array to be written.

   Post: An attempt is made to copy num chars from chrarr to the output 
         buffer.
         If the buffer becomes full during the process, an attempt is made
         to write the buffer to the output file.
         If an error occurs, FALSE is returned, otherwise TRUE is returned.
*/

int putarr (PLTFILE *pltfp, int num, char *chrarr);
