/*			      SUPINTVL.CC
			A C++ implementation of
			   "super precision"
			  interval arithmetic.

                Copyright (c) 1997 William T. Wyatt, Jr.

	"Super precision" objects are implementations of a floating-point
data type having user-selectable precision and radix.  The intent is to
allow extremely long multiple-precision mantissas in the floating-point
representation.

	This file contains a supplemental class definition for a "super
precision interval" floating-point data type with similar properties.  Interval
arithmetic maintains an upper bound and a lower bound on the result at each
step of the computation.  The "sharpness" of the upper and lower bounds depends
upon details of the implementation.  This implementation endeavors to guarantee
the validity of the interval endpoints as "bounds" for the computed value.
To guarantee this, however, some operations extend the interval by an ulp
upward and downward to account for any possible roundoff.  Also, division by
zero or by an interval containing zero may be treated as a fatal error if the
class super_precision static variable "permissive_error" is FALSE.  If it is
TRUE an attempt will be made to obtain simple arithmetic results using a result
interval (-Inf, +Inf), but many functions regard irregular arguments as error
conditions and terminate.

*/

#ifndef already_super
#include "super.cc"
#endif


class super_interval					{ // CLASS DEFINITION
protected:
	static super_interval *zzi;	// returnable scratch object

public:
	super_precision upper;		// upper boundary of interval
	super_precision lower;		// lower boundary of interval

	super_interval ( void);					// constructor
	super_interval ( const int32, const int32);		// constructor
//	~super_interval ( void);				// destructor
	super_interval ( const int32);			//conversion constructor
	super_interval ( const double);			//conversion constructor
	void *operator new ( size_t);			// overload new
	void operator delete ( void *);			// overload delete
	static void setup ( const int32, const short i=10);	//initialization
	super_interval ( const super_interval &);	// ovld copy-constructor
	super_interval & operator= ( const super_interval &);	// overload a=b
	super_interval & operator= ( const int32);		// overload a=i
	super_interval & operator= ( const double);		// overload a=d
	super_interval & operator= ( const super_precision &);	// overload a=s
	super_interval operator- ( void);			// overload -b
	super_interval operator+ ( void);			// overload +b
	super_interval operator+ ( const super_interval &) const; //overload a+b
	super_interval & operator+=( const super_interval &);   // overload a+=b
	super_interval operator+ ( const int32) const;		// overload a+i
	super_interval & operator+=( const int32);		// overload a+=i
	super_interval operator+ ( const super_precision &) const;//overload a+s
	super_interval & operator+=( const super_precision &);	// overload a+=s
	super_interval operator- ( const super_interval &) const; //overload a-b
	super_interval & operator-=( const super_interval &);	// overload a-=b
	super_interval operator- ( const int32) const;		// overload a-i
	super_interval & operator-=( const int32);		// overload a-=i
	super_interval operator- ( const super_precision &) const;//overload a-s
	super_interval & operator-=( const super_precision &);	// overload a-=s
	super_interval operator* ( const super_interval &) const; //overload a*b
	super_interval & operator*=( const super_interval &);	// overload a*=b
	super_interval operator* ( const int32) const;		// overload a*i
	super_interval & operator*=( const int32);		// overload a*=i
	super_interval operator* ( const super_precision &) const;//overload a*s
	super_interval & operator*=( const super_precision &);	// overload a*=s
	super_interval operator/ ( const super_interval &) const; //overload a/b
	super_interval & operator/=( const super_interval &);	// overload a/=b
	super_interval operator/ ( const int32) const;		// overload a/i
	super_interval & operator/=( const int32);		// overload a/=i
	super_interval operator/ ( const super_precision &) const;//overload a/s
	super_interval & operator/=( const super_precision &);	// overload a/=s
	friend super_interval operator+ ( const int32,
				const super_interval &); 	// overload i+a
	friend super_interval operator- ( const int32,
				const super_interval &); 	// overload i-a
	friend super_interval operator* ( const int32,
				const super_interval &); 	// overload i*a
	friend super_interval operator/ ( const int32,
				const super_interval &); 	// overload i/a
	friend super_interval operator+ ( const super_precision &,
				const super_interval &); 	// overload s+a
	friend super_interval operator- ( const super_precision &,
				const super_interval &); 	// overload s-a
	friend super_interval operator* ( const super_precision &,
				const super_interval &); 	// overload s*a
	int operator==( const super_interval &) const;		// overload a==b
	int operator!=( const super_interval &) const;		// overload a!=b
	int operator==( const int32) const;			// overload a==i
	int operator!=( const int32) const;			// overload a!=i
	friend int operator==( const int32, const super_interval &);//ovrld i==a
	friend int operator!=( const int32, const super_interval &);//ovrld i!=a
	int operator==( const super_precision &) const;		// overload a==s
	int operator!=( const super_precision &) const;		// overload a!=s
	friend int operator==( const super_precision &,
				const super_interval &);	// overload s==a
	friend int operator!=( const super_precision &,
				const super_interval &);	// overload s!=a
	int operator< ( const super_interval &) const;		// overload a<b
	int operator>=( const super_interval &) const;		// overload a>=b
	int operator< ( const int32) const;			// overload a<i
	int operator>=( const int32) const;			// overload a>=i
	friend int operator< ( const int32, const super_interval &);//ovrld i<a
	friend int operator>=( const int32, const super_interval &);//ovrld i>=a
	int operator< ( const super_precision &) const;		// overload a<s
	int operator>=( const super_precision &) const;		// overload a>=s
	friend int operator< ( const super_precision &,
				const super_interval &);	// overload s<a
	friend int operator>=( const super_precision &,
				const super_interval &);	// overload s>=a
	int operator> ( const super_interval &) const;		// overload a>b
	int operator<=( const super_interval &) const;		// overload a<=b
	int operator> ( const int32) const;			// overload a>i
	int operator<=( const int32) const;			// overload a<=i
	friend int operator> ( const int32, const super_interval &);//ovrld i>a
	friend int operator<=( const int32, const super_interval &);//ovrld i<=a
	int operator> ( const super_precision &) const;		// overload a>s
	int operator<=( const super_precision &) const;		// overload a<=s
	friend int operator> ( const super_precision &,
				const super_interval &);	// overload s>a
	friend int operator<=( const super_precision &,
				const super_interval &);	// overload s<=a
	void super_dump ( void) const;			// dump super_interval
	friend super_interval pow ( const super_interval &,
				const int32);			// overload pow
	friend super_interval pow ( const super_interval &,
				const super_precision &);	// overload pow
	friend super_interval pow ( const super_interval &,
				const super_interval &);	// overload pow
	friend super_interval pow ( const super_precision &,
				const super_interval &);	// overload pow
	friend super_interval pow ( const int32,
				const super_interval &);	// overload pow
	friend super_interval interval ( const super_precision &,
				const super_precision &);	//store interval
	friend super_interval interval_upside (
				const super_precision &);	//store interval
	friend super_interval interval_downside (
				const super_precision &);	//store interval
	friend super_precision interval_width
				( const super_interval &);	// show width
	friend super_interval fabs ( const super_interval &);	// overload abs
	friend super_interval sqrt ( const super_interval &);	// overload sqrt
	friend super_interval exp ( const super_interval &);	// overload exp
	friend super_interval log ( const super_interval &);	// overload log
	friend super_interval sin ( const super_interval &);	// overload sin
	friend super_interval cos ( const super_interval &);	// overload cos
	friend super_interval tan ( const super_interval &);	// overload tan
	friend ostream & operator<<						// OVERLOAD <<
			( ostream &, const super_interval &);	// overload <<

};			// END OF CLASS DEFINITION of super_interval.

super_interval :: super_interval ( void)			// CLASS CONSTRUCTOR
{
}

super_interval :: super_interval ( const int32 i,		// CLASS CONSTRUCTOR
					const int32 j)
{
	if ( i > j)
	{
		this->upper = i;
		this->lower = j;
	}
	else
	{
		this->upper = j;
		this->lower = i;
	}
}

void *super_interval :: operator new			// OVERLOAD new
				( size_t size)
{
	return (new super_precision [2]);
}

void super_interval :: operator delete			// OVERLOAD delete
				( void *ptr)
{
	delete [] ptr;
	ptr = 0;			// set ptr to NULL
}

super_interval :: super_interval		      // OVERLOAD copy-constructor
				( const super_interval &a)
{
	upper = a.upper;
	lower = a.lower;
}

super_interval :: super_interval			  //CONVERSION CONSTRUCTOR
				( const int32 i)
{
	this->upper = i;			// Convert integer to super_interval.
	this->lower = i;
}

super_interval :: super_interval			  //CONVERSION CONSTRUCTOR
				( const double d)
{
	this->upper = d;			// Convert double to super_interval.
	this->lower = d;
}


//
// ALLOCATE CLASS STATIC OBJECTS AT FILE SCOPE.
//
super_interval *super_interval :: zzi;


void super_interval :: setup					// PUBLIC FN
			( const int32 digits_, const short radix_)
{							// initializes stuff
	// Invoke super_precision setup function.
	super_precision :: setup ( digits_, radix_);

	// Allocate *zzi scratch super_interval object.
	/* super_interval * */ super_interval :: zzi = new super_interval;

	return;
}

super_interval & super_interval :: operator=			// OVERLOAD =
				( const super_interval &b)
{								// ASSIGN a=b
	this->upper = b.upper;
	this->lower = b.lower;
	return ( *this);
}

super_interval & super_interval :: operator=			// OVERLOAD =
				( const int32 i)
{								// ASSIGN a=i
	this->upper = i;
	this->lower = i;
	return ( *this);
}

super_interval & super_interval :: operator=			// OVERLOAD =
				( const double d)
{								// ASSIGN a=d
	this->upper = d;
	this->lower = d;
	return ( *this);
}

super_interval & super_interval :: operator=			// OVERLOAD =
				( const super_precision &s)
{								// ASSIGN a=s
	this->upper = s;
	this->lower = s;
	return ( *this);
}

super_interval super_interval :: operator-			// OVERLOAD -
				( void)
{								// unary -b
	zzi->upper = -this->lower;
	zzi->lower = -this->upper;
	return ( *zzi);
}

super_interval super_interval :: operator+			// OVERLOAD +
				( void)
{								// unary +b
	zzi->upper = this->upper;
	zzi->lower = this->lower;
	return ( *zzi);
}

super_interval super_interval :: operator+			// OVERLOAD +
				( const super_interval &b) const
{								// a + b
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper + b.upper;
	if ( zzi->upper - b.upper != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper + b.upper;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower + b.lower;
	if ( zzi->lower - b.lower != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower + b.lower;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator+=			// OVERLOAD +=
				( const super_interval &b)
{								// a += b
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += b.upper;
	if ( this->upper - b.upper != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += b.upper;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += b.lower;
	if ( this->lower - b.lower != zzi->lower)
	{		// It was not exact because it is not reversible.
		super_precision :: set_rounding_method ( FALSE);  // Truncated.
		this->lower += b.lower;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator-			// OVERLOAD -
				( const super_interval &b) const
{								// a - b
	// No widening of the interval occurs if the subtraction is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper - b.lower;
	if ( zzi->upper + b.upper != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper - b.upper;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower - b.upper;
	if ( zzi->lower + b.lower != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower - b.lower;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator-=			// OVERLOAD -=
				( const super_interval &b)
{								// a -= b
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += b.lower;
	if ( this->upper - b.lower != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += b.lower;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += b.upper;
	if ( this->lower - b.upper != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->lower;	// Restore this and try again.
		this->lower += b.upper;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator+			// OVERLOAD +
				( const int32 i) const
{								// a + i
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper + i;
	if ( zzi->upper - i != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper + i;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower + i;
	if ( zzi->lower - i != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower + i;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator+=			// OVERLOAD +=
				( const int32 i)
{								// a += i
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += i;
	if ( this->upper - i != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += i;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += i;
	if ( this->lower - i != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->lower;	// Restore this and try again.
		this->lower += i;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator+			// OVERLOAD +
				( const super_precision &s) const
{								// a + s
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper + s;
	if ( zzi->upper - s != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper + s;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower + s;
	if ( zzi->lower - s != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower + s;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator+=			// OVERLOAD +=
				( const super_precision &s)
{								// a += s
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += s;
	if ( this->upper - s != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += s;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += s;
	if ( this->lower - s != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->lower;	// Restore this and try again.
		this->lower += s;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator-			// OVERLOAD -
				( const int32 i) const
{								// a - i
	// No widening of the interval occurs if the subtraction is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper - i;
	if ( zzi->upper + i != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper - i;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower - i;
	if ( zzi->lower + i != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower - i;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator-=			// OVERLOAD -=
				( const int32 i)
{								// a -= i
	zzi->upper = i;			// Use zzi->upper for i.
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += zzi->upper;
	if ( this->upper - zzi->upper != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += zzi->upper;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += zzi->upper;
	if ( this->lower - zzi->upper != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->lower;	// Restore this and try again.
		this->lower += zzi->upper;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator-			// OVERLOAD -
				( const super_precision &s) const
{								// a - s
	// No widening of the interval occurs if the subtraction is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper - s;
	if ( zzi->upper + s != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper - s;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower - s;
	if ( zzi->lower + s != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower - s;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator-=			// OVERLOAD -=
				( const super_precision &s)
{								// a -= s
	// No widening of the interval occurs if the addition is exact.

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->lower = this->upper;	// Use zzi->lower for scratch.
	this->upper += s;
	if ( this->upper - s != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->lower;	// Restore this and try again.
		this->upper += s;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower;	// Use zzi->lower for scratch.
	this->lower += s;
	if ( this->lower - s != zzi->lower)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->lower;	// Restore this and try again.
		this->lower += s;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.
	
	return ( *this);
}

super_interval super_interval :: operator*			// OVERLOAD *
				( const super_interval &b) const
{								// a * b
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper * b.upper;
	zzi->lower = this->lower * b.lower;
	if ( zzi->lower > zzi->upper)
	{
		x = zzi->upper;
		zzi->upper = zzi->lower;
		zzi->lower = x;
	}
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();

	x = this->lower * b.upper;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}

	x = this->upper * b.lower;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}
	super_precision :: set_rounding_method ( TRUE);

	return ( *zzi);
}

super_interval & super_interval :: operator*=			// OVERLOAD *=
				( const super_interval &b)
{								// a *= b
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper * b.upper;
	zzi->lower = this->lower * b.lower;
	if ( zzi->lower > zzi->upper)
	{
		x = zzi->upper;
		zzi->upper = zzi->lower;
		zzi->lower = x;
	}
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();

	x = this->lower * b.upper;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}

	x = this->upper * b.lower;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}
	super_precision :: set_rounding_method ( TRUE);

	*this = *zzi;

	return ( *this);
}

super_interval super_interval :: operator*			// OVERLOAD *
				( const int32 i) const
{								// a * i
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper * i;
	if ( zzi->upper / i != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper * i;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower * i;
	if ( zzi->lower / i != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower * i;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator*=			// OVERLOAD *=
				( const int32 i)
{								// a *= i
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper;	// Use zzi->upper for scratch.
	this->upper *= i;
	if ( this->upper / i != zzi->upper)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->upper;
		this->upper *= i;
		if ( this->upper > 0) this->upper.increment_mantissa ();
	}
	
	zzi->upper = this->lower;	// Use zzi->upper for scratch.
	this->lower *= i;
	if ( this->lower / i != zzi->upper)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->upper;
		this->lower *= i;
		if ( this->lower < 0) this->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *this);
}

super_interval super_interval :: operator*			// OVERLOAD *
				( const super_precision &s) const
{								// a * s
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper * s;
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	zzi->lower = this->lower * s;
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator*=			// OVERLOAD *=
				( const super_precision &s)
{								// a *= s
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	this->upper *= s;
	if ( this->upper > 0) this->upper.increment_mantissa ();
	this->lower *= s;
	if ( this->lower < 0) this->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *this);
}

super_interval operator+					// OVERLOAD +
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION 					// i + a
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	super_interval :: zzi->upper = a.upper;
	super_interval :: zzi->upper += i;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = a.lower;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval operator-					// OVERLOAD -
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION 					// i - a

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
			// zzi cannot be the same object as a.
	super_interval :: zzi->upper = i;
	super_interval :: zzi->upper -= a.upper;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = a.lower;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval operator*					// OVERLOAD *
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION 					// i * a

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	super_interval :: zzi->upper = a.upper;
	super_interval :: zzi->upper *= i;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = a.lower;
	super_interval :: zzi->lower *= i;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval operator+					// OVERLOAD +
		( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION 					// s + a
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	super_interval :: zzi->upper = s + a.upper;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = a.lower;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval operator-					// OVERLOAD -
		( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION 					// s - a
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	super_interval :: zzi->upper = s - a.upper;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = a.lower;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval operator*					// OVERLOAD *
		( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION 					// s * a

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	super_interval :: zzi->upper = s * a.upper;
	if ( super_interval :: zzi->upper > 0)
			super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = s * a.lower;
	if ( super_interval :: zzi->lower < 0)
			super_interval :: zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *super_interval :: zzi);
}

super_interval super_interval :: operator/			// OVERLOAD /
				( const super_interval &b) const
{								// a / b
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

		// Test to see if divisor interval contains zero.
	if ( b.lower == 0 || b.upper == 0 ||
		( b.lower < 0 && b.upper > 0))
	{		// Yup, it contains zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			return *zzi;
		}
		else
		{
			cout << "Dividing by an interval containing zero -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}
	
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper / b.lower;
	zzi->lower = this->lower / b.upper;
	if ( zzi->lower > zzi->upper)
	{
		x = zzi->upper;
		zzi->upper = zzi->lower;
		zzi->lower = x;
	}
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();

	x = this->lower / b.lower;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}

	x = this->upper / b.upper;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}
	super_precision :: set_rounding_method ( TRUE);

	return ( *zzi);
}

super_interval & super_interval :: operator/=			// OVERLOAD /=
				( const super_interval &b)
{								// a /= b
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

		// Test to see if divisor interval contains zero.
	if ( b.lower == 0 || b.upper == 0 ||
		( b.lower < 0 && b.upper > 0))
	{		// Yup, it contains zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			*this = *zzi;
			return *this;
		}
		else
		{
			cout << "Dividing by an interval containing zero -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}
	
	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper / b.lower;
	zzi->lower = this->lower / b.upper;
	if ( zzi->lower > zzi->upper)
	{
		x = zzi->upper;
		zzi->upper = zzi->lower;
		zzi->lower = x;
	}
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();

	x = this->lower / b.lower;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}

	x = this->upper / b.upper;
	if ( x > zzi->upper)
	{
		zzi->upper = x;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( x < zzi->lower)
		{
			zzi->lower = x;
			if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
		}
	}
	super_precision :: set_rounding_method ( TRUE);

	*this = *zzi;

	return ( *this);
}

super_interval super_interval :: operator/			// OVERLOAD /
				( const int32 i) const
{								// a / i
		// Test to see if divisor equals zero.
	if ( i == 0)
	{		// Yup, it equals zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			return *zzi;
		}
		else
		{
			cout << "Dividing an interval by zero integer -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper / i;
	if ( zzi->upper * i != this->upper)
	{		// It was not exact because it is not reversible.
		zzi->upper = this->upper / i;
		if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	}
	
	zzi->lower = this->lower / i;
	if ( zzi->lower * i != this->lower)
	{		// It was not exact because it is not reversible.
		zzi->lower = this->lower / i;
		if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator/=			// OVERLOAD /=
				( const int32 i)
{								// a /= i
		// Test to see if divisor equals zero.
	if ( i == 0)
	{		// Yup, it equals zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			*this = *zzi;
			return *this;
		}
		else
		{
			cout << "Dividing an interval by zero integer -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper;	// Use zzi->upper for scratch.
	this->upper /= i;
	if ( this->upper * i != zzi->upper)
	{		// It was not exact because it is not reversible.
		this->upper = zzi->upper;
		this->upper /= i;
		if ( this->upper > 0) this->upper.increment_mantissa ();
	}
	
	zzi->upper = this->lower;	// Use zzi->upper for scratch.
	this->lower /= i;
	if ( this->lower * i != zzi->upper)
	{		// It was not exact because it is not reversible.
		this->lower = zzi->upper;
		this->lower /= i;
		if ( this->lower < 0) this->lower.decrement_mantissa ();
	}
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *this);
}

super_interval super_interval :: operator/			// OVERLOAD /
				( const super_precision &s) const
{								// a / s
		// Test to see if divisor equals zero.
	if ( s == 0)
	{		// Yup, it equals zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			return *zzi;
		}
		else
		{
			cout << "Dividing an interval by zero integer -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	zzi->upper = this->upper / s;
	if ( zzi->upper > 0) zzi->upper.increment_mantissa ();
	zzi->lower = this->lower / s;
	if ( zzi->lower < 0) zzi->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *zzi);
}

super_interval & super_interval :: operator/=			// OVERLOAD /=
				( const super_precision &s)
{								// a /= s
		// Test to see if divisor equals zero.
	if ( s == 0)
	{		// Yup, it equals zero -- so it's a divide-by-zero.
		if ( super_precision :: permissive_error)
		{
			zzi->upper = super_NaN ( +1);		// +Inf
			zzi->lower = super_NaN ( +2);		// -Inf
			*this = *zzi;
			return *this;
		}
		else
		{
			cout << "Dividing an interval by zero integer -- ";
			cout << "fatal error." << endl;
			exit (1);
		}
	}

	super_precision :: set_rounding_method ( FALSE);  // Truncated.
	this->upper /= s;
	if ( this->upper > 0) this->upper.increment_mantissa ();
	this->lower /= s;
	if ( this->lower < 0) this->lower.decrement_mantissa ();
	super_precision :: set_rounding_method ( TRUE);   // Rounded.

	return ( *this);
}

int super_interval :: operator==				// OVERLOAD a==b
				( const super_interval &b) const
{								// a == b
// Returns a TRUE if the intervals (this->lower, this->upper) and
// (b.lower, b.upper) OVERLAP, including equality of any endpoints.
// Otherwise, returns a FALSE.
	int i = 0;
	if ( this->lower >= b.lower && this->lower <= b.upper) i = 1;
	if ( this->upper >= b.lower && this->upper <= b.upper) i = 1;

	return i;
}

int super_interval :: operator!=				// OVERLOAD a!=b
				( const super_interval &b) const
{								// a != b
// Returns a TRUE if the intervals (this->lower, this->upper) and
// (b.lower, b.upper) DO NOT OVERLAP, including nonequality of any endpoints.
// Otherwise, returns a FALSE.
	int i = 1;
	if ( this->lower >= b.lower && this->lower <= b.upper) i = 0;
	if ( this->upper >= b.lower && this->upper <= b.upper) i = 0;

	return i;
}

int super_interval :: operator==				// OVERLOAD a==i
				( const int32 i) const
{								// a == i
// Returns a TRUE if the super_interval (this->lower, this->upper) contains i.
// Otherwise, returns a FALSE.

	super_interval :: zzi->upper = i;
	return ( this->lower <= super_interval :: zzi->upper
		&& this->upper >= super_interval :: zzi->upper);
}

int super_interval :: operator!=				// OVERLOAD a!=i
				( const int32 i) const
{								// a != i
// Returns a FALSE if the super_interval (this->lower, this->upper) contains i.
// Otherwise, returns a TRUE.

	super_interval :: zzi->upper = i;
	return ( this->lower > super_interval :: zzi->upper
		|| this->upper < super_interval :: zzi->upper);
}

int operator==							// OVERLOAD i==a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i == a
// Returns a TRUE if the super_interval (a.lower, a.upper) contains i.
// Otherwise, returns a FALSE.

	super_interval :: zzi->upper = i;
	return ( a.lower <= super_interval :: zzi->upper
		&& a.upper >= super_interval :: zzi->upper);
}

int operator!=							// OVERLOAD i!=a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i != a
// Returns a FALSE if the super_interval (a.lower, a.upper) contains i.
// Otherwise, returns a TRUE.
	super_interval :: zzi->upper = i;
	return ( a.lower > super_interval :: zzi->upper
		|| a.upper < super_interval :: zzi->upper);
}

int super_interval :: operator==				// OVERLOAD a==s
				( const super_precision &s) const
{								// a == s
// Returns a TRUE if the super_interval (this->lower, this->upper) contains s.
// Otherwise, returns a FALSE.

	return ( this->lower <= s && this->upper >= s);
}

int super_interval :: operator!=				// OVERLOAD a!=s
				( const super_precision &s) const
{								// a != s
// Returns a FALSE if the super_interval (this->lower, this->upper) contains s.
// Otherwise, returns a TRUE.

	return ( this->lower > s || this->upper < s);
}

int operator==							// OVERLOAD s==a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s == a
// Returns a TRUE if the super_interval (a.lower, a.upper) contains s.
// Otherwise, returns a FALSE.

	return ( a.lower <= s && a.upper >= s);
}

int operator!=							// OVERLOAD s!=a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s != a
// Returns a FALSE if the super_interval (a.lower, a.upper) contains s.
// Otherwise, returns a TRUE.
	return ( a.lower > s || a.upper < s);
}

int super_interval :: operator<					// OVERLOAD a<b
				( const super_interval &b) const
{								// a < b
// Returns a TRUE if the interval (this->lower, this->upper) is ENTIRELY less
// than (b.lower, b.upper), including any endpoints.
// Otherwise, returns a FALSE.

	return ( this->upper < b.lower);
}

int super_interval :: operator>=				// OVERLOAD a>=b
				( const super_interval &b) const
{								// a >= b
// Returns a FALSE if the interval (this->lower, this->upper) is ENTIRELY less
// than (b.lower, b.upper), including any endpoints.
// Otherwise, returns a TRUE.

	return ( this->upper >= b.lower);
}

int super_interval :: operator<					// OVERLOAD a<i
				( const int32 i) const
{								// a < i
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than i, including the endpoints.
// Otherwise, returns a FALSE.

	return ( this->upper < i);
}

int super_interval :: operator>=				// OVERLOAD a>=i
				( const int32 i) const
{								// a >= i
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than i, including the endpoints.
// Otherwise, returns a TRUE.

	return ( this->upper >= i);
}

int operator<							// OVERLOAD i<a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i < a
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than i, including the endpoints.
// Otherwise, returns a FALSE.

	return ( i < a.lower);
}

int operator>=							// OVERLOAD i>=a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i >= a
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than i, including the endpoints.
// Otherwise, returns a TRUE.

	return ( i >= a.upper);
}

int super_interval :: operator<					// OVERLOAD a<s
				( const super_precision &s) const
{								// a < s
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than s, including the endpoints.
// Otherwise, returns a FALSE.

	return ( this->upper < s);
}

int super_interval :: operator>=				// OVERLOAD a>=s
				( const super_precision &s) const
{								// a >= s
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than s, including the endpoints.
// Otherwise, returns a TRUE.

	return ( this->upper >= s);
}

int operator<							// OVERLOAD s<a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s < a
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than s, including the endpoints.
// Otherwise, returns a FALSE.

	return ( s < a.lower);
}

int operator>=							// OVERLOAD s>=a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s >= a
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than s, including the endpoints.
// Otherwise, returns a TRUE.
	return ( s >= a.lower);
}

int super_interval :: operator>					// OVERLOAD a>b
				( const super_interval &b) const
{								// a > b
// Returns a TRUE if the interval (this->lower, this->upper) is ENTIRELY greater
// than (b.lower, b.upper), including any endpoints.
// Otherwise, returns a FALSE.

	return ( this->lower > b.upper);
}

int super_interval :: operator<=				// OVERLOAD a<=b
				( const super_interval &b) const
{								// a <= b
// Returns a FALSE if the interval (this->lower, this->upper) is ENTIRELY
// greater than (b.lower, b.upper), including any endpoints.
// Otherwise, returns a TRUE.

	return ( this->lower <= b.upper);
}

int super_interval :: operator>					// OVERLOAD a>i
				( const int32 i) const
{								// a > i
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than i, including the endpoints.
// Otherwise, returns a FALSE.

	return ( this->lower > i);
}

int super_interval :: operator<=				// OVERLOAD a<=i
				( const int32 i) const
{								// a <= i
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than i, including the endpoints.
// Otherwise, returns a TRUE.

	return ( this->lower <= i);
}

int operator>							// OVERLOAD i>a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i > a
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than i, including the endpoints.
// Otherwise, returns a FALSE.

	return ( i > a.upper);
}

int operator<=							// OVERLOAD i<=a
				( const int32 i, const super_interval &a)
{	// FRIEND FUNCTION					// i <= a
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than i, including the endpoints.
// Otherwise, returns a TRUE.

	return ( i <= a.upper);
}

int super_interval :: operator>					// OVERLOAD a>s
				( const super_precision &s) const
{								// a > s
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than s, including the endpoints.
// Otherwise, returns a FALSE.

	return ( this->lower > s);
}

int super_interval :: operator<=				// OVERLOAD a<=s
				( const super_precision &s) const
{								// a <= s
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// greater than s, including the endpoints.
// Otherwise, returns a TRUE.

	return ( this->upper <= s);
}

int operator>							// OVERLOAD s>a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s > a
// Returns a TRUE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than s, including the endpoints.
// Otherwise, returns a FALSE.

	return ( s > a.upper);
}

int operator<=							// OVERLOAD s<=a
			( const super_precision &s, const super_interval &a)
{	// FRIEND FUNCTION					// s <= a
// Returns a FALSE if the super_interval (this->lower, this->upper) is ENTIRELY
// less than s, including the endpoints.
// Otherwise, returns a TRUE.
	return ( s <= a.upper);
}

void super_interval :: super_dump				// PUBLIC FN
				( void) const
{							// (const) dump a
	cout << "\n("; lower.super_dump ();
	upper.super_dump ();
	cout << ")" << endl;

	return;
}

super_interval pow						// FRIEND FN
			( const super_interval &b, const int32 ia)
{	// FRIEND FUNCTION					// a = pow(b,ia)
// This function calculates the super_interval b raised to the ia power.
	int32 i, k, l;				// temporaries

	if ( b == 0 && ia <= 0)
	{		// Zero or negative power of interval containing zero.
		if ( ia == 0)
		{
			cout << "Fatal error -- pow taking zero power";
			cout << " of interval containing zero." << endl;
			exit (1);
		}
		if ( super_precision :: permissive_error)
		{
			super_interval :: zzi->upper = super_NaN ( +1);	// +Inf
			super_interval :: zzi->lower = super_NaN ( +2);	// -Inf
			return *super_interval :: zzi;
		}
		else
		{
			cout << "Fatal error -- pow taking negative power";
			cout << " of interval containing zero." << endl;
			exit (1);
		}
	}
	else
	{
		k = abs ( ia);
		if ( k == 0)
		{			// zeroth power yields unity
			if ( b.upper.disclose_validity ()
				|| b.lower.disclose_validity ())
			{
				cout << "Fatal error -- pow taking zeroth ";
				cout << "power of irregular form." << endl;
				exit (1);
			}
			super_interval :: zzi->upper = 1;
			super_interval :: zzi->lower = 1;
			return ( *super_interval :: zzi);
		}
		else
		{
			int first_store = 1;
			super_interval zv = b, zi;
			while ( k)
			{
				i = k / 2;
				l = k - i - i;
				if ( l)
				{
					if ( first_store)
					{
						first_store = 0;
						zi = zv;
					}
					else
					{
						zi *= zv;
					}
				}
				if ( i) zv *= zv;
				k = i;
			}
			if ( ia > 0)
			{			// positive exponent
				if ( zi.lower > zi.upper)
				{
					super_precision_pool_ptr x_;
					super_precision &x = *x_;
					x = zi.lower;
					zi.lower = zi.upper;
					zi.upper = x;
				}
				return ( zi);
			}
			else
			{			// negative exponent
				zv = 1;
				zi = zv / zi;
				if ( zi.lower > zi.upper)
				{
					super_precision_pool_ptr x_;
					super_precision &x = *x_;
					x = zi.lower;
					zi.lower = zi.upper;
					zi.upper = x;
				}
				return ( zi);
			}
		}
	}
}			// End of function pow (b,ia).

super_interval pow						// FRIEND FN
			( const super_interval &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// a = pow(b,s)
// This function calculates the super_interval b raised to the c power,
// where c is a super_precision object.

	if ( b == 0 && c <= 0)
	{		// Zero or negative power of interval containing zero.
		if ( c == 0)
		{
			cout << "Fatal error -- pow taking zero power";
			cout << " of interval containing zero." << endl;
			exit (1);
		}
		if ( super_precision :: permissive_error)
		{
			super_interval :: zzi->upper = super_NaN ( +1);	// +Inf
			super_interval :: zzi->lower = super_NaN ( +2);	// -Inf
			return *super_interval :: zzi;
		}
		else
		{
			cout << "Fatal error -- pow taking negative power";
			cout << " of interval containing zero." << endl;
			exit (1);
		}
	}
	else
	{
		if ( c == 0)
		{			// zeroth power yields unity
			if ( b.upper.disclose_validity ()
				|| b.lower.disclose_validity ())
			{
				cout << "Fatal error -- pow taking zeroth ";
				cout << "power of irregular form." << endl;
				exit (1);
			}
			super_interval :: zzi->upper = 1;
			super_interval :: zzi->lower = 1;
			return ( *super_interval :: zzi);
		}
		else
		{
			super_interval :: zzi->upper = log ( b.upper);
			super_interval :: zzi->upper *= c;
			super_interval :: zzi->upper =
					exp ( super_interval :: zzi->upper);
			super_interval :: zzi->lower = log ( b.lower);
			super_interval :: zzi->lower *= c;
			super_interval :: zzi->lower =
					exp ( super_interval :: zzi->lower);
			if ( super_interval :: zzi->lower >
				super_interval :: zzi->upper)
			{
				super_precision_pool_ptr x_;
				super_precision &x = *x_;
				x = super_interval :: zzi->lower;
				super_interval :: zzi->lower =
						super_interval :: zzi->upper;
				super_interval :: zzi->upper = x;
			}
			return ( *super_interval :: zzi);
		}
	}
}

super_interval pow						// FRIEND FN
			( const super_interval &b,
			const super_interval &c)
{	// FRIEND FUNCTION					// a = pow(b,c)
// This function calculates the super_interval b raised to the c power,
// where c is a super_interval object.

	b.upper.assert_invalid_quit ( b.lower, "super_interval pow");
	c.upper.assert_invalid_quit ( c.lower, "super_interval pow");

	if ( b.lower < 0)
	{
		cout << "Fatal error -- pow taking power of interval with ";
		cout << "negative lower bound." << endl;
		exit (1);
	}
	else
	{
		if ( b.lower == 0)
		{
				// Lower base is zero, so exponent must be > 0.
			if ( c.lower <= 0)
			{
				cout << "Fatal error -- pow taking nonpositive";
				cout << " power of interval containing zero.";
				cout << endl;
				exit (1);
			}
			else
			{
				super_interval :: zzi->lower = 0;
				super_precision_pool_ptr x_, y_;
				super_precision &x = *x_, &y = *y_;
				x = log ( b.upper);
				y = c.upper * x;
				x *= c.lower;
				if ( x > y) y = x;
				super_interval :: zzi->upper = exp ( y);
				return ( *super_interval :: zzi);
			}
		}
		else
		{	// b is entirely > 0.
			super_precision_pool_ptr x_, y_;
			super_precision &x = *x_, &y = *y_;
			x = log ( b.lower);
			y = x * c.upper;
			x *= c.lower;
			if ( x > y)
			{
				super_interval :: zzi->upper = x;
				super_interval :: zzi->lower = y;
			}
			else
			{
				super_interval :: zzi->upper = y;
				super_interval :: zzi->lower = x;
			}
			x = log ( b.upper);
			y = x * c.upper;
			x *= c.lower;
			if ( x > y) y = x;
			if ( y > super_interval :: zzi->upper)
					super_interval :: zzi->upper = y;
			if ( x < super_interval :: zzi->lower)
					super_interval :: zzi->lower = x;
			super_interval :: zzi->upper =
					exp ( super_interval :: zzi->upper);
			super_interval :: zzi->lower =
					exp ( super_interval :: zzi->lower);

			return ( *super_interval :: zzi);
		}
	}
}
	
super_interval pow						// FRIEND FN
			( const super_precision &b,
			const super_interval &c)
{	// FRIEND FUNCTION					// a = pow(s,c)
// This function calculates the super_precision object b raised to the c power,
// where c is a super_interval object.

	b.assert_invalid_quit ( "super_interval pow");
	c.upper.assert_invalid_quit ( c.lower, "super_interval pow");

	if ( b == 0)
	{
		if ( c.lower > 0)
		{			// Result is zero.
			*super_interval :: zzi = 0;
			return ( *super_interval :: zzi);
		}
		else
		{
			cout << "Fatal error -- pow raising zero to a power ";
			cout << "having a nonpositive lower part." << endl;
			exit (1);
		}
	}
	else
	{
		super_interval :: zzi->lower = log ( b);
		super_interval :: zzi->upper =
					super_interval :: zzi->lower * c.upper;
		super_interval :: zzi->upper =
					exp ( super_interval :: zzi->upper);
		super_interval :: zzi->lower *= c.lower;
		super_interval :: zzi->lower =
					exp ( super_interval :: zzi->lower);
		if ( super_interval :: zzi->lower >
			super_interval :: zzi->upper)
		{
			super_precision_pool_ptr x_;
			super_precision &x = *x_;
			x = super_interval :: zzi->lower;
			super_interval :: zzi->lower =
					super_interval :: zzi->upper;
			super_interval :: zzi->upper = x;
		}
		return ( *super_interval :: zzi);
	}
}
	
super_interval pow						// FRIEND FN
			( const int32 i,
			const super_interval &c)
{	// FRIEND FUNCTION					// a = pow(i,c)
// This function calculates the signed-integer i raised to the c power,
// where c is a super_interval object.

	c.upper.assert_invalid_quit ( c.lower, "super_interval pow");

	if ( i == 0)
	{
		if ( c.lower > 0)
		{			// Result is zero.
			*super_interval :: zzi = 0;
			return ( *super_interval :: zzi);
		}
		else
		{
			cout << "Fatal error -- pow raising zero to a power ";
			cout << "having a nonpositive lower part." << endl;
			exit (1);
		}
	}
	else
	{
		super_interval :: zzi->lower = i;
		super_interval :: zzi->lower =
					log ( super_interval :: zzi->lower);
		super_interval :: zzi->upper =
					super_interval :: zzi->lower * c.upper;
		super_interval :: zzi->upper =
					exp ( super_interval :: zzi->upper);
		super_interval :: zzi->lower *= c.lower;
		super_interval :: zzi->lower =
					exp ( super_interval :: zzi->lower);
		if ( super_interval :: zzi->lower >
			super_interval :: zzi->upper)
		{
			super_precision_pool_ptr x_;
			super_precision &x = *x_;
			x = super_interval :: zzi->lower;
			super_interval :: zzi->lower =
					super_interval :: zzi->upper;
			super_interval :: zzi->upper = x;
		}

		return ( *super_interval :: zzi);
	}
}
	
super_interval interval ( const super_precision &a,		// FRIEND FN
				const super_precision &b)
{							// store interval
	super_interval :: zzi->upper = a;
	super_interval :: zzi->lower = b;
	if ( super_interval :: zzi->lower > super_interval :: zzi->upper)
	{
		super_precision_pool_ptr x_;
		super_precision &x = *x_;
		x = super_interval :: zzi->lower;
		super_interval :: zzi->lower = super_interval :: zzi->upper;
		super_interval :: zzi->upper = x;
	}

	return *super_interval :: zzi;
}

super_precision interval_width ( const super_interval &a)	// FRIEND FN
{							// get interval width
// This function returns the interval width (upper - lower).
	return a.upper - a.lower;
}

super_interval interval_upside ( const super_precision &a)	// FRIEND FN
{				// store interval upside of the argument
// If the argument is nonzero, the interval width is epsilon.
	super_interval :: zzi->upper = a;
	super_interval :: zzi->lower = a;
	if ( a > 0)
	{
		super_interval :: zzi->upper.increment_mantissa ();
	}
	else
	{
		if ( a != 0)
		{
			super_interval :: zzi->upper.decrement_mantissa ();
		}
	}

	return *super_interval :: zzi;
}

super_interval interval_downside ( const super_precision &a)	// FRIEND FN
{				// store interval downside of the argument
// If the argument is nonzero, the interval width is epsilon.
	super_interval :: zzi->upper = a;
	super_interval :: zzi->lower = a;
	if ( a > 0)
	{
		super_interval :: zzi->lower.decrement_mantissa ();
	}
	else
	{
		if ( a != 0)
		{
			super_interval :: zzi->lower.increment_mantissa ();
		}
	}

	return *super_interval :: zzi;
}

super_interval fabs ( const super_interval &a)			// FRIEND FN
{
// Returns an interval containing every possible "absolute value" of every
// number on the argument interval.  If the argument interval contains zero,
// then the lower end of the result interval is zero.
	if ( a.lower <= 0 && a.upper >= 0)
	{
		super_interval :: zzi->lower = 0;
		if ( -a.lower <= a.upper)
		{
			super_interval :: zzi->upper = a.upper;
		}
		else
		{
			super_interval :: zzi->upper = -a.lower;
		}
	}
	else
	{
		if ( a.upper > 0)
		{
			*super_interval :: zzi = a;
		}
		else
		{
			super_interval :: zzi->upper = -a.lower;
			super_interval :: zzi->lower = -a.upper;
		}
	}

	return *super_interval :: zzi;
}

super_interval sqrt ( const super_interval &a)			// FRIEND FN
{							// overload sqrt
// Returns an interval containing every possible "positive square root" of every
// number on the argument interval.  If the argument interval contains a
// negative number, then an error condition results.
	if ( a.lower < 0)
	{
		cout << "sqrt interval contains a negative number -- fatal ";
		cout << "error." << endl;
		exit (1);
	}
	else
	{
		super_interval :: zzi->upper = sqrt ( a.upper);
		super_interval :: zzi->upper.increment_mantissa ();
		super_interval :: zzi->lower = sqrt ( a.lower);
		super_interval :: zzi->lower.decrement_mantissa ();
	}

	return *super_interval :: zzi;
}

super_interval exp ( const super_interval &a)			// FRIEND FN
{							// overload exp
// Returns an interval containing the exponential of every number on the
// argument interval.
	super_interval :: zzi->upper = exp ( a.upper);
	super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower = exp ( a.lower);
	super_interval :: zzi->lower.decrement_mantissa ();

	return *super_interval :: zzi;
}
	
super_interval log ( const super_interval &a)			// FRIEND FN
{							// overload log
// Returns an interval containing the natural logarithm of every number on the
// argument interval.
	super_interval :: zzi->upper = log ( a.upper);
	super_interval :: zzi->lower = log ( a.lower);
	super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower.decrement_mantissa ();

	return *super_interval :: zzi;

}

super_interval sin ( const super_interval &a)			// FRIEND FN
{							// overload sin
// Returns an interval containing the sin of every number on the argument
// interval.
	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	if ( interval_width ( a) >= 2 * pi ())
	{		// Simple case.
		b = 1;
		*super_interval :: zzi = interval ( -b, b);

		return *super_interval :: zzi;
	}

		// Must determine if interval contains pi/2 + n*2*pi.
	b = 2 * pi ();
	super_interval :: zzi->upper = a.upper % b;
	if ( a.upper < 0) super_interval :: zzi->upper += b;
	super_interval :: zzi->lower = super_interval :: zzi->upper -
					( a.upper - a.lower);
		// No conversion required, so is safe to use zzi == super:
	if ( *super_interval :: zzi == pi () / 2)
	{
		super_interval :: zzi->upper = 1;
	}
	else
	{
		super_interval :: zzi->upper =
					sin ( super_interval :: zzi->upper);
	}

		// Must determine if interval contains 3pi/2 + n*2*pi.
	if ( *super_interval :: zzi == (3 * pi ()) / 2)
	{
		super_interval :: zzi->lower = -1;
	}
	else
	{
		super_interval :: zzi->lower =
					sin ( super_interval :: zzi->lower);
	}
	
	if ( super_interval :: zzi->lower > super_interval :: zzi->upper)
	{
		b = super_interval :: zzi->upper;
		super_interval :: zzi->upper = super_interval :: zzi->lower;
		super_interval :: zzi->lower = b;
	}
	b = 1;
	if ( super_interval :: zzi->upper < b)
	{
		super_interval :: zzi->upper.increment_mantissa ();
	}
	b = -b;
	if ( super_interval :: zzi->lower > b)
	{
		super_interval :: zzi->lower.decrement_mantissa ();
	}

	return *super_interval :: zzi;
}

super_interval cos ( const super_interval &a)			// FRIEND FN
{							// overload cos
// Returns an interval containing the cos of every number on the argument
// interval.
	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	if ( interval_width ( a) >= 2 * pi ())
	{		// Simple case.
		b = 1;
		*super_interval :: zzi = interval ( -b, b);

		return *super_interval :: zzi;
	}

		// Must determine if interval contains n*2*pi.
	b = 2 * pi ();
	super_interval :: zzi->upper = a.upper % b;
	if ( a.upper < 0) super_interval :: zzi->upper += b;
	super_interval :: zzi->lower = super_interval :: zzi->upper -
					( a.upper - a.lower);
		// Using *zzi so no conversion permitted, avoid *zzi == 0:
	if ( *super_interval :: zzi == b)
	{
		super_interval :: zzi->upper = 1;
	}
	else
	{
		b = 0;
		if ( *super_interval :: zzi == b)
		{
			super_interval :: zzi->upper = 1;
		}
		else
		{
			super_interval :: zzi->upper =
					cos ( super_interval :: zzi->upper);
		}
	}

		// Must determine if interval contains pi + n*2*pi.
	if ( *super_interval :: zzi == pi ())
	{
		super_interval :: zzi->lower = -1;
	}
	else
	{
		super_interval :: zzi->lower =
					cos ( super_interval :: zzi->lower);
	}

	if ( super_interval :: zzi->lower > super_interval :: zzi->upper)
	{
		b = super_interval :: zzi->upper;
		super_interval :: zzi->upper = super_interval :: zzi->lower;
		super_interval :: zzi->lower = b;
	}

	b = 1;
	if ( super_interval :: zzi->upper < b)
	{
		super_interval :: zzi->upper.increment_mantissa ();
	}
	b = -b;
	if ( super_interval :: zzi->lower > b)
	{
		super_interval :: zzi->lower.decrement_mantissa ();
	}

	return *super_interval :: zzi;
}

super_interval tan ( const super_interval &a)			// FRIEND FN
{	// FRIEND FUNCTION				// overload tan
// Returns an interval containing the tan of every number on the argument
// interval.
	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	if ( interval_width ( a) >= pi ())
	{		// Simple case.
		cout << "Interval argument for tan contains singularity point";
		cout << " -- fatal error." << endl;
		exit (1);
	}

		// Must determine if interval contains n*pi.
		// Function is periodic with period equal to pi.
	b = pi ();
	super_interval :: zzi->upper = a.upper % b;
	if ( a.upper < 0) super_interval :: zzi->upper += b;
	super_interval :: zzi->lower = super_interval :: zzi->upper -
					( a.upper - a.lower);
		// No conversion required, so is safe to use zzi == super:
	if ( *super_interval :: zzi == pi () / 2)
	{
		cout << "Interval argument for tan contains singularity point";
		cout << " -- fatal error." << endl;
		exit (1);
	}
	else
	{
		super_interval :: zzi->upper =
					tan ( super_interval :: zzi->upper);
		super_interval :: zzi->lower =
					tan ( super_interval :: zzi->lower);
	}

	if ( super_interval :: zzi->lower > super_interval :: zzi->upper)
	{
		b = super_interval :: zzi->upper;
		super_interval :: zzi->upper = super_interval :: zzi->lower;
		super_interval :: zzi->lower = b;
	}
	super_interval :: zzi->upper.increment_mantissa ();
	super_interval :: zzi->lower.decrement_mantissa ();

	return *super_interval :: zzi;
}

ostream & operator<<						// OVERLOAD <<
			( ostream &ost, const super_interval &a)
{	// FRIEND FUNCTION					// ost << a

	ost << "(" << a.lower << ", " << a.upper << ")";

	return ost;
}
