/* area:  Areacode Finder v1.6

  Program searches for telephone area codes/state information by
  area code, state name, city name, or postal codes.

  Author:  Jason Mathews <mathews@nssdc.gsfc.nasa.gov>
	   NASA/Goddard Space Flight Center - Code 633
	   Greenbelt Road
	   Greenbelt, MD 20771  USA

  Any comments about this program would be appreciated. Please do not
  contact me just to tell me about a new area code since new ones
  appear all the time, but actual errors in the data should be reported.

  For the latest area codes try http://www.bellcore.com/NANP/
  and http://www.555-1212.com/aclookup.html

  Copyright (C) 1992-97 by Jason Mathews. Permission is granted to any
  individual or institution to use, copy or redistribute this software so long
  as it is not sold for profit, provided this copyright notice is retained.

  Usage: area                   - help
	 area *                 - list all area codes
	 area XXX [XXX...]      - find match
	  where XXX is an area code, state, city, or postal code.

    States and cities are found by wildcard match if wildcard characters
    ('*' and/or '?') are specified, otherwise a substring comparison
    is made.  All comparisons are case insensitive; e.g., "NEW" and "New"
    are equivalent.

    - Two letter state/postal codes like TX for Texas, CA for California
      can be used, otherwise type in the state name.
    - Enter two word state names like "New*Jersey" or "New York".
    - Enter three digit area code to find a state such as 301 for Maryland.
    - Enter city names in between quotes like "Los Angeles"
      or substrings such as "cisco" for "San Francisco".
    - Enter "*" for a list of all Area Codes.

  Compilers: This program will compile with Turbo C 2.0, Borland C++,
	     UNIX gcc, VAX/VMS C, and other ANSI compilers.

    bcc -O area.c           (MS-DOS/Borland)
    gcc -O area.c -o area   (Unix)

  History:

  V1.0 26-Nov-92  - Original version for MS-DOS/UNIX.

  V1.1 20-Apr-94  - Optimized postal code search.
		  - Added 210, 407, 508, 516, 610, 708, 810, 903, 908,
		    909, 910, and 917 area codes (effective 1/94).
		  - Updated 706 and 905 area codes.
		  - Added Canada, Bahamas (BA), Bermuda (BM) postal codes.

  V1.2 24-May-94  - Fixed several database corrections.
		  - Added city search as suggested by Peter Mauzey.
		  - Added strstr function for some "non-ANSI" compilers.

  V1.3 1-Jun-94   - Combined city and state search for faster searches.
		  - Improved search with wildcard expression or substring
		  - search depending on the target string.
		  - Fixed wildcard evaluation with "*?" expression.
		  - Added JM, NT postal codes + more cities.

  V1.4 8-Feb-95   - Added 630 area code and updated Illinois info
		    thanks to Dave Beyerl.

  V1.5 29-Jun-95  - Added 520 (AZ), 360 (WA), and 334 (AL)
		    thanks to comments from James Arnone and Loyd Bulmur.

  V1.6 16-Jun-97  - Updated old information and added 87 new area codes
                    thanks to Don Hawkinson and Bellcore.

 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* end-of-list marker */
#define LAST_AREACODE 99

#ifdef __TURBOC__

# define ToLower(c) tolower(c)
# define ToUpper(c) toupper(c)

#else /* ?unix */

/* Avoid using const if compiler does not support it */
# ifndef __STDC__
#   define const
# endif

/*  some compilers don't check if a character is already upper or lower
 *  case in the toupper/tolower functions, so we define our own here.
 */

# ifndef _tolower
#   define _tolower(c) ((c) + 'a' - 'A')
# endif

# ifndef _toupper
#   define _toupper(c) ((c) + 'A' - 'a')
# endif

# define ToLower(c) (isupper(c) ? _tolower(c) : (c))
# define ToUpper(c) (islower(c) ? _toupper(c) : (c))

#endif /* ?__TURBOC__ */

#ifdef VMS
#  define EXIT exit
#else
#  define EXIT return
#endif /* VMS */


/* Global variables */

const char copyright[] =
	"   ** Area Code Finder v1.6 **    (C) Jason Mathews 1992-97\n";

int zone = -1;  /* zone code */

typedef struct zone_s {
    char *code;
    char *stateName;
} zone_t;

zone_t areazones[] = {
"AL","Alabama",
"AK","Alaska",
"AZ","Arizona",
"AR","Arkansas",
"BC","British Columbia",
"CA","California",
"CN","Canada",
"CO","Colorado",
"CT","Connecticut",
"DE","Delaware",
"DC","Washington, District of Columbia",
"FL","Florida",
"GA","Georgia",
"HI","Hawaii",
"ID","Idaho",
"IL","Illinois",
"IN","Indiana",
"IA","Iowa",
"KS","Kansas",
"KY","Kentucky",
"LA","Louisiana",
"ME","Maine",
"MD","Maryland",
"MA","Massachusetts",
"MI","Michigan",
"MN","Minnesota",
"MS","Mississippi",
"MO","Missouri",
"MT","Montana",
"NE","Nebraska",
"NV","Nevada",
"NH","New Hampshire",
"NJ","New Jersey",
"NM","New Mexico",
"NY","New York",
"NC","North Carolina",
"ND","North Dakota",
"OH","Ohio",
"OK","Oklahoma",
"OR","Oregon",
"PA","Pennsylvania",
"PR","Puerto Rico",
"RI","Rhode Island",
"SC","South Carolina",
"SD","South Dakota",
"TN","Tennessee",
"TX","Texas",
"UT","Utah",
"VT","Vermont",
"VI","Virgin Islands",
"VA","Virginia",
"WA","Washington",
"WV","West Virginia",
"WI","Wisconsin",
"WY","Wyoming",
"  ","Wide area (tollfree)",
"  ","International Access",
"  ","Pay-Per-Call caller pays",
"  ","Caribbean Islands, Virgin Islands, and more",
"JM","Jamaica",
"AB","Alberta",
"MB","Manitoba",
"NB","New Brunswick",
"NF","Newfoundland",
"NS","Nova Scotia",
"NT","N.W. Territory",
"ON","Ontario",
"PE","Prince Edward Island",
"PO","Quebec",
"SK","Saskatchewan",
"YT","Yukon Territory",

/* areacodes with no zone category: e.g. 700 US Gov't */
"  ","Service",

0,0 /* sentinel entry marks end of zone list */

};

typedef struct code_s {
    short zcode;        /* index into zonelist (0..n) */

    short areaCode;     /* local area code.
			 * if areacode is -1 then the list continues from
			 * the previous entry.
			 */

    char *list;         /* list of cities, counties, ... with same areacode.
			 * if list is NULL (0) then this area code covers
			 * the entire area.
			 */
} code_t;

code_t areacodes[] = {
0,205,"Northern including Birmingham",
0,334,"Southern including Auburn, Mobile, Montgomery, and Selma",
1,907,0,

/* Arizona */
2,520,"Flagstaff, Kingman, Prescott, Tucson, Sierra Vista, Yuma",
2,602,"Phoenix metropolitan area",
2, -1,"Buckeye, Chandler, Glendale, Mesa, Scottsdale, Tempe",

3,501,"Northern Arkansas including Little Rock",
3,870,"Southest Arkansas",

/* British Columbia */
4,250,"Interior of British Columbia, Vancouver Island",
4,604,"Vancouver and the Lower Mainland,",
4, -1,"including Sunshine and Whistler",

/* California */
5,213,"Downtown Los Angeles",
5,209,"Fresno",
5,310,"Beverly Hills, Long Beach west of Los Angeles River, Whittier",
5,323,"Area surrounding Downtown Los Angeles (was 213) effective 6/13/98",
5,408,"San Jose metropolitan area, Sunnyvale",
5,415,"San Francisco area, Marin County",
5, -1,"part of Brisbane, eastern part of Daly City",
5,510,"Oakland, Berkeley, Fremont, Richmond",
5,530,"Old 916 area code area excluding Sacramento metropolitan area",
5, -1,"effective date TBD",
5,562,"Long Beach east of the Los Angeles River, Paramount,",
5, -1,"Downey, and Pico Rivera",
5,619,"San Diego metropolitan area",
5,626,"Pasadena and eastern part of old 818 area code",
5,650,"part of Brisbane, western part of Daly City",
5, -1,"was 415 area code south of San Francisco",
5,707,"Eureka, Napa, Santa Rosa",
5,714,"Anaheim, Orange County",
5,760,"Southeastern part of the state extending northward",
5, -1,"to eastern slope of Sierra Nevada (was 619)",
5, -1,"Palm Springs",
5,805,"Bakersfield, Ventura, Simi Valley",
5,818,"Burbank, Van Nuys, Glendale, La Canada Flintridge",
5,831,"Old 408 areacode excluding San Jose metropolitan area",
5, -1,"effective 7/11/98",
5,909,"Riverside (Southwestern CA)",
5,916,"Sacramento and (until 530 is effective) South Lake Tahoe",
5,925,"Martinez, Concord, Dublin, and Livermore",
5, -1,"eastern part of old 510 area code effective 3/14/98",
5,949,"Irvine, Newport Beach and Costa Mesa",

/* Canada */
6,250,"Interior of British Columbia, Vancouver Island",
6,604,"Vancouver and the Lower Mainland,",
6, -1,"including Sunshine and Whistler",
6,403,"Alberta and Yukon",
6,306,"Saskatchewan",
6,204,"Manitoba",
6,416,"Toronto, Ont.",
6,519,"London, Ont.",
6,613,"Ottawa, Ont.",
6,705,"East Ontario",
6,807,"West Ontario",
6,905,"Southern Ontario",
6,418,"Northern Quebec",
6,450,"Quebec effective date TBD",
6,819,"NW Quebec",
6,514,"Montreal",
6,709,"Newfoundland",
6,506,"New Brunswick",
6,902,"Nova Scotia, Prince Edward Island",

7,303,"Denver metropolitan area ",
7,719,"Colorado Springs, Pueblo",
7,970,"Aspen, Fort Collins, Grand Junction",

/* Connecticut */
8,203,"New Haven, Fairfield and New Haven counties",
8,860,"Hartford, Litchfield, Middlesex, New London, Tolland",
8, -1,"and Windham counties ",

9,302,0,
10,202,0,

/* Florida */
11,305,"Miami, Key West, Dade and Monroe Counties",
11,352, "Gainesville (Northern)",
11,407,"Orlando, West Palm Beach",
11,561,"Boca Raton, Sebastian (Central)",
11,813,"Tampa, St. Petersburg, Winter Haven",
11,850,"Florida",
11,904,"Daytona Beach, Jacksonville, Pensacola",
11, -1,"Panama City, Tallahassee",
11,941,"Ft. Myers, Naples, Sarasota (Southwest)",
11,954,"Fort Lauderdale, Broward County",

/* Georgia */
12,404,"Atlanta, Rome",
12,706,"Athens, Augusta, Columbus",
12,770,"Atlanta suburbs",
12,912,"Macon, Savannah, Waycross",

13,808,0,
14,208,0,

/* Illinois */
15,217,"Champaign-Urbana, Decatur, Springfield",
15,309,"Galesburg, Macomb, Peoria",
15,312,"Chicago's central commercial area (the Loop)",
15,618,"Alton, Carbondale, Centralia, Mount Vernon",
15,630,"Suburban Chicago, Aurora, Naperville, Oak Brook",
15, -1,"DuPage County, southern Kane County, part of Kendall County",
15,708,"Chicago suburbs, Evanston, Waukegan",
15,773,"Chicago,  O'hare Airport",
15,815,"De Kalb, Joliet, Moline, Ottawa, Rockford",
15,847,"Chicago - northern suburbs",

16,219,"Gary, South Bend, Warsaw",
16,317,"Indianapolis metro area, Kokomo",
16,765,"outside Indianapolis metro area",
16,812,"Evansville",

17,712,"Council Bluffs, Sioux City",
17,515,"Des Moines",
17,319,"Dubuque",

18,785,"Abilene, Goodland, Hays, Lawrence, Manhatten, Salina, Topeka",
18,316,"Dodge City, Emporia, Liberal, Parsons, Wichita",
18,913,"Kansas City,",

19,502,"Louisville, Frankfort, Paducah, Shelbyville",
19,606,"Ashland, Lexington, Middlesboro, Winchester",

20,504,"Baton Rouge, New Orleans",
20,318,"Lake Charles, Monroe, Shreveport",
21,207,0,

/* Maryland */
22,240,"Western Maryland",
22,301,"Bethesda, Cumberland, Frederick, Greenbelt,",
22, -1,"Hagerstown, Rockville, Silver Spring",
22,410,"Annapolis, Baltimore, Ellicott City, Salisbury",
22,443,"Eastern Maryland",

23,413,"Pittsfield, Springfield",
23,508,"Fall River, Framingham, New Bedford, Worchester",
23,617,"Boston, Cambridge, Plymouth",
23,781,"Central Boston",
23,978,"Overflow for 508 area effective date TBD",

24,313,"Detroit and inner suburbs",
24,616,"Battle Creek, Grand Rapids, Kalamazoo",
24,517,"Jackson, Lansing",
24,810,"Flint, Macomb, St. Clair, Sanilac, Genesee, Lapeer counties",
24,906,"Escanaba, Sault Ste. Marie",
24,248,"Oakland county",
24,734,"Ann Arbor, Livonia, Wayne and Ypsilant (effective 12/97)",

/* Minnesota */
25,218,"Duluth",
25,320,"Alexandria, St. Cloud",
25,612,"Minneapolis / St. Paul metropolitan area",

25,507,"Mankato, Rochester",

26,601,"Jackson Local Access and Transport Area",
26,228,"Mississippi",

/* Missouri */

27,314,"St. Louis metropolitan area",
27, -1,"Jefferson City, Mexico, Poplar Bluff, Rolla",
27,417,"Joplin, Springfield",

27,573,"Cape Girardeau, Columbia, Fulton, Hannibal,Jefferson City",

/*
27,816,"Belton, Independence, Kansas City, Marshall, St. Joseph, Sedalia",
27,660,"Belton, Independence, Marshall, Sedalia effective date TBD",
*/

27,816,"Belton, Independence, Kansas City, Marshall, St. Joseph, Sedalia",
27,660,"Kansas City",

28,406,0,
29,402,"Omaha, Lincoln",
29,308,"North Platte, Scottsbluff",
30,702,0,
31,603,0,

32,201,"Newark, Hackensack, Jersey City, Patterson",
32,609,"Atlantic City, Camden, Trenton, Wildwood",
32,908,"Elizabeth, New Brunswick, Warren,",
32, -1,"Hunterdon, and portions of Somerset counties",
32,732,"Central New Jersey including Middlesex,",
32, -1,"Monmouth, Ocean and parts of Somerset counties",
32,973,"North New Jersey, Bergen and Hudson counties",

33,505,0,

34,518,"Albany, Schenectady",
34,607,"Binghamton, Corning",
34,716,"Buffalo, Niagara Falls, Rochester",
34,516,"Hempstead, Long Island area",
34,914,"Mount Vernon, White Plains",
34,212,"New York City (Manhattan and Bronx)",
34,718,"New York City (Queens, Brooklyn and Staten Island)",
34,917,"New York City (All Locations)",
34,315,"Syracuse, Watertown",
35,704,"Asheville, Charlotte, Salisbury",
35,910,"Greensboro, Winston-Salem",
35,919,"Raleigh",
36,701,0,

/* Ohio */
37,216,"Cleveland",
37,330,"Eastern Ohio, Akron",
37,419,"Lima, Toledo",
37,513,"Cincinnati, Dayton, Springfield",
37,614,"Columbus, Marion, Zanesville",
37,937,"Area north of, but not including Cincinnati",
37,440,"Akron, Canton, Youngstown",
37,740,"Columbus",
37,740,"Old 614 area code excluding Columbus metropolitan area,",
37, -1,"Franklin, Delaware, Marion, Union, and",
37, -1,"Madison counties (efective 11/97)",

38,918,"Bartlesville, McAlester, Muskogee, Tulsa",
38,405,"Enid, Oklahoma City, Norman, Ponca City, Stillwater",

/* Oregon */
39,503,"Portland, Salem",
39,541,"Southern and Eastern Oregon",

/* Pennsylvania */
40,215,"Levittown, Philadelphia, Quakertown",
40,412,"Butler, Indiana, New Castle, Pittsburgh, Uniontown",
40,610,"Allentown, Bethlehem, Reading, Swarthmore, Valleyforge",
40,717,"Gettysburg, Harrisburg, Hershey, Lancaster, Scranton",
40,814,"Bradford, Clarion, Erie, Punxsutawney, Williamsburg",

/* 40,724,"Pennsylvania", /* TBD */

41,787,0, /*  Puerto Rico */

42,401,0,

/* South Carolina */
43,803,"Augusta, Charlotte, Charleston, Columbia",
43,864,"Greenville",
43,843,"Charleston and Florence, Atlantic coast area",
43, -1,"(effective 3/98)",

44,605,0,

45,423,"Chattanooga, Knoxville",
45,615,"Nashville",
45,901,"Jackson, Memphis",
45,931,"South Tennessee (effective 9/97)",

/* Texas */
46,210,"San Antonio metropolitan area",
46,214,"Central Dallas",
46,254,"Temple, Waco",
46,281,"Outer Houston, Baytown, Pasadena",
46,409,"Bay City, Beaumont, Bryan, College Station, Galveston, Huntsville",
46,512,"Austin, Brownsville, Corpus Christi, Del Rio, Eagle Pass,",
46, -1,"Laredo, McAllen, Victoria",
46,713,"Houston, Baytown, Pasadena",
46,806,"Amarillo, Dalhart, Lubbock",
46,817,"Fort Worth metropolitan area, including Arlington",
46,830,"Uvalde, New Braunfels, Fredericksburg, and Kerrville",
46,903,"Texarkana, Tyler",
46,915,"West Texas",
46,940,"Denton, Wichita Falls",
46,956,"Laredo, Brownsville, McAllen",
46,972,"Dallas suburbs, Irving, Ennis, Greenville,",
46, -1,"Jefferson, Longview, Sherman",

/* Utah */
47,801,"Salt Lake City, Ogden, Provo",
47,435,"Outside Salt Lake City",

48,802,0,

49,284,"British Virgin Islands",
49,340,"US Virgin Islands",

/* Virginia */
50,540,"Roanoke",
50,703,"Alexandria, Arlington, Fairfax, Fredricksburg,",
50, -1,"Roanoke, Staunton, Vienna, Winchester",
50,757,"Norfolk",
50,804,"Charlottesville, Lynchburg, Newport News,",
50, -1,"Portsmouth, Richmond, Williamsburg",

/* Washington */
51,206,"Seattle, Richmond Beach ",
51,253,"South Washington, Auburn, Des Moines, Tacoma",
51,425,"Everett",
51,360,"Olympia, Vancover",
51,509,"Spokane, Walla Walla, Yakima",

52,304,0,

/* Wisconsin */
53,414,"Kenosha Milwaukee, Racine",
53,608,"Beloit, La Crosse, Madison",
53,715,"Eau Claire, Wausau",
53,920,"Appleton, Green Bay, Oshkosh",

/* Wyoming */
54,307,0,

55,800,"Toll-Free free for caller",
55,888,"Toll-Free free for caller",

56,11,0,  /* International */
56,670,"Commonwealth of Northern Mariana Islands (CNMI)",
56,671,"Guam",

57,900,0, /* Pay-Per-Call */

58,242,"Bahamas",
58,246,"Barbados",
58,264,"Anguilla",
58,268,"Antigua and Barbuda",
58,284,"British Virgin Islands",
58,340,"US Virgin Islands",
58,345,"Cayman Islands",
58,441,"Bermuda",
58,473,"Grenada",
58,649,"Turks and Caicos",
58,664,"Montserrat",
58,758,"St. Lucia",
58,767,"Dominica",
58,784,"St. Vincent and the Grenadines",
58,809,"Dominican Republic",
58,868,"Trinidad and Tobago",
58,869,"St. Kitts and Nevis",

59,876,0, /* Jamaica */

/* Canada */

60,403,0, /* Alberta */
61,204,0, /* Manitoba */
62,506,0, /* New Brunswick */
63,709,0, /* Newfoundland */
64,902,0, /* Nova Scotia */
65,867,"Yukon and Northwest Territories",
66,416,"Toronto", /* Ontario */
66,519,"London",
66,613,"Ottawa",
66,705,"North Bay",
66,807,"Fort William, Thunder Bay",
66,905,"Southern Ontario",
67,902,0, /* Prince Edward */
68,418,"Quebec City",
68,450,"Quebec",
68,514,"Montreal",
68,819,"Sherbrooke",
69,306,0, /* Saskatchewan */
70,867,"Yukon and Northwest Territories",

/* Generic service #'s */

71,456,"International Inbound",
71,500,"Personal Communications Services",
71,600,"Canada/Services",
71,700,"IC Services",
71,710,"US Government",
71,880,"Paid 800 service",
71,881,"Paid 888 service",

LAST_AREACODE,0,0    /* sentinel entry */
};

#ifndef __TURBOC__
/*---------------------------------------------------------------------*
 * strlwr - converts a string to lower-case
 *
 * Returns: pointer to the converted string
 *---------------------------------------------------------------------*/
char* strlwr( char* s )
{
  if (s != 0)
  {
      register char *ps = s;
      while (*ps)
      {
	  if (isupper((unsigned char)*ps))
	    *ps = _tolower((unsigned char)*ps);
	  ps++;
	}
    }
  return s;
}
#endif /* ?!__TURBOC__ */

/*-----------------------------------------------------------------------*
 * stristr - Scans a string for the occurrence of a given string
 *           ignoring case
 *
 * Returns:  1 if str1 contains str2, otherwise 0.
 *-----------------------------------------------------------------------*/
int stristr( const char *str1, const char *str2 )
{
  register const char *s1, *s2;
  char c1, c2;

  if (!*str2) return 1;           /* return 1 if str2 empty */

  do
  {
    s1 = str1; c1 = *s1;
    s2 = str2; c2 = *s2;
    while (c1 && c2)
    {
	if (isupper(c1)) c1 = _tolower(c1);
	if (isupper(c2)) c2 = _tolower(c2);
	if (c1==c2)    /* str1[i] == str2[i] ? */
	{
	    c1 = *(++s1);  c2 = *(++s2);
	}
	else break;
     }
     if (!*s2) return 1;    /* target parsed - match found */
     ++str1;    /* try next position of str1 */
  } while (*str1);
  return 0;
}

/*-----------------------------------------------------------------------*
 * hasWildCard: Checks if wildcard characters are present in string s
 *
 * Returns:     1 if '*' or '?' are found, 0 otherwise.
 *-----------------------------------------------------------------------*/
int hasWildCard( const char *s )
{
    while (*s)
    {
	if (*s == '*' || *s == '?') return 1;
	s++;
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * match:       match string s with wildcard pattern
 *
 * Returns:     1 if the pattern matches the string
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int match (const char *target, const char *pattern)
{
    register char symbol;
    register const char *s = target;
    const char *old_pat = 0;  /* address of last wildcard pattern */
    const char *old_target;   /* address of target string */

  /*  Parse each string character by character until
   *  a mismatch is found or the end of one is reached.
   */
 matchRetry:
    while (*pattern != 0)
    {
    switch (*pattern) {
	   case '?':
		goto matchNext; /* match any symbol, except NULL */

	   case '*':
		old_pat = pattern++;
		while (*pattern != 0)
		{
		  switch (*pattern) {
		   case '?':
			if (!*s) return 0; /* no match? */
			s++;  pattern++;
			continue;
		   case '*': /* handle multiple *'s */
			pattern++;
			continue;
		  } /* switch */
		  break;
		} /* while */
		/*  If the end of the expression was parsed
		 *  and the last character was a wildcard (*) then
		 *  the rest of the string is accepted.
		 */
		if (!*pattern) return 1;
		for (symbol = ToLower(*pattern);; s++)
		{
		    if (!*s) return 0; /* no match */
		    if (ToLower(*s) == symbol)
		    {
			old_target = s + 1;
			break;
		    }
		}
	goto matchNext2;

	   default:
		symbol = ToLower(*pattern);
	} /* switch */

	if (ToLower(*s) != symbol) break; /* mismatch? */

  matchNext:
	if (!*s) break;
  matchNext2:
	s++;  pattern++;   /* increment strings */
	symbol = ToLower(*pattern);
    } /* while */

    /* check if match failed */
    if (ToLower(*s) != symbol && old_pat && *old_target != 0)
    {
	/*  check if a wildcard was previously parsed, if so back up
	 *  to that pattern and try again.
	 */
	pattern = old_pat; /* reset pattern */
	s = old_target++;  /* reset target to last position */
	goto matchRetry;
    }

  /*    Check if the last two characters match:
   *    either strings must be NULL to end the loop,
   *    so they both must equal NULL for a match.
   */
    return (*pattern == *s);
}

/*-----------------------------------------------------------------------*
 * Printcode:   Print the entry for the specified areacode
 *
 *              Keep track of the previous zone and
 *              only print the state information when this changes.
 *-----------------------------------------------------------------------*/
void PrintCode (int code)
{
    int newzone = areacodes[code].zcode;

    /*  check if entry is continued from the previous one
     *  (identified by the -1 areacode) and already printed.
     */
    if (areacodes[code].areaCode < 0) return;
    if (newzone != zone)
    {
	zone = newzone;
	printf("\n%s  %s area code(s):\n",
		areazones[zone].code,
	       areazones[zone].stateName);
      }

    printf("    %03d %s\n", areacodes[code].areaCode,
	areacodes[code].list ? areacodes[code].list : "All locations.");

    /* Check if areacode list continues on next entry */
    while (areacodes[++code].areaCode == -1)
	printf("        %s\n", areacodes[code].list);
}

/*-----------------------------------------------------------------------*
 * FindAbbrev:  Find postal code in list of states matching the string s
 *
 * returns:     1 if match found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindAbbrev (const char *s)
{
    int  i;
    char ch1, ch2;      /* target characters in uppercase */
    ch1 = ToUpper(*s);
    ch2 = ToUpper(*(s+1));
    for (i=0; areazones[i].code != 0; i++)
    {
	if (ch1==areazones[i].code[0] && ch2==areazones[i].code[1])
	{
		int code = 0;
		int found = 0;
	    /*  Locate areacode in list and
	     *  print all entries for that state.
	     */
		while (areacodes[code].zcode < i) code++;
		while (areacodes[code].zcode == i)
		{
			PrintCode(code++);
			found = 1;
		}
		return found;
	}
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * FindCode:    Find areacode in list
 *
 * returns:     1 if areacode is found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindCode (const char *s)
{
    int i;
    int code = atoi(s);
    if (code == 0) return 0; /* invalid code */

    for (i=0; areacodes[i].zcode != LAST_AREACODE; i++)
    {
      if (areacodes[i].areaCode == code)
	{
	  PrintCode(i);
	  return 1;
	}
    }
    return 0;
}

/*-----------------------------------------------------------------------*
 * FindString:  Find all states and/or cities matching the pattern s
 *
 * Returns:     1 if string is found.
 *              0 otherwise.
 *-----------------------------------------------------------------------*/
int FindString (const char *s)
{
    int i;
    int rc = 0;  /* return code */
    int code, pos;
    char target[30];
    int wildcards = hasWildCard(s);
    int (*cmp)(const char*, const char*) = wildcards ? match : stristr;

    if (*s == '*' && !*(s+1))   /* show all codes and return */
    {
      for (code=0; areacodes[code].zcode != LAST_AREACODE; code++) {
	PrintCode(code);
      }
      return 1;
    }

    strncpy(target, s, sizeof(target)); /* copy target string */
    strlwr(target);                     /* convert to lower case */

    for (code=0,i=0; areazones[i].code != 0; i++)
    {
	if (cmp(areazones[i].stateName, target))
	{
	    /* print all codes for found zone */
	    while (areacodes[code].zcode==i)
	    {
		PrintCode(code++);
	    }
	    rc = 1;
	}
	else /* try city search */
	{
	    while (areacodes[code].list != 0 && areacodes[code].zcode == i)
	    {
		if (cmp(areacodes[code].list, target))
		{
		    pos = code;
		    while (areacodes[pos].areaCode < 0) pos--;
		    PrintCode(pos);
		    rc = 1;
		}
		code++;
	    }
	    /* set zone position */
	    while (areacodes[code].zcode <= i) code++;
	}
    } /* for */
    return rc;
}


int main (int argc, char **argv)
{
    int i;

    printf(copyright);
    if (argc==1)
    {
	printf("\nProgram searches for telephone area codes,\n");
	printf("   as area xxx xxx xxx etc.\n");
	printf("   xxx is an Area Code, State name, or City.\n\n");
	printf("   Two letter state/postal codes like TX for Texas, CA for California\n");
	printf("      can be used, otherwise type in the state or city name.\n");
	printf("   Enter wildcard expressions such as: New* or *Green?b*\n");
	printf("   Enter substrings like: cisco or east\n");
	printf("   Enter multiple words in between quotes like: \"Los Angeles\" or \"New York\"\n");
	printf("   Enter area * for a list of all Area Codes.\n");
	EXIT(0);
    }

    for (i=1; i < argc; i++)
    {
	zone = -1;      /* reset zone number */
	if (hasWildCard(argv[i])) goto others;
	switch (strlen(argv[i])) {
	 case 2: /* find state zone: CA, MD, NY, ...
		  * if there's no match then try the
		  * general string search.
		  */
		if (!FindAbbrev(argv[i])) goto others;
		break;

	 case 3: /* find area code: 301, 718, ... */
		if (isdigit(*argv[i]) && FindCode(argv[i]))
			break;
		/* otherwise try general search */

	 default: /* state name: new*york, maryland, ... */
	 others:  /* or city name: Greenbelt, Fresno, ... */
		if (!FindString(argv[i]))
		    printf("\nNot found, %s.\n", argv[i]);
     } /* switch */
    } /* for each argument */

    EXIT(0);
}

/*
 * end area.c
 */
