/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"

#include "idm.h"
#include <stdarg.h>
#ifdef CTL3D
    #include <ctl3d.h>
#endif CTL3D


GLOBALOPTIONS   global;
FILEOPTIONS     *fileoptions = NULL;
FILEOPTIONS     *curfile = NULL;
KANJIFONT       *InstalledFonts = NULL;
ASCIIFONT       DefAsciiFont = { "Times New Roman", 12 };
HANDLE          hInstance = NULL;
HMENU           hmenu = NULL;
HMENU           FullMenu, MinimalMenu;
BOOL            Dialogs3D = FALSE;

static int		Untitled = 0;
static BOOL     AllInitialized = FALSE;

static TOOLBARICON toolbar[] = {
	{ ID_KANA, 180, TBBORDER+2, 24, 24, 1, 0, NULL, "Kana_Undep", "Kana_Dep", "Kana_Grayed", "Kana_Pressed" },
	{ ID_ASCII, 210, TBBORDER+2, 24, 24, 0, 0, NULL, "Ascii_Undep", "Ascii_Dep", "Ascii_Grayed", "Ascii_Pressed" },
	{ ID_CONVERT, 250, TBBORDER+2, 24, 24, 0, 1, "CVT_Disabled", "CVT_Undep", NULL, NULL, "CVT_Pressed" },
	{ ID_BOLD, 290, TBBORDER+2, 24, 24, -1, 2, "Bold_Disabled", "Bold_Undep", "Bold_Dep", "Bold_Grayed", "Bold_Pressed" },
	{ ID_ITALICS, 320, TBBORDER+2, 24, 24, -1, 2, "Italics_Disabled", "Italics_Undep", "Italics_Dep", "Italics_Grayed", "Italics_Pressed" },
	{ ID_UNDERLINE, 350, TBBORDER+2, 24, 24, -1, 2, "Und_Disabled", "Und_Undep", "Und_Dep", "Und_Grayed", "Und_Pressed" },
    { ID_REVERSE, 380, TBBORDER+2, 24, 24, -1, 2, "Rev_Disabled", "Rev_Undep", "Rev_Dep", "Rev_Grayed", "Rev_Pressed" },
    { ID_HOLLOW, 410, TBBORDER+2, 24, 24, -1, 2, "Hollow_Disabled", "Hollow_Undep", "Hollow_Dep", "Hollow_Grayed", "Hollow_Pressed" },
	{ ID_SHADOW, 440, TBBORDER+2, 24, 24, -1, 2, "Shadow_Disabled", "Shadow_Undep", "Shadow_Dep", "Shadow_Grayed", "Shadow_Pressed" },
    { ID_GLOSSARY, 480, TBBORDER+2, 24, 24, 0, 0, NULL, "Glossary_Undep", "Glossary_Dep", "Glossary_Grayed", "Glossary_Pressed" },
    { ID_SPECIAL, 510, TBBORDER+2, 24, 24, 0, 0, NULL, "Special_Undep", "Special_Dep", "Special_Grayed", "Special_Pressed" },
    { ID_LEFTRIGHT, 550, TBBORDER+2, 24, 24, -1, 0, "LR_Disabled", "LR_Undep", "LR_Dep", "LR_Grayed", "LR_Pressed" },
    { ID_UPDOWN, 580, TBBORDER+2, 24, 24, -1, 0, "UD_Disabled", "UD_Undep", "UD_Dep", "UD_Grayed", "UD_Pressed" }
};



LONG FAR PASCAL MainWinProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL FileParentWinProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL FileWinProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL JeditProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL JtextProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL RulerProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL ConvProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL StatusProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL StatisticsProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FileOpenDlgProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FileSaveFormatProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FileSaveAsDlgProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL DictionaryProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL BigKanjiProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL KanjiInfoProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL JISInputProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL LookupProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL LookupListProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL JISTableDlgProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL JISTableProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL HotKeysProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL ChangeHotKeyProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL OptionsProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL GlossaryProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL GlossaryListProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL AboutProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL SimpleProc (HWND, WORD, WORD, LONG);
extern LONG FAR PASCAL InputModeIconProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL InstalledFontsProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL UserConversionProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL EditUserConversionProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL SearchProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL ReplaceProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL ReplaceDlgProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL PrintProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL PageSetupProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL PrinterSetupProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL SummaryProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL HeaderProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL EditHeaderProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FileFindProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL RunProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL TemplateProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FormatParagraphProc (HWND, WORD, WORD, LONG);


static void InitializeEverything(void)
{
	fileoptions = NULL;

    ReadOptionsFile();
	InitPrinting();
	InitFonts();
	InitConversion();
	ReadGlossary(global.glossary);
	ReadConversionCache(global.convcache);
}



void ErrorMessage(HWND hwnd, char *fmt, ...)
{
	va_list argptr;
	char buffer[MAXLINELEN];

    va_start(argptr, fmt);
    vsprintf(buffer, fmt, argptr);
    va_end(argptr);

    MessageBeep(0);
    MessageBox (hwnd, buffer, "Something is wrong...", MB_ICONEXCLAMATION | MB_OK);
}



int Confirmation (HWND hwnd, char *fmt, ...)
{
    va_list argptr;
    char buffer[MAXLINELEN];

    va_start(argptr, fmt);
	vsprintf(buffer, fmt, argptr);
    va_end(argptr);

    MessageBeep(0);
    return (MessageBox (hwnd, buffer, "Please Confirm...", MB_ICONQUESTION | MB_YESNOCANCEL));
}



int YesNo(HWND hwnd, char *fmt, ...)
{
    va_list argptr;
    char buffer[MAXLINELEN];

    va_start(argptr, fmt);
	vsprintf(buffer, fmt, argptr);
    va_end(argptr);

	MessageBeep(0);
    return (MessageBox (hwnd, buffer, "Please Confirm...", MB_ICONQUESTION | MB_YESNO));
}



BOOL RetryMessage(char *fmt, ...)
{
    va_list argptr;
    char buffer[MAXLINELEN];

    va_start(argptr, fmt);
	vsprintf(buffer, fmt, argptr);
    va_end(argptr);

    MessageBeep(0);
    if (MessageBox (global.hwnd, buffer, "Something is wrong...",
        MB_ICONEXCLAMATION | MB_RETRYCANCEL) == IDCANCEL) return (FALSE);
    else
        return (TRUE);
}



static void AdjustAllChildWindows (int width, int height)
{
    int i;
    int ToolBarHeight = 0, StatusBarHeight = 0;
    RECT rect;

    if (global.tbhwnd != NULL) {
        MoveWindow(global.tbhwnd, -1, -1, width + 2, TBHEIGHT + 2, TRUE);
        ToolBarHeight = TBHEIGHT + 1;
    }

    if (global.statushwnd != NULL) {
        MoveWindow(global.statushwnd, -1, height - STATUSHEIGHT,
					width + 2, STATUSHEIGHT + 2, TRUE);
        StatusBarHeight = STATUSHEIGHT;
    }

    if (global.clienthwnd == NULL || global.convhwnd == NULL) return;


	switch (global.listposition) {
        case 0: MoveWindow(global.clienthwnd, 0, ToolBarHeight, width,
                            height - StatusBarHeight - ToolBarHeight, TRUE);

                if (!(GetWindowLong(global.convhwnd, GWL_STYLE) & WS_POPUP)) {
                    DestroyWindow(global.convhwnd);
                    global.convhwnd = CreateWindow("JWP Conversion Bar", NULL,
                                                    WS_THICKFRAME | WS_VISIBLE | WS_POPUP,
                                                    global.convbar.left, global.convbar.top,
                                                    global.convbar.right, 0,
                                                    global.hwnd, NULL, hInstance, NULL);
                }
                break;

        case 1: i = SYSFONT->height + 2 * BORDERSPACE +
						GetSystemMetrics(SM_CXVSCROLL) +
                        2 * GetSystemMetrics(SM_CYBORDER);

                if (GetWindowLong(global.convhwnd, GWL_STYLE) & WS_POPUP) {
                    DestroyWindow(global.convhwnd);
                    global.convhwnd = CreateWindow("JWP Conversion Bar", NULL,
                                                    WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE | WS_BORDER,
                                                    -1, ToolBarHeight - 1, width + 2, i,
                                                    global.hwnd, 3, hInstance, NULL);
                } else {
                    MoveWindow(global.convhwnd, -1, ToolBarHeight - 1, width + 2, i, TRUE);
                }

                MoveWindow(global.clienthwnd, 0, ToolBarHeight - 1 + i, width,
                            height - StatusBarHeight - ToolBarHeight + 1 - i, TRUE);

                break;

        case 2: i = SYSFONT->height + 2 * BORDERSPACE +
                        GetSystemMetrics(SM_CXVSCROLL) +
                        2 * GetSystemMetrics(SM_CYBORDER);

                if (GetWindowLong(global.convhwnd, GWL_STYLE) & WS_POPUP) {
                    DestroyWindow(global.convhwnd);
                    global.convhwnd = CreateWindow("JWP Conversion Bar", NULL,
                                                    WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE | WS_BORDER,
                                                    -1, height - StatusBarHeight - i, width + 2, i,
                                                    global.hwnd, 3, hInstance, NULL);
                } else {
                    MoveWindow(global.convhwnd, -1, height - StatusBarHeight - i,
                                width + 2, i, TRUE);
                }

                MoveWindow(global.clienthwnd, 0, ToolBarHeight, width,
                            height - StatusBarHeight - i - ToolBarHeight, TRUE);
                break;

	}

    InvalidateRect(global.hwnd, NULL, TRUE);
    InvalidateRect(global.clienthwnd, NULL, TRUE);
}



int PASCAL WinMain (HANDLE hThisInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
	MSG                 msg;
	HDC                 hdc;
	WNDCLASS            wndclass;
    RECT                rect;
    HCURSOR             hcursor;
	CLIENTCREATESTRUCT  clientcreate;
    HWND                signonhwnd;
    HANDLE              haccel;
    extern BOOL         Read3DDialogOption(void);


	if (!(GetWinFlags() & WF_PMODE)) {
        ErrorMessage(global.hwnd, "This program requires protected mode.");
        return (-1);
    }

    hInstance = hThisInstance;

    hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);

	FullMenu = LoadMenu(hInstance, "FileMenu");
    MinimalMenu = LoadMenu(hInstance, "NoFileMenu");


#ifdef CTL3D
    /* 3D dialogs */

    Dialogs3D = Read3DDialogOption();

    if (Dialogs3D) {
        Dialogs3D = Ctl3dRegister(hInstance);
        if (Dialogs3D) Ctl3dAutoSubclass(hInstance);
    }
#endif CTL3D


    /* Register window classes */

	if (!hPrevInstance) {
		wndclass.style          = CS_HREDRAW | CS_VREDRAW;
		wndclass.lpfnWndProc    = MainWinProc;
		wndclass.cbClsExtra     = 0;
		wndclass.cbWndExtra     = 0;
        wndclass.hInstance      = hThisInstance;
        wndclass.hIcon          = LoadIcon(hThisInstance, "Icon1");
		wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
        wndclass.hbrBackground  = COLOR_APPWORKSPACE + 1;
        wndclass.lpszMenuName   = NULL;
		wndclass.lpszClassName  = "JWP Frame";

        RegisterClass(&wndclass);

        wndclass.cbWndExtra     = sizeof(FILEOPTIONS *);
		wndclass.lpfnWndProc    = FileParentWinProc;
        wndclass.hIcon          = LoadIcon(hThisInstance, "DocIcon");
        wndclass.hbrBackground  = COLOR_WINDOW + 1;
		wndclass.lpszClassName  = "JWP File Parent";

		RegisterClass(&wndclass);

        wndclass.style          = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
		wndclass.lpfnWndProc    = FileWinProc;
		wndclass.hCursor        = LoadCursor(NULL, IDC_IBEAM);
        wndclass.lpszClassName  = "JWP File";

		RegisterClass(&wndclass);

        wndclass.style          = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
        wndclass.cbWndExtra     = sizeof(FILEOPTIONS *) + 2 * sizeof(WORD) + sizeof(HWND);
        wndclass.lpfnWndProc    = JeditProc;
        wndclass.hIcon          = NULL;
		wndclass.lpszClassName  = "JWP Japanese Edit Control";

		RegisterClass(&wndclass);

        wndclass.cbWndExtra     = 0;
        wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
		wndclass.lpfnWndProc    = RulerProc;
		wndclass.lpszClassName  = "JWP Ruler";

		RegisterClass(&wndclass);

        wndclass.cbWndExtra     = 0;
		wndclass.lpfnWndProc    = LookupListProc;
        wndclass.lpszClassName  = "JWP Kanji Lookup List";

        RegisterClass(&wndclass);

        wndclass.cbWndExtra     = sizeof(KANJI *) + sizeof(int) + sizeof(BOOL);
        wndclass.lpfnWndProc    = JtextProc;
        wndclass.hbrBackground  = NULL;
        wndclass.lpszClassName  = "JWP Text Box";

        RegisterClass(&wndclass);

        wndclass.cbWndExtra     = 0;
		wndclass.lpfnWndProc    = JISTableProc;
        wndclass.lpszClassName  = "JWP JIS Table";

		RegisterClass(&wndclass);

        wndclass.style          = CS_HREDRAW | CS_VREDRAW;
        wndclass.cbWndExtra     = sizeof(WORD);
		wndclass.lpfnWndProc    = BigKanjiProc;
        wndclass.lpszClassName  = "JWP Big Kanji";

        RegisterClass(&wndclass);

        wndclass.cbWndExtra     = sizeof(HWND);
        wndclass.lpfnWndProc    = InputModeIconProc;
        wndclass.lpszClassName  = "JWP Input Mode Icon";

		RegisterClass(&wndclass);

        wndclass.cbWndExtra     = 0;
		wndclass.lpfnWndProc    = ConvProc;
        wndclass.hbrBackground  = NULL;
        wndclass.lpszClassName  = "JWP Conversion Bar";

		RegisterClass(&wndclass);

		wndclass.lpfnWndProc    = StatusProc;
        wndclass.lpszClassName  = "JWP Status";

		RegisterClass(&wndclass);
	}


    global.clienthwnd = NULL;

    {
        BOOL maximized;
        int x, y, width, length;

        maximized = ReadProgramSizeOptions (&x, &y, &width, &length);

        if (maximized) {
			global.hwnd = CreateWindow("JWP Frame", PROGNAME,
                                        WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_MAXIMIZE,
                                        CW_USEDEFAULT, CW_USEDEFAULT, width, length,
                                        NULL, NULL, hThisInstance, NULL);
            nCmdShow = SW_SHOWMAXIMIZED;
        } else {
            global.hwnd = CreateWindow("JWP Frame", PROGNAME,
                                        WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
										x, y, width, length,
                                        NULL, NULL, hThisInstance, NULL);
        }
    }

	ShowWindow(global.hwnd, nCmdShow);
    UpdateWindow(global.hwnd);


    /* Create the sign-on message */

    signonhwnd = CreateDialog(hInstance, "SignOn", global.hwnd, SimpleProc);


    /* Load accelerators */

    haccel = LoadAccelerators(hInstance, "Accelerator");
    

    /* Initialize variables */

    global.mode = M_KANA;
    global.active = NULL;
	global.convsel = -1;
    global.glisthwnd = NULL;

	global.cpos = (POINT far *) BlockAlloc(CPOSPAGESIZE);
    global.cpossize = CPOSPAGESIZE;
	global.cpospara = NULL;
    global.cposcount = 0;

    InitializeEverything();

    global.leading = (int) ((double) PRINTFONT->leading * global.dispscale);
    if (global.leading % 2 != 0) global.leading++;

    global.spacing = (int) ((double) PRINTFONT->spacing * global.dispscale);
    if (global.spacing % 2 != 0) global.spacing++;

    SwitchMenu(0);      /* Full menu */


    /* Take away the sign-on message */

    DestroyWindow(signonhwnd);


    /* Others */

    global.gloslistdlg = MakeProcInstance(GlossaryListProc, hInstance);

	hdc = GetDC(global.hwnd);
    GetTextMetrics(hdc, &(global.textmetric));
	global.borderspace = AVGWIDTH;
	ReleaseDC(global.hwnd, hdc);


    if (global.draftview) CheckMenuItem(hmenu, IDM_VIEWDRAFT, MF_CHECKED | MF_BYCOMMAND);
    if (global.showstatus) SendMessage(global.hwnd, WM_COMMAND, IDM_VIEWSTATUS, 0L);
    if (global.showribbon) SendMessage(global.hwnd, WM_COMMAND, IDM_VIEWRIBBON, 0L);
    if (global.showruler) CheckMenuItem(hmenu, IDM_VIEWRULER, MF_CHECKED | MF_BYCOMMAND);
    if (global.showspecial) SendMessage(global.tbhwnd, BM_SETSTATE, TRUE, ID_SPECIAL);

	/* Create the conversion bar */

	switch (global.listposition) {
        case 0: global.convhwnd = CreateWindow("JWP Conversion Bar", "Kanji List",
                                                WS_VISIBLE | WS_THICKFRAME | WS_POPUP,
                                                0, 0, 0, 0, global.hwnd, NULL, hInstance, NULL);
                ShowWindow(global.convhwnd, SW_SHOW);
				break;
		case 1:
		case 2: global.convhwnd = CreateWindow("JWP Conversion Bar", NULL,
												WS_CHILD | WS_VISIBLE | WS_BORDER,
												0, 0, 0, 0, global.hwnd, 3, hInstance, NULL);
				break;
	}


	/* Create the Client window */

    clientcreate.hWindowMenu = GetSubMenu(FullMenu, 6);
	clientcreate.idFirstChild = ID_FIRSTFILE;

    global.clienthwnd = CreateWindow ("MDICLIENT", NULL,
                                      WS_CHILD | WS_VISIBLE,
									  0, 0, 0, 0, global.hwnd, 1, hInstance,
									  (LPSTR) &clientcreate);

    /* Create the glossary list */

    if (global.glistvisible) SendMessage(global.hwnd, WM_COMMAND, IDM_VIEWGLOSSARY, 0L);

    SetActiveWindow(global.hwnd);


    GetClientRect(global.hwnd, &rect);
    AllInitialized = TRUE;
    AdjustAllChildWindows(rect.right, rect.bottom);
    UpdateWindow(global.hwnd);

    ShowCursor(FALSE);
    SetCursor(hcursor);


    /* Now create an empty file window? */

    {
        char buffer[MAXLINELEN];
        char word[MAXLINELEN];
        char remains[MAXLINELEN];

        word[0] = '\0';
        remains[0] = '\0';
        _fstrcpy(buffer, lpszCmdParam);
        sscanf(buffer, "%s %[^\n]", word, remains);

        if (!word[0]) {
            global.active = NewFile (FN_NORMAL, TRUE);

            if (global.active != NULL) {
                char buffer[50];

                global.active->filename[0] = '\0';
                sprintf(buffer, "Untitled #%d", ++Untitled);
                ShowFileWindow(buffer, global.active);
            }
        } else {
            do {
                if (!DoFileOpen(word)) {
                    ErrorMessage(global.hwnd, "Cannot open file '%s'.", word);
                }

                strcpy(buffer, remains);
                word[0] = '\0';
                remains[0] = '\0';
                sscanf(buffer, "%s %[^\n]", word, remains);
            } while (word[0]);
        }
    }


    if (Dialogs3D != global.ctl3d) {
        ErrorMessage(global.hwnd, PROGNAME " cannot create 3D dialog boxes!  "
                                  "Maybe you are not running Windows 3.1?");
    }


    /* Message loop */

	while (GetMessage(&msg, NULL, 0, 0)) {
        if (!TranslateAccelerator (global.hwnd, haccel, &msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
	}

#ifdef CTL3D
	if (Dialogs3D) Ctl3dUnregister(hInstance);
#endif CTL3D

	return (msg.wParam);
}



void CenterDialogBox (HWND hwnd)
{
    RECT rect;

    GetWindowRect(hwnd, &rect);
    OffsetRect(&rect, -rect.left, -rect.top);
    MoveWindow(hwnd, ((GetSystemMetrics(SM_CXSCREEN) - rect.right ) / 2 + 4) & ~7,
                     (GetSystemMetrics(SM_CYSCREEN) - rect.bottom) / 2,
                     rect.right, rect.bottom, FALSE);
}



BOOL FAR PASCAL CloseAllEnumProc (HWND hwnd, LONG lParam)
{
    FILEOPTIONS *f;

	if (GetWindow(hwnd, GW_OWNER)) return (TRUE);      /* Check for icon title */

    for (f = fileoptions; f != NULL; f = f->next) if (f->parent == hwnd) break;

    if (f == NULL) return (TRUE);       /* Not a valid window */


    SendMessage(global.clienthwnd, WM_MDIMAXIMIZE, hwnd, 0L);

    if (!SendMessage(hwnd, WM_QUERYENDSESSION, 0, 0L)) return (FALSE);

    SendMessage(global.clienthwnd, WM_MDIDESTROY, hwnd, 0L);

    return (TRUE);
}



static BOOL RedisplayFile (FILEOPTIONS *f)
{
    HDC hdc, hdcmem;
    HBITMAP hbitmap;
    HBRUSH hbrush;
    RECT rect;

	if (IsIconic(f->hwnd) || IsIconic(f->parent)) return (FALSE);

    hdc = GetDC(f->hwnd);
    hdcmem = CreateCompatibleDC(hdc);
    SetTextColor(hdcmem, GetSysColor(COLOR_WINDOWTEXT));
    SetBkColor(hdcmem, GetSysColor(COLOR_WINDOW));

    GetClientRect(f->hwnd, &rect);

    hbitmap = CreateCompatibleBitmap (hdc, rect.right, rect.bottom);
    if (hbitmap == NULL) {
        InvalidateRect(f->hwnd, NULL, TRUE);
        UpdateWindow(f->hwnd);
        return (FALSE);
    }

    SelectObject(hdcmem, hbitmap);
    hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
    FillRect(hdcmem, &rect, hbrush);
    DeleteObject(hbrush);

    HideCaret(f->hwnd);
    RedrawFile(f, hdcmem, 0, 0, &rect, global.showspecial);

    BitBlt(hdc, 0, 0, rect.right, rect.bottom, hdcmem, 0, 0, SRCCOPY);

    if (f == global.active) {
        if (FindCaret(f, TRUE))
            DoCaret(f, CURX(f), CURY(f) - CURLINE(f)->height, TRUE);
        else
            DoCaret(f, 0, 0, FALSE);
    }

    ShowCaret(f->hwnd);

    ReleaseDC(f->hwnd, hdc);
    DeleteDC(hdcmem);
    DeleteObject(hbitmap);

    return (TRUE);
}



LONG FAR PASCAL MainWinProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	int         i;
	RECT		rect;
    char        buffer[15], far *cp;
	HCURSOR		hcursor;
    FILEOPTIONS *f;
    static      LastSearchIsReplace = FALSE;


	/* Takes care of menu items */

	switch (message) {

    case WM_MOVE:
        if (AllInitialized) {
            GetWindowRect(hwnd, &rect);

            if (global.glisthwnd != NULL) {
                SetWindowPos(global.glisthwnd, NULL,
                             rect.left + global.glistsize.left,
                             rect.top + global.glistsize.top,
                             0, 0, SWP_NOSIZE | SWP_NOZORDER);
            }
            if (global.convhwnd != NULL && global.listposition == 0) {
                SetWindowPos(global.convhwnd, NULL,
                             rect.left + global.convbar.left,
                             rect.top + global.convbar.top,
                             0, 0, SWP_NOSIZE | SWP_NOZORDER);
            }

            SetActiveWindow(hwnd);
        }
        break;

	case WM_SIZE:
        if (AllInitialized) {
            AdjustAllChildWindows (LOWORD(lParam), HIWORD(lParam));

            for (f = fileoptions; f != NULL; f = f->next) {
                SELPARA1(f) = SELPARA2(f) = NULL;
                SELPOS1(f) = SELPOS2(f) = 0;
                SELTYPE(f) = SEL_SELECTION;
                SELNOW(f) = FALSE;
            }
        }
        return (0);

	case WM_KEYDOWN:
    case WM_CHAR:
        if (global.active != NULL)
            SendMessage(global.active->hwnd, message, wParam, lParam);
		return (0);

    case WM_INITMENUPOPUP: {
        if (hmenu == MinimalMenu) {
            /* Nothing */
        } else if (hmenu == FullMenu) {
            if (global.active == NULL) return (0);

            switch (LOWORD(lParam)) {
                case 2:     /* Edit */
                    if (SELPARA1(global.active) == NULL) {
                        EnableMenuItem(hmenu, IDM_EDITCUT, MF_BYCOMMAND | MF_GRAYED);
                        EnableMenuItem(hmenu, IDM_EDITCOPY, MF_BYCOMMAND | MF_GRAYED);
                    } else {
                        EnableMenuItem(hmenu, IDM_EDITCUT, MF_BYCOMMAND | MF_ENABLED);
                        EnableMenuItem(hmenu, IDM_EDITCOPY, MF_BYCOMMAND | MF_ENABLED);
                    }
                    if (ClipboardFull()) {
                        EnableMenuItem(hmenu, IDM_EDITPASTE, MF_BYCOMMAND | MF_ENABLED);
                    } else {
                        EnableMenuItem(hmenu, IDM_EDITPASTE, MF_BYCOMMAND | MF_GRAYED);
                    }
                    if (global.active->undo == NULL) {
                        EnableMenuItem(hmenu, IDM_EDITUNDO, MF_BYCOMMAND | MF_GRAYED);
                    } else {
                        EnableMenuItem(hmenu, IDM_EDITUNDO, MF_BYCOMMAND | MF_ENABLED);
                    }
                    if (global.active->redo == NULL) {
                        EnableMenuItem(hmenu, IDM_EDITREDO, MF_BYCOMMAND | MF_GRAYED);
                    } else {
                        EnableMenuItem(hmenu, IDM_EDITREDO, MF_BYCOMMAND | MF_ENABLED);
                    }
                    break;

                case 4:     /* Kanji */
                    if (SELPARA1(global.active) == NULL) {
                        EnableMenuItem(hmenu, IDM_KANJIINFO, MF_BYCOMMAND | MF_GRAYED);
                    } else {
                        EnableMenuItem(hmenu, IDM_KANJIINFO, MF_BYCOMMAND | MF_ENABLED);
                    }
                    if (SELPARA1(global.active) == NULL) {
                        EnableMenuItem(hmenu, IDM_KANJICONVERT, MF_BYCOMMAND | MF_GRAYED);
                    } else {
                        EnableMenuItem(hmenu, IDM_KANJICONVERT, MF_BYCOMMAND | MF_ENABLED);
                    }
                    break;
            }
        }
        return (0);
    }

	case WM_COMMAND:

        if (HIBYTE(wParam) == ID_TOOLBAR) {
            switch (LOBYTE(wParam)) {
                case ID_GLOSSARY:
                    SendMessage(hwnd, WM_COMMAND, IDM_VIEWGLOSSARY, 0L);
                    break;

                case ID_SPECIAL:
                    SendMessage(hwnd, WM_COMMAND, IDM_VIEWSPECIAL, 0L);
                    break;

                /*
				case ID_LEFTRIGHT:
                    i = SendMessage(global.tbhwnd, BM_GETSTATE, 0, ID_LEFTRIGHT);
					if (!i) {
                        SendMessage(global.tbhwnd, BM_SETSTATE, TRUE, ID_LEFTRIGHT);
                        SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_UPDOWN);
					}
					break;

				case ID_UPDOWN:
                    i = SendMessage(global.tbhwnd, BM_GETSTATE, 0, ID_UPDOWN);
					if (!i) {
                        SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_LEFTRIGHT);
                        SendMessage(global.tbhwnd, BM_SETSTATE, TRUE, ID_UPDOWN);
					}
					break;
                */

                case ID_KANA:
                    if (global.mode != M_KANA) ToggleInputMode();
                    if (global.active != NULL)
                        TakeCareOfThings(global.active, FALSE);
					break;

                case ID_ASCII:
                    if (global.mode != M_ASCII) ToggleInputMode();
                    if (global.active != NULL)
                        TakeCareOfThings(global.active, FALSE);
					break;

				case ID_CONVERT:
                    if (global.active != NULL) {
                        SendMessage(global.active->hwnd, WM_COMMAND, IDM_KANJICONVERT, 0L);
                    }
					break;
			}

			return (0);
        }


		switch (wParam) {
            case IDM_EDITUNDO:
            case IDM_EDITREDO:
            case IDM_EDITCUT:
            case IDM_EDITCOPY:
            case IDM_EDITPASTE:
            case IDM_EDITCLEAR:
            case IDM_EDITSELECTALL:
            case IDM_KANJICONVERT:
            case IDM_KANJILCONVERT:
			case IDM_KANJIINFO:
                if (global.active != NULL)
                    SendMessage(global.active->hwnd, WM_COMMAND, wParam, 0L);
                return (0);

            case IDM_TOGGLEMODE:
                ToggleInputMode ();
                if (global.active != NULL)
                    TakeCareOfThings(global.active, TRUE);
                return (0);

			case IDM_TOGGLEGLOSSARY:
                global.dynamicglossary = !global.dynamicglossary;
				return (0);

            case IDM_FILENEW: {
                extern char far *TplChosen;

                i = DialogBox (hInstance, "FileNew", hwnd, TemplateProc);

                switch (i) {
                    case -2: return (0);    /* Cancel */
                    case -1: break;         /* Use None */
                    default: if (TplChosen != NULL && TplChosen[0]) {
                                CreateUsingTemplate(TplChosen, ++Untitled);
                                return (0);
                             }
                             break;
                }

                global.active = NewFile (FN_NORMAL, TRUE);
                if (global.active != NULL) {
                    global.active->filename[0] = '\0';
					sprintf(buffer, "Untitled #%d", ++Untitled);
                    ShowFileWindow(buffer, global.active);
                } else {
                    ErrorMessage(global.hwnd, "Cannot open any more file windows!");
                }
				return (0);
            }

			case IDM_FILEOPEN:
                DoFileOpen(NULL);
				return (0);

			case IDM_FILECLOSE:
                if (global.active != NULL) {
                    if (!SendMessage(global.active->parent, WM_QUERYENDSESSION, 0, 0L))
                        return (0);

                    global.active->changed = FALSE;

                    SendMessage(global.clienthwnd, WM_MDIDESTROY, global.active->parent, 0L);
                }
                return (0);

			case IDM_FILESAVE:
				DoFileSaveAs(FALSE);
                return (0);

			case IDM_FILESAVEAS:
                DoFileSaveAs(TRUE);
				return (0);

            case IDM_FILEPRINT:
				if (global.active != NULL) {
                    i = DialogBox (hInstance, "Print", hwnd, PrintProc);

                    hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
                    ShowCursor(TRUE);
                    if (i) PrintFile (global.active);
                    ShowCursor(FALSE);
                    SetCursor(hcursor);
                }
                return (0);

            case IDM_FILEPRINTER:
                i = DialogBox (hInstance, "PrinterSetup", hwnd, PrinterSetupProc);

                if (i) ChangePrinterOrFonts();
                return (0);

            case IDM_FILESUMMARY:
                if (global.active != NULL) {
                    DialogBox (hInstance, "Summary", hwnd, SummaryProc);
                }
                return (0);


            case IDM_FILEHEADERS:
                if (global.active != NULL) {
                    DialogBox (hInstance, "Header", hwnd, HeaderProc);
                }
                return (0);

            case IDM_FILEFIND:
                DialogBox (hInstance, "FileFind", hwnd, FileFindProc);
                return (0);

			case IDM_FILEEXIT:
                if (SendMessage(hwnd, WM_COMMAND, IDM_WINDOWCLOSE, 0L)) {
                    SendMessage(hwnd, WM_DESTROY, 0, 0L);
                }
				return (0);

            case IDM_FILEABOUT:
                DialogBox (hInstance, "About", hwnd, AboutProc);
                return (0);

            case IDM_VIEWDRAFT: {
                FILEOPTIONS *f;
                HCURSOR hcursor;

                if (global.draftview) {
                    CheckMenuItem(hmenu, IDM_VIEWDRAFT, MF_UNCHECKED | MF_BYCOMMAND);
                    global.draftview = FALSE;
				} else {
                    CheckMenuItem(hmenu, IDM_VIEWDRAFT, MF_CHECKED | MF_BYCOMMAND);
                    global.draftview = TRUE;
                }

				/* Try to redraw the entire window */

                hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
                ShowCursor(TRUE);

                for (f = fileoptions; f != NULL; f = f->next) {
                    ReformatFile(f);
                    RedisplayFile(f);
                }

                ShowCursor(FALSE);
                SetCursor(hcursor);

                return (0);
            }

            case IDM_VIEWRIBBON: {
                RECT rect;

                GetClientRect(global.hwnd, &rect);

                if (global.tbhwnd != NULL) {
                    CheckMenuItem(hmenu, IDM_VIEWRIBBON, MF_UNCHECKED | MF_BYCOMMAND);
                    DestroyWindow(global.fonthwnd);
                    DestroyWindow(global.tbhwnd);
                    global.tbhwnd = global.fonthwnd = NULL;
                } else {
                    CheckMenuItem(hmenu, IDM_VIEWRIBBON, MF_CHECKED | MF_BYCOMMAND);

                    global.tbhwnd = CreateToolbar (global.hwnd, -1, -1, rect.right + 2, TBHEIGHT + 2,
                                                    1, ID_TOOLBAR, sizeof(toolbar) / sizeof(TOOLBARICON),
                                                    hInstance, toolbar, "X_Cursor");

                    /* Create the font list in the toolbar */

                    global.fonthwnd = CreateWindow("combobox", "Font List", WS_CHILD | WS_VISIBLE | CBS_DROPDOWNLIST,
                                                    2 * TBBORDER, TBBORDER+2, 160 - 2 * TBBORDER, 200, global.tbhwnd, ID_FONTBOX, hInstance, NULL);
                    EnableWindow(global.fonthwnd, FALSE);
                }

                AdjustAllChildWindows (rect.right, rect.bottom);
                return (0);
            }

            case IDM_VIEWSTATUS: {
                RECT rect;

                GetClientRect(global.hwnd, &rect);

                if (global.statushwnd != NULL) {
                    CheckMenuItem(hmenu, IDM_VIEWSTATUS, MF_UNCHECKED | MF_BYCOMMAND);
                    DestroyWindow(global.statushwnd);
                    global.statushwnd = NULL;
                } else {
                    CheckMenuItem(hmenu, IDM_VIEWSTATUS, MF_CHECKED | MF_BYCOMMAND);
                    global.statushwnd = CreateWindow ("JWP Status", "Status Bar",
                                                      WS_CHILD | WS_VISIBLE | WS_BORDER,
                                                      0, 0, 0, 0, global.hwnd, 2, hInstance, NULL);
                }

                AdjustAllChildWindows (rect.right, rect.bottom);
                return (0);
            }

            case IDM_VIEWRULER: {
                BOOL On;
                FILEOPTIONS *f;

                On = !global.showruler;
                global.showruler = On;

                if (On) {
                    CheckMenuItem(hmenu, IDM_VIEWRULER, MF_CHECKED | MF_BYCOMMAND);
                } else {
                    CheckMenuItem(hmenu, IDM_VIEWRULER, MF_UNCHECKED | MF_BYCOMMAND);
                }

                for (f = fileoptions; f != NULL; f = f->next) {
                    DisplayRuler(f, On);
                    //InvalidateRect(f->parent, NULL, TRUE);
                }
                return (0);
            }

            case IDM_VIEWGLOSSARY:
                if (global.glisthwnd != NULL) {
                    SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_GLOSSARY);
                    CheckMenuItem(hmenu, IDM_VIEWGLOSSARY, MF_UNCHECKED | MF_BYCOMMAND);
                    DestroyWindow(global.glisthwnd);
					global.glisthwnd = NULL;
				} else {
                    SendMessage(global.tbhwnd, BM_SETSTATE, TRUE, ID_GLOSSARY);
					CheckMenuItem(hmenu, IDM_VIEWGLOSSARY, MF_CHECKED | MF_BYCOMMAND);
                    global.glisthwnd = CreateDialog(hInstance, "GlossaryList", hwnd, global.gloslistdlg);
                    SetActiveWindow(hwnd);
                }
                return (0);

            case IDM_VIEWSPECIAL: {
                FILEOPTIONS *f;
                HCURSOR hcursor;

				if (global.showspecial) {
					SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_SPECIAL);
                    CheckMenuItem(hmenu, IDM_VIEWSPECIAL, MF_UNCHECKED | MF_BYCOMMAND);
					global.showspecial = FALSE;
				} else {
					SendMessage(global.tbhwnd, BM_SETSTATE, TRUE, ID_SPECIAL);
					CheckMenuItem(hmenu, IDM_VIEWSPECIAL, MF_CHECKED | MF_BYCOMMAND);
					global.showspecial = TRUE;
                }

				/* Try to redraw the entire window */

                hcursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
                ShowCursor(TRUE);
                for (f = fileoptions; f != NULL; f = f->next) RedisplayFile(f);
                ShowCursor(FALSE);
                SetCursor(hcursor);
                return (0);
            }

            case IDM_EDITSEARCH:
                if (global.active == NULL) return (0);
                i = DialogBox (hInstance, "Search", hwnd, SearchProc);
                if (i) {
                    DoSearch();
                    LastSearchIsReplace = FALSE;
                }
                return (0);

            case IDM_CONTINUESEARCH:
                if (global.active == NULL) return (0);
                if (LastSearchIsReplace) DoReplace(); else DoSearch();
                return (0);

            case IDM_EDITREPLACE:
                if (global.active == NULL) return (0);
                i = DialogBox (hInstance, "Replace", hwnd, ReplaceProc);
                if (i) {
                    DoReplace();
                    LastSearchIsReplace = TRUE;
                }
                return (0);

            case IDM_WINDOWTILE:
                SendMessage(global.clienthwnd, WM_MDITILE, 0, 0L);
                return (0);

			case IDM_WINDOWCASCADE:
				SendMessage(global.clienthwnd, WM_MDICASCADE, 0, 0L);
                return (0);

			case IDM_WINDOWARRANGE:
				SendMessage(global.clienthwnd, WM_MDIICONARRANGE, 0, 0L);
                return (0);

			case IDM_WINDOWCLOSE: {
                BOOL DoIt;

				/* Any file opened? */
				if (fileoptions == NULL) return (TRUE);

                DoIt = EnumChildWindows(global.clienthwnd, CloseAllEnumProc, 0L);

				if (DoIt) SwitchMenu (1);     /* Minimal menu */
				return (DoIt);
			}

			case IDM_UTILITIESSTAT:
                DialogBox (hInstance, "Statistics", hwnd, StatisticsProc);
                return (0);

            case IDM_KANJIINPUT:
                if (global.active != NULL) curfile = global.active;
				else curfile = NULL;

                DialogBox (hInstance, "KanjiInput", global.hwnd, JISInputProc);
                return (0);

            case IDM_KANJILOOKUP:
                if (global.active != NULL) curfile = global.active;
                else curfile = NULL;

                DialogBox (hInstance, "KanjiLookup", global.hwnd, LookupProc);
				return (0);

            case IDM_KANJITABLE:
                if (global.active != NULL) curfile = global.active;
                else curfile = NULL;

                DialogBox (hInstance, "JISTable", global.hwnd, JISTableDlgProc);
                return (0);

            case IDM_UTILITIESOPTIONS:
                DialogBox (hInstance, "Options", hwnd, OptionsProc);

                /* 3D dialog boxes */

#ifdef CTL3D
                if (Dialogs3D != global.ctl3d) {
                    if (Dialogs3D) {
                        Ctl3dUnregister(hInstance);
                        Dialogs3D = FALSE;
                    } else {
                        Dialogs3D = Ctl3dRegister(hInstance);
                        if (Dialogs3D) {
                            Ctl3dAutoSubclass(hInstance);
                        } else {
                            ErrorMessage(global.hwnd, PROGNAME " cannot create 3D dialog boxes!  "
                                                      "Maybe you are not running Windows 3.1?");
                            global.ctl3d = FALSE;
                        }
                    }
                }
#endif CTL3D

                return (0);

            case IDM_UTILITIESDICT: {
                INPUTMODE oldmode;

				oldmode = global.mode;
                DialogBox (hInstance, "Dictionary", hwnd, DictionaryProc);

				if (oldmode != global.mode) ToggleInputMode();

                return (0);
            }

            case IDM_UTILITIESHOTKEYS:
                DialogBox (hInstance, "HotKeys", hwnd, HotKeysProc);
                return (0);

            case IDM_UTILITIESGLOSSARY: {
                INPUTMODE oldmode;
				BOOL dynamic;

                oldmode = global.mode;
                dynamic = global.dynamicglossary;
				global.dynamicglossary = FALSE;

                DialogBox (hInstance, "Glossary", hwnd, GlossaryProc);

				if (oldmode != global.mode) ToggleInputMode();
                global.dynamicglossary = dynamic;

				return (0);
            }

            case IDM_UTILITIESCONVERT:
                DialogBox (hInstance, "UserDict", hwnd, UserConversionProc);
                return (0);

            case IDM_UTILITIESFONTS:
                i = DialogBox (hInstance, "KanjiFonts", hwnd, InstalledFontsProc);

                if (i) ChangePrinterOrFonts();
				return (0);

            case IDM_FILERUN:
                DialogBox (hInstance, "RunProg", hwnd, RunProc);
                return (0);

            case IDM_FILE1:
            case IDM_FILE2:
            case IDM_FILE3:
            case IDM_FILE4: {
                char *cp;
                extern char *GetQuickFile(int);

                cp = GetQuickFile(wParam - IDM_FILE1);
                if (cp != NULL) {
                    if (!DoFileOpen(cp)) {
                        ErrorMessage(global.hwnd, "Cannot open file '%s'.", cp);
                    }
                }
                return (0);
            }

            case IDM_FORMATPAGE: {
                FILEOPTIONS *f;

                i = DialogBox (hInstance, "PageSetup", hwnd, PageSetupProc);

                switch (i) {
                    case 0: break;

                    case 1: if (global.active != NULL) {
                                global.active->linelen = CalcLineLength(global.active);
                                ReformatFile(global.active);
                                InvalidateRect(global.active->hwnd, NULL, TRUE);
                            }
                            break;

                    case 2: for (f = fileoptions; f != NULL; f = f->next) {
                                f->linelen = CalcLineLength(f);
                                ReformatFile(f);
                                InvalidateRect(f->hwnd, NULL, TRUE);
                            }
                            break;
                }

                return (0);
            }

            case IDM_FORMATPARAGRAPH: {
                POSITION p;

                if (global.active == NULL) return (0);
                i = DialogBox (hInstance, "FormatParagraph", hwnd, FormatParagraphProc);

                if (i) {
                    if (SELPARA1(global.active) != NULL) {
                        PARAOF(p) = SELPARA1(global.active);
                        LINEOF(p) = PARAOF(p)->lines;
                        POSOF(p) = 0;

                        ReformatParagraph(global.active, p, SELPARA2(global.active)->next, OP_UPDATE | OP_MOVETOEND);
                    } else {
                        PARAOF(p) = CURPARA(global.active);
                        LINEOF(p) = PARAOF(p)->lines;
                        POSOF(p) = 0;

                        ReformatParagraph(global.active, p, PARAOF(p)->next, OP_UPDATE | OP_MOVETOEND);
                    }
                    SetReformatProc(NULL);
                }
                return (0);
            }

            case IDM_HELPINDEX:
            case IDM_HELPINTRODUCTION:
            case IDM_HELPENTERTEXT:
            case IDM_HELPMOVING:
            case IDM_HELPKANJIS:
            case IDM_HELPSPECIAL:
            case IDM_HELPFILES:
            case IDM_HELPPRINTING:
            case IDM_HELPTECHSUPPORT: {
                switch (wParam) {
                    case IDM_HELPINDEX:
                        WinHelp(global.hwnd, global.help, HELP_INDEX, NULL);
                        return (0);
                    case IDM_HELPONHELP:
                        WinHelp(global.hwnd, NULL, HELP_HELPONHELP, NULL);
                        //WinHelp(global.hwnd, "C:\\WINDOWS\\WINHELP.HLP", HELP_INDEX, NULL);
                        return (0);
                    case IDM_HELPSEARCH:
						WinHelp(global.hwnd, global.help, HELP_KEY, NULL);
                        return (0);
                    case IDM_HELPINTRODUCTION:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_Introduction);
                        return (0);
                    case IDM_HELPENTERTEXT:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_EnterText);
                        return (0);
                    case IDM_HELPMOVING:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_MovingAround);
                        return (0);
                    case IDM_HELPKANJIS:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_ManagingKanjis);
                        return (0);
                    case IDM_HELPSPECIAL:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_SpecialFeatures);
                        return (0);
                    case IDM_HELPFILES:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_WorkingWithFiles);
                        return (0);
                    case IDM_HELPPRINTING:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_Printing);
                        return (0);
                    case IDM_HELPTECHSUPPORT:
                        WinHelp(global.hwnd, global.help, HELP_CONTEXT, Topic_TechnicalSupport);
                        return (0);
                }
            }
        }
        break;

    case WM_SYSCOMMAND:
        switch (wParam & 0xfff0) {
            case SC_CLOSE:
                if (!SendMessage(hwnd, WM_COMMAND, IDM_WINDOWCLOSE, 0L)) return (0);
                break;
        }
        break;

    case WM_WININICHANGE: {
        HDC hdc;
        char buf[MAXLINELEN];

        cp = (char far *) lParam;
        if (cp == NULL || !_fstricmp(cp, "windows")) {
            hdc = GetPrinterDC(TRUE, buf);
            if (hdc == NULL) {
                FreeMem(global.printer);
                global.printer = "(None)";
                break;
            }
            if (stricmp(buf, global.printer)) ChangePrinterOrFonts();
            DeleteDC(hdc);
        }

        break;
    }


	case WM_DESTROY:
        if (!SendMessage(hwnd, WM_COMMAND, IDM_WINDOWCLOSE, 0L)) return (0);

        FreeProcInstance (global.gloslistdlg);

		WriteOptionsFile(FALSE);
        WriteGlossary(global.glossary);
        WriteConversionCache(global.convcache);
        WriteUserConversions(global.userdict);

		FreeAllMemory();

        /* Free some ad hoc stuff */

        DeleteObject(DefAsciiFont.hfont);       /* ASCII font */
        DeleteObject(FullMenu);                 /* The menus */
        DeleteObject(MinimalMenu);
        DisplayKanjiBitmap(NULL, 0, 0, 0, 0, -1, 0, NULL);

        WinHelp(global.hwnd, global.help, HELP_QUIT, NULL);
        PostQuitMessage(0);
		return (0);

	case WM_RENDERFORMAT:
    case WM_DESTROYCLIPBOARD:
	case WM_RENDERALLFORMATS:
    case WM_ASKCBFORMATNAME:
    case WM_SIZECLIPBOARD:
    case WM_PAINTCLIPBOARD:
    case WM_HSCROLLCLIPBOARD:
    case WM_VSCROLLCLIPBOARD:
        return (ProcessClipboardMessage(hwnd, message, wParam, lParam));
    }

	return (DefFrameProc(hwnd, global.clienthwnd, message, wParam, lParam));
}
