/*************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPLIB.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/3/30
**
** DESCRIPTION: C-library for the IP package.
**              Includes conversion routines, checksum calculation etc.
**              Machine dependent code is concentrated in four routines:
**              htons, htonl, ntohs, ntohl.
**              These routines convert machine byte order to network
**              byte order and vice versa.
**
**************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** CHANGER:     $Author:   etstjan  $
** WORKFILE:    $Workfile:   IPLIB.C  $
** LOGFILE:     $Logfile:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPLIB.C_V  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPLIB.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:24:02   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:14:50   etstjan
**              No explicit note
*************************************************************************/

#include      <stdlib.h>
#include      <string.h>
#include      <stdio.h>
#include      <time.h>
#include      <sys/timeb.h>
#include      <beholder.h>

#include      "ipcodes.h"                 /* predefined return codes  */
#include      "ip.h"                      /* the IP types             */
#include      "iplib.h"                   /* this header              */

typedef union {                           /* byte/word/lword pointer  */
          BYTE      *BytePtr;
          USHORT    *WrdPtr;
          ULONG     *LWrdPtr;
        } PNTRUNION;


/**************************************************************
** NAME:        OnesSum
** SYNOPSIS:    USHORT OnesSum(ULONG Source);
** 
** DESCRIPTION: Converts Source to a USHORT, according to one's
**              complements rules. 
** RETURNS:     The USHORT result.
**************************************************************/
USHORT OnesSum(ULONG Source)
{
  Source = (Source & 0xffffl) + ((Source >> 16) & 0xffffl);
  return (USHORT)~((Source & 0xffffl) + ((Source >> 16) & 0xffffl));
}

/**************************************************************
** NAME:        CompSum
** SYNOPSIS:    ULONG CompSum(BYTE *Data, USHORT Len);
**  
** DESCRIPTION: Calculates the sum of Data as the ones
**              complement sum of the data in 16 bit words.
**              Len contains #bytes, when Len is odd, the sum
**              is calculated with a padding zero-byte.
** RETURNS:     The calculated sum. 
** SEE ALSO:    CompCheck
** USES:
**************************************************************/
ULONG CompSum(BYTE *Data, USHORT Len)
{
  USHORT   i;
  ULONG    Sum = 0;

  for (i = 0; i < Len - 1; i += 2) {
    Sum += *(USHORT *)(Data + i);
    }
  if ((Len & 1) == 1) {                 /* if Len is odd */
    Sum += *(Data + Len - 1);           /* pad with zero byte */
    }
  return Sum;
}

/*************************************************************************
** NAME:        CompCheck
** SYNOPSIS:    USHORT CompCheck(BYTE *Data, USHORT Len);
**
** DESCRIPTION: Calculates the checksum of Data as the ones
**              complement of the ones complement sum of the
**              data in 16 bit words.
**              (The IP default).
**              Len contains #bytes, when Len is odd, the checksum
**              is calculated with a padding zero-byte.
** RETURNS:     The calculated checksum. When the checksum
**              is stored in a byte stream, the order will be
**              the requested <high order><low order>.
** SEE ALSO:
** USES:        CompSum
*************************************************************************/
USHORT CompCheck(BYTE *Data, USHORT Len)
{
  ULONG  Check = CompSum(Data, Len);
  return OnesSum(Check);
}


/**************************************************************
** NAME:        LinkCheckSum
** SYNOPSIS:    ULONG LinkCheckSum(DATALINK *ThisLink)
**  
** DESCRIPTION: Calculates the sum of the data over all the
**              datablocks in the datalink ThisLink.
** RETURNS:     The computed sum.
**************************************************************/
ULONG LinkCheckSum(DATALINK *ThisLink)
{
  ULONG  Check   = 0L;
  int    Odd     = 0;

  while (ThisLink != NULL) {
    if (ThisLink->Length != 0) {
      if (Odd) {
        Check += (USHORT)*(ThisLink->DataStart) << 8;
        Check += CompSum(ThisLink->DataStart + 1, ThisLink->Length - 1);
        }
      else {
        Check += CompSum(ThisLink->DataStart, ThisLink->Length);
        }
      Odd ^= (ThisLink->Length & 1);
      }
    ThisLink = ThisLink->NextOne;
    }
  return Check;
}

/**************************************************************
** NAME:        AddressSwap
** SYNOPSIS:    void AddressSwap(ADDRESS_T *Address1,
**                               ADDRESS_T *Address2);
** DESCRIPTION: Swaps Address1 and Address2.
** RETURNS:   
**************************************************************/
void AddressSwap(ADDRESS_T *Address1, ADDRESS_T *Address2)
{
  ADDRESS_T  TempAdd = *Address1;
  *Address1 = *Address2;
  *Address2 = TempAdd;
}

/*************************************************************************
** NAME:        CipToByte
** SYNOPSIS:    int CipToByte(BYTE *Destin, CIPHEAD *Source);
**
** DESCRIPTION: Converts the IP header in Source (C structure format)
**              to a byte stream in Destin. Destin must be large enough
**              to accept all bytes (at most 64 bytes);
**              The checksum field is calculated and filled as well.
** RETURNS:     NO_ERR    ->  No Problem
**              PARAMERR  ->  Parameter error in Source
** SEE ALSO:    ByteToCip, OptionsToCip, CipToOptions
** USES:        CheckSum
*************************************************************************/
int CipToByte(BYTE *Destin, CIPHEAD *Source)
{
  USHORT        Count;
  PNTRUNION     TmpPnt;

  TmpPnt.BytePtr = Destin;

  if ((Source->Version != VERSION) || (Source->HLength % 4)) 
    return PARAMERR;

  *TmpPnt.BytePtr++ = (BYTE)((Source->Version << 4) +
                             ((Source->HLength / 4) & 0x0f));

  if ((*TmpPnt.BytePtr++ = Source->Tos) & 0xc0)    /* set type of service */
    return PARAMERR;

  
  *TmpPnt.WrdPtr++ = htons(Source->TLength);                /* and length */
  *TmpPnt.WrdPtr++ = htons(Source->Ident);                  /* and Ident */

  if ((Source->Flag & 0xf8) || (Source->Offset & 0xd000)) 
    return PARAMERR;

  *TmpPnt.WrdPtr++ = htons((Source->Flag << 5) * 256 + (Source->Offset & 0x1fff));

  *TmpPnt.BytePtr++ = Source->Time;              /* set time and protocol */
  *TmpPnt.BytePtr++ = Source->Protocol;

  *TmpPnt.WrdPtr++ = htons(0);                   /* clear checksum space */

  /* copy source address      */
  *TmpPnt.LWrdPtr++ = Source->Source;
  /* and destination address  */
  *TmpPnt.LWrdPtr++ = Source->Destin;

  /* copy the option space */
  for (Count = 20; Count < Source->HLength; Count++) 
    *TmpPnt.BytePtr++ = Source->Options[Count - 20];

  /* fill the checksum field */
  *(USHORT *)(Destin + 10) = CompCheck(Destin, Source->HLength);

  return NO_ERR;
}


/*************************************************************************
** NAME:        ByteToCip
** SYNOPSIS:    int ByteToCip(CIPHEAD *Destin, BYTE *Source);
**
** DESCRIPTION: Converts the IP header in Source (byte stream)
**              to a C structure format in Destin.
** RETURNS:     NO_ERR    ->  No Problem
**              VERERR    ->  Version number not correct
**              CHECKERR  ->  Checksum error
** SEE ALSO:    CipToByte, OptionsToCip, CipToOptions
** USES:        CheckSum
*************************************************************************/
int ByteToCip(CIPHEAD *Destin, BYTE *Source)
{
  USHORT      Count;
  PNTRUNION   TmpPnt;
  int         RetVal = NO_ERR;

  TmpPnt.BytePtr = Source;

  Destin->Version = (BYTE)(*TmpPnt.BytePtr >> 4);     /* get version number */
  Destin->HLength = (BYTE)((*TmpPnt.BytePtr++ & 0xf) * 4);

  if (Destin->Version != VERSION) 
    RetVal = VERERR;

  Destin->Tos = *TmpPnt.BytePtr++;                   /* get type of service */
  Destin->TLength = ntohs(*TmpPnt.WrdPtr++);     /* get total packet length */
  Destin->Ident = ntohs(*TmpPnt.WrdPtr++);            /* get identification */
  Destin->Flag = (BYTE)(*TmpPnt.BytePtr >> 5);    /* get flag bits (3 bits) */
  Destin->Offset = ntohs(*TmpPnt.WrdPtr++) & 0x7f;  /* fragm. offset (13 b) */
  Destin->Time = *TmpPnt.BytePtr++;                     /* get time to live */
  Destin->Protocol = *TmpPnt.BytePtr++;                     /* and protocol */
  Destin->CheckSum = ntohs(*TmpPnt.WrdPtr++);             /* store checksum */

  /* check header and total length */
  if ((USHORT)Destin->HLength > Destin->TLength) 
    RetVal = HLENERR;

  /* check the checksum */
  if (CompCheck(Source, Destin->HLength) != 0) 
    RetVal = CHECKERR;

  /* get source address */
  Destin->Source = *TmpPnt.LWrdPtr++;
  Destin->Destin = *TmpPnt.LWrdPtr++;

  /* set the option field length */
  Destin->OLength = Destin->HLength - 20;
  /* clear option state */
  Destin->OpState = 0;
  /* copy the option space */
  for (Count = 0; (Count < Destin->OLength) && (Count < 40); Count++) 
    Destin->Options[Count] = *TmpPnt.BytePtr++;

  return RetVal;
}


/*************************************************************************
** NAME:        OptionsToCip
** SYNOPSIS:    int OptionsToCip(CIPHEAD *Destin,
**                               BYTE *Source,
**                               int Length);
**
** DESCRIPTION: Adds the Options in the Byte stream Source
**              to the options in the Source C structure.
**              If the option contains a 'Source Route' option,
**              the first hop address is used as destination
**              address and removed from the option.
** RETURNS:     NO_ERR    ->  No Problem
**              LENGTHERR ->  Option field to long
** SEE ALSO:    CipToOptions, CipToByte, ByteToCip
** USES:
*************************************************************************/
int OptionsToCip(CIPHEAD *Destin, BYTE *Source, USHORT Length)
{
  USHORT   i, OptLen, OffSet = Destin->OLength;

  /* append new options to options already available in header */
  for (i = 0; (i < Length) && (Source[i] != ENDOFLIST); i += OptLen) {
    if ((i + Destin->OLength) > 44)
      return LENGTHERR;
    if (Source[i] == NOP) {
      OptLen = 1;
      Destin->Options[OffSet + i] = Source[i];
      }
    else {
      OptLen = Source[i + 1];
      if (OptLen + i > Length) {
        return OPTERR;
        }
      if ((Source[i] == STRICT) || (Source[i] == LOOSE)) {
        if (OptLen < 7) {
          return OPTERR;
          }
        Destin->OpState |= (Source[i] == STRICT) ? STRICT_SRC : LOOSE_SRC;
        Destin->Destin = *(ADDRESS_T *)(Source + i + 3);
        memmove(Source + i + 3, Source + i + 7, Length - i - 7); 
        Source[i + 1] -= 4;
        OptLen -= 4;
        Length -= 4;
        }
      memcpy(Destin->Options + OffSet + i, Source + i, OptLen);
      }
    }
  Destin->OLength += i;
  return NO_ERR;
}

/**************************************************************
** NAME:        AllignOptions
** SYNOPSIS:    void AllignOptions(CIPHEAD *ThisHeader);
**           
** DESCRIPTION: Alligns the options in ThisHeader with
**              ENDOFLIST. The IPHeader, Options and Total
**              Packet Length are updated accordingly.
** RETURNS:   
**************************************************************/
void AllignOptions(CIPHEAD *ThisHeader)
{
  while ((ThisHeader->OLength % 4) != 0) {
    ThisHeader->Options[ThisHeader->OLength++] = ENDOFLIST;
     }
  ThisHeader->TLength -= ThisHeader->HLength;
  ThisHeader->HLength = (BYTE)(20 + ThisHeader->OLength);
  ThisHeader->TLength += ThisHeader->HLength;
}

/**************************************************************
** NAME:        RemoveNonCopyOptions
** SYNOPSIS:    void RemoveNonCopyOptions(CIPHEAD *ThisHeader);
**           
** DESCRIPTION: Removes options from Thisheader that should
**              not be copied on fragmentation. The options
**              are alligned as well, updating HLength,
**              OLength and TLength.
**
** RETURNS:   
**************************************************************/
void RemoveNonCopyOptions(CIPHEAD *ThisHeader)
{
  USHORT i = 0;
  int    OptLen, Remove;

  while (i < ThisHeader->OLength) {
    Remove = 0;
    switch (ThisHeader->Options[i]) {
      case ENDOFLIST :
      case NOP       : OptLen = Remove = 1;
                       break;
      default        : OptLen = ThisHeader->Options[i + 1];
                       Remove = ((ThisHeader->Options[i] & COPY_MASK) == 0);
                       break;
      }
    if (Remove) {
      RemoveOption(ThisHeader, i);
      }
    else i += OptLen;
    }
  /* alligning options takes care of Header, Options and Packet size */
  AllignOptions(ThisHeader);
}

/**************************************************************
** NAME:        RemoveOption
** SYNOPSIS:    void RemoveOption(CIPHEAD *ThisHeader,
**                                USHORT Distance);
**           
** DESCRIPTION: Removes the option at relative distance
**              Distance from ThisHeader.
**              HLength, OLength and TLength are updated.
** RETURNS:   
**************************************************************/
void RemoveOption(CIPHEAD *ThisHeader, USHORT Distance)
{
  int  OptLen;

  OptLen = ((ThisHeader->Options[Distance] == ENDOFLIST) ||
            (ThisHeader->Options[Distance] == NOP)) ?
           1 : ThisHeader->Options[Distance + 1];

  memmove(ThisHeader->Options + Distance,
          ThisHeader->Options + Distance + OptLen,
          ((ThisHeader->OLength -= OptLen)) - Distance);
}

/**************************************************************
** NAME:        MakeRespondOptions
** SYNOPSIS:    void MakeRespondOptions(CIPHEAD *ThisHeader);
**           
** DESCRIPTION: Uses ThisHeader->OpState to determine which
**              options must reside in the IP header.
**              These options are: Record Route, Timestamp,
**              Strict Source Route and Loose Source Route.
**              The Source Route Options are reversed.
**              Other options are deleted and the header
**              length etc. are updated.
**              Options are alligned as well.
** RETURNS:    
**************************************************************/
void MakeRespondOptions(CIPHEAD *ThisHeader)
{
  BYTE      SaveOption[40], SaveAdd[4];
  USHORT    i, j, OptLen;

  for (i = 0; i < ThisHeader->OLength; i+= OptLen) {
    OptLen = ThisHeader->Options[i + 1];
    switch(ThisHeader->Options[i]) {
      case RECORD    : 
      case TIMESTAMP : break;
      case LOOSE     : 
      case STRICT    : memcpy(SaveOption, ThisHeader->Options + i + 3,
                              OptLen - 3);
                       if (memcmp((char *)&(ThisHeader->Source),
                                   SaveOption, 4) == 0) {
                         /* check, see rfc1122 page 36 */
                         ThisHeader->Options[i + 1] -= 4;
                         memmove(SaveOption, SaveOption + 4,
                                 (OptLen -= 4) - 3);
                         memmove(ThisHeader->Options + i + OptLen,
                                 ThisHeader->Options + i + OptLen + 4,
                                 (ThisHeader->OLength -= 4) - i - OptLen);
                         }
                       memcpy(SaveAdd, SaveOption + OptLen - 7, 4);
                       memmove(SaveOption + 4, SaveOption, OptLen - 7);
                       memcpy(SaveOption, (char *)&(ThisHeader->Source), 4);
                       memcpy((char *)&(ThisHeader->Source), SaveAdd, 4);
                       for (j = 0; j < OptLen - 3 ; j += 4) {
                         memcpy(ThisHeader->Options + i + j + 3,
                                SaveOption + OptLen - 7 - j, 4);
                         }
                       ThisHeader->Options[i + 2] = 4;
                       break;
      default        : RemoveOption(ThisHeader, i);
                       OptLen = 0;
                       break;
      }
    }
  AllignOptions(ThisHeader);
}

/*************************************************************************
** NAME:        CipToOptions
** SYNOPSIS:    int OptionsToCip(BYTE *Destin,
**                               CIPHEAD *Source);
**
** DESCRIPTION: Converts the Options in the Source
**              C structure to a Byte stream in Destin.
** RETURNS:     NO_ERR    ->  No Problem
** SEE ALSO:    OptionsToCip, CipToByte, ByteToCip
** USES:
*************************************************************************/
int CipToOptions(BYTE *Destin, CIPHEAD *Source)
{
  USHORT  i;

  for (i = 0; i < Source->OLength; i++) {
    Destin[i] = Source->Options[i];
    }
  return NO_ERR;
}


/*************************************************************************
** NAME:        CudpToByte
** SYNOPSIS:    int CudpToByte(BYTE *Destin, CUDPHEAD *Source);
**
** DESCRIPTION: Converts the UDP header in Source (C structure format)
**              to a byte stream in Destin.
**              Destin must be large enough to accept all bytes
**              (at most 8 bytes);
**              The checksum field is filled with 0's, should be
**              filled later if a checksum is desired.
**
** RETURNS:     NO_ERR
** SEE ALSO:    CipToByte, ByteToCip, ByteToCudp
** USES:
*************************************************************************/
int CudpToByte(BYTE *Destin, CUDPHEAD *Source)
{
  PNTRUNION    TmpPnt;

  TmpPnt.BytePtr = Destin;

  /* set source and destination port  and UDP length */
  *TmpPnt.WrdPtr++ = htons(Source->SPort);
  *TmpPnt.WrdPtr++ = htons(Source->DPort);
  *TmpPnt.WrdPtr++ = htons(Source->DLength);
  *TmpPnt.WrdPtr = htons(0);                   /* clear checksum space     */
  return NO_ERR;
}


/*************************************************************************
** NAME:        ByteToCudp
** SYNOPSIS:    int ByteToCudp(CUDPHEAD *Destin, BYTE *Source);
**
** DESCRIPTION: Converts the UDP header in Source (byte stream)
**              to a C structure format in Destin.
** RETURNS:     NO_ERR
** SEE ALSO:    CudpToByte, CipToByte, ByteToCip
** USES:
*************************************************************************/
int ByteToCudp(CUDPHEAD *Destin, BYTE *Source)
{
  PNTRUNION    TmpPnt;

  TmpPnt.BytePtr = Source;

  /* get source and destination port, UDP length and checksum */
  Destin->SPort = ntohs(*TmpPnt.WrdPtr++);
  Destin->DPort = ntohs(*TmpPnt.WrdPtr++);
  Destin->DLength = ntohs(*TmpPnt.WrdPtr++);
  Destin->CheckSum = ntohs(*TmpPnt.WrdPtr);
  return NO_ERR;
}


/**************************************************************
** NAME:        UDPCheckSum
** SYNOPSIS:    USHORT UDPCheckSum(CIPHEAD *IPHead,
**                               DATALINK *ThisLink)
** 
** DESCRIPTION: Calculates the checksum of an UDP packet.
**              Info is derived from the IP header IPHead
**              and from the UDP packet in ThisLink.
** RETURNS:     The calculated checksum.
**************************************************************/
USHORT UDPCheckSum(CIPHEAD *IPHead, DATALINK *ThisLink)
{
  ULONG    TempCheck;

  /* first, the pseudo header */
  TempCheck = CompSum((BYTE *)&(IPHead->Source), 4);
  TempCheck += CompSum((BYTE *)&(IPHead->Destin), 4);

  TempCheck += (USHORT)(IPHead->Protocol) << 8;
  TempCheck += CompSum(ThisLink->DataStart + 4, 2);
  /* then the UDP packet, including the UDP header */
  TempCheck += LinkCheckSum(ThisLink);
  return OnesSum(TempCheck);
}


/*************************************************************************
** NAME:        DLinkToByte
** SYNOPSIS:    USHORT DLinkToByte(BYTE *Destin,
**                               DATALINK *Source);
**
** DESCRIPTION: Converts the data in the DATALINK Source
**              to a continuous byte stream in Destin.
**              Destin must be large enough to contain
**              all bytes.
** RETURNS:     Length of byte stream.
** SEE ALSO:    OptionsToCip, CipToByte, ByteToCip, CipToOptions
** USES:
*************************************************************************/
USHORT DLinkToByte(BYTE *Destin, DATALINK *Source)
{
  DATALINK     *Temp;
  USHORT       Index = 0;

  for (Temp = Source; Temp != NULL; Temp = Temp->NextOne) {
    memcpy(Destin + Index, Temp->DataStart, Temp->Length);
    Index += Temp->Length;
    }
  return Index;
}

/*************************************************************************
** NAME:        DLinkLen
** SYNOPSIS:    USHORT int DLinkLen(DATALINK *Source);
**
** DESCRIPTION: Calculates the total number of bytes
**              in the datalink Source.
** RETURNS:     Total number of bytes in datalink.
** SEE ALSO:    DLinkToByte
** USES:
*************************************************************************/
USHORT DLinkLen(DATALINK *Source)
{
  USHORT  Len = 0;

  while (Source != NULL) {
    Len += Source->Length;
    Source = Source->NextOne;
    }
  return Len;
}


/**************************************************************
** NAME:        SplitDataLink
** SYNOPSIS:    DATALINK *SplitDataLink(DATALINK *ThisLink,
**                                      DATALINK **LastLink, 
**                                      USHORT Size);
**  
** DESCRIPTION: Splits a datalink in two parts; one with
**              'Size' bytes and one with the remainder.
**              A new DATALINK is created if necessary.
**              'LastLink' is set to the last DATALINK in
**              the first DATALINK chain.
** RETURNS:     A pointer to the DATALINK at the head
**              of the remaining part.
**************************************************************/
DATALINK *SplitDataLink(DATALINK *ThisLink,
                        DATALINK **LastLink, USHORT Size)
{
  DATALINK *NewLink;
  DATALINK *PrevLink = ThisLink;
  USHORT   CollSize = 0;

  while ((ThisLink != NULL) && (CollSize < Size)) {
    CollSize += ThisLink->Length;
    PrevLink = ThisLink;
    ThisLink = ThisLink->NextOne;
    }
                    
  if (CollSize < Size) {
    return NULL;
    }

  *LastLink = PrevLink;
  if ((CollSize != Size) || (Size == 0)) {
    if (Size == 0) {                          /* size 0 is special */
      CollSize = PrevLink->Length;
      }
    if ((NewLink = IPDataLinkGet(CollSize - Size)) == NULL) {
      return NULL;
      }
    NewLink->ThisType = PrevLink->ThisType;
    NewLink->NextOne = PrevLink->NextOne;
    PrevLink->Length -= CollSize - Size;
    memcpy(NewLink->DataStart, PrevLink->DataStart + PrevLink->Length,
           CollSize - Size);
    PrevLink->NextOne = NULL;
    return NewLink;
    }
  else {
    PrevLink->NextOne = NULL;
    return ThisLink;
    }
}


/**************************************************************
** NAME:        DiscardAll
** SYNOPSIS:    void DiscardAll(DATALINK *LinkPnt);
** 
** DESCRIPTION: Frees all data in the datalink LinkPnt.
** RETURNS: 
**************************************************************/
void DiscardAll(DATALINK *LinkPnt)
{
  DATALINK *TmpPnt = LinkPnt;

  while (LinkPnt != NULL) {
    TmpPnt = LinkPnt;
    LinkPnt = LinkPnt->NextOne;
    IPDataLinkFree(TmpPnt);
    }
}

/**************************************************************
** NAME:        IPDataLinkGet
** SYNOPSIS:    DATALINK *IPDataLinkGet(USHORT Size);
** 
** DESCRIPTION: Allocates a DATALINK structure and Size bytes
**              of storage from the IP buffer management.
**              The DataArea pointer is set to the Size bytes
**              of storage, as well as the DataStart pointer.
**              Length contains Size, NextOne and NextLink
**              are initialized to NULL. SaveData is set to
**              0 (= do not save).
** RETURNS:     NULL --> no storage left
**              else     pointer to DATALINK structure
**************************************************************/
DATALINK *IPDataLinkGet(USHORT Size)
{
  DATALINK *TmpPnt;
  
  if (((TmpPnt = IPBufGet(sizeof(DATALINK))) == NULL) ||
      ((TmpPnt->DataArea = IPBufGet(Size)) == NULL)) {
    if (TmpPnt != NULL) {
      IPBufFree(TmpPnt);
      }
    return NULL;
    }
  TmpPnt->SaveData = 0;
  TmpPnt->DataStart = TmpPnt->DataArea;
  TmpPnt->NextOne = TmpPnt->NextLink = NULL;
  TmpPnt->Length = Size;
  return TmpPnt;
}

/**************************************************************
** NAME:        IPDataLinkFree
** SYNOPSIS:    int IPDataLinkFree(DATALINK *ThisLink);
** 
** DESCRIPTION: Frees the DATALINK ThisLink and the
**              associated data block.
** RETURNS:     NO_ERR  -->  no error
**              else         IP Buffer Management error code
**************************************************************/
int IPDataLinkFree(DATALINK *ThisLink)
{
  int RetCode;

  RetCode = (ThisLink->SaveData) ? NO_ERR : IPBufFree(ThisLink->DataArea);
  if (RetCode != NO_ERR) {
    IPBufFree(ThisLink);
    return RetCode;
    }
  else {
    return IPBufFree(ThisLink);
    }  
}
                    
/*************************************************************************
** NAME:        CicmpToByte
** SYNOPSIS:    int CicmpToByte(BYTE *Destin, CICMPHEAD *Source);
**
** DESCRIPTION: Converts the ICMP header in Source (C structure format)
**              to a byte stream in Destin.
**              Destin must be large enough to accept all bytes.
**              (8 bytes);
**              The checksum field is cleared.
** RETURNS:     NO_ERR    ->  No Problem
** SEE ALSO:    ByteToCicmp
** USES:
*************************************************************************/
int CicmpToByte(BYTE *Destin, CICMPHEAD *Source)
{
  PNTRUNION    TmpPnt;

  TmpPnt.BytePtr = Destin;

  *TmpPnt.BytePtr++ = Source->Type;       /* set message type         */
  *TmpPnt.BytePtr++ = Source->Code;       /* set message code         */
  *TmpPnt.WrdPtr++ = htons(0);            /* clear checksum space     */

  switch (Source->Type) {
    case PARAMPROB :  *TmpPnt.BytePtr = Source->MoreInfo.Pointer;
                      break;
    case REDIRECT :   *TmpPnt.LWrdPtr = Source->MoreInfo.GateWayAdd;
                      break;
    default :         /* not used field may be filled */
                      *TmpPnt.WrdPtr++ = htons(Source->MoreInfo.IdSeq[0]);
                      *TmpPnt.WrdPtr = htons(Source->MoreInfo.IdSeq[1]);
                      break;
    }
  return NO_ERR;
}


/*************************************************************************
** NAME:        ByteToCicmp
** SYNOPSIS:    int ByteToCicmp(CICMPHEAD *Destin, BYTE *Source);
**
** DESCRIPTION: Converts the ICMP header in Source (byte stream)
**              to a C structure format in Destin. The checksum is
**              not checked, as this requires knowledge about the
**              data part of the packet.
** RETURNS:     NO_ERR    ->  No Problem
** SEE ALSO:    CicmpToByte
** USES:
*************************************************************************/
int ByteToCicmp(CICMPHEAD *Destin, BYTE *Source)
{
  PNTRUNION    TmpPnt;

  TmpPnt.BytePtr = Source;

  Destin->Type = *TmpPnt.BytePtr++;           /* get type of message   */
  Destin->Code = *TmpPnt.BytePtr++;           /* and code */
  Destin->CheckSum = *TmpPnt.WrdPtr++;        /* store checksum */

  switch (Destin->Type) {
    case PARAMPROB :  Destin->MoreInfo.Pointer = *TmpPnt.BytePtr;
                      break;
    case REDIRECT :   Destin->MoreInfo.GateWayAdd = *TmpPnt.LWrdPtr;
                      break;
    default :         /* not used field may be filled */
                      Destin->MoreInfo.IdSeq[0] = ntohs(*TmpPnt.WrdPtr++);
                      Destin->MoreInfo.IdSeq[1] = ntohs(*TmpPnt.WrdPtr);
                      break;
    }

  return NO_ERR;
}

/**************************************************************
** NAME:        GetNetMask
** SYNOPSIS:    ADDRESS_T GetNetMask(ADDRESS_T ThisAddress);
**            
** DESCRIPTION: Returns the network mask for IP Address
**              'ThisAddress'. If the IP Address isn't of
**              the A, B or C type, 0 is returned.
** RETURNS:     The Network mask for 'ThisAddress'.
**************************************************************/
ADDRESS_T GetNetMask(ADDRESS_T ThisAddress)
{
  return (IN_CLASSA(ThisAddress)) ? ~AHOSTMASK :
         (IN_CLASSB(ThisAddress)) ? ~BHOSTMASK :
         (IN_CLASSC(ThisAddress)) ? ~CHOSTMASK : ~INADDR_ANY;
}

#ifdef LITTLE_ENDIAN
#define SWAP(a) ((USHORT)((((ULONG)(a)) << 16 | ((ULONG)(a))) >> 8))
/**************************************************************
** NAME:        htonl
** SYNOPSIS:    ULONG htonl(ULONG hostlong);
**  
** DESCRIPTION: Converts 32 bit words between host byte order
**              and network byte order.
** RETURNS:     The converted 32 bit word.
**************************************************************/
ULONG htonl(ULONG hostlong)
{
  return ((ULONG)SWAP((USHORT)(hostlong >> 16)))
         | ((ULONG)SWAP((USHORT)hostlong) << 16);
}

/**************************************************************
** NAME:        htons
** SYNOPSIS:    USHORT htons(USHORT hostshort);
**  
** DESCRIPTION: Converts 16 bit words between host byte order
**              and network byte order.
** RETURNS:     The converted 16 bit word.
**************************************************************/
USHORT htons(USHORT hostshort)
{
  return SWAP(hostshort);
}

/**************************************************************
** NAME:        ntohl
** SYNOPSIS:    ULONG ntohl(ULONG netlong);
**  
** DESCRIPTION: Converts 32 bit words between network byte
**              order and host byte order.
** RETURNS:     The converted 32 bit word.
**************************************************************/
ULONG ntohl(ULONG netlong)
{
  return ((ULONG)SWAP((USHORT)(netlong >> 16)))
         | ((ULONG)SWAP((USHORT)netlong) << 16);
}

/**************************************************************
** NAME:        ntohs
** SYNOPSIS:    USHORT ntohs(USHORT netshort);
**  
** DESCRIPTION: Converts 16 bit words between network byte
**              order and host byte order.
** RETURNS:     The converted 16 bit word.
**************************************************************/
USHORT ntohs(USHORT netshort)
{
  return SWAP(netshort);
}
#endif
  
/**************************************************************
** NAME:        TimeMilSec
** SYNOPSIS:    ULONG TimeMilSec(void);
**         
** DESCRIPTION: Returns the time in milliseconds since
**              midnight. The high order bit of the time is
**              set, in order to indicate it ain't with
**              respect to midnight UT.
** RETURNS:     The calculated time.
**************************************************************/
ULONG TimeMilSec(void)
{
  struct tm     *TmTime;
  struct timeb  UnixTime;

  ftime(&UnixTime);
  TmTime = localtime(&(UnixTime.time));
  return (1000l * (
          (ULONG)(TmTime->tm_hour) * 3600l +
          (ULONG)(TmTime->tm_min)    * 60l +
          (ULONG)(TmTime->tm_sec))         +
          UnixTime.millitm) | 0x80000000l; 
}

