// HexView - Hex File Viewer for NT/WIN32 Using Mapped File Functions
// Copyright (C) 1992 Ray Duncan
// PC Magazine * Ziff Davis Publications

#define dim(x) (sizeof(x) / sizeof(x[0]))   // returns no. of elements
#define EXENAMESIZE 256                     // max length of path+filename
#define BPL 16                              // bytes per line

#include "stdlib.h"
#include "windows.h"
#include "string.h"
#include "commdlg.h"
#include "hexview.h"

HANDLE hInst;                               // module instance handle
HWND hFrame;                                // handle for frame window
HWND hChild;                                // handle for child window

HFONT hFont;                                // handle for nonprop. font
INT CharX, CharY;                           // character dimensions
INT LPP;                                    // lines per page
INT BPP;                                    // bytes per page
INT ThumbInc;                               // bytes per thumb unit
INT WinWidth;                               // width of frame client area

INT hFile = -1;                             // handle for current file
char szFileName[EXENAMESIZE+1];             // name of current file
LONG FileSize;                              // length of current file
LONG FileIndex;                             // index from thumb tracking

LONG ViewPtr;                               // addr, first line current page
LONG TopAddr;                               // address, top of last page
LONG TopLine;                               // line num, top of last page

HANDLE hMap = 0;                            // handle, file mapping object
LPSTR lpMap = NULL;                         // base address, mapping object

char szFrameClass[] = "HexView";            // classname for frame window
char szChildClass[] = "HexViewChild";       // classname for child window
char szAppName[] = "Hex File Viewer";       // long application name
char szMenuName[] = "HexViewMenu";          // name of menu resource
char szIconName[] = "HexViewIcon";          // name of icon resource
char szIni[] = "Hexview.Ini" ;              // initialization filename

char *szFilter[] = {                        // filters for Open dialog
    "All Files (*.*)", "*.*",               
    "Executable files (*.EXE)", "*.EXE",
    "Object Modules (*.OBJ)", "*.OBJ",
    "" };

//
// Table of window messages supported by FrameWndProc()
// and the functions which correspond to each message.
//
struct decodeWord frameMsgs[] = {
    WM_PAINT, DoPaint,
    WM_SIZE, DoSize,
    WM_COMMAND, DoCommand,
    WM_CLOSE, DoClose,
    WM_DESTROY, DoDestroy,
    WM_VSCROLL, DoVScroll, } ;

//
// Table of window messages supported by ChildWndProc()
// and the functions which correspond to each message.
//
struct decodeWord childMsgs[] = {
    WM_PAINT, DoChildPaint, } ;

//
// Table of menubar item IDs and their corresponding functions.
//
struct decodeWord menuitems[] = {
    IDM_OPEN, DoMenuOpen,
    IDM_EXIT, DoMenuExit,
    IDM_ABOUT, DoMenuAbout, } ;

//
// WinMain -- entry point for this application from Windows.
//
INT APIENTRY WinMain(HANDLE hInstance,
    HANDLE hPrevInstance, LPSTR lpCmdLine, INT nCmdShow)
{
    MSG msg;                                // receives message packet

    hInst = hInstance;                      // save instance handle

    if(!hPrevInstance)                      // if first instance,
        if(!InitApp(hInstance))             // register window class
        {
            MessageBox(hFrame, "Can't initialize HexView!", szAppName,
                MB_ICONSTOP | MB_OK);
            return(FALSE);
        }

    if(!InitInstance(hInstance, nCmdShow))  // create instance window
    {
        MessageBox(hFrame, "Can't initialize HexView!", szAppName,
            MB_ICONSTOP | MB_OK);
        return(FALSE);
    }

    while(GetMessage(&msg, NULL, 0, 0))     // while message != WM_QUIT
    {
        TranslateMessage(&msg);             // translate virtual key codes
        DispatchMessage(&msg);              // dispatch message to window
    }

    TermInstance(hInstance);                // clean up for this instance
    return(msg.wParam);                     // return code = WM_QUIT value
}

//
// InitApp --- global initialization code for this application.
// Registers window classes for frame and child windows.
//
BOOL InitApp(HANDLE hInstance)
{
    WNDCLASS  wc;                           // window class data
    BOOL bParent, bChild;                   // status from RegisterClass

    // set parameters for frame window class
    wc.style = CS_HREDRAW|CS_VREDRAW;       // class style
    wc.lpfnWndProc = FrameWndProc;          // class callback function
    wc.cbClsExtra = 0;                      // extra per-class data
    wc.cbWndExtra = 0;                      // extra per-window data
    wc.hInstance = hInstance;               // handle of class owner
    wc.hIcon = LoadIcon(hInst, "HexViewIcon");      // application icon
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       // default cursor
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); // background color 
    wc.lpszMenuName =  szMenuName;          // name of menu resource
    wc.lpszClassName = szFrameClass;        // name of window class

    bParent = RegisterClass(&wc);           // register parent window class

    // modify some of the parameters for the child window class
    wc.style = 0;                           // class style
    wc.lpfnWndProc = ChildWndProc;          // class callback function
    wc.lpszMenuName =  NULL;                // name of menu resource
    wc.lpszClassName = szChildClass;        // name of window class

    bChild = RegisterClass(&wc);            // register child window class

    return(bChild && bParent);              // return combined status
}

//
// InitInstance --- instance initialization code for this application.
// Gets information about system nonproportional font.  Allocates memory 
// to use as file I/O buffer.  Creates frame window, gets initialization 
// information (if any) from INI file (Windows 3.x) or registration 
// database (WIN32/NT). Positions and sizes window, opens and positions
// file if initialization info found.
//
BOOL InitInstance(HANDLE hInstance, INT nCmdShow)
{
    HDC hdc;                                // handle for device context
    TEXTMETRIC tm;                          // font information
    RECT rect;                              // window position & size
    char buff[80];                          // scratch buffer

    hFrame = CreateWindow(                  // create frame window
        szFrameClass,                       // window class name
        szAppName,                          // text for title bar
        WS_OVERLAPPEDWINDOW | WS_VSCROLL,   // window style
        CW_USEDEFAULT, CW_USEDEFAULT,       // default position
        CW_USEDEFAULT, CW_USEDEFAULT,       // default size
        NULL,                               // no parent window
        NULL,                               // use class default menu
        hInstance,                          // window owner
        NULL);                              // unused pointer

    if(!hFrame) return(FALSE);              // error, can't create window

    hdc = GetDC(hFrame);                    // get device context
    hFont = GetStockObject(SYSTEM_FIXED_FONT);  
    SelectObject(hdc, hFont);               // realize nonproportional
    GetTextMetrics(hdc, &tm);               // font, get character size,
    CharX = tm.tmAveCharWidth;              // and calculate window width
    CharY = tm.tmHeight + tm.tmExternalLeading;
    WinWidth = (CharX * 75) + GetSystemMetrics(SM_CXVSCROLL);
    ReleaseDC(hFrame, hdc);                 // release device context

    GetWindowRect(hFrame, &rect);           // get window position, size

    // read saved position/size profile for our frame window, if any
    rect.left = GetPrivateProfileInt("Frame", "xul", rect.left, szIni);
    rect.top = GetPrivateProfileInt("Frame", "yul", rect.top, szIni);
    rect.right = GetPrivateProfileInt("Frame", "xlr", rect.right, szIni);
    rect.bottom = GetPrivateProfileInt("Frame", "ylr", rect.bottom, szIni);

    MoveWindow(hFrame, rect.left, rect.top, // force window position, size  
        WinWidth, rect.bottom-rect.top, TRUE);

    // get saved filename and file offset for display, if any
    GetPrivateProfileString("File", "filename", "", szFileName, 
        EXENAMESIZE, szIni);
    GetPrivateProfileString("File", "fileptr", "", buff, sizeof(buff), szIni);

    if(szFileName[0])                       // if filename and file offset
    {                                       // was saved from previous
        OpenDataFile();                     // execution, open the file
        SetFilePosition(atol(buff));
    }
    
    ShowWindow(hFrame, nCmdShow);           // make frame window visible
    UpdateWindow(hFrame);                   // force WM_PAINT message
    return(TRUE);                           // return success flag
}

//
// TermInstance -- instance termination code for this application.
// This is a general-purpose opportunity to clean up after ourselves.
//
BOOL TermInstance(HANDLE hinstance)
{
    CloseDataFile();                        // close file, destroy mapping
    return(TRUE);                           // return success flag
}

//
// FrameWndProc --- callback function for application frame window.
// Decodes window message using table frameMsgs[] and runs corresponding
// function.  If no match found, passes message to Windows DefWindowProc().
//
LONG CALLBACK FrameWndProc(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    INT i;                                  

    for(i = 0; i < dim(frameMsgs); i++)     
    {                                       
        if(wMsg == frameMsgs[i].Code)
            return((*frameMsgs[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// ChildWndProc --- callback function for application child window.
// Works like FrameWndProc, except uses table childMsgs[].
//
LONG CALLBACK ChildWndProc(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    INT i;                                  

    for(i = 0; i < dim(childMsgs); i++)     
    {                                       
        if(wMsg == childMsgs[i].Code)
            return((*childMsgs[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoCommand -- processes WM_COMMAND messages for frame window. 
// Decodes the menubar item with the menuitems[] array, then
// runs the corresponding function to process the command.
// If no match found, passes message to Windows DefWindowProc().
// 
LONG DoCommand(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    INT i;                       

    for(i = 0; i < dim(menuitems); i++)     
    {                                       
        if(wParam == menuitems[i].Code)
            return((*menuitems[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoDestroy -- processes the WM_DESTROY message for frame window by
// posting a WM_QUIT message to the same window, forcing WinMain
// to fall out of the event loop.
// 
LONG DoDestroy(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    PostQuitMessage(0);          
    return(FALSE);                   
}

//
// DoClose -- processes a WM_CLOSE message for frame window by
// saving the current window position, size, filename, and file 
// offset, then forcing a WM_DESTROY message.
// 
LONG DoClose(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    UpdateFrameProfile();                   // save window information
    DestroyWindow(hWnd);                    // then close down the app
    return(FALSE);                              
}

//
// DoVScroll -- process various WM_VSCROLL messages for frame window.
// The user can generate these messages by dragging thumb, clicking
// in scrollbar, or clicking arrows at both ends of scrollbar.
// 
LONG DoVScroll(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    RECT rect;

    switch(LOWORD(wParam))                  // LOWORD is vital for Win32
    {                                       
        case SB_TOP:                        // go to top of file if
            if(ViewPtr)                     // we aren't there already
            {
                SetFilePosition(0);
                Repaint();
            }
            break;

        case SB_BOTTOM:                     // go to bottom of file if
            SetFilePosition(FileSize);      // we aren't there already
            Repaint();
            break;

        case SB_LINEUP:                     // scroll up by one line if
            if(ViewPtr)                     // we aren't already at top
            {   
                SetFilePosition(ViewPtr - BPL);
                ScrollWindow(hWnd, 0, CharY, NULL, NULL);
                UpdateWindow(hWnd);
            }
            break;

        case SB_LINEDOWN:                   // scroll down by one line if
            if(ViewPtr < TopAddr)           // we aren't already at bottom
            {
                SetFilePosition(ViewPtr + BPL);
                ScrollWindow(hWnd, 0, -CharY, NULL, NULL);
                GetClientRect(hWnd, &rect);
                rect.top = max(0, (LPP-1) * CharY);
                InvalidateRect(hWnd, &rect, TRUE);
                UpdateWindow(hWnd);
            }
            break;

        case SB_PAGEUP:                     // scroll up by one page
            SetFilePosition(ViewPtr - BPP);
            Repaint();  
            break;

        case SB_PAGEDOWN:                   // scroll down by one page
            SetFilePosition(ViewPtr + BPP);
            Repaint();
            break;

        case SB_THUMBPOSITION:              // reposition file by thumb
            SetFilePosition(ThumbInc * (LONG) THUMBPOS);
            Repaint();
            break;

        case SB_THUMBTRACK:                 // track drag of thumb
            ThumbTrack(THUMBPOS);
            break;
    }

    return(FALSE);                              
}

//
// DoPaint -- process WM_PAINT message for frame window by
// painting hex and ASCII data dump for file offsets that fall
// within the window.  We make no attempt to optimize this for
// painting regions that are smaller than the window.
// 
LONG DoPaint(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    HDC hdc;                                // scratch device context
    PAINTSTRUCT ps;                         // scratch paint structure
    INT i;

    hdc = BeginPaint(hWnd, &ps);            // get device context
    SelectObject(hdc, hFont);               // realize nonprop. font

    if(hFile != -1)                         // paint all lines that
    {                                       // fall within the window
        for(i = 0; i < LPP; i++)
            DisplayLine(hdc, i);
    }

    EndPaint(hWnd, &ps);                    // release device context
    return(FALSE);
}

//
// DoChildPaint -- process WM_PAINT message for child window.
// These occur during thumb drag; we put up a tiny window that
// displays the file offset corresponding to the thumb position.
// 
LONG DoChildPaint(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    HDC hdc;
    PAINTSTRUCT ps;
    RECT rect;
    char buff[256];

    hdc = BeginPaint(hWnd, &ps);            // get device context
    GetClientRect(hWnd, &rect);             // get client area dimensions
    SelectObject(hdc, hFont);               // select nonproportional font
    wsprintf(buff, "%08lX", FileIndex);     // format file index from thumb
    DrawText(hdc, buff, -1,                 // paint index into window
        &rect, DT_CENTER | DT_VCENTER | DT_SINGLELINE); 
    EndPaint(hWnd, &ps);                    // release device context
    return(FALSE);
}

//
// DoSize -- process WM_SIZE message for frame window by calculating
// the number of lines per page and bytes per page for new window size.
// Also reposition the display if we were already at end of file and
// the window grew.
//
LONG DoSize(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    LPP = HIWORD(lParam) / CharY;           // calc lines per page
    BPP = LPP * BPL;                        // calc bytes per page
    ConfigDisplay();                        // calc display parameters
    if(ViewPtr > TopAddr)                   // make sure window refilled
        SetFilePosition(TopAddr);
    return(FALSE);
}

//
// DoMenuOpen -- process File-Open command from menu bar. All
// the hard work is done by the OpenFile common dialog.
//
LONG DoMenuOpen(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    OPENFILENAME ofn;                       // used by common dialogs

    szFileName[0]  = '\0';                  // init filename buffer

    ofn.lStructSize = sizeof(OPENFILENAME); // length of structure
    ofn.hwndOwner = hWnd;                   // handle for owner window
    ofn.lpstrFilter = szFilter[0];          // address of filter list
    ofn.lpstrCustomFilter = NULL;           // custom filter buffer address
    ofn.nFilterIndex = 1;                   // pick default filter
    ofn.lpstrFile = szFileName;             // buffer for path+filename
    ofn.nMaxFile = EXENAMESIZE;             // length of buffer
    ofn.lpstrFileTitle = NULL;              // buffer for filename only
    ofn.lpstrInitialDir = NULL;             // initial directory for dialog
    ofn.lpstrTitle = NULL;                  // title for dialog box
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
    ofn.lpstrDefExt = NULL;                 // default extension 

    if(GetOpenFileName(&ofn))               // display open dialog
    {
        OpenDataFile();                     // open file for viewing
        Repaint();                          // force display of data
    }

    return(FALSE);
}

//
// DoMenuExit -- process File-Exit command from menu bar.  This
// is simply handled by sending a WM_CLOSE message as though Close
// had been picked on the System menu to shut down the app.
// 
LONG DoMenuExit(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    SendMessage (hWnd, WM_CLOSE, 0, 0L);    
    return(FALSE);                          
}

//
// DoMenuAbout -- process File-About command from menu bar. We
// allocate a thunk for the dialog callback routine, display the
// dialog, then release the thunk after the dialog is dismissed.
// 
LONG DoMenuAbout(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WNDPROC lpProcAbout;                    

    lpProcAbout = MakeProcInstance((WNDPROC)AboutDlgProc, hInst);
    DialogBox(hInst, "AboutBox", hWnd, lpProcAbout);         
    FreeProcInstance(lpProcAbout);
    return(FALSE);                              
}

//
// AboutDlgProc -- This is the callback routine for the About... dialog.
//
BOOL CALLBACK AboutDlgProc (HWND hwnd, UINT msg, UINT wParam, LONG lParam)
{
    if((msg==WM_COMMAND) && (wParam==IDOK)) // dismiss dialog if OK 
        EndDialog(hwnd, 0);                 // button was clicked, 
    return(FALSE);                          // otherwise do nothing
}

//
// SetWindowCaption -- concatenate the specified filename with the 
// application name, then update the frame window's title bar.
// If called with NULL pointer, removes any previous filename from
// the title bar, leaving only the application name.
//
VOID SetWindowCaption(char * szName)
{
    char szTemp[EXENAMESIZE+1];             

    if(szName == NULL)                      // if null pointer, just
        SetWindowText(hFrame, szAppName);   // display application name
    else
    {                                       // otherwise...
        strcpy(szTemp, szAppName);          // get application name
        strcat(szTemp, " - ");              // add separator
        strcat(szTemp, szName);             // add filename
        SetWindowText(hFrame, szTemp);      // put result into title bar
    }
}

//
// Repaint -- force refresh of formatted output in main window. This
// gets called at various points when the file position is changed.
//
VOID Repaint(VOID)
{
    InvalidateRect(hFrame, NULL, TRUE);     // repaint entire window 
}

//
// SetFilePosition - called during processing of various vertical scrollbar 
// messages to set ViewPtr to a valid value.
//
VOID SetFilePosition(LONG NewViewPtr)
{
    if(hChild)                              // destroy the child window
    {                                       // if it happens to be active
        DestroyWindow(hChild);
        hChild = (HWND) 0;
    }

    if(hFile == -1)                         // bail out if no file open
        return;

    ViewPtr = NewViewPtr & 0x0FFFFFFF0;     // offset is multiple of 16
                                            
    if(ViewPtr > TopAddr)                   // enforce legal file offset
        ViewPtr = TopAddr;
    if(ViewPtr < 0)
        ViewPtr = 0;

    SetScrollPos(hFrame, SB_VERT,           // set thumb on scrollbar
        ViewPtr/ThumbInc, TRUE);
}

//
// ConfigDisplay -- Configure various display parameters and scrollbar
// range according to current window size and file size.
//
VOID ConfigDisplay(VOID)
{
    // calc address and line number of first line, last page
    TopAddr = max(((FileSize + 15) & 0x0FFFFFFF0) - BPP, 0);
    TopLine = TopAddr / BPL;

    // calculate bytes per scrollbar increment
    ThumbInc = BPL * ((TopLine/32767) + 1);

    // configure vertical scrollbar or make it disappear if not needed
    if(BPP >= FileSize)                     // file fits within window
    {
        SetScrollRange(hFrame, SB_VERT, 0, 0, FALSE);
        SetScrollPos(hFrame, SB_VERT, 0, TRUE);
        if(ViewPtr) SetFilePosition(0);
    }
    else                                    // file is too big for window
    {
        SetScrollRange(hFrame, SB_VERT, 0, ((TopLine*BPL)/ThumbInc), FALSE);
        SetScrollPos(hFrame, SB_VERT, ViewPtr/ThumbInc, TRUE);
    }
}

//
// ThumbTrack() - called during processing of WM_VSCROLL message to track
// thumb position, also forcing update of file offset in child window.
//
VOID ThumbTrack(INT pos)
{
    RECT rect;
    INT SizeX = CharX * 10;                 // calc size of child window
    INT SizeY = CharY * 2;

    if(!hChild)                             // create child window 
    {
        GetClientRect(hFrame, &rect);       // get client area of frame

        hChild = CreateWindow(              // create child window
            szChildClass,                   // window class name
            NULL,                           // text for title bar
            WS_CHILD | WS_BORDER | WS_VISIBLE,  // window style
            (rect.right-SizeX) / 2,         // x position
            (rect.bottom-SizeY) / 2,        // y position
            SizeX,                          // window width
            SizeY,                          // window height
            hFrame,                         // frame is parent window
            (HMENU) 1,                      // child window ID
            hInst,                          // window owner module
            NULL);                          // unused pointer
    }

    // calculate file location to display, force repaint of child window
    FileIndex = min((LONG) pos * (LONG) ThumbInc, TopAddr);   
    InvalidateRect(hChild, NULL, TRUE);     
}

//
// DisplayLine -- format and display a single line of hex/ASCII dump
// using the device context and relative window line number supplied
// by the caller.  The file data offset is calculated from the
// relative window line and the offset corresponding to the line 
// currently at the top of the window.
//
VOID DisplayLine(HDC hdc, INT line)
{
    INT i;
    char buff[256], c, *p;
    LONG x;

    if((ViewPtr + (LONG) line*BPL) < FileSize)
    {
        // format file offset as 8-digit hex number
        p = buff + wsprintf(buff, "%08lX ", ViewPtr + (LONG) line*BPL);

        // format this 16-bytes as hex data
        for(i = 0; i < BPL; i++)  
        {
            x = ViewPtr + (LONG)(line * BPL) + i;

            if(x < FileSize)
                p += wsprintf(p, "%02X ", lpMap[x] & 0x0ff);
            else
                p += wsprintf(p, "   ");
        }

        // format same 16-bytes as ASCII, using "." for control characters
        for(i = 0; i < BPL; i++)           
        {
            x = ViewPtr + (LONG)(line * BPL) + i;

            if(x < FileSize)
            {
                c = lpMap[x];
                if(c < 0x20) 
                    c = '.';
            }
            else c = ' ';
            *p++ = c;
        }

        // append a null byte, then paint the formatted data
        *p = '\0';                          
        TextOut(hdc, 0, line*CharY, buff, strlen(buff));
    }
}

//
// OpenDataFile -- open specified file for viewing, save handle. The
// filename was previously placed in szFileName by OpenFile common dialog.
//
VOID OpenDataFile(VOID)
{
    char temp[256];

    if(hFile != -1)                         // close previous file if any
        CloseDataFile();                    // and destroy its mappings

    hFile = _lopen(szFileName, OF_READ);    // try and open the new file

    if(hFile == -1)                         // bail out if no such file
    {
        wsprintf(temp, "Can't open file: %s", szFileName);
        MessageBox(hFrame, temp, szAppName, MB_ICONSTOP | MB_OK);
        return;
    }

    FileSize = _llseek(hFile, 0, 2);        // get size of file
    
    if(FileSize == 0)                       // bail out if file is empty
    {
        MessageBox(hFrame, "File is empty!", szAppName, 
            MB_ICONSTOP | MB_OK);
        CloseDataFile();
        return;
    }

    // create file mapping object
    hMap = CreateFileMapping((HANDLE) hFile,
        (LPSECURITY_ATTRIBUTES) NULL, PAGE_READONLY, 0, 0, (LPSTR) NULL);

    if(hMap == 0)                           // bail out if no mapping object
    {
        MessageBox(hFrame, "Can't create file mapping object!", 
            szAppName, MB_ICONSTOP | MB_OK);
        CloseDataFile();
        return;
    }

    lpMap = (LPSTR) MapViewOfFile(hMap, FILE_MAP_READ, 0, 0, 0);

    if(lpMap == 0)                          // bail out if no mapping object
    {
        MessageBox(hFrame, "Can't map view of file!", 
            szAppName, MB_ICONSTOP | MB_OK);
        CloseDataFile();
        return;
    }

    ViewPtr = 0;                            // reset window address pointer
    ConfigDisplay();                        // calc display parameters
    SetWindowCaption(szFileName);           // put filename in title bar
}

//
// CloseDataFile -- close any previously opened file, destroy mappings
// if necessary.
//
VOID CloseDataFile(VOID)
{
    if(lpMap)       UnmapViewOfFile(lpMap); // destroy mapping
    if(hMap)        CloseHandle(hMap);      // release mapping object
    if(hFile != -1) _lclose(hFile);         // release file handle

    hFile = -1;                             // reset everything
    lpMap = NULL;
    hMap = 0;

    // remove any previous filename from title bar
    SetWindowCaption(NULL);
}

//
// UpdateFrameProfile() --  Update frame window profile in 
// registration database by saving current window position, 
// window size, name of file being displayed, and file offset.
//
VOID UpdateFrameProfile(VOID)
{
    RECT rect;
    char temp[20];

    if(IsIconic(hFrame) || IsZoomed(hFrame)) return;

    GetWindowRect(hFrame, &rect);           // get position of frame window

    wsprintf(temp,"%d", rect.left);
    WritePrivateProfileString("Frame", "xul", temp, "HexView.ini");

    wsprintf(temp,"%d", rect.top);
    WritePrivateProfileString("Frame", "yul", temp, "HexView.ini");

    wsprintf(temp,"%d", rect.right);
    WritePrivateProfileString("Frame", "xlr", temp, "HexView.ini");

    wsprintf(temp,"%d", rect.bottom);
    WritePrivateProfileString("Frame", "ylr", temp, "HexView.ini");

    if(hFile == -1) return;

    WritePrivateProfileString("File", "filename", szFileName, "HexView.ini");

    wsprintf(temp,"%ld", ViewPtr);
    WritePrivateProfileString("File", "fileptr", temp, "HexView.ini");
}



