{ FILESEL.INC: Routines for an interactive file selector }

procedure SORTFILES (var Fnames: filelist; Nfiles: integer);

{ Shell sort a list of file names.
  Procedure as published in Tanenbaum, "Structured Computer Organization",
  Prentice-Hall, Englewood Cliffs, NJ, 1976.
}
var Dist: integer;              { sorting distance }
    K, I: integer;              { genl sorting indexes }
    Tmp:  filename;

begin

{ Determine the initial value of Dist by finding the largest power
  of 2 less than Nfiles, and subtracting 1 from it. The final step in
  this calculation is performed inside the main sorting loop.
}
  Dist := 4;
  while (Dist < Nfiles) do
    Dist := Dist + Dist;
  Dist := Dist - 1;

{ Main sorting loop. The outer loop is executed once per pass. }
  while (Dist > 1) do begin
    Dist := Dist div 2;
    for K := 1 to (Nfiles - Dist) do begin
      I := K;
      while (I > 0) do begin
        { This stmt. is the comparison. It also controls moving values
          upward after an exchange.
        }
        if (Fnames[I] > Fnames[I+Dist]) then begin
          { This is the swap }
          tmp := Fnames[I];
          Fnames[I] := Fnames[I+Dist];
          Fnames[I+Dist] := tmp;
        end else
          I := 0;      { stop the while loop! }
        I := I - Dist;
      end; { while }
    end; { for K }
  end; { while Dist }

end; { procedure SORTFILES }

{ DISP_NAME: Display a single filename in the file selector window with
  the specified foreground and background colors.
}
procedure DISP_NAME (var Fnames: filelist; Fnum, Dispfrst, Fcolor,
  Bcolor: integer);
var Row: integer;
    i: integer;

begin
  Row := Fnum - Dispfrst + 1;
  textbackground (Bcolor);
  textcolor (Fcolor);
  gotoxy (1, Row);
  write(Fnames[Fnum]);
  { Fill out rest of line with blanks }
  for i := 1 to (12 - length(Fnames[Fnum])) do
    write (' ');
  { Position cursor at start of this filename }
  gotoxy (1, Row);
end; { procedure DISP_NAME }

{ REFRESH_FILES: Refresh the list of files in the file selector window }
procedure REFRESH_FILES (var Fnames: filelist; Select, Dispfrst,
  Displast: integer);
var i: integer;
    bg: integer;    { temp for BGcol }
    txt: integer;   { temp for Textcol }

begin
  if RevVideo then begin
    bg := Textcol;
    txt := BGcol;
  end else begin
    bg := BGcol;
    txt := Textcol;
  end;
  openwin (30, 4, 45, Displast - Dispfrst + 4, TRUE);
  for i := Dispfrst to Displast do begin
    if (i <> Select) then
      disp_name (Fnames, i, Dispfrst, txt, bg);      { normal }
  end;

  { We draw the selected filename last, so the cursor is on that row: }
  disp_name (Fnames, Select, Dispfrst, bg, txt)      { highlighted }
end; { procedure REFRESH_FILES }

procedure FILE_SELECT (Fmask: text80; var Flnm: text80);
const MAXFWIN = 20;       { Max # files to display in window }

var Srch: SearchRec;
    Fnames: filelist;
    Nfiles: integer;
    Select: integer;
    Dispfrst: integer;
    Displast: integer;
    msg: text80;
    ch: char;
    Selected: boolean;
    Surfdir: text80;
    Dir: dirstr;
    Name: namestr;
    Ext: extstr;
    txt: integer;
    bg: integer;

label DONE;

begin
  Nfiles := 0;
  Flnm := '';
  if RevVideo then begin
    bg := Textcol;
    txt := BGcol;
  end else begin
    bg := BGcol;
    txt := Textcol;
  end;

  Surfdir := getenv ('SURFDIR');
  if (Surfdir <> '') then
    Fmask := Surfdir + '\' + Fmask;

  repeat
    { Read all the file names that match the wildcard spec }
    findfirst (Fmask, 0, Srch);
    while (Doserror = 0) and (Nfiles < MAXFILES) do begin
      Nfiles := Nfiles + 1;
      with Srch do
        Fnames[Nfiles] := Name;
      findnext (Srch);
    end;

    if (Nfiles < 1) then begin
      clrscr;
      writeln('No files found for file mask: ', Fmask);
      write('Enter new file mask, or Enter to abort: ');
      readln(Fmask);
      if (Fmask = '') then
        goto DONE;
    end;
  until (Nfiles > 0);

  sortfiles (Fnames, Nfiles);

  Select := 1;
  Dispfrst := 1;
  if (Nfiles > MAXFWIN) then
    Displast := MAXFWIN
  else
    Displast := Nfiles;
  refresh_files (Fnames, Select, Dispfrst, Displast);
  Flnm := Fnames[Select];
  Selected := FALSE;

  repeat
    ch := readkey;
    if (ch = chr(0)) then begin
      { Function or arrow key pressed, get 2nd code }
      ch := readkey;
      case ch of
        'H': if (Select > 1) then begin     { Up Arrow }
            Select := Select - 1;
            if (Select < Dispfrst) then begin
              Dispfrst := Dispfrst - 1;
              Displast := Displast - 1;
              refresh_files (Fnames, Select, Dispfrst, Displast);
            end else begin
              { Don't need to refresh entire list - just move highlight }
              { First set old name to normal: }
              disp_name (Fnames, Select+1, Dispfrst, txt, bg);
              { Then set new name to highlight: }
              disp_name (Fnames, Select, Dispfrst, bg, txt);
            end;
          end; { if Select > 1 }

        'P': if (Select < Nfiles) then begin    { Down Arrow }
            Select := Select + 1;
            if ( Select > Displast) then begin
              Dispfrst := Dispfrst + 1;
              Displast := Displast + 1;
              refresh_files (Fnames, Select, Dispfrst, Displast);
            end else begin
              { Don't need to refresh entire list - just move highlight }
              { First set old name to normal: }
              disp_name (Fnames, Select-1, Dispfrst, txt, bg);
              { Then set new name to highlight: }
              disp_name (Fnames, Select, Dispfrst, bg, txt);
            end;
          end; { if Select < Nfiles }

        'G': if (Select > 1) then begin         { Home key }
            Select := 1;
            Dispfrst := 1;
            if (Nfiles > MAXFWIN) then
              Displast := MAXFWIN
            else
              Displast := Nfiles;
            refresh_files (Fnames, Select, Dispfrst, Displast);
          end; { if Select > 1 }

        'O': if (Select < Nfiles) then begin    { End key }
            Select := Nfiles;
            Displast := Nfiles;
            if (Nfiles > MAXFWIN) then
              Dispfrst := Displast - MAXFWIN + 1
            else
              Dispfrst := Displast - Nfiles + 1;
            refresh_files (Fnames, Select, Dispfrst, Displast);
          end; { if Select < Nfiles }

        'I': if (Select > 1) then begin         { PgUp key }
            Select := Select - MAXFWIN + 1;
            if (Select < 1) then
              Select := 1;
            Dispfrst := Dispfrst - MAXFWIN + 1;
            if (Dispfrst < 1) then
              Dispfrst := 1;
            if (Nfiles > MAXFWIN) then
              Displast := Dispfrst + MAXFWIN - 1
            else
              Displast := Dispfrst + Nfiles - 1;
            refresh_files (Fnames, Select, Dispfrst, Displast);
          end; { if Select > 1 }

        'Q': if (Select < Nfiles) then begin    { PgDn key }
            Select := Select + MAXFWIN - 1;
            if (Select > Nfiles) then
              Select := Nfiles;
            Displast := Displast + MAXFWIN - 1;
            if (Displast > Nfiles) then
              Displast := Nfiles;
            if (Nfiles > MAXFWIN) then
              Dispfrst := Displast - MAXFWIN + 1
            else
              Dispfrst := Displast - Nfiles + 1;
            refresh_files (Fnames, Select, Dispfrst, Displast);
          end; { if Select < Nfiles }

        else          { Invalid keypress }
          write (^G);
      end; { case ch of }
    end else begin    { if ch = 0 }
      case ch of
        chr(13): begin      { Enter key }
          Flnm := Fnames[Select];
          Selected := TRUE;
        end;

        chr(27): begin      { Escape key }
          Flnm := '';
          Selected := TRUE;
        end;

        else
          write (^G);
      end; { case ch of }
    end; { if ch = 0 }
  until Selected;

  if (Flnm <> '') then begin
    { Prepend the path }
    fsplit (Fmask, Dir, Name, Ext);
    Flnm := Dir + Flnm;
  end;

DONE:
  { This is the old window size from the parameters menu: }
  openwin (13,6,67,21, TRUE);

end; { procedure FILE_SELECT }
