{$F+} { Compiler Directive: Generate far procedure calls: On } { DO NOT CHANGE! }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Menu
    version 5.25

    This unit holds an easy to use menu system which allows programs to offer
    quick selections and choices.

    Purpose:
      To supply a menu system that is created at run time and ensures maximum
      versatility.

    How it works:
      First, the menu is initialized and stored.
      Then the program offers the selection.
      Making a selection returns a selection code.
      Finally, the menu is destroyed when it's no longer needed.

    Features:
      Two styles of menus are included:
        The box menu ( Borland style )
        The bar menu ( Lotus style )
      This unit will automatically link with the Windows unit.
      This unit will automatically link with the Lock unit.

  Versions
    1.1 implements scrolling and page movements.
    1.2 adds movement for windows.
    2.0 adds support for bar menus.
    2.1 adds support for the status line.
    3.0 cleans up the memory allocation supply to the menu system.
        expands the selection codes to bar menus.
    3.1 adds first character search keys for faster menu selection.
    4.0 allows the menu to operate in sensitive mode.
    5.1 allows the text attributes to blend by default.
    5.2 allows the arrows keys to optionally terminate the box menu.
        and allows for additional bars to be inserted into the menu system.
    5.21 repairs a bug in which the pointer could move the current selection
         outside the choices.
    5.22 alters the code to work with Pointer version 1.11
    5.23 adds a new pre-selection box menu search routine.
    5.24 updated to work with revamped pointer unit.
    5.25 added code to compile and operate under Speed Pascal/2.

  Copyright 1989, 1993, All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Menu;

  Interface

    Uses
      CRT,
      Core,
      Keyboard;

(***********************************************************

  Maximum possible choices for the bar menu.

***********************************************************)

    Const
      Max_Bar_Choices = 83;

(***********************************************************

  Possible values which will cause the menu to terminate.

***********************************************************)

      Up_Key     = Keyboard.Press_Up_Arrow;
      Down_Key   = Keyboard.Press_Down_Arrow;
      Left_Key   = Keyboard.Press_Left_Arrow;
      Right_Key  = Keyboard.Press_Right_Arrow;
      Enter_Key  = Keyboard.Press_Enter;
      Escape_Key = Keyboard.Press_Escape;
      Insert_Key = Keyboard.Press_Insert;
      Delete_Key = Keyboard.Press_Delete;

(***********************************************************

  Move_Windows determines if the top window can be moved
    using standard commands.  It works only if the Windows
    unit is used, otherwise it does nothing.

***********************************************************)

      Move_Windows: Boolean = True;

(***********************************************************

  Lock_System determines if the Lock system is invoked if
    the command is entered.  It works only if the Lock unit
    is used, otherwise it does nothing.

***********************************************************)

      Lock_System: Boolean = True;

(***********************************************************

  Normal_Color holds the normal color of the menu text in
    the menu display.

***********************************************************)

      Normal_Color: Byte = 7;

(***********************************************************

  Selector_Color holds the special color used in high
    lighting the text in the menu display.  This should be
    distinguishable from the normal color.

************************************************************)

      Selector_Color: Byte = 112;

(***********************************************************

  First_Character_Color holds a color which is used only on
    the first character of the selection.  Activated only
    for the bar menu system.

***********************************************************)

      First_Character_Color: Byte = 10;

(***********************************************************

  Leave_Selection_On determines if the selected choice will
    remain in a high lighted mode after selected, or if the
    normal color will be reinstated.

***********************************************************)

      Leave_Choice_On: Boolean = True;

(***********************************************************

  Sensitive_Mode will allow the menu to exit as soon as one
    of the character choices are made.  Instead of waiting
    for enter to be pressed, this mode assumes that the user
    decided on the menu item by the character.

***********************************************************)

      Sensitive_Mode: Boolean = False;

(***********************************************************

  Blend_Attributes will determine if the current screen
    attributes are to be blended together with the default
    attributes.

***********************************************************)

      Blend_Attribute: Boolean = True;

(***********************************************************

  Exit_On_Arrows will determine if the current box menu will
    exit when the left and right arrows are pressed, or if
    they will act as originally intended.

***********************************************************)

      Exit_On_Arrows: Boolean = False;

(***********************************************************

  HighLight letter determines if the current box menu will
    highlight the first letter of the choices or leave them
    all the same color.

***********************************************************)

      HighLight_Letter: Boolean = False;

(***********************************************************

  Menu type structures.
    These menu type structures are needed to store the data
    of the menus.  Each is built to need only the minimum
    amount of memory.

***********************************************************)

    Type
     { Small record used for the bar menu. }
      Choice_Record = Record
                        Start_HighLight,
                        Finish_HighLight: Byte;
                      End;
     { Bar menu type stores the data as a string. }
      Bar_Menu_Type = Record
                        Data: String;
                        Last_Choice,
                        Choice_Amount: Byte;
                        Choices: Array[ 1 .. Max_Bar_Choices ] of Choice_Record;
                      End;
     { Box menu type stores the data as a list of strings. }
      Box_Menu_Type = Record
                        Top,
                        Last,
                        Bottom: Pointer;
                      End;

(***********************************************************

  Command holds the last command issued in the menu.

***********************************************************)

    Var
      Command: Byte;

(**********************************************************

  Help Screen.
    This variable procedure points to the procedure which
    will be called when the F1 key is pressed.  The program
    may change this pointer if necessary.
    ( Not supported with Pascal version 4 )

***********************************************************)

     {$IFNDEF VER40}
      Help_Screen: Procedure( Selection: Word );

(***********************************************************

  Display Status.
    This variable procedure points to the procedure which
    will be called when the particular selection is high-
    lighted on the screen.
    ( Not supported with Pascal version 4 )

***********************************************************)

      Display_Status: Procedure( Selection: Word );
     {$ENDIF}

(***********************************************************

  Procedure: Initialize the box menu.

    This procedure must be called before adding any item to
    the box menu.  It initializes the box menu variable for
    later use.

***********************************************************)

    Procedure Initialize_Box_Menu( Var Menu: Box_Menu_Type );

(***********************************************************

  Function: Insert choice on top of the box menu.

    This function accepts a string holding the choice to be
    displayed on the screen and a code which will be
    returned to the program in the event that this choice is
    selected.  Insertion functions only allocate the minimum
    amount of memory necessary to store the data to prevent
    waste.  The previous top item is moved down the list.
    If it fails, it returns false.

***********************************************************)

    Function Insert_Top_Box_Menu( Var Menu: Box_Menu_Type; New_String: String; New_Code: Word ): Boolean;

(***********************************************************

  Function: Insert choice on bottom of the box menu.

    This function accepts a string holding the choice to be
    displayed on the screen and a code which will be
    returned to the program in the event that this choice is
    selected.  The previous bottom item is moved up the
    list.  If it fails, it returns false.

***********************************************************)

    Function Insert_Bottom_Box_Menu( Var Menu: Box_Menu_Type; New_String: String; New_Code: Word ): Boolean;

(***********************************************************

  Procedure: Dispose of the allocated box menu.

    This procedure will deallocate all the memory taken up
    by the given menu and return it for program use.  It
    then will reinitialize the menu.

***********************************************************)

    Procedure Dispose_Box_Menu( Var Menu: Box_Menu_Type );

(***********************************************************

  Procedure: Remove the top choice on the box menu.

    This procedure will remove and deallocate the memory of
    the top choice on the menu.  All succeeding choices are
    moved upward.

***********************************************************)

    Procedure Remove_Top_Box_Menu( Var Menu: Box_Menu_Type );

(***********************************************************

  Procedure: Remove the bottom choice on the box menu.

    This procedure will remove and deallocate the memory of
    the bottom choice on the menu.  All preceding choices
    remain the same.

***********************************************************)

    Procedure Remove_Bottom_Box_Menu( Var Menu: Box_Menu_Type );

(***********************************************************

  Procedure: Offer the box menu.

    This procedure will offer the box menu for a selection
    to be made.  Standard function keys are supported such
    as Up, Down, Left, Right, Page_Up, Page_Down, End and
    Home.  This procedure will return when either Insert,
    Delete, Enter or Escape is pressed and optionally, when
    Left or Right is pressed.  The code returned in
    Selection signifies the last selection made.  Special
    first character search keys are also supported.  The F1
    key will call up the help window.  Window movement keys
    are fully supported.  Lock function key is fully
    supported.

***********************************************************)

    Procedure Offer_Box_Menu( Var Menu: Box_Menu_Type; Var Code: Word );

(***********************************************************

  Function: Insert bar on top of the box menu.

    This function inserts a bar on top of the box menu.  The
    previous top item is moved down the list.  If it fails,
    it returns false.

***********************************************************)

    Function Insert_Bar_Top_Box_Menu( Var Menu: Box_Menu_Type; Data: Byte ): Boolean;

(***********************************************************

  Function: Insert bar on bottom of the box menu.

    This function inserts a bar on the bottom of the box
    menu.  The previous bottom item is moved up the list.
    If it fails, it returns false.

***********************************************************)

    Function Insert_Bar_Bottom_Box_Menu( Var Menu: Box_Menu_Type; Data: Byte ): Boolean;

(***********************************************************

  Procedure: Initialize the bar menu.
    This procedure will initialize the bar menu and prepare
    it for offering the choices.  The procedure accepts data
    as the menu.  The menu is defined as a string consisting
    of the choices separated by two or more spaces. A single
    space will not be registered as separating two choices.
    If the string does not begin with two spaces, they will
    be added.

***********************************************************)

    Procedure Initialize_Bar_Menu( Var Menu: Bar_Menu_Type; Data: String );

(***********************************************************

  Procedure: Offer the bar menu.
    This procedure will offer the bar menu for a selection
    to be made.  Standard function keys are supported such
    as Right, Left, End & Home.  This procedure will return
    when either Enter, Escape, Up or Down is entered and
    will return the selection as a number from 1 to
    Max_Bar_Choices.  Window movement keys are fully
    supported.  Lock function key is fully supported.  The
    F1 key will call up the help screen.  When escape is
    pressed, the code returned is 0.

      Example:
        '  File  Edit  Find error  '  Choice    Selection is
             ^     ^     ^    ^
             |     |     |    |       File          1
             |     |   Single choice  Edit          2
             |     |        |         Find error    3
             Possible Choices

***********************************************************)

    Procedure Offer_Bar_Menu( Var Menu: Bar_Menu_Type; Var Selection: Byte );

(***********************************************************

  Procedure: Offer the bar menu update.
    Like Offer_Bar_Menu, this procedure will offer the bar
    menu for a selection to be made, but this variation also
    will take the command and perform it as if the user had
    entered it like his first command.

      How it works:
        If Command is Left arrow, then the routine will
        move the selection bar over to the next arrow on the
        left and select that.  If Command is Right arrow,
        then the right selection is taken.  If Command is
        Press_Capital_Letters, Press_Lower_Letters or
        Press_Numbers then the first choice of the supplied
        character is chosen.

      Example

        Menu.Command := 0
        ...
        Repeat
          Offer_Bar_Menu_Update( ... Menu.Command );
          ...
          Open and close the appropriate submenus here
          ...
        Until ( Bar menu code is zero )

***********************************************************)

    Procedure Offer_Bar_Menu_Update( Var Menu: Bar_Menu_Type; Var Selection: Byte; Last_Command: Byte; Character: Char );

(***********************************************************

  Function: Get from the top of the box menu.
    This function is designed to return the string at the
    top of the box menu.

***********************************************************)

    Function Get_Top_Box_Menu( Var Menu: Box_Menu_Type ): String;

(***********************************************************

  Procedure: Offer the box menu with pre-search.

    This procedure will offer the box menu for a selection
    to be made but first searches for a choice of the given
    character.  Standard function keys are supported such
    as Up, Down, Left, Right, Page_Up, Page_Down, End and
    Home.  This procedure will return when either Insert,
    Delete, Enter or Escape is pressed and optionally, when
    Left or Right is pressed.  The code returned in
    Selection signifies the last selection made.  Special
    first character search keys are also supported.  The F1
    key will call up the help window.  Window movement keys
    are fully supported.  Lock function key is fully
    supported.

***********************************************************)

    Procedure Offer_Box_Menu_With_Search( Var Menu: Box_Menu_Type; Var Code: Word; First_Character: Char );

{----------------------------------------------------------------------------}

  Implementation

    Const
     { This holds the code used to signal lines. }
      Line_Code = 65535;
     { This holds the exit values. }
      Standard_Exit_Values = [ Press_Enter, Press_Escape, Press_Insert, Press_Delete ];
      Extra_Exit_Values = [ Press_Left_Arrow, Press_Right_Arrow ];

    Type
     { This type holds the data for finding a single choice part. }
      Node_Data_Type = Record
                         Code: Word;
                         Next_Menu_Item,
                         Previous_Menu_Item: Pointer;
                       End;
     { This type holds a single item choice for a box menu. }
      Menu_Item_Node_Type = Record
                              Data: Node_Data_Type;
                              String_Data: String;
                            End;
      Menu_Item_Node_Pointer_Type = ^Menu_Item_Node_Type;
     { This type holds the box menu dimensions. }
      Info_Type = Record
                    Size,
                    Top,
                    Left,
                    Right,
                    Bottom,
                    Width,
                    Row,
                    Column: Byte;
                    High: Pointer;
                  End;

    Var
     { This holds the attribute necessary for the box lines. }
      Line_Attribute: Byte;

{////////////////////////////////////////////////////////////////////////////}

  {$I Menu2.Pas}

{////////////////////////////////////////////////////////////////////////////}

(*************************************************

  Procedure: Initialize bar menu.
    As previously defined.

*************************************************)

    Procedure Initialize_Bar_Menu;
      Begin
        Menu.Data := Data;
        Menu.Choice_Amount := 0;
        With Menu do
          Begin
            If ( Data[ 1 ] <> ' ' )
              then
                Insert( ' ', Data, 1 );
            If ( Data[ 2 ] <> ' ' )
              then
                Insert( ' ', Data, 1 );
            If ( Data[ Length( Data ) ] <> ' ' )
              then
                Data := Data + ' ';
            If ( Data[ Pred( Length( Data ) ) ] <> ' ' )
              then
                Data := Data + ' ';
            Get_Choice_Info( Menu );
            Menu.Last_Choice := 1;
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer bar menu.
    As previously defined.

*************************************************)

    Procedure Offer_Bar_Menu( Var Menu: Bar_Menu_Type; Var Selection: Byte );
      Var
        Store_Mode: Byte;
      Begin
        Store_Mode := TextAttr;
        If Blend_Attribute
          then
            Show_Bar_Menu( Menu, Selection, WhereY, Combine( TextAttr, Normal_Color, True, False ),
                           Combine( TextAttr, Selector_Color, False, True ),
                           Combine( TextAttr, First_Character_Color, True, False ), 0, #0 )
          else
            Show_Bar_Menu( Menu, Selection, WhereY, Normal_Color, Selector_Color, First_Character_Color, 0, #0 );
        If ( Command = Escape_Key )
          then
            Selection := 0;
        TextAttr := Store_Mode;
       {$IFDEF OS2}
        TextBackground( TextAttr shr 4 );
       {$ENDIF}
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer bar menu update.
    As previously defined.

*************************************************)

    Procedure Offer_Bar_Menu_Update( Var Menu: Bar_Menu_Type; Var Selection: Byte; Last_Command: Byte; Character: Char );
      Var
        Store_Mode: Byte;
      Begin
        Store_Mode := TextAttr;
        If Blend_Attribute
          then
            Show_Bar_Menu( Menu, Selection, WhereY, Combine( TextAttr, Normal_Color, True, False ),
                           Combine( TextAttr, Selector_Color, False, True ),
                           Combine( TextAttr, First_Character_Color, True, False ), Last_Command, Character )
          else
            Show_Bar_Menu( Menu, Selection, WhereY, Normal_Color, Selector_Color, First_Character_Color, Last_Command,
                           Character );
        If ( Command = Escape_Key )
          then
            Selection := 0;
        TextAttr := Store_Mode;
       {$IFDEF OS2}
        TextBackground( TextAttr shr 4 );
       {$ENDIF}
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get menu item node.
    This function allocates a new node of the
    given size for the menu box.  If it fails,
    it returns false;

*************************************************)

    Function Get_Menu_Item_Node( Var New_Menu_Item: Pointer; Size: Byte ): Boolean;
      Var
        Total_Size: Word;
      Begin
        Total_Size := ( SizeOf( Node_Data_Type ) + Size );
        GetMem( New_Menu_Item, Total_Size );
        If ( New_Menu_Item <> Nil )
          then
            Begin
              Put_Next( New_Menu_Item, Nil );
              Put_Previous( New_Menu_Item, Nil );
              Get_Menu_Item_Node := True;
            End
          else
            Get_Menu_Item_Node := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose menu item node.
    This procedure disposes of the memory node of
    the given size.

*************************************************)

    Procedure Dispose_Menu_Item_Node( Var Old_Menu_Item_Node_Type: Pointer; Size: Byte );
      Begin
        If ( Old_Menu_Item_Node_Type <> Nil )
          then
            FreeMem( Old_Menu_Item_Node_Type, ( SizeOf( Node_Data_Type ) + Size ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Down size data.
    This procedure gets rid of the strings
    trailing blank spaces.

*************************************************)

    Procedure DownSize_Data( Var The_String: String );
      Var
        String_Length: Byte absolute The_String;
      Begin
        While ( ( String_Length > 0 ) and ( The_String[ String_Length ] = ' ' ) ) do
          Dec( String_Length );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert bottom.
    This procedure inserts the given node into the
    menu structure at the bottom.

*************************************************)

    Procedure Insert_Bottom( Var Menu: Box_Menu_Type; Var New_Item: Pointer );
      Begin
        If ( Menu.Top = Nil )
          then
            Begin
              Menu.Top := New_Item;
              Menu.Bottom := New_Item;
              Menu.Last := New_Item;
            End
          else
            Begin
              Put_Next( Menu.Bottom, New_Item );
              Put_Previous( New_Item, Menu.Bottom );
              Put_Next( New_Item, Nil );
              Menu.Bottom := New_Item;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Insert bottom box menu.
    As previously defined.

*************************************************)

    Function Insert_Bottom_Box_Menu( Var Menu: Box_Menu_Type; New_String: String; New_Code: Word ): Boolean;
      Var
        Okay: Boolean;
        New_Menu_Item_Node: Pointer;
      Begin
        DownSize_Data( New_String );
        Okay := Get_Menu_Item_Node( New_Menu_Item_Node, Succ( Length( New_String ) ) );
        If Okay
          then
            Begin
              Put_Data( New_Menu_Item_Node, New_String );
              Put_Code( New_Menu_Item_Node, New_Code );
              Insert_Bottom( Menu, New_Menu_Item_Node )
            End;
        Insert_Bottom_Box_Menu := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize data.
    This function initializes the data string for
    the menu lines.

*************************************************)

    Procedure Initialize_Data( Data: Byte; Var New_String: String );
      Begin
        Case Data of
          1: New_String := '-';
          2: New_String := '=';
          3: New_String := '';
          4: New_String := '';
        End; { Case }
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Insert bar bottom box menu.
    As previously defined.

*************************************************)

    Function Insert_Bar_Bottom_Box_Menu( Var Menu: Box_Menu_Type; Data: Byte ): Boolean;
      Var
        Okay: Boolean;
        New_String: String;
        New_Menu_Item_Node: Pointer;
      Begin
        Initialize_Data( Data, New_String );
        Okay := Get_Menu_Item_Node( New_Menu_Item_Node, Succ( Length( New_String ) ) );
        If Okay
          then
            Begin
              Put_Data( New_Menu_Item_Node, New_String );
              Put_Code( New_Menu_Item_Node, Line_Code );
              Insert_Bottom( Menu, New_Menu_Item_Node );
            End;
        Insert_Bar_Bottom_Box_Menu := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert top.
    This procedure inserts the given node into the
    menu structure at the top.

*************************************************)

    Procedure Insert_Top( Var Menu: Box_Menu_Type; Var New_Item: Pointer );
      Begin
        If ( Menu.Top = Nil )
          then
            Begin
              Menu.Top := New_Item;
              Menu.Bottom := New_Item;
            End
          else
            Begin
              Put_Next( New_Item, Menu.Top );
              Put_Previous( Menu.Top, New_Item );
              Put_Previous( New_Item, Nil );
              Menu.Top := New_Item;
            End;
        Menu.Last := Menu.Top;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Insert top box menu.
    As previously defined.

*************************************************)

    Function Insert_Top_Box_Menu( Var Menu: Box_Menu_Type; New_String: String; New_Code: Word ): Boolean;
      Var
        Okay: Boolean;
        New_Menu_Item_Node: Pointer;
      Begin
        DownSize_Data( New_String );
        Okay := Get_Menu_Item_Node( New_Menu_Item_Node, Succ( Length( New_String ) ) );
        If Okay
          then
            Begin
              Put_Data( New_Menu_Item_Node, New_String );
              Put_Code( New_Menu_Item_Node, New_Code );
              Insert_Top( Menu, New_Menu_Item_Node );
            End;
        Insert_Top_Box_Menu := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Insert bar top box menu.
    As previously defined.

*************************************************)

    Function Insert_Bar_Top_Box_Menu( Var Menu: Box_Menu_Type; Data: Byte ): Boolean;
      Var
        Okay: Boolean;
        New_String: String;
        New_Menu_Item_Node: Pointer;
      Begin
        Initialize_Data( Data, New_String );
        Okay := Get_Menu_Item_Node( New_Menu_Item_Node, Succ( Length( New_String ) ) );
        If Okay
          then
            Begin
              Put_Data( New_Menu_Item_Node, New_String );
              Put_Code( New_Menu_Item_Node, Line_Code );
              Insert_Top( Menu, New_Menu_Item_Node );
            End;
        Insert_Bar_Top_Box_Menu := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Eliminate menu.
    This routine is called when the last node of
    the menu is supposed to be eliminated.

*************************************************)

    Procedure Eliminate_Menu( Var Menu: Box_Menu_Type );
      Var
        The_String: String;
      Begin
        The_String := Get_Data( Menu.Top );
        Dispose_Menu_Item_Node( Menu.Top, Succ( Length( The_String ) ) );
        Menu.Bottom := Nil;
        Menu.Top := Nil;
        Menu.Last := Nil;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove bottom box menu.
    As previously defined.

*************************************************)

    Procedure Remove_Bottom_Box_Menu( Var Menu: Box_Menu_Type );
      Var
        The_String: String;
        Old_Menu_Item_Node: Pointer;
      Begin
        If ( Menu.Bottom <> Nil )
          then
            If ( Menu.Bottom <> Menu.Top )
              then
                Begin
                  Old_Menu_Item_Node := Menu.Bottom;
                  If ( Menu.Last = Menu.Bottom )
                    then
                      Menu.Last := Get_Previous( Menu.Bottom );
                  Menu.Bottom := Get_Previous( Menu.Bottom );
                  Put_Next( Menu.Bottom, Nil );
                  The_String := Get_Data( Old_Menu_Item_Node );
                  Dispose_Menu_Item_Node( Old_Menu_Item_Node, Succ( Length( The_String ) ) );
                End
              else
                Eliminate_Menu( Menu );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove top box menu.
    As previously defined.

*************************************************)

    Procedure Remove_Top_Box_Menu( Var Menu: Box_Menu_Type );
      Var
        The_String: String;
        Old_Menu_Item_Node: Pointer;
      Begin
        If ( Menu.Top <> Nil )
          then
            If ( Menu.Bottom <> Menu.Top )
              then
                Begin
                  Old_Menu_Item_Node := Menu.Top;
                  If ( Menu.Last = Menu.Top )
                    then
                      Menu.Last := Get_Next( Menu.Top );
                  Menu.Top := Get_Next( Menu.Top );
                  Put_Previous( Menu.Top, Nil );
                  The_String := Get_Data( Old_Menu_Item_Node );
                  Dispose_Menu_Item_Node( Old_Menu_Item_Node, Succ( Length( The_String ) ) );
                End
              else
                Eliminate_Menu( Menu );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose box menu.
    As previously defined.

*************************************************)

    Procedure Dispose_Box_Menu( Var Menu: Box_Menu_Type );
      Var
        The_String: String;
        Old_Node_Pointer: Pointer;
      Begin
        While ( Menu.Top <> Nil ) do
          Begin
            Old_Node_Pointer := Menu.Top;
            Menu.Top := Get_Next( Menu.Top );
            The_String := Get_Data( Old_Node_Pointer );
            Dispose_Menu_Item_Node( Old_Node_Pointer, Succ( Length( The_String ) ) );
          End;
        Menu.Bottom := Nil;
        Menu.Last := Nil;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write truncated.
    This procedure writes the given data on the
    screen window, after truncating it to the
    current window size.

*************************************************)

    Procedure Write_Truncated( Field: String; Start, Row, Normal_Color, First_Color, Width: Byte );
      Var
        Count: Byte;
      Begin
        Truncate_String( Field, Start, Width );
        GotoXY( 1, Row );
        If HighLight_Letter
          then
            Begin
              TextAttr := First_Color;
             {$IFDEF OS2}
              TextBackground( TextAttr shr 4 );
             {$ENDIF}
              Write( Screen, Field[ 1 ] );
              Delete( Field, 1, 1 );
            End;
        TextAttr := Normal_Color;
       {$IFDEF OS2}
        TextBackground( TextAttr shr 4 );
       {$ENDIF}
        Write( Screen, Field );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Decide attribute.
    This procedure decides which attribute to
    display for the particular line of the box
    menu.

*************************************************)

    Procedure Decide_Attribute( Field: String; Start, Row, Normal_Color, First_Color, Width: Byte; Code: Word );
      Begin
        If ( Code = Line_Code )
          then
            Write_Truncated( Field, Start, Row, Line_Attribute, Line_Attribute, Width )
          else
            Write_Truncated( Field, Start, Row, Normal_Color, First_Color, Width );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize screen.
    This procedure displays the initial box menu
    on the screen window.

*************************************************)

    Procedure Initialize_Screen( Var Info: Info_Type; Menu: Box_Menu_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Var
        Value: Byte;
        Pointer_1,
        Pointer_2: Pointer;
      Begin
        Info.Row := Info.Top;
        Pointer_1 := Menu.Top;
        Pointer_2 := Menu.Top;
        While ( Pointer_2 <> Info.High ) do
          Begin
            Pointer_2 := Get_Next( Pointer_2 );
            Inc( Info.Row );
            If ( Info.Row > Info.Bottom )
              then
                Begin
                  Info.Row := Info.Bottom;
                  Pointer_1 := Get_Next( Pointer_1 );
                End;
          End;
        Value := Pred( Info.Top );
        While ( Pointer_1 <> Nil ) and ( Value < Info.Bottom ) do
          Begin
            Inc( Value );
            Decide_Attribute( Get_Data( Pointer_1 ), Info.Column, Value, Normal_Color, First_Color, Info.Width,
                              Get_Code( Pointer_1 ) );
            Pointer_1 := Get_Next( Pointer_1 );
          End;
        HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update.
    This procedure updates the current screen
    window after something is modified.

*************************************************)

    Procedure Update( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Var
        Counter: Byte;
        The_Pointer: Pointer;
      Begin
        Write_Truncated( Get_Data( Info.High ), Info.Column, Info.Row, Normal_Color, First_Color, Info.Width );
        HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
        The_Pointer := Info.High;
        If ( Info.Row <> Info.Top )
          then
            For Counter := Pred( Info.Row ) downto Info.Top do
              If ( The_Pointer <> Nil )
                then
                  Begin
                    The_Pointer := Get_Previous( The_Pointer );
                    If ( The_Pointer <> Nil )
                      then
                        Decide_Attribute( Get_Data( The_Pointer ), Info.Column, Counter, Normal_Color, First_Color, Info.Width,
                                          Get_Code( The_Pointer ) );
                  End;
        The_Pointer := Info.High;
        If ( Info.Row <> Info.Bottom )
          then
            For Counter := Succ( Info.Row ) to Info.Bottom do
              If ( The_Pointer <> Nil )
                then
                  Begin
                    The_Pointer := Get_Next( The_Pointer );
                    If ( The_Pointer <> Nil )
                      then
                        Decide_Attribute( Get_Data( The_Pointer ), Info.Column, Counter, Normal_Color, First_Color, Info.Width,
                                          Get_Code( The_Pointer ) );
                  End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Goto search.
    This function moves the current selection bar
    to the next choice that begins with the given
    selection character.  If it fails, it returns
    false.

*************************************************)

    Function Goto_Search( Character: Char; Var Info: Info_Type; Menu: Box_Menu_Type; Normal_Color, Selector_Color,
                          First_Color: Byte ): Boolean;
      Var
        Mark: Pointer;
        Look: String;
      Begin
        Goto_Search := False;
        Mark := Info.High;
        Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
        Repeat
          If ( Get_Next( Info.High ) <> Nil )
            then
              Begin
                Info.High := Get_Next( Info.High );
                Inc( Info.Row );
              End;
          Look := Get_Data( Info.High );
        Until ( ( Get_Next( Info.High ) = Nil ) or ( Look[ 1 ] = Character ) );
        If ( Look[ 1 ] <> Character )
          then
            Begin
              While ( Info.High <> Menu.Top ) do
                Begin
                  Info.High := Get_Previous( Info.High );
                  Dec( Info.Row );
                End;
              Look := Get_Data( Info.High );
              While ( ( Info.High <> Mark ) and ( Look[ 1 ] <> Character ) ) do
                Begin
                  If ( Get_Next( Info.High ) <> Nil )
                    then
                      Begin
                        Info.High := Get_Next( Info.High );
                        Inc( Info.Row );
                      End;
                  Look := Get_Data( Info.High );
                End;
            End;
        Goto_Search := ( Look[ 1 ] = Character );
        If ( ( Info.Row > Info.Bottom ) or ( Info.Row < Info.Top ) )
          then
            Initialize_Screen( Info, Menu, Normal_Color, Selector_Color, First_Color );
        HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto down.
    This procedure moves the selection bar down
    one choice.

*************************************************)

    Procedure Goto_Down( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte; Amount: Integer );
      Var
        Done: Boolean;
      Begin
        If ( Find_Next( Info.High ) <> Nil )
          then
            Begin
              Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
              Repeat
                If ( Info.Row < Info.Bottom )
                  then
                    Begin
                      Inc( Info.Row );
                      Info.High := Get_Next( Info.High );
                    End
                  else
                    Begin
                      Scroll_Window_Up;
                      Info.High := Get_Next( Info.High );
                      Decide_Attribute( Get_Data( Info.High ), Info.Column, Info.Bottom, Normal_Color, First_Color, Info.Width,
                                        Get_Code( Info.High ) );
                    End;
                Dec( Amount );
                Done := ( ( Get_Code( Info.High ) <> Line_Code ) and ( Amount < 1 ) );
              Until ( Done or ( Get_Next( Info.High ) = Nil ) );
            End;
        HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto up.
    This procedure moves the selection bar up
    one choice.

*************************************************)

    Procedure Goto_Up( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte; Amount: Integer );
      Var
        Done: Boolean;
      Begin
        If ( Find_Previous( Info.High ) <> Nil )
          then
            Begin
              Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
              Repeat
                If ( Info.Row > Info.Top )
                  then
                    Begin
                      Dec( Info.Row );
                      Info.High := Get_Previous( Info.High );
                    End
                  else
                    Begin
                      Scroll_Window_Down;
                      Info.High := Get_Previous( Info.High );
                      Decide_Attribute( Get_Data( Info.High ), Info.Column, Info.Top, Normal_Color, First_Color, Info.Width,
                                        Get_Code( Info.High ) );
                    End;
                Dec( Amount );
                Done := ( ( Get_Code( Info.High ) <> Line_Code ) and ( Amount < 1 ) );
              Until ( Done or ( Get_Previous( Info.High ) = Nil ) );
            End;
        HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto page down.
    This procedure moves the selection screen down
    one page, then moves the selection bar to the
    appropriate row.

*************************************************)

    Procedure Goto_Page_Down( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte; Menu: Box_Menu_Type );
      Var
        Counter: Byte;
        Point: Pointer;
      Begin
        If ( Find_Next( Info.High ) <> Nil )
          then
            Begin
              If ( Info.Row = Info.Bottom )
                then
                  Begin
                    For Counter := Info.Top to Pred( Info.Bottom ) do
                      If ( Find_Next( Info.High ) <> Nil )
                        then
                          Info.High := Get_Next( Info.High );
                    If ( Get_Code( Info.High ) = Line_Code )
                      then
                        If ( Find_Previous( Info.High ) <> Nil )
                          then
                            Info.High := Find_Previous( Info.High )
                          else
                            If ( Find_Next( Info.High ) <> Nil )
                              then
                                Info.High := Find_Next( Info.High )
                              else
                                Write_Error( 204, 'Goto_Page_Down' );
                    ClrScr;
                    Point := Info.High;
                    Info.Row := Pred( Info.Bottom );
                    Decide_Attribute( Get_Data( Point ), Info.Column, Info.Bottom, Normal_Color, First_Color, Info.Width,
                                      Get_Code( Point ) );
                    HighA( Info.Bottom, Info.Left, Info.Size, Selector_Color );
                    While ( ( Get_Previous( Point ) <> Nil ) and ( Info.Row >= Info.Top ) ) do
                      Begin
                        Point := Get_Previous( Point );
                        Decide_Attribute( Get_Data( Point ), Info.Column, Info.Row, Normal_Color, First_Color, Info.Width,
                                          Get_Code( Point ) );
                        Dec( Info.Row );
                      End;
                    Info.Row := Info.Bottom;
                  End
                else
                  Begin
                    Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
                    While ( ( Info.Row < Info.Bottom ) and ( Find_Next( Info.High ) <> Nil ) ) do
                      Begin
                        Info.High := Get_Next( Info.High );
                        Inc( Info.Row );
                      End;
                    If ( Get_Code( Info.High ) = Line_Code )
                      then
                        Begin
                          If ( Find_Next( Info.High ) <> Nil )
                            then
                              Info.High := Find_Next( Info.High )
                            else
                              If ( Find_Previous( Info.High ) <> Nil )
                                then
                                  Info.High := Find_Previous( Info.High )
                                else
                                  Write_Error( 204, 'Goto_Page_Down' );
                          Initialize_Screen( Info, Menu, Normal_Color, Selector_Color, First_Color );
                        End;
                    HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto page up.
    This procedure moves the selection screen up
    one page, then moves the selection bar to the
    appropriate row.

*************************************************)

    Procedure Goto_Page_Up( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte; Menu: Box_Menu_Type );
      Var
        Counter: Byte;
        Point: Pointer;
      Begin
        If ( Find_Previous( Info.High ) <> Nil )
          then
            Begin
              If ( Info.Row = Info.Top )
                then
                  Begin
                    For Counter := Info.Top to Pred( Info.Bottom ) do
                      If ( Find_Previous( Info.High ) <> Nil )
                        then
                          Info.High := Get_Previous( Info.High );
                    If ( Get_Code( Info.High ) = Line_Code )
                      then
                        If ( Find_Next( Info.High ) <> Nil )
                          then
                            Info.High := Find_Next( Info.High )
                          else
                            If ( Find_Previous( Info.High ) <> Nil )
                              then
                                Info.High := Find_Previous( Info.High )
                              else
                                Write_Error( 204, 'Goto_Page_Up' );
                    ClrScr;
                    Point := Info.High;
                    Info.Row := Succ( Info.Top );
                    Decide_Attribute( Get_Data( Point ), Info.Column, Info.Top, Normal_Color, First_Color, Info.Width,
                                      Get_Code( Point ) );
                    HighA( Info.Top, Info.Left, Info.Size, Selector_Color );
                    While ( ( Get_Next( Point ) <> Nil ) and ( Info.Row <= Info.Bottom ) ) do
                      Begin
                        Point := Get_Next( Point );
                        Decide_Attribute( Get_Data( Point ), Info.Column, Info.Row, Normal_Color, First_Color, Info.Width,
                                          Get_Code( Point ) );
                        Inc( Info.Row );
                      End;
                    Info.Row := Info.Top;
                  End
                else
                  Begin
                    Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
                    For Counter := Succ( Info.Top ) to Info.Row do
                      If ( Find_Previous( Info.High ) <> Nil )
                        then
                          Info.High := Get_Previous( Info.High );
                    Info.Row := Info.Top;
                    If ( Get_Code( Info.High ) = Line_Code )
                      then
                        Begin
                          If ( Find_Previous( Info.High ) <> Nil )
                            then
                              Info.High := Find_Previous( Info.High )
                            else
                              If ( Find_Next( Info.High ) <> Nil )
                                then
                                  Info.High := Find_Next( Info.High )
                                else
                                  Write_Error( 204, 'Goto_Page_Up' );
                          ClrScr;
                          Point := Info.High;
                          Repeat
                            Decide_Attribute( Get_Data( Point ), Info.Column, Info.Row, Normal_Color, First_Color, Info.Width,
                                              Get_Code( Point ) );
                            Point := Get_Next( Point );
                            Inc( Info.Row );
                          Until ( Info.Row > Info.Bottom  ) or ( Point = Nil );
                          Info.Row := Info.Top;
                        End;
                    HighA( Info.Row, Info.Left, Info.Size, Selector_Color );
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto right.
    This procedure moves the selection screen to
    the right by one character.

*************************************************)

    Procedure Goto_Right( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Begin
        If ( Info.Column < ( 255 - Info.Right ) )
          then
            Begin
              Inc( Info.Column );
              Update( Info, Normal_Color, Selector_Color, First_Color );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto left.
    This procedure moves the selection screen to
    the left by one character.

*************************************************)

    Procedure Goto_Left( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Begin
        If ( Info.Column > 1 )
          then
            Begin
              Dec( Info.Column );
              Update( Info, Normal_Color, Selector_Color, First_Color );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto home.
    This procedure moves the selection screen to
    the home position.  The selector screen starts
    at position one.

*************************************************)

    Procedure Goto_Home( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Begin
        Info.Column := 1;
        Update( Info, Normal_Color, Selector_Color, First_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Goto end.
    This procedure moves the selection screen to
    the end position.  The selector screen ends
    at the last position.

*************************************************)

    Procedure Goto_End( Var Info: Info_Type; Normal_Color, Selector_Color, First_Color: Byte );
      Begin
        ClrScr;
        Info.Column := ( 255 - Info.Right );
        Update( Info, Normal_Color, Selector_Color, First_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Show box menu.
    This procedure is the core procedure that
    allows the selection of one of the choices
    from the possibilities.

*************************************************)

    Procedure Show_Box_Menu( Menu: Box_Menu_Type; Var Selection: Pointer; Normal_Color, Selector_Color,
                             First_Color: Byte; Allow_Arrows: Boolean; First_Character: Char );
      Var
        Okay: Boolean;
        Info: Info_Type;
        Character: Char;
        Exit_Values: Set of Byte;
      Begin
        If ( Menu.Top = Nil )
          then
            Begin
              Selection := Nil;
              Command := Press_Escape;
            End
          else
            Begin
              If Allow_Arrows
                then
                  Exit_Values := Standard_Exit_Values
                else
                  Exit_Values := Standard_Exit_Values + Extra_Exit_Values;
              Line_Attribute := TextAttr;
              TextAttr := Normal_Color;
             {$IFDEF OS2}
              TextBackground( TextAttr shr 4 );
             {$ENDIF}
              Info.Column := 1;
              Info.Top := 1;
              Info.Left := 1;
              Info.Right := Succ( Right_Of_Window^ - Left_Of_Window^ );
              Info.Bottom := Succ( Bottom_Of_Window^ - Top_Of_Window^ );
              Info.Size := Succ( Info.Right - Info.Left );
              Info.High := Menu.Last;
              Info.Width := Succ( Right_Of_Window^ - Left_Of_Window^ );
              ClrScr;
              Initialize_Screen( Info, Menu, Normal_Color, Selector_Color, First_Color );
              If ( First_Character <> ' ' )
                then
                  Okay := Goto_Search( UpCase( First_Character ), Info, Menu, Normal_Color, Selector_Color, First_Color );
              Repeat
                Get_Command( Character, Command );
                Case Command of
                  Pointer_Up:
                    Goto_Up( Info, Normal_Color, Selector_Color, First_Color, Adjust_Amount );
                  Pointer_Down:
                    Goto_Down( Info, Normal_Color, Selector_Color, First_Color, Adjust_Amount );
                  Press_Numbers,
                  Press_Capital_Letters,
                  Press_Lower_Letters:
                    Begin
                      Okay := Goto_Search( UpCase( Character ), Info, Menu, Normal_Color, Selector_Color, First_Color );
                      If ( Sensitive_Mode and Okay )
                        then
                          Command := Press_Enter;
                    End;
                  Press_Down_Arrow,
                  Press_Tab:
                    Goto_Down( Info, Normal_Color, Selector_Color, First_Color, 1 );
                  Press_Up_Arrow,
                  Press_Shift_Tab:
                    Goto_Up( Info, Normal_Color, Selector_Color, First_Color, 1 );
                  Press_Page_Down:
                    Goto_Page_Down( Info, Normal_Color, Selector_Color, First_Color, Menu );
                  Press_Page_Up:
                    Goto_Page_Up( Info, Normal_Color, Selector_Color, First_Color, Menu );
                  Press_Right_Arrow,
                  Press_Word_Right:
                    If Allow_Arrows
                      then
                        Goto_Right( Info, Normal_Color, Selector_Color, First_Color );
                  Press_Left_Arrow,
                  Press_Word_Left:
                    If Allow_Arrows
                      then
                        Goto_Left( Info, Normal_Color, Selector_Color, First_Color );
                  Press_Home:
                    Goto_Home( Info, Normal_Color, Selector_Color, First_Color );
                  Press_End:
                    Goto_End( Info, Normal_Color, Selector_Color, First_Color );
                  Pointer_Button1_Double:
                    Command := Press_Enter;
                  else
                    Extended_Capabilities( Command, Get_Code( Info.High ) );
                End; { Case }
              Until ( Command in Exit_Values );
              Selection := Info.High;
              If ( Not Leave_Choice_On )
                then
                  Norm( Info.Row, Info.Left, Info.Size, Normal_Color, First_Color );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer box menu.
    As previously defined.

*************************************************)

    Procedure Offer_Box_Menu( Var Menu: Box_Menu_Type; Var Code: Word );
      Var
        Old_Attribute,
        First_Attribute,
        Normal_Attribute,
        HighLight_Attribute: Byte;
        The_Selection: Pointer;
      Begin
        Old_Attribute := TextAttr;
        If Blend_Attribute
          then
            Begin
              First_Attribute := Combine( TextAttr, First_Character_Color, True, False );
              Normal_Attribute := Combine( TextAttr, Normal_Color, True, False );
              HighLight_Attribute := Combine( TextAttr, Selector_Color, False, True );
            End
          else
            Begin
              First_Attribute := First_Character_Color;
              Normal_Attribute := Normal_Color;
              HighLight_Attribute := Selector_Color;
            End;
        Show_Box_Menu( Menu, The_Selection, Normal_Attribute, HighLight_Attribute, First_Attribute, ( not Exit_On_Arrows ),
                       ' ' );
        Menu.Last := The_Selection;
        Case Command of
          Left_Key,
          Right_Key,
          Escape_Key: Code := 0;
          else Code := Get_Code( The_Selection );
        End; { Case }
        TextAttr := Old_Attribute;
       {$IFDEF OS2}
        TextBackground( TextAttr shr 4 );
       {$ENDIF}
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer box menu.
    As previously defined.

*************************************************)

    Procedure Offer_Box_Menu_With_Search( Var Menu: Box_Menu_Type; Var Code: Word; First_Character: Char );
      Var
        Old_Attribute,
        First_Attribute,
        Normal_Attribute,
        HighLight_Attribute: Byte;
        The_Selection: Pointer;
      Begin
        Old_Attribute := TextAttr;
        If Blend_Attribute
          then
            Begin
              First_Attribute := Combine( TextAttr, First_Character_Color, True, False );
              Normal_Attribute := Combine( TextAttr, Normal_Color, True, False );
              HighLight_Attribute := Combine( TextAttr, Selector_Color, False, True );
            End
          else
            Begin
              First_Attribute := First_Character_Color;
              Normal_Attribute := Normal_Color;
              HighLight_Attribute := Selector_Color;
            End;
        Show_Box_Menu( Menu, The_Selection, Normal_Attribute, HighLight_Attribute, First_Attribute, ( not Exit_On_Arrows ),
                       First_Character );
        Menu.Last := The_Selection;
        Case Command of
          Left_Key,
          Right_Key,
          Escape_Key: Code := 0;
          else Code := Get_Code( The_Selection );
        End; { Case }
        TextAttr := Old_Attribute;
       {$IFDEF OS2}
        TextBackground( TextAttr shr 4 );
       {$ENDIF}
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure Initialize box menu.
    As previously defined.

*************************************************)

    Procedure Initialize_Box_Menu( Var Menu: Box_Menu_Type );
      Begin
        Menu.Top := Nil;
        Menu.Bottom := Nil;
        Menu.Last := Nil;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get from the top of the box menu.
    As previously defined.

*************************************************)

    Function Get_Top_Box_Menu( Var Menu: Box_Menu_Type ): String;
      Begin
        If ( Menu.Top <> Nil )
          then
            Get_Top_Box_Menu := Get_Data( Menu.Top )
          else
            Get_Top_Box_Menu := '';
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Main initializing section.
    First, initialize the default help screens.
    Then link up with core unit.

*************************************************)

    Begin
     {$IFNDEF VER40}
      Help_Screen := Default_Help_Screen;
      Display_Status := Default_Help_Screen;
     {$ENDIF}
    End.


