{$F+} {Compiler Directive: Generate far procedure calls: On }
{$O+} {Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  String_Utilities (formally Strings)
    version 2.21

  This unit holds several procedures which involve the use of strings.

  Purpose:
    This unit holds a collection of procedures and functions which perform
    operations on strings or generate strings as output.

  How it works:
    Get Time & Get Date  - Gets system values and converts them into
                           readable strings for output.
    Write Out Number     - Converts any integer into a readable string.
    Remove Blanks        - Removes unwanted blanks from the string.
    Spread Out String    - Combines both left and right justification.
    Comma a String       - Adds commas (or Delimiters) to a string of numbers.
    Add, Subtract, Mod,
    Multiply, Divide     - Performs mathematical functions on numbers
                           represented by strings

  Features:
    Mathematical functions can support integers limited only by the size of
      the strings.  This allows very large integer variables.
    Write out number supports integers up to the maximum size of long
      integers.
    Routines optimized for speed and simplicity.

  Limitations:
    Mathematics does not support scientific exponents.
    Can only process strings up to the maximum string length.
    Due to methods, mathematics may perform slowly for large strings.

  Versions:
    2.0 - Optimized the number to text conversion routines.
    2.1 - Added the Expand and Center procedures.
    2.11 - Optimized Remove_All_Blanks to work faster.
    2.2 - Updated unit to allow for alternate country formats.
    2.21 - Added coding to compile under Speed Pascal/2.

  Copyright 1989, 1992, 1996,  All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

{$IFNDEF OS2}
 Unit String_Utilities;
{$ELSE}
 Unit String_U;
{$ENDIF}

  Interface

    Uses
     {$IFDEF OS2}
      BseDOS,
      BseErr,
     {$ENDIF}
      DOS;

(***********************************************************

  American only
    This constant is defined to allow the programmer to 
    override the enhancement to this unit that was 
    introduced with version 1.2.  Set to true to override.

***********************************************************)

    Const
      American_Only = False;

(***********************************************************

  Procedure: Get the time.

    This procedure gets the time from the system and returns
    it as a readable string.
    Example:
      If time is 2:30 pm
      Procedure returns 'Two Thirty PM'

***********************************************************)

    Procedure Get_Time( Var Data: String );

(***********************************************************

  Procedure: Get the date.

    This procedure gets the data from the system and returns
    it as a readable string.
    Example:
      If date is 1-2-1990
      Procedure returns 'Tuesday, January Second, Nineteen
                         Hundred and Ninety'

***********************************************************)

    Procedure Get_Date( Var Data: String );

(***********************************************************

  Function: Write out the number.

    This function takes the number and returns it as a
    readable string.
    Example:
      If number is 189458596
      Function returns 'One Hundred Eighty Nine Million,
                        Four Hundred Fifty Eight Thousand,
                        Five Hundred and Ninety Six'

***********************************************************)

    Function Write_Out_Number( Number: LongInt ): String;

(***********************************************************

  Procedure: Remove all blanks from string.

    This procedure takes any string and removes all spaces,
    #32, from that string.

***********************************************************)

    Procedure Remove_All_Blanks( Var Data: String );

(***********************************************************

  Procedure: Remove double blanks from string.

    This procedure takes any string and removes all
    consecutive spaces from that string replacing them with
    single spaces.

***********************************************************)

    Procedure Remove_Double_Blanks( Var Data: String );

(***********************************************************

  Procedure: Spread out the string.

    This procedure takes a data string and spreads it out
    evenly to fit the size by adding spaces internally.

***********************************************************)

    Procedure Spread_Out_String( Var Data: String; Size: Byte );

(***********************************************************

  Procedure: Comma a string.

    This procedure takes a number string and adds comma or
    the country specific separator to put it in a standard
    numerical form.

***********************************************************)

    Procedure Comma_The_String( Var Data: String );

(***********************************************************

  Procedure: Capitalize the string.

    This procedure takes a alphanumeric string and changes
    all the small letters into capitals.

***********************************************************)

    Procedure Capitalize( Var Data: String );

(***********************************************************

  Function: Add integer strings.

    This function takes two strings representing numerical
    integers and adds them together returning a third string
    with the result.

***********************************************************)

    Function Add_Integer_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Subtract integer strings.

    This function takes two strings representing numerical
    integers and subtracts the second from the first
    returning a third string with the result.

***********************************************************)

    Function Subtract_Integer_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Multiply integer strings.

    This function takes two strings representing numerical
    integers and multiplies them together returning a third
    string with the result.

***********************************************************)

    Function Multiply_Integer_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Divide integer strings.

    This function takes two strings representing numerical
    integers and divides the second from the first returning
    a third string with the result.

***********************************************************)

    Function Divide_Integer_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Mod integer strings.

    This function takes two strings representing numerical
    integers and divides the second from the first,
    returning a third string with the remainder as a result.

***********************************************************)

    Function Mod_Integer_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Add real strings.

    This function takes two strings representing numerical
    floating point numbers and adds them together, returning
    a third string with the result.

***********************************************************)

    Function Add_Real_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Subtract real strings.

    This function takes two strings representing numerical
    floating point numbers and subtracts the second from the
    first returning a third string with a result.

***********************************************************)

    Function Subtract_Real_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Multiply real strings.

    This function takes two strings representing numerical
    floating point numbers and multiplies them together,
    returning a third string with a result.

***********************************************************)

    Function Multiply_Real_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Divide real strings.

    This function takes two strings representing numerical
    floating point numbers and divides the second from the
    first, returning the result in the third string.
    This function may take a long time to calculate the
      result depending on the given values and the
      processor's speed.

***********************************************************)

    Function Divide_Real_String( Number1, Number2: String ): String;

(***********************************************************

  Function: Expand.
    This function expands the given character to create a
    string of Width length.

***********************************************************)

    Function Expand( Data: Char; Width: Byte ): String;

(***********************************************************

  Function: Center.
    This function returns a string containing the given
    string extended on both sides to fit the given Width.

***********************************************************)

    Function Center( Data: String; Width: Byte; Fill: Char ): String;

(***********************************************************

  Function: Push to.
    This function returns a string extended with the Fill
    character to allow both given strings to fit the given
    limit.

***********************************************************)

    Function Push_To( Front, Back: String; Limit: Byte; Fill: Char ): String;

{-----------------------------------------------------------------------------}

  Implementation

    {$DEFINE Quick} { Generates alternate code to speed processes up. }

    Const
      { These constants make converting numbers to strings faster. }
      Singles: Array[ 0 .. 19 ] of String[ 10 ] =
        ( '', 'One ', 'Two ', 'Three ', 'Four ', 'Five ', 'Six ', 'Seven ',
          'Eight ', 'Nine ', 'Ten ', 'Eleven ', 'Twelve ', 'Thirteen ',
          'Fourteen ', 'Fifteen ', 'Sixteen ', 'Seventeen ', 'Eighteen ',
          'Nineteen ' );
      Tens: Array[ 0 .. 9 ] of String[ 10 ] =
        ( '', '', 'Twenty ', 'Thirty ', 'Forty ', 'Fifty ', 'Sixty ',
          'Seventy ', 'Eighty ', 'Ninety ' );
      WeekDays: Array[ 0 .. 6 ] of String[ 11 ] =
        ( 'Sunday, ', 'Monday, ', 'Tuesday, ', 'Wednesday, ', 'Thursday, ',
          'Friday, ', 'Saturday, ' );
      Months: Array[ 1 .. 12 ] of String[ 12 ] =
        ( 'January ', 'February ', 'March ', 'April ', 'May ', 'June ',
          'July ', 'August ', 'September ', 'October ', 'November ', 'December ' );
      Days: Array[ 1 .. 31 ] of String[ 18 ] =
        ( 'First, ', 'Second, ', 'Third, ', 'Forth, ', 'Fifth, ', 'Sixth, ',
          'Seventh, ', 'Eighth, ', 'Ninth, ', 'Tenth, ', 'Eleventh, ',
          'Twelfth, ', 'Thirteenth, ', 'Fourteenth, ', 'Fifteenth, ',
          'Sixteenth, ', 'Seventeenth, ', 'Eighteenth, ', 'Nineteenth, ',
          'Twentieth, ', 'Twenty First, ', 'Twenty Second, ', 'Twenty Third, ',
          'Twenty Forth, ', 'Twenty Fifth, ', 'Twenty Sixth, ',
          'Twenty Seventh, ', 'Twenty Eighth, ', 'Twenty Ninth, ',
          'Thirtieth, ', 'Thirty First, ' );

    Type
     { Sign variable type. }
      Sign = ( Positive, Negative );
     { These are defined for getting the country specific infomation from the DOS. }
      Z_String_2 = Packed array[ 1 .. 2 ] of Char;
      Z_String_5 = Packed array[ 1 .. 5 ] of Char;
      Country_Info = Packed Record
                              Date_Format: Integer;
                              Case Byte of
                                0: ( Currency_Symbol_1,
                                     Thousands_Separator_1,
                                     Decimal_Separator_1: Z_String_2;
                                     Extra_1: Array[ 1 .. 30 ] of Byte );
                                1: ( Currency_Symbol_2: Z_String_5;
                                     Thousands_Separator_2,
                                     Decimal_Separator_2: Z_String_2;
                                     Date_Separator: Z_String_2;
                                     Time_Separator: Z_String_2;
                                     Currency_Symbol_Location,
                                     Currency_Decimal_Places: Byte;
                                     Time_format: Byte;
                                     Extended_ASCII_Map: Pointer;
                                     List_Separator: Z_String_2;
                                     Extra_2: Array[ 1 .. 14 ] of Byte );
                            End;

    Var
     { These variable holds the country specific information. }
      Date_Format: ( American, European, Japanese );
      Currency_Symbol: String[ 4 ];
      Thousands_Separator: Char;
      Decimal_Separator: Char;
      Date_Separator: Char;
      Time_Separator: Char;
      Currency_Location: ( Before, After, Spaced_Before, Spaced_After );
      Currency_Decimal_Places: Byte;
      Time_Format: ( Hour_12, Hour_24 );
      List_Separator: Char;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write error.
    If a value other than zero is passed to it,
    this procedure writes an error message to the
    screen and halts.

*************************************************)

    Procedure Write_Error( Result: Word; Sentence: String );
      Begin
        If ( Result <> 0 )
          then
            Begin
              WriteLn( 'Error ', Result, ' in ', Sentence, '.' );
             {$IFNDEF VER40}
              RunError( Result );
             {$ELSE}
              Halt( Result );
             {$ENDIF}
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Single.
    This function returns a string corresponding
    to a number from zero to nineteen.

*************************************************)

    Function Single( Number: ShortInt ): String;
      Begin
        Single := Singles[ Number ];
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Ten.
    This function returns a string corresponding
    to a number from zero to ninety nine.

*************************************************)

    Function Ten( Number: ShortInt ): String;
      Begin
        If ( Number > 19 )
          then
            Ten := Tens[ Number Div 10 ] + Single( Number Mod 10 )
          else
            Ten := Single( Number );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Hundreds.
    This function returns a string corresponding
    to a number from zero to nine hundred and
    ninety nine.

*************************************************)

    Function Hundreds( Number: Integer ): String;
      Var
        Small: Integer;
      Begin
        If ( Number < 100 )
          then
            Hundreds := Ten( Number )
          else
            Begin
              Small := ( Number Mod 100 );
              If Small = 0
                then
                  Hundreds := Singles[ Number Div 100 ] + 'hundred '
                else
                  Hundreds := Singles[ Number Div 100 ] + 'hundred and ' + Ten( Number Mod 100 );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Thousands.
    This function returns a string corresponding
    to a number from zero to nine hundred and
    ninety nine thousand, nine hundred and ninety
    nine.

*************************************************)

    Function Thousands( Number: LongInt ): String;
      Var
        Small,
        Large: Integer;
      Begin
        If ( Number < 1000 )
          then
            Thousands := Hundreds( Number )
          else
            Begin
              Small := ( Number Mod 1000 );
              Large := ( Number Div 100 );
              If ( Small = 0 )
                then
                  Thousands := Hundreds( Number Div 1000 ) + 'thousand '
                else
                  Case Large of
                    10: Thousands := Hundreds( Number Div 1000 ) + 'thousand and ' + Hundreds( Small );
                    11,12,13,
                    14,15,16,
                    17,18,19: Thousands := Hundreds( Number );
                    else Thousands := Hundreds( Number Div 1000 ) + 'thousand, ' + Hundreds( Small );
                  End; { Case }
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Millions.
    This function returns a string corresponding
    to a number from zero to nine hundred and
    ninety nine million, nine hundred and ninety
    nine thousand, nine hundred and ninety nine.

*************************************************)

    Function Millions( Number: LongInt ): String;
      Begin
        If ( Number < 1000000 )
          then
            Millions := Thousands( Number )
          else
            Millions := Hundreds( Number Div 1000000 ) + 'million, ' + Thousands( Number Mod 1000000 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Write out number.
    As previously defined.

*************************************************)

    Function Write_Out_Number( Number: LongInt ): String;
      Begin
        Write_out_number := Millions( Abs( Number ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get date.
    As previously defined.

*************************************************)

    Procedure Get_Date( Var Data: String );
      Var
        Day,
        Year,
        Month,
        WeekDay: Word;
      Begin
        GetDate( Year, Month, Day, WeekDay );
        Data := WeekDays[ WeekDay ] + Months[ Month ] + Days[ Day ] + Thousands( Year );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get time.
    As previously defined.

*************************************************)

    Procedure Get_Time( Var Data: String );
      Var
        Hour,
        Minute,
        Second,
        Seconds: Word;
        Late: Boolean;
      Begin
        GetTime( Hour, Minute, Second, Seconds );
        If ( Hour > 12 )
          then
            Begin
              Late := True;
              Hour := ( Hour - 12 );
            End
          else
            Late := False;
        Data := ( Single( Hour ) + Ten( Minute ) );
        If Late
          then
            Data := ( Data + 'PM' )
          else
            Data := ( Data + 'AM' );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove double blanks.
    As previously defined.

*************************************************)

    Procedure Remove_Double_Blanks( Var Data: String );
      Var
        Count: Byte;
      Begin
        Count := Pos( '  ', Data );
        While ( Count <> 0 ) do
          Begin
            Delete( Data, Count, 1 );
            Count := Pos( '  ', Data );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Spread out string.
    As previously defined.

*************************************************)

    Procedure Spread_Out_String( Var Data: String; Size: Byte );
      Var
        Count: Integer;
      Begin
        Remove_Double_Blanks( Data );
        For Count := 1 to Length( Data ) do
          If ( Data[ Count ] = ' ' )
            then
              Data[ Count ] := #0;
        If ( Pos( #0, Data ) <> 0 )
          then
            While ( Length( Data ) < Size ) do
              Begin
                Count := 1;
                While ( ( Count < Length( Data ) ) and ( Length( Data ) < Size ) ) do
                  Begin
                    If ( Data[ Count ] = #0 )
                      then
                        Insert( ' ', Data, Succ( Count ) );
                    Inc( Count );
                  End;
              End;
        For Count := 1 to Length( Data ) do
          If ( Data[ Count ] = #0 )
            then
              Data[ Count ] := ' ';
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove all blanks.
    As previously defined.

*************************************************)

    Procedure Remove_All_Blanks( Var Data: String );
      Var
        Count: Byte;
      Begin
        For Count := Length( Data ) downto 1 do
          If ( Data[ Count ] = ' ' )
            then
              Delete( Data, Count, 1 );
      End;


{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Comma the string.
    As previously defined.

*************************************************)

    Procedure Comma_The_String( Var Data: String );
      Var
        Look,
        The_Length: Byte;
      Begin
        Look := Pos( Decimal_Separator, Data );
        If ( Look = 0 )
          then
            The_Length := Length( Data )
          else
            The_Length := Pred( Look );
        While ( The_Length > 0 ) and ( not ( Data[ The_Length ] in [ '0' .. '9' ] ) ) do
          Dec( The_Length );
        While ( ( The_Length - 3 ) > 0 ) and ( Data[ The_Length - 3 ] in [ '0' .. '9' ] ) do
          Begin
            Insert( Thousands_Separator, Data, ( The_Length - 2 ) );
            The_Length := ( The_Length - 3 )
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Capitalize.
    As previously defined.

*************************************************)

    Procedure Capitalize( Var Data: String );
      Var
        Point: Byte;
      Begin
        For Point := 1 to Length( Data ) do
          If ( Data[ Point ] in [ 'a' .. 'z' ] )
            then
              Data[ Point ] := Chr( Ord( Data[ Point ] ) + ( Ord( 'A' ) - Ord( 'a' ) ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Clean string.
    This function cleans the numerical string of
    any foreign characters.  It returns the sign
    of the string separately.  Useful for clearing
    away commas and other characters.

*************************************************)

    Function Clean_String( Var Number: String ): Sign;
      Var
        Signs,
        Counter: Byte;
      Begin
        Signs := 0;
        Counter := 1;
        While ( Counter <= Length( Number ) ) do
          Begin
            If ( Number[ Counter ] = '-' )
              then
                Inc( Signs );
            If not ( Number[ Counter ] in [ '0'..'9' ] )
              then
                Delete( Number, Counter, 1 )
              else
                Inc( Counter );
          End;
        If Odd( Signs )
          then
            Clean_String := Negative
          else
            Clean_String := Positive;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Extend string.
    This procedure takes two integer strings and
    extends one of them until they are both of
    equal size.

*************************************************)

    Procedure Extend_String( Var Number1, Number2: String );
      Begin
        While ( Length( Number1 ) < Length( Number2 ) ) do
          Number1 := '0' + Number1;
        While ( Length( Number1 ) > Length( Number2 ) ) do
          Number2 := '0' + Number2;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove front zeros.
    This procedure removes the front zeros of the
    given string.

*************************************************)

    Procedure Remove_Front_Zeros( Var Number: String );
      Begin
        While ( ( Length( Number ) > 0 ) and ( Number[ 1 ] = '0' ) ) do
          Delete( Number, 1, 1 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Clear string.
    This function takes a numerical string and
    clears away all unnecessary characters.
    It returns the string's sign separately.  This
    function makes the string a real number.

*************************************************)

    Function Clear_String( Var Number: String ): Sign;
      Var
        Signs,
        Counter: Byte;
        Points: Boolean;
      Begin
        Signs := 0;
        Points := False;
        Counter := 1;
        While ( Counter <= Length( Number ) ) do
          Begin
            If ( Number[ Counter ] = '-' )
              then
                Inc( Signs );
            If not ( Number[ Counter ] in [ '0' .. '9', Decimal_Separator ] )
              then
                Delete( Number, Counter, 1 )
              else
                Begin
                  If ( Number[ Counter ] = Decimal_Separator )
                    then
                      If Points
                        then
                          Delete( Number, Counter, 1 )
                        else
                          Begin
                            Points := True;
                            Inc( Counter );
                          End
                    else
                      Inc( Counter );
                End;
          End;
        If Odd( Signs )
          then
            Clear_String := Negative
          else
            Clear_String := Positive;
        If ( not Points )
          then
            Number := Number + Decimal_Separator;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Position point.
    This function returns the position of the
    decimal point or the country delimitor as a
    value from the end of the string.

*************************************************)

    Function Position_Point( Data: String ): Byte;
      Begin
        Position_Point := ( Length( Data ) - Pos( Decimal_Separator, Data ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Expand strings.
    This procedure takes two real numerical
    strings and expands the smaller until it is
    the same size as the large.

*************************************************)

    Procedure Expand_Strings( Var Number1, Number2: String );
      Var
        Position1,
        Position2: Byte;
      Begin
        Position1 := Pos( Decimal_Separator, Number1 );
        Position2 := Pos( Decimal_Separator, Number2 );
        While ( Position1 < Position2 ) do
          Begin
            Inc( Position1 );
            Number1 := ( '0' + Number1 );
          End;
        While ( Position1 > Position2 ) do
          Begin
            Inc( Position2 );
            Number2 := ( '0' + Number2 );
          End;
        Position1 := Length( Number1 );
        Position2 := Length( Number2 );
        While ( Position1 < Position2 ) do
          Begin
            Inc( Position1 );
            Number1 := ( Number1 + '0' );
          End;
        While ( Position1 > Position2 ) do
          Begin
            Inc( Position2 );
            Number2 := ( Number2 + '0' );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Equalize strings.
    This function takes two numerical real strings
    and makes them equal so that they can be
    operated upon evenly.

*************************************************)

    Function Equalize_Strings( Var Number1, Number2: String; Var Sign1, Sign2: Sign ): Byte;
      Begin
        Sign1 := Clear_String( Number1 );
        Sign2 := Clear_String( Number2 );
        Expand_Strings( Number1, Number2 );
        Equalize_Strings := Position_Point( Number1 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove point.
    This procedure removes the decimal separator
    from the numerical real strings.

*************************************************)

    Procedure Remove_Point( Var Number: String );
      Var
        Position: Byte;
      Begin
        Position := Pos( Decimal_Separator, Number );
        If ( Position = 0 )
          then
            Write_Error( 207, 'Remove_Point: Numerical string lacks decimal separator' )
          else
            Delete( Number, Position, 1 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert point.
    This procedure inserts a decimal separator
    character into the string at the given
    location from the end.

*************************************************)

    Procedure Insert_Point( Var Number: String; Position: Byte );
      Var
        Where: Byte;
      Begin
        Where := Succ( Length( Number ) - Position );
        Insert( Decimal_Separator, Number, Where )
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Convert to byte.
    This function returns the byte value
    associated with the given string.

*************************************************)

    Function Convert_To_Byte( Data: Char ): Byte;
     {$IFNDEF VER60}
      Begin
        Convert_To_Byte := ( Byte( Data ) and $0F );
      End;
     {$ELSE}
      Assembler;
      Asm
        MOV   Al, Data
        AND   Al, $0F
      End;
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Convert to char.
    This function converts the given byte to a
    character.

*************************************************)

    Function Convert_To_Char( Data: Byte ): Char;
     {$IFNDEF VER60}
      Begin
        Convert_To_Char := Chr( Data or $30 );
      End;
     {$ELSE}
      Assembler;
      Asm
        MOV   Al, Data
        OR    Al, $30
      End;
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add unit.
    This function adds the two numeric characters
    together and returns the result and the carry
    flag.

*************************************************)

    Function Add_Units( Num1, Num2: Char; Var Carry: Byte ): Char;
     {$IFNDEF VER60}
      Var
        Hold: Byte;
      Begin
        Hold := ( Convert_To_Byte( Num1 ) + Convert_To_Byte( Num2 ) + Carry );
        Carry := ( Hold Div 10 );
        Add_Units := Convert_To_Char( Hold Mod 10 );
      End;
     {$ELSE}
      Assembler;
      Asm
        LEs Di, Carry          { Let Es:Di point to the carry Variable }
        XOR Ah, Ah             { Clear the upper part of Ax }
        MOV Al, Num1           { Al gets Number 1 }
        MOV Bl, Num2           { Bl gets Number 2 }
        RCR Byte [ Es:Di ], 1  { Shift the carry bit into the flag }
        ADC Al, Bl             { Add Bl to Al with the carry flag }
        AAA                    { Adjust the result to an Ascii Character }
        OR Al, $30             { Finish the Ascii adjustment }
        MOV [ Es:Di ], Ah      { Store the carry result }
      End;                     { The result is returned in Al }
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract units.
    This function subtracts the second character
    from the first and returns the result and the
    borrow flag.

*************************************************)

    Function Subtract_Units( Num1, Num2: Char; Var Borrow: Byte ): Char;
     {$IFNDEF VER60}
      Var
        Hold: ShortInt;
      Begin
        Hold := ( Convert_To_Byte( Num1 ) - Convert_To_Byte( Num2 ) - Borrow );
        If ( Hold < 0 )
          then
            Begin
              Borrow := 1;
              Hold := ( Hold + 10 );
            End
          else
            Borrow := 0;
        Subtract_Units := Convert_To_Char( Hold Mod 10 );
      End;
     {$ELSE}
      Assembler;
      Asm
        LEs Di, Borrow         { Let Es:Di point to the Borrow Variable }
        XOR Ah, Ah             { Clear the upper part of Ax }
        MOV Al, Num1           { Al gets Number 1 }
        MOV Bl, Num2           { Bl gets Number 2 }
        RCR Byte [ Es:Di ], 1  { Shift the Borrow bit into the flag }
        SBB Al, Bl             { Subtract Bl from Al and the borrow flag }
        AAS                    { Adjust the result to an ASCii Character }
        OR Al, $30             { Finish the ASCii adjustment }
        MOV [ Es:Di ], Ah      { Store the Borrow result }
      End;                     { The result is returned in Al }
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply units.
    This function multiplies the given character
    by the given value and returns the result.

*************************************************)

    Function Multiply_Units( Number: Char; Value: Byte; Var Add: Byte ): Char;
     {$IFNDEF VER60}
      Var
        Hold: Byte;
      Begin
        Hold := ( ( Convert_To_Byte( Number ) * Value ) + Add );
        Add := ( Hold Div 10 );
        Multiply_Units := Convert_To_Char( Hold Mod 10 );
      End;
     {$ELSE}
      Assembler;
      Asm
        LEs Di, Add            { Let Es:Di point to the Add Variable }
        MOV Al, Number         { Let Al hold Number }
        AND AL, $0F            { Convert Al to Binary }
        MUL Value              { Multiply the converted number by Value }
        ADD Al, [ Es:Di ]      { Add the Extra Value }
        ADC Ah, 0              { And extend the carry if any }
        MOV BX, 10             { Get ready to separate the result }
        DIV BL
        MOV [ Es:Di ], Al      { Store the result }
        MOV Al, Ah
      End;                     { The result is returned in Al }
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal add.
    This procedure adds two string numbers
    together and returns the result in result.

*************************************************)

    Procedure Internal_Add( Var Number1, Number2, Result: String );
      Var
        Carry,
        Counter: Byte;
      Begin
        Carry := 0;
        Result[ 0 ] := Number1[ 0 ];
        For Counter := Length( Number1 ) downto 1 do
          Result[ Counter ] := Add_Units( Number1[ Counter ], Number2[ Counter ], Carry );
        If ( Carry = 1 )
          then
            Result := '1' + Result;
        Remove_Front_Zeros( Result );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal subtract.
    This procedure subtracts the second numerical
    string from the first and returns the result.

*************************************************)

    Procedure Internal_Subtract( Var Number1, Number2, Result: String );
      Var
        Borrow,
        Counter: Byte;
      Begin
        Borrow := 0;
        Result[ 0 ] := Number1[ 0 ];
        For Counter := Length( Number1 ) downto 1 do
          Result[ Counter ] := Subtract_Units( Number1[ Counter ], Number2[ Counter ], Borrow );
        If ( Borrow = 1 )
          then
            Write_Error( 205, 'Internal_Subtract: Subtractors value too low' );
        Remove_Front_Zeros( Result );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal multiply.
    This procedure multiplies the numerical string
    by the given value.

*************************************************)

    Procedure Internal_Multiply( Var Number: String; Value: Byte );
      Var
        Add,
        Counter: Byte;
        Result: String;
      Begin
        If ( Value <> 1 )
          then
            If ( Value = 0 )
              then
                Begin
                  For Counter := Length( Number ) downto 1 do
                    Number[ Counter ] := '0';
                End
              else
                Begin
                  Add := 0;
                  Result[ 0 ] := Succ( Number[ 0 ] );
                  For Counter := Length( Number ) downto 1 do
                    Result[ Succ( Counter ) ] := Multiply_Units( Number[ Counter ], Value, Add );
                  Result[ 1 ] := Convert_To_Char( Add );
                  Number := Result;
                End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add easy.
    This function returns the result of adding
    the two given numbers.

*************************************************)

    Function Add_Easy( Number1, Number2: String ): String;
      Var
        Result: String;
      Begin
        Extend_String( Number1, Number2 );
        Internal_Add( Number1, Number2, Result );
        Add_Easy := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract easy.
    This function returns the result of
    subtracting the second given number from the
    first.

*************************************************)

    Function Subtract_Easy( Number1, Number2: String ): String;
      Var
        Result: String;
      Begin
        Extend_String( Number1, Number2 );
        Internal_Subtract( Number1, Number2, Result );
        Subtract_Easy := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add clean string.
    This function adds two strings together, or
    subtracts one from the other, depending on
    the signs.

*************************************************)

    Function Add_Clean_String( Number1, Number2: String; Sign1, Sign2: Sign ): String;
      Var
        Result: String;
      Begin
        Extend_String( Number1, Number2 );
        If ( Sign1 = Sign2 )
          then
            Begin
              Internal_Add( Number1, Number2, Result );
              If ( Sign1 = Negative )
                then
                  Result := '-' + Result;
            End
         else
            If ( Number1 > Number2 )
              then
                Begin
                  Internal_Subtract( Number1, Number2, Result );
                  If ( Sign1 = Negative )
                    then
                      Result := '-' + Result;
                End
              else
                Begin
                  Internal_Subtract( Number2, Number1, Result );
                  If ( Sign2 = Negative )
                    then
                      Result := '-' + Result;
                End;
        Add_Clean_String := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply clean string.
    This function multiplies the two given strings
    together, then adjusts their signs.

*************************************************)

    Function Multiply_Clean_String( Number1, Number2: String; Sign1, Sign2: Sign ): String;
      Var
        Value,
        Counter: Byte;
        Temp,
        Total,
        Zeros: String;
      Begin
        Total := '0';
        Zeros := '';
        For Counter := Length( Number2 ) downto 1 do
          Begin
            Value := Convert_To_Byte( Number2[ Counter ] );
            If ( Value <> 0 )
              then
                Begin
                  Temp := Number1;
                  Internal_Multiply( Temp, Value );
                  Temp := Temp + Zeros;
                  Total := Add_Easy( Temp, Total );
                End;
            Zeros := Zeros + '0';
          End;
        If ( Sign1 <> Sign2 )
          then
            Total := ( '-' + Total );
        Multiply_Clean_String := Total;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Divide internal.
    This procedure divides the Remainder by the
    Value using the classic shift and subtract
    strategy and returns the Result.

*************************************************)

    Procedure Divide_Internal( Value: String; Var Remainder, Result: String );
      Var
        Count: Byte;
      Begin
        Extend_String( Remainder, Value );
        Count := 0;
        While ( Remainder > Value ) do
          Begin
            Value := ( Value + '0' );
            Remove_Front_Zeros( Remainder );
            Remove_Front_Zeros( Value );
            Extend_String( Remainder, Value );
            Inc( Count );
          End;
        If ( Count > 0 )
          then
            Delete( Value, Length( Value ), 1 );
        Repeat
          Extend_String( Remainder, Value );
          If ( Remainder > Value )
            then
              Repeat
                Inc( Result[ Length( Result ) ] ) ;
                Remainder := Subtract_Easy( Remainder, Value );
                Extend_String( Remainder, Value );
              Until ( Remainder < Value );
          While ( ( Remainder < Value ) and ( Count > 0 ) ) do
            Begin
              Result := Result + '0';
              Dec( Count );
              Delete( Value, Length( Value ), 1 );
              Extend_String( Remainder, Value );
            End;
        Until ( Count < 1 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Divide clean strings.
    This procedure divides the second number into
    the first then adjusts the signs.

*************************************************)

    Procedure Divide_Clean_String( Number1, Number2: String; Sign1, Sign2: Sign; Var Result, Remainder: String );
      Begin
        Result := '0';
        Remainder := Number1;
        Extend_String( Number1, Number2 );
        If ( Number1 >= Number2 )
          then
            Divide_Internal( Number2, Remainder, Result );
        If ( Length( Result ) > 1 )
          then
            Dec( Result[ 0 ] );
        Remove_Front_Zeros( Remainder );
        If ( Sign1 <> Sign2 )
          then
            Begin
              Result := ( '-' + Result );
              Remainder := ( '-' + Remainder );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Div clean strings.
    This procedure div the second number into
    the first then adjusts the signs.
    The division is performed using the classic
    shift and subtract strategy.

*************************************************)

    Procedure Div_Clean_String( Number1, Number2: String; Sign1, Sign2: Sign; Var Result, Remainder: String );
      Begin
        Result := '0';
        Remainder := Number1;
        Extend_String( Number1, Number2 );
        If ( Number1 >= Number2 )
          then
            Divide_Internal( Number2, Remainder, Result );
        If ( Length( Result ) > 1 )
          then
            Dec( Result[ 0 ] );
        Remove_Front_Zeros( Remainder );
        If ( Remainder <> '' )
          then
            Begin
              Result := ( Result + Decimal_Separator );
              While ( ( Remainder <> '' ) and ( Length( Result ) < 254 ) ) do
                Begin
                  Repeat
                    Remove_Front_Zeros( Remainder );
                    Remove_Front_Zeros( Number2 );
                    Result := Result + '0';
                    Remainder := Remainder + '0';
                    Extend_String( Remainder, Number2 );
                  Until ( Remainder > Number2 );
                  Repeat
                    Remove_Front_Zeros( Remainder );
                    Remove_Front_Zeros( Number2 );
                    Inc( Byte( Result[ Length( Result ) ] ) );
                    Remainder := Subtract_Integer_String( Remainder, Number2 );
                    Extend_String( Remainder, Number2 );
                  Until ( Remainder < Number2 ) ;
                End;
            End;
        If ( Sign1 <> Sign2 )
          then
            Begin
              Result := ( '-' + Result );
              Remainder := ( '-' + Remainder );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add integer string.
    As previously defined.

*************************************************)

    Function Add_Integer_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
      Begin
        Sign1 := Clean_String( Number1 );
        Sign2 := Clean_String( Number2 );
        Add_Integer_String := Add_Clean_String( Number1, Number2, Sign1, Sign2 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract integer string.
    As previously defined.

*************************************************)

    Function Subtract_Integer_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
      Begin
        Sign1 := Clean_String( Number1 );
        Sign2 := Clean_String( Number2 );
        Case Sign2 of
          Positive: Sign2 := Negative;
          Negative: Sign2 := Positive;
        End; { Case }
        Subtract_Integer_String := Add_Clean_String( Number1, Number2, Sign1, Sign2 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply integer string.
    As previously defined.

*************************************************)

    Function Multiply_Integer_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
      Begin
        Sign1 := Clean_String( Number1 );
        Sign2 := Clean_String( Number2 );
        Multiply_Integer_String := Multiply_Clean_String( Number1, Number2, Sign1, Sign2 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Divide integer string.
    As previously defined.

*************************************************)

    Function Divide_Integer_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result,
        Remainder: String;
      Begin
        Sign1 := Clean_String( Number1 );
        Sign2 := Clean_String( Number2 );
        Divide_Clean_String( Number1, Number2, Sign1, Sign2, Result, Remainder );
        Divide_Integer_String := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Mod integer string.
    As previously defined.

*************************************************)

    Function Mod_Integer_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result,
        Remainder: String;
      Begin
        Sign1 := Clean_String( Number1 );
        Sign2 := Clean_String( Number2 );
        Divide_Clean_String( Number1, Number2, Sign1, Sign2, Result, Remainder );
        Mod_Integer_String := Remainder;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add real string.
    As previously defined.

*************************************************)

    Function Add_Real_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result: String;
        Position: Byte;
      Begin
        Position := Equalize_Strings( Number1, Number2, Sign1, Sign2 );
        Remove_Point( Number1 );
        Remove_Point( Number2 );
        Result := Add_Clean_String( Number1, Number2, Sign1, Sign2 );
        Insert_Point( Result, Position );
        Add_Real_String := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract real string.
    As previously defined.

*************************************************)

    Function Subtract_Real_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result: String;
        Position: Byte;
      Begin
        Position := Equalize_Strings( Number1, Number2, Sign1, Sign2 );
        Remove_Point( Number1 );
        Remove_Point( Number2 );
        Case Sign2 of
          Positive: Sign2 := Negative;
          Negative: Sign2 := Positive;
        End; { Case }
        Result := Add_Clean_String( Number1, Number2, Sign1, Sign2 );
        Insert_Point( Result, Position );
        Subtract_Real_String := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply real string.
    As previously defined.

*************************************************)

    Function Multiply_Real_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result: String;
        Position1,
        Position2: Byte;
      Begin
        Sign1 := Clear_String( Number1 );
        Sign2 := Clear_String( Number2 );
        Position1 := Position_Point( Number1 );
        Position2 := Position_Point( Number2 );
        Remove_Point( Number1 );
        Remove_Point( Number2 );
        Result := Multiply_Clean_String( Number1, Number2, Sign1, Sign2 );
        Insert_Point( Result, ( Position1 + Position2 ) );
        Multiply_Real_String := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Divide real string.
    As previously defined.

*************************************************)

    Function Divide_Real_String( Number1, Number2: String ): String;
      Var
        Sign1,
        Sign2: Sign;
        Result,
        Remainder: String;
        Position: Byte;
      Begin
        Position := Equalize_Strings( Number1, Number2, Sign1, Sign2 );
        Remove_Point( Number1 );
        Remove_Point( Number2 );
        Div_Clean_String( Number1, Number2, Sign1, Sign2, Result, Remainder );
        Divide_Real_String := Result;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand.
    As previously defined.

*************************************************)

    Function Expand( Data: Char; Width: Byte ): String;
      Var
        Count: Byte;
        Temporary_Data: String;
      Begin
       {$IFDEF Quick}
        Temporary_Data[ 0 ] := Char( Width );
        FillChar( Temporary_Data[ 1 ], Width, Data );
       {$ELSE}
        Temporary_Data := '';
        For Count := 1 to Width do
          Temporary_Data := Temporary_Data + Data;
       {$ENDIF}
        Expand := Temporary_Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Center.
    As previously defined.

*************************************************)

    Function Center( Data: String; Width: Byte; Fill: Char ): String;
      Var
        Front: Boolean;
      Begin
        Front := True;
        While ( Length( Data ) < Width ) do
          Begin
            If Front
              then
                Data := Fill + Data
              else
                Data := Data + Fill;
            Front := not Front;
          End;
        Center := Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Push to.
    As previously defined.

*************************************************)

    Function Push_To( Front, Back: String; Limit: Byte; Fill: Char ): String;
      Begin
        While ( ( Length( Front ) + Length( Back ) ) < Limit ) do
          Front := Front + Fill;
        Push_To := Front;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Poll_Country_Info.
    This procedure gets country specific
    information from the operating system so that
    the code can convert from one country format
    to another.

*************************************************)

    Procedure Poll_Country_Info( Var Data: Country_Info );
     {$IFNDEF OS2}
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ds := Seg( Data );
        The_Registers.Dx := Ofs( Data );
        The_Registers.Ah := $38;
        The_Registers.Al := 0;
        The_Registers.Bx := 0;
        MSDOS( The_Registers );
        If ( The_Registers.Bx = 0 )
          then
            Begin
              WriteLn( 'Error in Poll_Country_Info: System error.' );
              Halt;
            End;
      End;
     {$ELSE}
      Var
        Result,
        Actual_Length,
        Reporting_Length: LongWord;
        Country_Code: CountryCode;
        Country_Information: CountryInfo;
      Begin
        Reporting_Length := SizeOf( CountryInfo );
        Country_Code.Country := 0;
        Country_Code.CodePage := 0;
        Result := DOSQueryCtryInfo( Reporting_Length, Country_Code, Country_Information, Actual_Length );
        If ( Result <> No_Error )
          then
            Begin
              WriteLn( 'Error in Poll_Country_Info: System error: ', Result );
              WriteLn( 'Press [Enter] to continue' );
              ReadLn;
              Halt;
            End
          else
            Begin
              Data.Date_Format := Country_Information.fsDateFmt;
              Data.Currency_Symbol_2 := Z_String_5( Country_Information.SzCurrency );
              Data.Thousands_Separator_2 := Z_String_2( Country_Information.szThousandsSeparator );
              Data.Decimal_Separator_2 := Z_String_2( Country_Information.szDecimal );
              Data.Date_Separator := Z_String_2( Country_Information.szDateSeparator );
              Data.Time_Separator := Z_String_2( Country_Information.szTimeSeparator );
              Data.Currency_Symbol_Location := Country_Information.fsCurrencyFmt;
              Data.Currency_Decimal_Places := Country_Information.cDecimalPlace;
              Data.Time_format := Country_Information.fsTimeFmt;
              Data.List_Separator := Z_String_2( Country_Information.szDataSeparator );
            End;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get_Country_Info.
    This procedure sets the global variables so
    that they reflect the particular format of
    the country they are to be used in.

*************************************************)

    Procedure Get_Country_Info;
      Var
        Data: Country_Info;
        Count: Byte;
      Begin
        Poll_Country_Info( Data );
       {$IFNDEF OS2}
        If ( Lo( DOSVersion ) = 2 ) or American_Only
          then
            Begin
              If American_Only
                then
                  Begin
                    Decimal_Separator := '.';
                    Currency_Symbol := '$';
                    Thousands_Separator := ',';
                  End
                else
                  Begin
                    Decimal_Separator := Data.Decimal_Separator_1[ 1 ];
                    Currency_Symbol := Data.Currency_Symbol_1[ 1 ];
                    Thousands_Separator := Data.Thousands_Separator_1[ 1 ];
                  End;
              Date_Format := American;
              Date_Separator := '/';
              Time_Separator := ':';
              Currency_Location := Before;
              Currency_Decimal_Places := 2;
              Time_Format := Hour_12;
              List_Separator := ',';
            End
          else
       {$ENDIF}
            Begin
              Decimal_Separator := Data.Decimal_Separator_2[ 1 ];
              Thousands_Separator := Data.Thousands_Separator_2[ 1 ];
              Currency_Symbol := '';
              For Count := 1 to 4 do
                If ( Data.Currency_Symbol_2[ Count ] <> #0 )
                  then
                    Currency_Symbol := Currency_Symbol + Data.Currency_Symbol_2[ Count ];
              Date_Separator := Data.Date_Separator[ 1 ];
              Time_Separator := Data.Time_Separator[ 1 ];
              Case Data.Date_Format of
                0: Date_Format := American;
                1: Date_Format := European;
                2: Date_Format := Japanese;
                else Date_Format := American;
              End;
              Case Data.Currency_Symbol_Location of
                0: Currency_Location := Before;
                1: Currency_Location := After;
                2: Currency_Location := Spaced_Before;
                3: Currency_Location := Spaced_After;
                4: Decimal_Separator := Currency_Symbol[ 1 ];
                else Currency_Location := Before;
              End;
              Currency_Decimal_Places := Data.Currency_Decimal_Places;
              Case Data.Time_Format of
                0: Time_Format := Hour_12;
                1: Time_Format := Hour_24;
                else Time_Format := Hour_24;
              End;
              List_Separator := Data.List_Separator[ 1 ];
            End;
      End;

{-----------------------------------------------------------------------------}

    Begin
      Get_Country_Info;
    End.

