(*************************************************

  Procedure: Update virtual window row.
    This procedure will update the screen for the
    specified row of the virtual window.

*************************************************)

    Procedure Update_Virtual_Window_Row( Where: Virtual_Window_Pointer_Type; Row, Screen: Byte );
      Var
        Source: Pointer;
        Screen_Length,
        Window_Length: Byte;
        Extra: Cell_Type;
      Begin
        With Where^ do
          Begin
            Source := Address_Storage( Storage, Row, Where_Left );
            Screen_Length := Succ( Screen_Right - Screen_Left );
            Window_Length := Succ( Window_Width - Where_Left );
            If ( Screen_Length <= Window_Length )
              then
               {$IFNDEF OS2}
                Write_Data( Screen, Screen_Left, Source^, Screen_Length )
               {$ELSE}
                Write_Data( Screen, Screen_Left, Source, Screen_Length )
               {$ENDIF}
              else
                Begin
                 {$IFNDEF OS2}
                  Write_Data( Screen, Screen_Left, Source^, Window_Length );
                 {$ELSE}
                  Write_Data( Screen, Screen_Left, Source, Window_Length );
                 {$ENDIF}
                  Extra.Character := ' ';
                  Extra.Attribute := Current_Window^.New_Attribute;
                 {$IFNDEF OS2}
                  Write_Data( Screen, Screen_Left + Window_Length, Extra, 1 );
                 {$ELSE}
                  Write_Data( Screen, Screen_Left + Window_Length, Addr( Extra ), 1 );
                 {$ENDIF}
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual window.
    This procedure will update the entire virtual
    window on the screen.

*************************************************)

    Procedure Update_Virtual_Window( Where: Virtual_Window_Pointer_Type );
      Var
        Row,
        Value: Byte;
      Begin;
        With Where^ do
          Begin
            Value := Where_Top;
            For Row := Screen_Top to Screen_Bottom do
              Begin
                If ( Value <= Window_Height )
                  then
                    Update_Virtual_Window_Row( Where, Value, Row );
                Inc( Value );
              End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update virtual character.
    This procedure updates the screen for a
    freshly written virtual character.  Thus, the
    entire virtual window doesn't need to be re-
    displayed for a single change in the window.
    If the character is out of the view range, it
    ignores it.

*************************************************)

    Procedure Update_Virtual_Character( Where: Virtual_Window_Pointer_Type; Row, Column: Byte; Character: Char );
      Var
        Value1,
        Value2: Byte;
      Begin
        With Where^ do
          Begin
            Value1 := ( Screen_Left + ( Column - Where_Left ) );
            Value2 := ( Screen_Top + ( Row - Where_Top ) );
            If ( Value1 <= Screen_Right ) and ( Value2 <= Screen_Bottom )
              then
                Put_Character_On_Screen( Value1, Value2, Character, Storage.Attribute );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write character on virtual window.
    This procedure takes the given character and
    stores it in the virtual window.  Then it
    writes the new character on the screen.  It
    also must update the whole virtual window if
    the character happens to be the last character
    on the last line.  In that case, the virtual
    window must scroll up one line.

*************************************************)

    Procedure Write_Character_on_Virtual_Window( Where: Virtual_Window_Pointer_Type; Character: Char );
      Begin
        With Where^ do
          Begin
            Storage.Attribute := VW_TextAttr;
            Put_To_Storage( Storage, Storage.Cursor_Row, Storage.Cursor_Column, Character, VW_TextAttr );
            Update_Virtual_Character( Where, Storage.Cursor_Row, Storage.Cursor_Column, Character );
            Inc( Storage.Cursor_Column );
            If ( Storage.Cursor_Column > Storage.Column_Length )
              then
                Begin
                  Storage.Cursor_Column := 1;
                  Inc( Storage.Cursor_Row );
                  If ( Storage.Cursor_Row > Storage.Row_Length )
                    then
                      Begin
                        Scroll_Storage_Up( Storage );
                        Update_Virtual_Window( Where );
                        Storage.Cursor_Row := Storage.Row_Length;
                      End;
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Window output.
    This function is part of the file managing
    routines.  Its basic function is to read the
    data that's passed to it and process it into
    the file.  { the screen }

*************************************************)

   {$IFNDEF OS2}
    Function Window_Output( Var The_File: TextRec ): Integer;
      Var
        Pointer: Word;
        Where: Virtual_Window_Pointer_Type;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Where := Current_Window^.Virtual_Window;
              With The_File do
                Begin
                  Pointer := 0;
                  While ( Pointer < BufPos ) do
                    Begin
                      Case BufPtr^[ Pointer ] of
                        #10: With Current_Window^.Virtual_Window^ do
                              Begin
                                Inc( Storage.Cursor_Row );
                                If ( Storage.Cursor_Row > Storage.Row_Length )
                                  then
                                    Begin
                                      Scroll_Storage_Up( Storage );
                                      Update_Virtual_Window( Where );
                                      Storage.Cursor_Row := Storage.Row_Length;
                                    End;
                              End;
                        #13: Where^.Storage.Cursor_Column := 1;
                        else Write_Character_On_Virtual_Window( Where, BufPtr^[ Pointer ] );
                      End; { Case }
                      Inc( Pointer );
                    End;
                  BufPos := 0;
                End;
            End;
        Window_Output := 0;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Window close.
    This function is part of the file management
    routines.  It merely closes the file.

*************************************************)

   {$IFNDEF OS2}
    Function Window_Close( Var The_File: TextRec ): Integer;
      Begin
        Window_Close := 0;
        The_File.Mode := FmClosed;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Window open.
    This function is part of the file management
    routines.  It opens the file and initialized
    the file management pointing routines.

*************************************************)

   {$IFNDEF OS2}
    Function Window_Open( Var The_File: TextRec ): Integer;
      Begin
        Window_Open := 0;
        The_File.Mode := FmOutput;
        The_File.InOutFunc := @Window_Output;
        The_File.FlushFunc := @Window_Output;
        The_File.CloseFunc := @Window_Close;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Functions: Virtual window move view left.
    As previously defined.

*************************************************)

    Function VW_Move_View_Left;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                If ( Where_Left > 1 )
                  then
                    Begin
                      Dec( Where_Left );
                      Dec( Where_Right );
                      Update_Virtual_Window( Current_Window^.Virtual_Window );
                      VW_Move_View_Left := True;
                    End
                  else
                    VW_Move_View_Left := False;
              End
          else
            VW_Move_View_Left := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Functions: Virtual window move view right.
    As previously defined.

*************************************************)

    Function VW_Move_View_Right;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                If ( Where_Right < Storage.Column_Length )
                  then
                    Begin
                      Inc( Where_Left );
                      Inc( Where_Right );
                      Update_Virtual_Window( Current_Window^.Virtual_Window );
                      VW_Move_View_Right := True;
                    End
                  else
                    VW_Move_View_Right := False;
              End
          else
            VW_Move_View_Right := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Functions: Virtual window move view up.
    As previously defined.

*************************************************)

    Function VW_Move_View_Up;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                If ( Where_Top > 1 )
                  then
                    Begin
                      Dec( Where_Top );
                      Dec( Where_Bottom );
                      Update_Virtual_Window( Current_Window^.Virtual_Window );
                      VW_Move_View_Up := True;
                    End
                  else
                    VW_Move_View_Up := False;
              End
          else
            VW_Move_View_Up := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Functions: Virtual window move view down.
    As previously defined.

*************************************************)

    Function VW_Move_View_Down;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                If ( Where_Bottom < Storage.Row_Length )
                  then
                    Begin
                      Inc( Where_Top );
                      Inc( Where_Bottom );
                      Update_Virtual_Window( Current_Window^.Virtual_Window );
                      VW_Move_View_Down := True;
                    End
                  else
                    VW_Move_View_Down := False;
              End
          else
            VW_Move_View_Down := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move view home.
    This function moves the virtual window to the
    home position.

*************************************************)

    Function Move_View_Home: Boolean;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                While ( Where_Left > 1 ) do
                  Begin
                    Dec( Where_Left );
                    Dec( Where_Right );
                  End;
                Update_Virtual_Window( Current_Window^.Virtual_Window );
                Move_View_Home := True;
              End
          else
            Move_View_Home := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move view end.
    This function moves the virtual window to the
    end position.

*************************************************)

    Function Move_View_End: Boolean;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                While ( Where_Right < Storage.Column_Length ) do
                  Begin
                    Inc( Where_Left );
                    Inc( Where_Right );
                  End;
                Update_Virtual_Window( Current_Window^.Virtual_Window );
                Move_View_End := True;
              End
          else
            Move_View_End := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move view page up.
    This function moves the virtual window to the
    page up position.

*************************************************)

    Function Move_View_PageUp( Amount: Byte ): Boolean;
      Var
        Count: Byte;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin;
                Get_The_Mode;
                For Count := 1 to Amount do
                  If ( Where_Top > 1 )
                    then
                      Begin
                        Dec( Where_Top );
                        Dec( Where_Bottom );
                      End;
                Update_Virtual_Window( Current_Window^.Virtual_Window );
                Move_View_PageUp := True;
              End
          else
            Move_View_PageUp := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move view page down.
    This function moves the virtual window to the
    page down position.

*************************************************)

    Function Move_View_PageDown( Amount: Byte ): Boolean;
      Var
        Count: Byte;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                For Count := 1 to Amount do
                  If ( Where_Bottom < Storage.Row_Length )
                    then
                      Begin
                        Inc( Where_Top );
                        Inc( Where_Bottom );
                      End;
                Update_Virtual_Window( Current_Window^.Virtual_Window );
                Move_View_PageDown := True;
              End
          else
            Move_View_PageDown := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window reposition.
    This procedure redraws the entire virtual
    window and updates all the variables.  It is
    necessary to call this after the underlying
    window is altered.

*************************************************)

    Procedure VW_Reposition;
      Var
        Old_Top,
        Old_Left,
        Old_Right,
        Old_Bottom: Byte;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Get_The_Mode;
              With Current_Window^.Virtual_Window^ do
                Begin
                  Old_Top := Screen_Top;
                  Old_Left := Screen_Left;
                  Old_Right := Screen_Right;
                  Old_Bottom := Screen_Bottom;
                  Screen_Right := Succ( Lo( WindMax ) );
                  Screen_Top := Succ( Hi( WindMin ) );
                  Screen_Left := Succ( Lo( WindMin ) );
                  Screen_Bottom := Succ( Hi( WindMax ) );
                  If ( Storage.Column_Length < Where_Right )
                    then
                      Begin
                        Where_Right := Storage.Column_Length;
                        Screen_Right := Pred( Where_Right + Screen_Left );
                      End;
                  If ( Storage.Row_Length < Where_Bottom )
                    then
                      Begin
                        Where_Bottom := Storage.Row_Length;
                        Screen_Bottom := Pred( Where_Bottom + Screen_Top );
                      End;
                  Where_Right := ( Screen_Right - Screen_Left ) + Where_Left;
                  Where_Bottom := ( Screen_Bottom - Screen_Top ) + Where_Top;
                  If ( Storage.Column_Length < Where_Right )
                    then
                      Where_Right := Storage.Column_Length;
                  If ( Storage.Row_Length < Where_Bottom )
                    then
                      Where_Bottom := Storage.Row_Length;
                  If ( Screen_Right > Old_Right )
                    then
                      Old_Right := Screen_Right;
                  If ( Screen_Left < Old_Left )
                    then
                      Old_Left := Screen_Left;
                  If ( Screen_Top < Old_Top )
                    then
                      Old_Top := Screen_Top;
                  If ( Screen_Bottom > Old_Bottom )
                    then
                      Old_Bottom := Screen_Bottom;
                End;
              Update_Virtual_Window( Current_Window^.Virtual_Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window goto x y.
    As previously defined.

*************************************************)

    Procedure VW_GotoXY( Column, Row: Byte );
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                If ( Column <= Storage.Column_Length ) or ( Column >= 1 ) or
                   ( Row <= Storage.Row_Length ) or ( Row >= 1 )
                  then
                    Begin
                      Storage.Cursor_Column := Column;
                      Storage.Cursor_Row := Row;
                    End;
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window clear screen.
    As previously defined.

*************************************************)

    Procedure VW_ClrScr;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                Clear_Storage( Storage );
                Storage.Cursor_Column := 1;
                Storage.Cursor_Row := 1;
                Update_Virtual_Window( Current_Window^.Virtual_Window );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window clear to end of line.
    As previously defined.

*************************************************)

    Procedure VW_ClrEOL;
      Var
        Column: Byte;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^.Virtual_Window^ do
              Begin
                Get_The_Mode;
                For Column := Storage.Cursor_Column to Storage.Column_Length do
                  Put_To_Storage( Storage, Storage.Cursor_Row, Column, ' ', Storage.Attribute );
                Update_Virtual_Window( Current_Window^.Virtual_Window );
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Virtual window where x.
    As previously defined.

*************************************************)

    Function VW_WhereX;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            VW_WhereX := Current_Window^.Virtual_Window^.Storage.Cursor_Column;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Virtual window where y.
    As previously defined.

*************************************************)

    Function VW_WhereY;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            VW_WhereY := Current_Window^.Virtual_Window^.Storage.Cursor_Row;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window scroll up.
    As previously defined.

*************************************************)

    Procedure VW_Scroll_Up;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Get_The_Mode;
              Scroll_Storage_Up( Current_Window^.Virtual_Window^.Storage );
              Update_Virtual_Window( Current_Window^.Virtual_Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window scroll down.
    As previously defined.

*************************************************)

    Procedure VW_Scroll_Down;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Get_The_Mode;
              Scroll_Storage_Down( Current_Window^.Virtual_Window^.Storage );
              Update_Virtual_Window( Current_Window^.Virtual_Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window scroll left.
    As previously defined.

*************************************************)

    Procedure VW_Scroll_Left;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Get_The_Mode;
              Scroll_Storage_Left( Current_Window^.Virtual_Window^.Storage );
              Update_Virtual_Window( Current_Window^.Virtual_Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window scroll right.
    As previously defined.

*************************************************)

    Procedure VW_Scroll_Right;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin
              Get_The_Mode;
              Scroll_Storage_Right( Current_Window^.Virtual_Window^.Storage );
              Update_Virtual_Window( Current_Window^.Virtual_Window );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Assign virtual window.
    As previously defined.

*************************************************)

   {$IFNDEF OS2}
    Procedure Assign_VW( Var The_File: Text );
      Begin
        With TextRec( The_File ) do
          Begin
            Mode := FmClosed;
            BufSize := SizeOf( Buffer );
            BufPtr := @Buffer;
            OpenFunc := @Window_Open;
            Name[ 0 ] := #0;
          End;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Calculate storage amount.
    This procedure calculates the amount of area
    in bytes needed to store the given window
    size.

*************************************************)

    Procedure Calculate_Storage_Amount( Var Storage: Storage_Record; New_Row_Length, New_Column_Length: Byte );
      Begin
        Storage.Row_Length := New_Row_Length;
        Storage.Column_Length := New_Column_Length;
        Storage.Amount := ( ( Storage.Row_Length * Storage.Column_Length ) * SizeOf( Cell_Type ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Virtual window create.
    As previously defined.

*************************************************)

    Function VW_Create( Row_Length, Column_Length: Byte ): Boolean;
      Var
        Okay: Boolean;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window = Nil )
          then
            With Current_Window^ do
              Begin
                Get_The_Mode;
                If ( Row_Length = 0 )
                  then
                    Row_Length := Succ( Hi( WindMax ) - Hi( WindMin ) );
                If ( Column_Length = 0 )
                  then
                    Column_Length := Succ( Lo( WindMax ) - Lo( WindMin ) );
                New( Virtual_Window );
                Okay := ( Virtual_Window <> Nil );
                If Okay
                  then
                    With Virtual_Window^ do
                      Begin
                        Screen_Right := Succ( Lo( WindMax ) );
                        Screen_Top := Succ( Hi( WindMin ) );
                        Screen_Bottom := Succ( Hi( WindMax ) );
                        Screen_Left := Succ( Lo( WindMin ) );
                        Storage.Cursor_Column := 1;
                        Storage.Cursor_Row := 1;
                        Storage.Attribute := TextAttr;
                        VW_TextAttr := TextAttr;
                        Window_Width := Column_Length;
                        Window_Height := Row_Length;
                        Where_Left := 1;
                        Where_Top := 1;
                        Where_Right := Succ( Screen_Right - Screen_Left );
                        Where_Bottom := Succ( Screen_Bottom - Screen_Top );
                        If ( Column_Length < Where_Right )
                          then
                            Begin
                              Where_Right := Column_Length;
                              Screen_Right := Pred( Where_Right + Screen_Left );
                            End;
                        If ( Row_Length < Where_Bottom )
                          then
                            Begin
                              Where_Bottom := Row_Length;
                              Screen_Bottom := Pred( Where_Bottom + Screen_Top );
                            End;
                        Calculate_Storage_Amount( Storage, Row_Length, Column_Length );
                        Okay := Allocate_Storage( Storage );
                      End;
                If Okay
                  then
                    Begin
                     {$IFNDEF VER40}
                      Reposition := VW_Reposition;
                     {$ENDIF}
                      Clear_Storage( Virtual_Window^.Storage );
                      Update_Virtual_Window( Virtual_Window );
                    End;
              End
          else
            Okay := False;
        VW_Create := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window destroy.
    As previously defined.

*************************************************)

    Procedure VW_Destroy;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            With Current_Window^ do
              Begin
                Deallocate_Storage( Virtual_Window^.Storage );
                Dispose( Virtual_Window );
                Virtual_Window := Nil;
               {$IFNDEF VER40}
                Reposition := Procedure_Default;
               {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Browse help screen.
    This procedure displays a help window on the
    screen and allows the user to take a look at
    it.

*************************************************)

    Procedure Browse_Help_Screen;
      Begin
        If Open_Window( Frame_1, Window_Pop, 1, 1, 75, 25, Blue_Character + White_background )
          then
            Begin
              Label_Window( White_Character + Blue_Background, 'Help' );
              WriteLn( 'Basic window moving commands are..' );
              WriteLn;
              WriteLn( ' <Up Arrow>            Moves the window contents up.' );
              WriteLn( ' <Down Arrow>          Moves the window contents down.' );
              WriteLn( ' <Left Arrow>          Moves the window contents left.' );
              WriteLn( ' <Right Arrow>         Moves the window contents right.' );
              WriteLn( ' <Home>                Moves the window contents to far left position.' );
              WriteLn( ' <End>                 Moves the window contents to far right position.' );
              WriteLn( ' <Page Up>             Moves the window contents up one page.' );
              WriteLn( ' <Page Down>           Moves the window contents down one page.' );
              WriteLn( ' <Escape>');
              WriteLn( ' <Enter>               Causes ending of browse.' );
              WriteLn;
              WriteLn( ' <Control Left Arrow>  Moves the window left.' );
              WriteLn( ' <Control Right Arrow> Moves the window right.' );
              WriteLn( ' <Control Page Up>     Moves the window up.' );
              WriteLn( ' <Control Page Down>   Moves the window down.' );
              WriteLn( ' <Control F1>          Reduces the window width.' );
              WriteLn( ' <Control F2>          Increases the window width.' );
              WriteLn( ' <Control F3>          Reduces the window height.' );
              WriteLn( ' <Control F4>          Increases the window height.' );
              TextAttr := Flashing + Red_Character + White_Background;
             {$IFDEF OS2}
              TextBackground( TextAttr shr 4 );
             {$ENDIF}
              WriteLn( '                       Press enter to return' );
              ReadLn;
              Close_Window( Window_Pop );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Virtual window browse.
    As previously defined.

*************************************************)

    Procedure VW_Browse;
      Var
        Okay,
        Done: Boolean;
        Top,
        Left,
        Right,
        Bottom,
        Result: Byte;
        Command: Char;
      Begin
        If ( Current_Window <> Nil ) and ( Current_Window^.Virtual_Window <> Nil )
          then
            Begin;
              Get_The_Mode;
              With Current_Window^.Virtual_Window^ do
                Begin
                  Right := Screen_Right;
                  Top := Screen_Top;
                  Left := Screen_Left;
                  Bottom := Screen_Bottom;
                End;
              Done := False;
              Cursor_Row := ( Bottom - Top  ) div 2;
              Cursor_Column_Start := ( Right - Left ) div 2;
              Cursor_Column_Finish := Cursor_Column_Start;
              While KeyPressed do
                Command := ReadKey;
              Repeat
                Get_Command( Command, Result );
                Case Result of
                  Press_Home:
                    Okay := Move_View_Home;
                  Press_End:
                    Okay := Move_View_End;
                  Outside_Down,
                  Press_Down_Arrow:
                    Okay := VW_Move_View_Down;
                  Outside_Up,
                  Press_Up_Arrow:
                    Okay := VW_Move_View_Up;
                  Outside_Right,
                  Press_Right_Arrow:
                    Okay := VW_Move_View_Right;
                  Outside_Left,
                  Press_Left_Arrow:
                    Okay := VW_Move_View_Left;
                  Press_Page_Down:
                    Okay := Move_View_PageDown( ( Bottom - Top ) );
                  Press_Page_Up:
                    Okay := Move_View_PageUp( ( Bottom - Top ) );
                  Press_F1:
                    Browse_Help_Screen;
                  Pressed_Move_Window_Up:
                    Okay := Move_Window_Up;
                  Pressed_Move_Window_Down:
                    Okay := Move_Window_Down;
                  Pressed_Move_Window_Right:
                    Okay := Move_Window_Right;
                  Pressed_Move_Window_Left:
                    Okay := Move_Window_Left;
                  Pressed_Reduce_Window_Height:
                    Okay := Reduce_Window_Up;
                  Pressed_Expand_Window_Height:
                    Okay := Expand_Window_Down;
                  Pressed_Reduce_Window_Width:
                    Okay := Reduce_Window_Left;
                  Pressed_Expand_Window_Width:
                    Okay := Expand_Window_Right;
                 {$IFNDEF VER40}
                  Pressed_Lock:
                    Core.Lock_Routine;
                 {$EndIf}
                  Press_Escape:
                    Done := True;
                  Press_Enter:
                    Done := True;
                End; {Case}
              Until Done;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reset window.
    This procedure resets the current window of
    the CRT unit.

*************************************************)

    Procedure Reset_Window;
      Begin
        With Current_Window^ do
          If ( Frame_Style = No_Frame )
            then
              Window( Safety_Left, Safety_Top, Safety_Right, Safety_Bottom )
            else
              If Shadow
                then
                  Window( Succ( Safety_Left ), Succ( Safety_Top ), Pred( Pred( Safety_Right ) ),
                          Pred( Pred( Safety_Bottom ) ) )
                else
                  Window( Succ( Safety_Left ), Succ( Safety_Top ), Pred( Safety_Right ), Pred( Safety_Bottom ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Make shadow.
    This procedure makes the character on the
    screen at the given row and column, a shadow
    attribute.

*************************************************)

    Procedure Make_Shadow( Row, Column, Count: Byte );
      Begin
        Dim_Screen_Attribute( Row, Column, Count, Shadow_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore row.
    This procedure restores the current row in
    memory back onto the screen.

*************************************************)

    Procedure Restore_Row( Row, Left, Right, Top: Byte );
      Var
        Where: Pointer;
      Begin
        Where := Address_Storage( Current_Window^.Storage, Succ( Row - Top ), 1 );
       {$IFNDEF OS2}
        Write_Data( Row, Left, Where^, Succ( Right - Left ) );
       {$ELSE}
        Write_Data( Row, Left, Where, Succ( Right - Left ) );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore row with shadow.
    This procedure restores the current row in
    memory back onto the screen and then adds a
    shadow.

*************************************************)

    Procedure Restore_Row_Shadow( Row, Left, Right, Top: Byte );
      Var
        Where: Pointer;
      Begin
        Where := Address_Storage( Current_Window^.Storage, Succ( Row - Top ), 1 );
       {$IFNDEF OS2}
        Write_Data( Row, Left, Where^, Succ( Right - Left ) );
       {$ELSE}
        Write_Data( Row, Left, Where, Succ( Right - Left ) );
       {$ENDIF}
        Make_Shadow( Row, Succ( Left ), ( Right - Left ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore column.
    This procedure restores the current column in
    memory back onto the screen.

*************************************************)

    Procedure Restore_Column( Column, Top, Bottom, Left: Byte );
      Var
        Row,
        Value,
        Attribute: Byte;
        Character: Char;
      Begin
        Value := Succ( Column - Left );
        For Row := Top to Bottom do
          Begin
            Get_from_Storage( Current_Window^.Storage, Succ( Row - Top ), Value, Character, Attribute );
            Put_Character_On_Screen( Column, Row, Character, Attribute );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore column with shadow.
    This procedure restores the current column in
    memory back onto the screen and then adds the
    shadow.

*************************************************)

    Procedure Restore_Column_Shadow( Column, Top, Bottom, Left: Byte );
      Var
        Row,
        Value,
        Attribute: Byte;
        Character: Char;
      Begin
        Value := Succ( Column - Left );
        Get_from_Storage( Current_Window^.Storage, 1, Value, Character, Attribute );
        Put_Character_On_Screen( Column, Top, Character, Attribute );
        For Row := Succ( Top ) to Bottom do
          Begin
            Get_from_Storage( Current_Window^.Storage, Succ( Row - Top ), Value, Character, Attribute );
            Put_Character_On_Screen( Column, Row, Character, Shadow_Attribute )
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Preserve the row.
    This procedure saves the given row of data on
    the screen in the memory of the window stack.

*************************************************)

    Procedure Preserve_Row( Row, Left, Right, Top: Byte );
      Var
        Where: Pointer;
      Begin
        Where := Address_Storage( Current_Window^.Storage, Succ( Row - Top ), 1 );
       {$IFNDEF OS2}
        Read_Data( Row, Left, Where^, Succ( Right - Left ) );
       {$ELSE}
        Read_Data( Row, Left, Where, Succ( Right - Left ) );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Preserve the column.
    This procedure saves the given column of data
    on the screen in the memory of the window
    stack.

*************************************************)

    Procedure Preserve_Column( Column, Top, Bottom, Left: Byte );
      Var
        Row,
        Value1,
        Attribute: Byte;
        Character: Char;
      Begin
        Value1 := Succ( Column - Left );
        For Row := Top to Bottom do
          Begin
            Get_Character_from_Screen( Column, Row, Character, Attribute );
            Put_To_Storage( Current_Window^.Storage, Succ( Row - Top ), Value1, Character, Attribute );
          End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Save the window.
    This procedure saves an image of the screen
    data in the given region on the window stack.
    It returns false if it fails.

*************************************************)

    Function Save_Window( Top, Bottom, Left, Right: Byte ): Boolean;
      Var
        Row: Byte;
        Okay: Boolean;
      Begin
        Okay := Allocate_Storage( Current_Window^.Storage );
        If Okay
          then
            For Row := Top to Bottom do
              Preserve_Row( Row, Left, Right, Top );
        Save_Window := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw top.
    This procedure creates a copy of the top of
    the frame in memory, then displays it on the
    screen in the given location.

*************************************************)

    Procedure Draw_Top( Var Frame: Frame_Type; Left, Top, Right: Byte );
      Var
        Hold,
        Count: Byte;
      Begin
        FillChar( Work_Area, SizeOf( Work_Area ), Frame.Attribute );
        Hold := Succ( Pred( Right ) - Succ( Left ) );
        For Count := 1 to Hold do
          Work_Area[ Count ].Character := Frame.Data[ 2 ];
       {$IFNDEF OS2}
        Write_Data( Top, Succ( Left ), Work_Area, Hold );
       {$ELSE}
        Write_Data( Top, Succ( Left ), Addr( Work_Area ), Hold );
       {$ENDIF}
        For Count := 1 to Hold do
          Work_Area[ Count ].Character := Frame.Data[ 5 ];
       {$IFNDEF OS2}
        Write_Data( Succ( Top ), Succ( Left ), Work_Area, Hold );
       {$ELSE}
        Write_Data( Succ( Top ), Succ( Left ), Addr( Work_Area ), Hold );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw top with shadow.
    This procedure creates a copy of the top of
    the frame in memory, then displays it on the
    screen in the given location.

*************************************************)

    Procedure Draw_Top_Shadow( Var Frame: Frame_Type; Left, Top, Right: Byte );
      Begin
        Draw_Top( Frame, Left, Top, Right );
        Make_Shadow( Succ( Top ), Succ( Right ), 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Do bottom.
    This procedure creates a copy of the bottom of
    the frame in memory, then displays it on the
    screen in the given location.

*************************************************)

    Procedure Do_Bottom( Var Frame: Frame_Type; Left, Bottom, Right, Hold: Byte );
      Var
        Count: Byte;
      Begin
        FillChar( Work_Area, SizeOf( Work_Area ), Frame.Attribute );
        For Count := 1 to Hold do
          Work_Area[ Count ].Character := Frame.Data[ 8 ];
       {$IFNDEF OS2}
        Write_Data( Bottom, Succ( Left ), Work_Area, Hold );
       {$ELSE}
        Write_Data( Bottom, Succ( Left ), Addr( Work_Area ), Hold );
       {$ENDIF}
        For Count := 1 to Hold do
          Work_Area[ Count ].Character := Frame.Data[ 5 ];
       {$IFNDEF OS2}
        Write_Data( Pred( Bottom ), Succ( Left ), Work_Area, Hold );
       {$ELSE}
        Write_Data( Pred( Bottom ), Succ( Left ), Addr( Work_Area ), Hold );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw bottom.
    This procedure displays the bottom of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Bottom( Var Frame: Frame_Type; Left, Bottom, Right: Byte );
      Var
        Hold: Byte;
      Begin
        Hold := Succ( Pred( Right ) - Succ( Left ) );
        Do_Bottom( Frame, Left, Bottom, Right, Hold );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw bottom with shadow.
    This procedure displays the bottom of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Bottom_Shadow( Var Frame: Frame_Type; Left, Bottom, Right: Byte );
      Var
        Hold,
        Count: Byte;
      Begin
        Hold := Succ( Pred( Right ) - Succ( Left ) );
        Do_Bottom( Frame, Left, Bottom, Right, Hold );
        Make_Shadow( Succ( Bottom ), Succ( Succ( Left ) ), Hold );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw left.
    This procedure displays the left side of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Left( Var Frame: Frame_Type; Top, Left, Bottom: Byte );
      Var
        Count: Byte;
        Data: Area_Type Absolute Work_Area;
      Begin
        FillChar( Data, 4, Frame.Attribute );
        Data[ 1 ].Character := Frame.Data[ 1 ];
        Data[ 2 ].Character := Frame.Data[ 2 ];
       {$IFNDEF OS2}
        Write_Data( Top, Left, Data, 2 );
       {$ELSE}
        Write_Data( Top, Left, Addr( Data ), 2 );
       {$ENDIF}
        Data[ 1 ].Character := Frame.Data[ 4 ];
        Data[ 2 ].Character := Frame.Data[ 5 ];
        For Count := Succ( Top ) to Pred( Bottom ) do
         {$IFNDEF OS2}
          Write_Data( Count, Left, Data, 2 );
         {$ELSE}
          Write_Data( Count, Left, Addr( Data ), 2 );
         {$ENDIF}
        Data[ 1 ].Character := Frame.Data[ 7 ];
        Data[ 2 ].Character := Frame.Data[ 8 ];
       {$IFNDEF OS2}
        Write_Data( Bottom, Left, Data, 2 );
       {$ELSE}
        Write_Data( Bottom, Left, Addr( Data ), 2 );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw left with shadow.
    This procedure displays the left side of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Left_Shadow( Var Frame: Frame_Type; Top, Left, Bottom: Byte );
      Begin
        Draw_Left( Frame, Top, Left, Bottom );
        Make_Shadow( Succ( Bottom ), Succ( Left ), 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw right.
    This procedure displays the right side of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Right( Var Frame: Frame_Type; Top, Right, Bottom: Byte );
      Var
        Count: Byte;
        Data: Area_Type Absolute Work_Area;
      Begin
        FillChar( Data, 4, Frame.Attribute );
        Data[ 1 ].Character := Frame.Data[ 2 ];
        Data[ 2 ].Character := Frame.Data[ 3 ];
       {$IFNDEF OS2}
        Write_Data( Top, Pred( Right ), Data, 2 );
       {$ELSE}
        Write_Data( Top, Pred( Right ), Addr( Data ), 2 );
       {$ENDIF}
        Data[ 1 ].Character := Frame.Data[ 5 ];
        Data[ 2 ].Character := Frame.Data[ 6 ];
        For Count := Succ( Top ) to Pred( Bottom ) do
         {$IFNDEF OS2}
          Write_Data( Count, Pred( Right ), Data, 2 );
         {$ELSE}
          Write_Data( Count, Pred( Right ), Addr( Data ), 2 );
         {$ENDIF}
        Data[ 1 ].Character := Frame.Data[ 8 ];
        Data[ 2 ].Character := Frame.Data[ 9 ];
       {$IFNDEF OS2}
        Write_Data( Bottom, Pred( Right ), Data, 2 );
       {$ELSE}
        Write_Data( Bottom, Pred( Right ), Addr( Data ), 2 );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw right with shadow.
    This procedure displays the right side of the
    frame on the screen in the given location.

*************************************************)

    Procedure Draw_Right_Shadow( Var Frame: Frame_Type; Top, Right, Bottom: Byte );
      Var
        Count: Byte;
        Data: Area_Type Absolute Work_Area;
      Begin
        FillChar( Data, 4, Frame.Attribute );
        Data[ 1 ].Character := Frame.Data[ 2 ];
        Data[ 2 ].Character := Frame.Data[ 3 ];
       {$IFNDEF OS2}
        Write_Data( Top, Pred( Right ), Data, 2 );
       {$ELSE}
        Write_Data( Top, Pred( Right ), Addr( Data ), 2 );
       {$ENDIF}
        Make_Shadow( Succ( Top ), Succ( Right ), 1 );
        Data[ 1 ].Character := Frame.Data[ 5 ];
        Data[ 2 ].Character := Frame.Data[ 6 ];
        For Count := Succ( Top ) to Pred( Bottom ) do
          Begin
           {$IFNDEF OS2}
            Write_Data( Count, Pred( Right ), Data, 2 );
           {$ELSE}
            Write_Data( Count, Pred( Right ), Addr( Data ), 2 );
           {$ENDIF}
            Make_Shadow( Succ( Count ), Succ( Right ), 1 );
          End;
        Data[ 1 ].Character := Frame.Data[ 8 ];
        Data[ 2 ].Character := Frame.Data[ 9 ];
       {$IFNDEF OS2}
        Write_Data( Bottom, Pred( Right ), Data, 2 );
       {$ELSE}
        Write_Data( Bottom, Pred( Right ), Addr( Data ), 2 );
       {$ENDIF}
        Make_Shadow( Succ( Bottom ), Succ( Right ), 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small top.
    This procedure draws only part of the top of
    the frame.

*************************************************)

    Procedure Draw_Small_Top( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Hold,
        Count: Byte;
      Begin
        If ( ( Left < Right ) and ( Top < Bottom ) )
          then
            Begin
              FillChar( Work_Area, SizeOf( Work_Area ), Frame.Attribute );
              Hold := Succ( Pred( Right ) - Succ( Left ) );
              For Count := 1 to Hold do
                Work_Area[ Count ].Character := Frame.Data[ 2 ];
             {$IFNDEF OS2}
              Write_Data( Top, Succ( Left ), Work_Area, Hold );
             {$ELSE}
              Write_Data( Top, Succ( Left ), Addr( Work_Area ), Hold );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small bottom.
    This procedure draws only part of the bottom
    of the frame.

*************************************************)

    Procedure Draw_Small_Bottom( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Hold,
        Count: Byte;
      Begin
        If ( ( Left < Right ) and ( Top < Bottom ) )
          then
            Begin
              FillChar( Work_Area, SizeOf( Work_Area ), Frame.Attribute );
              Hold := Succ( Pred( Right ) - Succ( Left ) );
              For Count := 1 to Hold do
                Work_Area[ Count ].Character := Frame.Data[ 8 ];
             {$IFNDEF OS2}
              Write_Data( Bottom, Succ( Left ), Work_Area, Hold );
             {$ELSE}
              Write_Data( Bottom, Succ( Left ), Addr( Work_Area ), Hold );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small left.
    This procedure draws only part of the left
    side of the frame.

*************************************************)

    Procedure Draw_Small_Left( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Left < Right ) and ( Top < Bottom ) )
          then
            For Count := Succ( Top ) to Pred( Bottom ) do
              Put_Character_On_Screen( Left, Count, Frame.Data[ 4 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small right.
    This procedure draws only part of the right
    side of the frame.

*************************************************)

    Procedure Draw_Small_Right( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Count: Byte;
      Begin
        If ( ( Left < Right ) and ( Top < Bottom ) )
          then
            For Count := Succ( Top ) to Pred( Bottom ) do
              Put_Character_On_Screen( Right, Count, Frame.Data[ 6 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw window side.
    This procedure draws both side of the window
    frame on the screen.

*************************************************)

    Procedure Draw_Window_Sides( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Count: Byte;
      Begin
        For Count := Succ( Top ) to Pred( Bottom ) do
          Begin
            Put_Character_On_Screen( Left, Count, Frame.Data[ 4 ], Frame.Attribute );
            Put_Character_On_Screen( Right, Count, Frame.Data[ 6 ], Frame.Attribute );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner left top.
    This procedure draws the left top corner of
    the frame on the screen.

*************************************************)

    Procedure Corner_Left_Top( Var Frame: Frame_Type; Left, Top: Byte );
      Begin
        Put_Character_On_Screen( Left, Top, Frame.Data[ 1 ], Frame.Attribute );
        Put_Character_On_Screen( Left, Succ( Top ), Frame.Data[ 4 ], Frame.Attribute );
        Put_Character_On_Screen( Succ( Left ), Top, Frame.Data[ 2 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner left bottom.
    This procedure draws the left bottom corner of
    the frame on the screen.

*************************************************)

    Procedure Corner_Left_Bottom( Var Frame: Frame_Type; Left, Bottom: Byte );
      Begin
        Put_Character_On_Screen( Left, Bottom, Frame.Data[ 7 ], Frame.Attribute );
        Put_Character_On_Screen( Left, Pred( Bottom ), Frame.Data[ 4 ], Frame.Attribute );
        Put_Character_On_Screen( Succ( Left ), Bottom, Frame.Data[ 8 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner left bottom with shadow.
    This procedure draws the left bottom corner of
    the frame on the screen.

*************************************************)

    Procedure Corner_Left_Bottom_Shadow( Var Frame: Frame_Type; Left, Bottom: Byte );
      Begin
        Corner_Left_Bottom( Frame, Left, Bottom );
        Make_Shadow( Succ( Bottom ), Succ( Left ), 2 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner right bottom.
    This procedure draws the right bottom corner
    of the frame on the screen.

*************************************************)

    Procedure Corner_Right_Bottom( Var Frame: Frame_Type; Right, Bottom: Byte );
      Begin
        Put_Character_On_Screen( Right, Bottom, Frame.Data[ 9 ], Frame.Attribute );
        Put_Character_On_Screen( Right, Pred( Bottom ), Frame.Data[ 6 ], Frame.Attribute );
        Put_Character_On_Screen( Pred( Right ), Bottom, Frame.Data[ 8 ], Frame.Attribute );
      End;


