#include <windows.h>
#include <stdlib.h>

#ifndef _CONVERT_INC
#define _CONVERT_INC
////////////////////////////////////////////////////////////////////////////////
//
//
//		CONVERT.H	- defines for Animate Custom Image Converters:
//
//		Animate Image Converters are Windows Dynamic Link Libraries with a .CNV 
//			extension.  They are used to provide an interface between different
//			file types and Animate.
//
//		Minimal Read Converter Requirements: (exported functions)		
//		------------------------------------
//		void CALLBACK GetInfo(LPCONVERTERINFO, LPCSTR);	
//		BOOL CALLBACK IsHandler(LPIMAGEINFO, void far *);		
//		BOOL CALLBACK ReadHeader(LPIMAGEINFO, void far *);
//		BOOL CALLBACK ReadRow(WORD, WORD, WORD, LPIMAGEINFO, void far *, void far *);
//		+ BOOL CALLBACK ReadFrame(WORD, WORD, LPIMAGEINFO, void far *, void far *);
//
//		+ required if File cannot be read by ReadRow (see note at IF_NOREADROW flag)
//
//
//
//		Minimal Write Converter Requirements: (exported functions)
//		-------------------------------------
//		void CALLBACK GetInfo(LPCONVERTERINFO, LPCSTR);
//		BOOL CALLBACK IsHandler(LPIMAGEINFO, void far *);		
//		BOOL CALLBACK WriteHeader(LPIMAGEINFO, void far *);	
//		BOOL CALLBACK WriteRow(WORD, WORD, WORD, LPIMAGEINFO, void far *, void far *);
//		+ BOOL CALLBACK WriteFrame(WORD, WORD, LPIMAGEINFO, void far *, void far *);
//
//		+ required if File cannot be written by WriteRow (see note at IF_NOWRITEROW flag)
//
//
//
//
//		Exported functions are called in the following order: 
//		-----------------------------------------------------
//				* = required function
//	
//								<DLL IS LOADED>
//
//								* CONVERTERID (defined in this file)
//				
//			---------------READING/WRITING STARTUP-------------
//						
//								* GETINFO		
//
//								MEMORYREQUEST
//								INITIALIZE		
//
//								* ISHANDLER         
//	
//			----READING----							----WRITING----
//
//			* READHEADER								WRITEDIALOG
//			READDIALOG									* WRITEHEADER
//
//		
//			------------START FRAME ACCESS LOOP-------------
//
//			READFRAMEHEADER							WRITEFRAMEHEADER
//			* READROW / READFRAME					* WRITEROW / WRITEFRAME
//
//			------------END FRAME ACCESS LOOP---------------
//
//                         
//
//			------------READING/WRITING CLEANUP-------------
//
//								? DEINITIALIZE
//
//			----------------DLL IS UNLOADED-----------------
//
//
//
//
//
#ifdef __cplusplus                                          

extern "C" {
#endif

#define	_MAX_DESCRIPTION	48 		// maximum size of CONVERTERINFO szDescription variable
#define	_MAX_RGB				256		// maximum # of lpRGB entries (ie,. wNumColors <=_MAX_RGB)
#define	_MAX_CONVERTER		13			// maximum sizeof CONVERTERINFO szConverter variable
//
//
//
typedef struct _tagConverterInfo
{  	
	char	szDescription[_MAX_DESCRIPTION];	// eg., "Windows Bitmap (*.BMP)"
	char	szExtension[_MAX_EXT];				// eg., "BMP"
	WORD	wCaps;			// eg., CC_READ | CC_WRITEMULTIFRAME
	DWORD	dwMem;			// amount of memory required for private memory block
	char	szConverter[_MAX_CONVERTER];		//	for internal use (by Animate).  
														//	Converter should not modify this field!!!!
	WORD	wIntFlags;		// reserved - converter should NOT modify!														
} CONVERTERINFO;

typedef CONVERTERINFO far * LPCONVERTERINFO;

//
//	Converter Capabilities:
//
#define	CC_READ					0x0001
#define	CC_WRITE					0x0002
#define	CC_WRITEMULTIFRAME	(0x0004 | CC_WRITE)
#define	CC_SETUPDIALOG			0x0008

//
//	IntFlags:
//
#define	CIF_INITIALIZED		0x0001
typedef struct _tagImageInfo
{  	   
			// 
			// IMAGEINFO:
			//	The following variables are passed to 
			//	the ReadHeader handler of an Animate Library
			//
	LPSTR			lpszPathName;			// the full path of the file 
													
	HFILE			hFile;					// File handle (always valid unless closed by converter)
												// converter must set hFile=0 if it calls _lclose(hFile)
																								
	WORD			wWidth;					// Number of pixels/row (hBuffer should be at least wRowPixels*

   WORD			wRowBytes;				// Used by DLL exports: ReadFrame and WriteFrame
												// ReadHeader should set this if IF_NOREADROW is set
												//			if left at 0, wRowBytes is calculated based on wWidth and iDataType
												//	This is set before a call to WriteFrame.  
												//			Currently, wRowBytes always = wWidth (during Write calls)

	WORD			wHeight;					//	Number of rows in hBuffer
												// ReadHeader >ReadFrame, >GetMinMax, >MakeDIB
												
	int			iDataType;				// Pixel data type (unsigned byte, word, double word, float,etc)
												//	ReadHeader >ReadFrame, >GetMinMax, >MakeDIB
												// Always = U_BYTE for Write Calls
												// See defines below (U_BYTE, S_BYTE, U_WORD, ...)

	WORD			wNumFrames;				// Number of image frames in this file
												// ReadHeader >

	WORD			wNumSequences;			// Number of Frame-sequences (most converters ignore this field)
												// (A single file may contain several different image sequences)
	
	LPSTR			lpszSequenceNames;	// a series of 0-terminated strings giving names for each Sequence
												// last string is terminated with two 0s
												// May be set to NULL
	
	int			iError;					// Set by the converter to indicate the error level
												// or to suppress default error report
												//	(CE_????? below)
   		
	WORD			wFlags;					// Flags (IF_???? below)
	
	WORD			wNumColors;				// indicates # of RGBQUAD structures pointed to by lpRGB
	
	RGBQUAD far	*lpRGB; 					// pointer to an array of wNumColors RGBQUAD structures

	void far 	*lpDIB;					// pointer to a packed DIB block (only on write)
												// 	Converter should NEVER modify this block
} IMAGEINFO;

typedef IMAGEINFO far * LPIMAGEINFO;      

//
//	iDataType values
//
#define	S_BYTE		-8		// SIGNED CHAR
#define	U_BYTE		8		// UNSIGNED CHAR 
#define	S_WORD		-16	// SIGNED INT
#define	U_WORD		16		// UNSIGNED INT
#define	S_DWORD		-32  	// SIGNED LONG
#define	U_DWORD		32		// UNSIGNED LONG
#define	FLOAT			33		// FLOATING POINT (_DWORDs)
#define	DOUBLE		64		// DOUBLE (64 bits)

//
//	Converter Error codes:
//
//		On error, Converter should
//			1) return FALSE,		
// 		2) set IMAGEINFO.iError = CE_??????
//	
#define	CE_INVALIDCOLORINFO_S	8
#define	CE_INVALIDDATATYPE_S	7   		// Currently, all positive error codes 
#define	CE_UNSUPPORTEDFORMAT_S	6		// have no effect (ie., = CE_UNDEFINED)
#define	CE_BADFORMAT_S			5
#define	CE_INVALIDHEADER_S	4
#define	CE_MEMORY_S				3			//    Positive error codes
#define	CE_FILEACCESS_S 		2			//			_Suppress default error reporting
#define	CE_CANCEL_S				1			// 
#define	CE_DEFAULT		0
#define	CE_UNDEFINED	0
#define	CE_OK				0
#define	CE_CANCEL				-1			//		Negative error codes
#define	CE_FILEACCESS			-2			//			cause a default error report
#define	CE_MEMORY				-3      	//
#define	CE_INVALIDHEADER		-4
#define	CE_BADFORMAT			-5   		// image data is badly formatted (file is corrupt)
#define	CE_UNSUPPORTEDFORMAT	-6			// image data ok, but can't read this format
#define	CE_INVALIDDATATYPE	-7			//
#define	CE_INVALIDCOLORINFO	-8			// color table is bad
													

//
//	IMAGEINFO Flags:
//
#define	IF_NOREADROW		0x0001	// indicates that ReadRow cannot be called to read this file
												// 	ReadFrame will be called instead
#define	IF_NOWRITEROW		0x0002	// indicates that WriteRow cannot be called to write this file
												// 	WriteFrame will be called instead
#define	IF_NOREADDIALOG	0x0004	//	indicates that ReadDialog should not be called 
												// 	(usually set during ReadHeader)
#define	IF_NOWRITEDIALOG	0x0008	// indicates that WriteDialog should not be called
												//		(usually set during Initialize)
#define	IF_READCOLORS		0x0010	// indicates that a color image is being read.  Animate will
												// 	not intensity rescale this image.  
												//		wNumColors and lpRGB members must be set accordingly
												//		lpRGB must point to a contiguous block of wNumColors RGBQUAD
												//		structures located in lpPrivate block
#define	IF_NOINTENSITYRESCALE	0x0020	// prevents intensity rescaling of image.  Might be used
														//	when reading U_BYTE data
#define	IF_FORCEREAD		0x0040	// forces Animate to read the image frame.  Animate caches frames so that
												// 	the same information will not be read again.  However, you might want
												// 	to Re-read the frame if the user has changed image reading options 
												//		during the ReadDialog call, for example
#define	IF_INVERTROWORDER	0x0080	// Controls order in which rows are written during WriteRow calls
												// 	or flips the order of the rows in lpBuffer during WriteFrame
												// Why? Some file formats save information lowest row first (eg., Windows DIBs)
												// if IF_INVERTROWORDER is not set (default), top row in image is written first
												// if IF_INVERTROWORDER is SET, bottom row in image is written first
  												// 	(usually set during WriteHeader)
  												//	NOTE: the wRow parameter is not affected by this flag.  It is always 0 at beginning
  												//			and incremented by 1 for each subsequent call.
//	
//	ConverterDialogParams:
//												

typedef struct _tagConverterDialogParams
{
	LPIMAGEINFO	lpImg;
	void far * 	lpPrivate;
} DIALOGPARAMS;

typedef DIALOGPARAMS far *	LPDIALOGPARAMS;

///////////////////////////////////////////////////////////////////////////
//
//
//		CONVERTER EXPORTED FUNCTIONS:
//
//
//
//
//
void CALLBACK GetInfo(LPCONVERTERINFO, LPCSTR);
BOOL CALLBACK IsHandler(LPIMAGEINFO, void far *);
	//
	//	Read Converter Functions:
	//
BOOL CALLBACK ReadHeader(LPIMAGEINFO, void far *);
BOOL CALLBACK ReadRow(WORD, WORD, WORD, LPIMAGEINFO, void far *, void far *);
BOOL CALLBACK ReadFrame(WORD, WORD, LPIMAGEINFO, void far *, void far *);
BOOL CALLBACK MinMaxInfo(WORD, WORD, LPIMAGEINFO, void far *, double far *, double far *);

	//
	//	Required for Write Converters:
	//
BOOL CALLBACK WriteHeader(LPIMAGEINFO, void far *);
BOOL CALLBACK WriteRow(WORD, WORD, WORD, LPIMAGEINFO, void far *, void far *);
	// Required if IF_NOREADROW flag is set in IMAGEINFO
BOOL CALLBACK WriteFrame(WORD, WORD, LPIMAGEINFO, void far *, void far *);

//
// Optional Exports
//
BOOL CALLBACK Initialize(LPIMAGEINFO, void far *);
BOOL CALLBACK Deinitialize(LPIMAGEINFO, void far *);
BOOL CALLBACK WriteFrameHeader(WORD, WORD, LPIMAGEINFO, void far *);
BOOL CALLBACK ReadFrameHeader(WORD, WORD, LPIMAGEINFO, void far *);
BOOL CALLBACK ReadDialog(HWND, LPIMAGEINFO, void far *);
BOOL CALLBACK WriteDialog(HWND, LPIMAGEINFO, void far *);
BOOL CALLBACK SetupDialog(HWND);
BOOL CALLBACK WriteFrameEnd(WORD, WORD, LPIMAGEINFO, void far *);
//
// Predefined mandatory export: ConverterID
//
#define	_ANIMATE_MAGIC	0x8765     
#ifndef 	_IMGCNV_INC
WORD CALLBACK _export ConverterID() { return _ANIMATE_MAGIC; }
#endif

//
//		END OF CONVERTER EXPORTS
//
///////////////////////////////////////////////////////////////////////////

	// DATABYTES: returns the # bytes for a datatype
	//		PARAMETER: x = {U_BYTE, U_WORD, S_WORD, U_DWORD, S_DWORD, FLOAT, DOUBLE}

#define DATABYTES(x) ((0xFFF8&abs(x))/8)	
#define DATABITS(x) (0xFFF8&abs(x))
#ifdef __cplusplus
}
#endif
// end of #ifndef _CONVERT_INC
#endif	


