// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: futils.cpp
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 01/25/2000
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

General-purpose platform independent file and directory functions
that need to be ported between WIN32 and UNIX platforms. 
*/
// ----------------------------------------------------------- // 
#include "futils.h"

// ----------------------------------------------------------- //
// Standalone platform indepentent file/directory functions    //
// ----------------------------------------------------------- //
int futils_exists(const char *name)
// Tests to see if the specified directory or file name 
// exists. Returns true if the file or directory exists. 
{
  struct stat buf;
  return stat(name, &buf) == 0;
}

int futils_isdirectory(const char *dir_name) 
// Returns true if the specified name is a directory
{
  struct stat buf;

#ifndef lstat
  if(stat(dir_name, &buf) != 0) return 0;
#else
  // Use lstat() rather than stat() so that symbolic links
  // pointing to directories can be identified correctly.
  if(lstat(fname, &buf) != 0) return 0;
#endif

#if defined (_S_IFMT) && defined(_S_IFDIR)
  if((buf.st_mode & _S_IFMT) == _S_IFDIR) return 1;
#else
  if((buf.st_mode & S_IFMT) == S_IFDIR) return 1;
#endif
  
 return 0;
}

int futils_isfile(const char *fname)
// Returns true if the specified name is any kind of file 
// or false if this is a directory.
{
 return futils_isdirectory(fname) == 0;
}

void futils_makeDOSpath(char *path)
// Make a DOS directory path by changing all
// forward slash path separators with back
// slashes.
{
  unsigned len = strlen(path);
  while(len--) {
    char c = *path;
    switch(c) {
      case '/' :
	*path = '\\';
	break;
      default:
	break;
    }
    path++;
  }
}

void futils_makeUNIXpath(char *path)
// Make a UNIX directory path by changing all
// back slash path separators with forward
// slashes.
{
  unsigned len = strlen(path);
  while(len--) {
    char c = *path;
    switch(c) {
      case '\\' :
	*path = '/';
	break;
      default:
	break;
    }
    path++;
  }
}

int futils_chdir(const char *dir_name)
// Change directory. Returns zero if successful.
{
  char *path = (char *)dir_name;
  
  if(!futils_exists(dir_name)) {
    return 1;
  }

#if defined (__WIN32__)
  futils_makeDOSpath(path);
#elif defined (__UNIX__)
  futils_makeUNIXpath(path);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif

  if(chdir(path) != 0) {
    return 1;
  }
  return 0;
}

int futils_rmdir(const char *dir_name)
// Remove the specified directory. Returns zero if successful.
// The directory must be empty and not be the current working
// directory or the root directory.
{
  char *path = (char *)dir_name;
  
  if(!futils_exists(dir_name)) {
    return 1;
  }

#if defined (__WIN32__)
  futils_makeDOSpath(path);
#elif defined (__UNIX__)
  futils_makeUNIXpath(path);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif

  if(rmdir(path) != 0) {
    return 1;
  }
  return 0;
}

int futils_hasdriveletter(const char *dir_name, char *letter)
// WIN32 function used to parse the drive letter from the
// specified directory. Returns false if the path does not
// contain a drive letter. If a drive letter is found it will
// be passed back in the "letter" variable.
{
  char *s = (char *)dir_name;
  
  *letter = 0; // Reset the letter variable
  s++; if(*s != ':') return 0; // No drive letter found
  
  // Get the drive letter following the first colon
  s--; *letter = *s;
  return 1;
}

int futils_mkdir(const char *dir_name)
// Make the specified directory (with sub-directories) 
// if the directory if it does not exist. Returns a 
// non-zero value if an error occurs. UNIX file systems 
// will use 755 permissions when new directories are 
// created.
{
  char *path = (char *)dir_name;
  char path_sep;
  char sbuf[futils_MAX_PATH_LENGTH];
  unsigned i = 0; int rv;  
  unsigned len;
  char *s;
  
#if defined (__WIN32__)
  char drive_letter;
  int has_drive_letter;
#endif
  
  if(futils_exists(dir_name)) {
    // Directory exists, return with no error reported
    return 0;
  }

#if defined (__WIN32__)
  has_drive_letter = futils_hasdriveletter(dir_name, &drive_letter);
  futils_makeDOSpath(path);
  path_sep = '\\';
#elif defined (__UNIX__)
  futils_makeUNIXpath(path);
  path_sep = '/';
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif

  // The mkdir function can only create one new directory per call
  // so the complete directory path can only be created one sub-
  // directory at a time.
  for(i = 0; i < futils_MAX_PATH_LENGTH; i++) sbuf[i] = '\0';

  len = strlen(path); s = path;
  if(len > futils_MAX_PATH_LENGTH) len = futils_MAX_PATH_LENGTH;

  for(i = 0; i < len; i++, s++) {

    // Get the whole directory path
    if(i == (len-1)) { memmove(sbuf, path, len); break; }

#if defined (__WIN32__)
    if((has_drive_letter) && (i == 1)) { // Skip over the drive letter
      i++; s++; 
      if(*s == path_sep) { i++; s++; }
    }
#endif

    if((*s == path_sep) && (i > 1)) { // Step past the root directory
      memmove(sbuf, path, i);
      if(!futils_exists(sbuf)) {
#if defined (__WIN32__)
	rv = mkdir(sbuf);
#elif defined (__UNIX__)
	rv = mkdir(sbuf,
		   S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
	if(rv != 0) {
	  return 1;
	}
      }      
    }
  }

  // Get rid of any terminating path separators
  if(sbuf[len-1] == path_sep) sbuf[len-1] = '\0';

  // Make the complete directory path if needed
  if(!futils_exists(sbuf)) {
#if defined (__WIN32__)
	rv = mkdir(sbuf);
#elif defined(__UNIX__)
    rv = mkdir(sbuf,
	       S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
    if(rv != 0) {
      return 1;
    }
  }      

  return 0;
}

int futils_remove(const char *fname)
// Remove the specified file
{
  if(!futils_exists(fname)) {
    return 1;
  }
  if(remove(fname) != 0) { 
   return 1;
  }
  return 0;
}

int futils_rename(const char *oldname, const char *newname)
// Rename the specified file
{
  if(!futils_exists(oldname)) {
    return 1;
  }
  if(rename(oldname, newname) != 0) {
    return 1;
  }
  return 0;
}

int futils_chmod(const char *fname, futilPermissionMode pmode)
// Change the file-permission settings. The "pmode"
// variable should be equal to futils_READONLY, futils_WRITEONLY,
// or futils_READWRITE. Returns a non-zero value if an 
// error occurs.
{
  if(!futils_exists(fname)) return 1;

  switch(pmode) {
    case futils_READONLY :
#if defined (_S_IREAD)
      if(chmod(fname, _S_IREAD) != 0) {
	return 1;
      }
#else
      if(chmod(fname, S_IREAD) != 0) {
	return 1;
      }
#endif
      break;

    case futils_WRITEONLY :
#if defined (_S_IWRITE)
      if(chmod(fname, _S_IWRITE) != 0) {
	return 1;
      }
#else
      if(chmod(fname, S_IWRITE) != 0) {
	return 1;
      }
#endif
      break;

    case futils_READWRITE :
#if defined (_S_IREAD) && defined (_S_IWRITE)
      if(chmod(fname, _S_IREAD | _S_IWRITE) != 0) {
	return 1;
      }
#else
      if(chmod(fname, S_IREAD | S_IWRITE) != 0) {
	return 1;
      }
#endif
      break;

    default:
      return 1;
  }

  return 0;
}

void futils_pathsimplify(const char *path, char *new_path, char path_sep)
// Function used to canonicalize a path and return a new path.
// The path separator should either be a forward slash for UNIX
// file systems or a backslash for DOS/WIN32 file systems. Multiple
// path separators will be collapsed to a single path separator.
// Leading `./' paths and trailing `/.' paths will be removed.
// Trailing path separators will be removed. All non-leading `../'
// paths and trailing `..' paths are handled by removing portions of
// the path. NOTE: This function assumes that the necessary memory has
// already been allocated for the "new_path" variable by the calling
// function.
{
  unsigned len ;
  char stub_char;
  int i, start;
  int ddot; // Directory immediately above the current directory position
  
  if(!*path) return;

  // Copy the complete path into the new_path variable
  len = strlen(path);
  memmove(new_path, path, len);
  new_path[len] = '\0'; // Ensure null termination
  stub_char = (*new_path == path_sep) ? path_sep : '.';

  // Remove all `./' paths preceding the string.  If `../' paths
  // precede, put `/' in front and remove them as well
  i = ddot = start = 0;
  while (1) {
    if(new_path[i] == '.' && new_path[i + 1] == path_sep)
      i += 2;
    else if (new_path[i] == '.' && new_path[i + 1] == '.' && \
	     new_path[i + 2] == path_sep) { 
      i += 3;
      ddot = 1;
    }
    else
      break;
  }
  if(i) strcpy(new_path, new_path + i - ddot);

  // Replace single `.' or `..' with `/'.
  if((new_path[0] == '.' && new_path[1] == '\0') 
     || (new_path[0] == '.' && new_path[1] == '.' && new_path[2] == '\0')) {
    new_path[0] = stub_char;
    new_path[1] = '\0';
    return;
  }

  // Walk along the path looking for paths to compact
  i = 0;
  while(1) {
    if(!new_path[i]) break;

    while(new_path[i] && new_path[i] != path_sep)  i++;

    start = i++;

    // If we didn't find any slashes, then there is nothing left to do
    if(!new_path[start]) break;

    // Handle multiple path separators in a row
    while(new_path[i] == path_sep) i++;

    if((start + 1) != i) {
      strcpy(new_path + start + 1, new_path + i);
      i = start + 1;
    }

    // Check for trailing path separator
    if(start && !new_path[i]) {
      new_path[--i] = '\0';
      break;
    }

    // Check for `../' paths, `./' paths or a trailing `.' by itself
    if(new_path[i] == '.') {
      // Handle trailing a `.' path
      if(!new_path[i + 1]) {
	new_path[--i] = '\0';
	break;
      }

      // Handle `./' paths
      if(new_path[i + 1] == path_sep) {
	strcpy(new_path + i, new_path + i + 1);
	i =(start < 0) ? 0 : start;
	continue;
      }

      // Handle `../' paths or a trailing `..' path by itself
      if(new_path[i + 1] == '.' &&
	 (new_path[i + 2] == path_sep || !new_path[i + 2])) {
	while(--start > -1 && new_path[start] != path_sep);
	strcpy(new_path + start + 1, new_path + i + 2);
	i = (start < 0) ? 0 : start;
	continue;
      }
    }
  }

  if(!*new_path) { // Nothing was left after the path was simplifed
     *new_path = stub_char;
    new_path[1] = '\0';
  }

  // Ensure null termination
  new_path[strlen(new_path)] = '\0';
}

int futils_genoutputfilename(const char *current_file, char *out_file,
			 char *extension)
// Generate a name for the output file using the "current_file" name with 
// the specified dot extension. NOTE: This function assumes that the 
// necessary memory has already been allocated for the "out_file"
// variable by the calling function. Returns a non-zero value if any
// errors occur.
{
  unsigned i = 0;
  unsigned len;
  char *p;
  
  if(!out_file) return 1;

  for(i = 0; i < futils_MAX_NAME_LENGTH; i++) out_file[i] = '\0';
  p = (char *)current_file;
  len = strlen(p);
  for(i = 0; i < len && i != futils_MAX_NAME_LENGTH; i++, p++) {
    if(*p == '.') break;
    out_file[i] = *p;
  }
  if((strlen(out_file) + strlen(extension)) > (futils_MAX_NAME_LENGTH - 1)) 
    return 1;
  strcat(out_file, extension); // Add the file extension (.xxx)
  return 0;
}

int futils_getcwd(char *dir, unsigned max_len)
// Passes back the present working directory in the "dir"
// variable. Returns 0 if no errors occur. NOTE: This
// function assumes that the required amount of memory
// has already been allocated for the "dir" pointer. The
// "max_len" value must be at least one byte greater than
// the length of the pathname to be returned.
{
  unsigned i = 0;
  
  // Clear the path name buffer
  for(i = 0; i < max_len; i++) dir[i] = '\0';

  if(getcwd(dir, max_len) == 0) {
    return 1;
  }

  // Ensure null termination
  dir[strlen(dir)] = '\0';
  return 0;
}

long futils_filelength(int fd)
// Returns the file size, in bytes, for the specified file descriptor.
// Return a value of -1 to indicate an error.
{
  struct stat buf;
  int result;

  // Get data associated with the file descriptor.
  result = fstat(fd, &buf);
  
  // Check if statistics are valid:
  if(result != 0) {
    return -1L;
  }
  else {
    return buf.st_size;
  }
}
// ----------------------------------------------------------- //

// ----------------------------------------------------------- //
// Standalone platform indepentent directory stream functions  //
// ----------------------------------------------------------- //
DIR *futils_opendir(const char *dirname)  
// Opens a directory stream corresponding to the directory named by 
// the dirname argument. The directory stream is positioned at the 
// first entry.
{
#if defined (__WIN32__)
  DIR *dirp;

  // Ensure that the directory exitsts
  if(futils_chdir(dirname) != 0) return NULL;

  // Allocate a memory for the DIR sructure
  dirp = (DIR *)malloc(sizeof(DIR));
  if(dirp == NULL) return NULL;

  // Position the directory stream to the first entry
  dirp->dd_buf.d_ino = FindFirstFile("*.*", &dirp->dd_buf.file);
  if(dirp->dd_buf.d_ino == INVALID_HANDLE_VALUE) {
    free(dirp);
    return NULL;
  }
  
  // Initialize the DIR varaibles
  strcpy(dirp->dd_buf.d_name, dirp->dd_buf.file.cFileName);
  dirp->dd_buf.d_reclen = dirp->dd_buf.file.nFileSizeLow;
  dirp->dd_buf.d_namlen = strlen(dirp->dd_buf.file.cFileName);
  dirp->dd_loc = 0;
  dirp->dd_fd = -1;
  dirp->dd_size = dirp->dd_buf.file.nFileSizeLow;
  strcpy(dirp->dd_dirname, dirname);
  
  return dirp;
#elif defined (__UNIX__)
  return opendir(dirname);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
}

int futils_closedir(DIR *dirp)
// Closes the directory stream referred to by the argument dirp.
// Returns zero upon successful completion or -1  if an error
// occurs.
{
#if defined (__WIN32__)
  if(!FindClose(dirp->dd_buf.d_ino)) return -1;
  free(dirp);
  return 0;
#elif defined (__UNIX__)
  return closedir(dirp);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
}

struct dirent *futils_readdir(DIR *dirp)
// Returns a pointer to a structure representing the directory entry 
// at the current position in the directory stream specified by the 
// argument dirp, and positions the directory stream at the next entry. 
// It returns a null pointer upon reaching the end of the directory 
// stream.
{
#if defined (__WIN32__)
  if(dirp->dd_loc == 0) { // We are at the first entry
    dirp->dd_loc++;
    return &dirp->dd_buf;
  }

  if(FindNextFile(dirp->dd_buf.d_ino, &dirp->dd_buf.file)) {
    dirp->dd_loc++;
    strcpy(dirp->dd_buf.d_name, dirp->dd_buf.file.cFileName);
    dirp->dd_buf.d_reclen = dirp->dd_buf.file.nFileSizeLow;
    dirp->dd_buf.d_namlen = strlen(dirp->dd_buf.file.cFileName);
    dirp->dd_fd = -1;
    dirp->dd_size = dirp->dd_buf.file.nFileSizeLow;
    return &dirp->dd_buf;
  }
  else {
    return NULL;
  }
#elif defined (__UNIX__)
  // NOTE: This note applies to C language version the file utitlity functions
  // 07/30/2000: The readdir() function will chop off the first
  // two characters of the file name under Solaris if not compiled
  // with the default native C compiler. Make sure you use the native
  // compiler (default /opt/SUNWspro/bin/cc, which may not be in your 
  // PATH), and not /usr/ucb/cc. See the following URL for more details: 
  // http://sunse.jinr.ru/local/solaris/solaris2.html#q6.17
  return readdir(dirp);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
}

void futils_rewinddir(DIR *dirp)
// Resets the position of the directory stream to which dirp refers
// to the beginning of the directory.
{
#if defined (__WIN32__)
  // Ensure that the directory still exitsts
  if(futils_chdir(dirp->dd_dirname) != 0) return;

  // Position the directory stream to the first entry
  dirp->dd_buf.d_ino = FindFirstFile("*.*", &dirp->dd_buf.file);
  if(dirp->dd_buf.d_ino == INVALID_HANDLE_VALUE) return;
  
  // Reset the DIR varaibles
  strcpy(dirp->dd_buf.d_name, dirp->dd_buf.file.cFileName);
  dirp->dd_buf.d_reclen = dirp->dd_buf.file.nFileSizeLow;
  dirp->dd_buf.d_namlen = strlen(dirp->dd_buf.file.cFileName);
  dirp->dd_loc = 0;
  dirp->dd_fd = -1;
  dirp->dd_size = dirp->dd_buf.file.nFileSizeLow;
#elif defined (__UNIX__)
  rewinddir(dirp);
#else
#error You must define a file system: __WIN32__ or __UNIX__
#endif
}
// ----------------------------------------------------------- //

// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

