unit Param2d;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Menus;

type
  TOnParseFunc = function (Sender: TObject;
    const Expression: string; var Code: integer): double of object;

  TParView  = class;
  TParModel = class;

  TParShape = class(TObject)
  private
    FPen: TPen;
    FBrush: TBrush;
    FParModel: TParModel;
    FSolid: boolean;
    procedure SetBrush(Value: TBrush);
    procedure SetPen(Value: TPen);
  protected
    procedure SetString(const Value: string); virtual;
    function GetString: string; virtual;
    procedure RegError(const Msg: string);
  public
    constructor Create;
    function CreateInstance: TParShape; virtual; abstract;
    procedure Register;
    destructor Destroy; override;
    function GetVal(const name: string): double; virtual;
    procedure Draw(v: TParView); virtual; abstract;
    function IsInside(const r: TRect; v: TParView): boolean;
    function IsCrossing(const r: TRect; v: TParView): boolean; virtual;
    procedure GetExt(var minx, miny, maxx, maxy: double); virtual; abstract;
    property Pen: TPen read FPen write SetPen;
    property Brush: TBrush read FBrush write SetBrush;
    property Solid: boolean read FSolid write FSolid;
    property Model: TParModel read FParModel write FParModel;
    property Desc: string read GetString write SetString;
  end;

  TParPoint = class(TParShape)
  private
    FStl: char;
    FX, FY: string;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParLine = class(TParShape)
  private
    FX1, FY1, FX2, FY2: string;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParRect = class(TParLine)
  public
    function CreateInstance: TParShape; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParRoundRect = class(TParRect)
  private
    FX3, FY3: string;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParEllipse = class(TParLine)
  public
    function CreateInstance: TParShape; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParArc = class(TParLine)
  private
    FX3, FY3, FX4, FY4: string;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParChord = class(TParArc)
  public
    function CreateInstance: TParShape; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParPie = class(TParArc)
  public
    function CreateInstance: TParShape; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParPolyline = class(TParShape)
  private
    FX, FY: TStrings;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    destructor Destroy; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParPolygon = class(TParPolyline)
  public
    function CreateInstance: TParShape; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
  end;

  TParText = class(TParShape)
  private
    FAl: string[2];
    FX, FY, FFmt, FVal: string;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParDim = class(TParShape)
  private
    FA: char; {[V]ertical, [H]orizontal, [A]ligned, [R]adial}
    FX1, FY1, FX2, FY2, FGL, FFmt: string;
    m_dx, m_dy: array [0..4] of double;
    m_dw: double;
  protected
    procedure SetString(const Value: string); override;
    function GetString: string; override;
  public
    constructor Create;
    function CreateInstance: TParShape; override;
    function GetVal(const name: string): double; override;
    procedure Draw(v: TParView); override;
    function IsCrossing(const r: TRect; v: TParView): boolean; override;
    procedure GetExt(var minx, miny, maxx, maxy: double); override;
  end;

  TParModel = class(TComponent)
  private
    FOnParse: TOnParseFunc;
    FVars: TStrings;
    FDesc: TStrings;
    FParShapes: TStrings;
    FErrors: TStrings;
    FSelection: TStrings;
    FOnError: TNotifyEvent;
    procedure SetStrings(Value: TStrings);
    function GetStrings: TStrings;
    procedure SetVars(Value: TStrings);
  protected
    procedure Update;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    procedure Draw(v: TParView);
    procedure RegError(const Msg: string; Sender: TParShape);
    function GetVal(const name: string): double;
    procedure SelWindow(const r: TRect; v: TParView);
    procedure SelCross(const r: TRect; v: TParView);
    function SelSingle(const r: TRect; v: TParView): string;
    procedure GetExt(var minx, miny, maxx, maxy: double);
    procedure SaveToFile(const FileName: string);
    procedure LoadFromFile(const FileName: string);
    property Shapes: TStrings read FParShapes;
    property Selection: TStrings read FSelection;
    property Errors: TStrings read FErrors;
  published
    property Vars: TStrings read FVars write SetVars;
    property Desc: TStrings read GetStrings write SetStrings;
    property OnError: TNotifyEvent read FOnError write FOnError;
    property OnParse: TOnParseFunc read FOnParse write FOnParse;
  end;

  TParView = class(TGraphicControl)
  private
    FSelPen: TPen;
    FParModel: TParModel;
    FOrgX, FOrgY, FZoom: double;
    FGridColor: TColor;
    FGridMajor, FGridMinor: double;
    procedure SetSelPen(Value: TPen);
    procedure SetModel(Value: TParModel);
    procedure SetGridMajor(Value: double);
    procedure SetGridMinor(Value: double);
    procedure SetGridColor(Value: TColor);
  protected
    procedure DrawCS; virtual;
    procedure DrawGrid; virtual;
    procedure Draw; virtual;
    procedure DrawFast; virtual;
    procedure SetOrgX(Value: double);
    procedure SetOrgY(Value: double);
    procedure SetZoom(Value: double);
  public
    procedure Paint; override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetOrg(x, y: double);
    function GetViewX(dx: double): integer;
    function GetViewY(dy: double): integer;
    function GetPaperX(x: integer): double;
    function GetPaperY(y: integer): double;
    procedure SelWindow(const r: TRect);
    procedure SelCross(const r: TRect);
    function SelSingle(const p: TPoint): string;
    procedure SelPoint(const p: TPoint);
    procedure ZoomAll;
    property Canvas;
  published
    {TControl properties}
    property Width default 200;
    property Height default 100;
    property Color default clWhite;
    property Align;
    property DragCursor;
    property DragMode;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    {TControl event handlers}
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    {New properties}
    property SelPen: TPen read FSelPen write SetSelPen;
    property Model: TParModel read FParModel write SetModel;
    property OrgX: double read FOrgX write SetOrgX;
    property OrgY: double read FOrgY write SetOrgY;
    property Zoom: double read FZoom write SetZoom;
    property GridMajor: double read FGridMajor write SetGridMajor;
    property GridMinor: double read FGridMinor write SetGridMinor;
    property GridColor: TColor read FGridColor write SetGridColor;
  end;

procedure Register;

implementation

{$R param2d}

procedure Register;
begin
  RegisterComponents('Samples', [TParView, TParModel]);
end;

var
  ParRegistry: TStrings;

type
  TCrossEval = class
    X, Y: double;
    rl, rt, rr, rb: double;
    function CrossLine(xc, yc, xd, yd: double): boolean; virtual; abstract;
    function CrossRect: boolean;
  end;

  TLineCrossEval = class(TCrossEval)
    xa, ya, xb, yb: double;
    function CrossLine(xc, yc, xd, yd: double): boolean; override;
  end;

  TEllipseCrossEval = class(TCrossEval)
    xc, yc, a, b: double;
    function CrossLine(xa, ya, xb, yb: double): boolean; override;
  end;

  TArcCrossEval = class(TCrossEval)
    xc, yc, a, b, x3, y3, x4, y4: double;
    function CrossLine(xa, ya, xb, yb: double): boolean; override;
  end;

{ TCrossEval }

function TCrossEval.CrossRect: boolean;
begin
  result := false;
       if CrossLine(rl, rt, rr, rt) then result := true
  else if CrossLine(rl, rt, rl, rb) then result := true
  else if CrossLine(rr, rt, rr, rb) then result := true
  else if CrossLine(rl, rb, rr, rb) then result := true;
end;

{ TLineCrossEval }

function TLineCrossEval.CrossLine(xc, yc, xd, yd: double): boolean;
var A, B: double;
begin
  result := false;
  if (xb <> xa) and (xd <> xc) then begin
    A := (yb-ya)/(xb-xa);
    B := (yd-yc)/(xd-xc);
    if A <> B then begin
      X := (A*xa-ya-B*xc+yc) / (A-B);
      Y := A * (X-xa) + ya;
      if ((X-xa)*(X-xb)<0) and ((X-xc)*(X-xd)<0)
        then result := true;
    end;
  end else begin
    if (xb = xa) and (xd <> xc) then begin
      X := xb;
      B := (yd-yc)/(xd-xc);
      Y := B * (X-xc) + yc;
      if ((xa-xc)*(xa-xd)<0) and ((ya-Y)*(yb-Y)<0)
        then result := true
    end
    else
    if (xb <> xa) and (xd = xc) then begin
      X := xd;
      A := (yb-ya)/(xb-xa);
      Y := A * (X-xa) + ya;
      if ((xc-xa)*(xc-xb)<0) and ((yc-Y)*(yd-Y)<0)
        then result := true;
    end;
  end;
end;

{ TEllipseCrossEval }

function TEllipseCrossEval.CrossLine(xa, ya, xb, yb: double): boolean;
var C, D, E, F, G, H, I: double;
begin
  result := false;
  if (xb <> xa) and (a*b > 0) then begin
    C := (yb-ya)/(xb-xa);
    D := -C*xa+ya-yc;
    E := C*C+b*b/a/a;
    F := (2*D*C-2*xc*b*b/a/a)/E;
    G := (D*D-b*b+b*b*xc*xc/a/a)/E;
    H := F*F/4-G;
    if H > 0 then begin
      I := sqrt(H);
      X := I - F/2;
      Y := C * (X-xa) + ya;
      if ((xa-X)*(xb-X)<0) then result := true
      else begin
        X := -I - F/2;
        Y := C * (X-xa) + ya;
        if ((xa-X)*(xb-X)<0) then result := true;
      end;
    end;
  end else if a*b > 0 then begin
    X := xa;
    C := b*b-b*b*(X-xc)*(X-xc)/a/a;
    if C > 0 then begin
      D := sqrt(C);
      Y := D + yc;
      if ((ya-Y)*(yb-Y)<0) then result := true
      else begin
        Y := -D + yc;
        if ((ya-Y)*(yb-Y)<0) then result := true;
      end;
    end;
  end;
end;

{ TArcCrossEval }

function TArcCrossEval.CrossLine(xa, ya, xb, yb: double): boolean;
var C, D, E, F, G, H, I, Z1, Z2: double;
begin
  result := false;
  if (xb <> xa) and (a*b > 0) then begin
    C := (yb-ya)/(xb-xa);
    D := -C*xa+ya-yc;
    E := C*C+b*b/a/a;
    F := (2*D*C-2*xc*b*b/a/a)/E;
    G := (D*D-b*b+b*b*xc*xc/a/a)/E;
    H := F*F/4-G;
    if H > 0 then begin
      I := sqrt(H);
      X := I - F/2;
      if ((xa-X)*(xb-X)<0) then begin
        Y := C*(X-xa)+ya;
        Z1 := (X-xc)*(y3-yc)-(Y-yc)*(x3-xc);
        Z2 := (X-xc)*(y4-yc)-(Y-yc)*(x4-xc);
        if (Z1<0) and (Z2>0) then result := true;
      end
      else begin
        X := -I - F/2;
        if ((xa-X)*(xb-X)<0) then begin
          Y := C*(X-xa)+ya;
          Z1 := (X-xc)*(y3-yc)-(Y-yc)*(x3-xc);
          Z2 := (X-xc)*(y4-yc)-(Y-yc)*(x4-xc);
          if (Z1<0) and (Z2>0) then result := true;
        end;
      end;
    end;
  end else if a*b > 0 then begin
    X := xa;
    C := b*b-b*b*(X-xc)*(X-xc)/a/a;
    if C > 0 then begin
      D := sqrt(C);
      Y := D + yc;
      if ((ya-Y)*(yb-Y)<0) then begin
        Z1 := (X-xc)*(y3-yc)-(Y-yc)*(x3-xc);
        Z2 := (X-xc)*(y4-yc)-(Y-yc)*(x4-xc);
        if (Z1<0) and (Z2>0) then result := true;
      end
      else begin
        Y := -D + yc;
        if ((ya-Y)*(yb-Y)<0) then begin
          Z1 := (X-xc)*(y3-yc)-(Y-yc)*(x3-xc);
          Z2 := (X-xc)*(y4-yc)-(Y-yc)*(x4-xc);
          if (Z1<0) and (Z2>0) then result := true;
        end;
      end;
    end;
  end;
end;

{ TParShape }

constructor TParShape.Create;
begin
  inherited Create;
  FPen := TPen.Create;
  FBrush := TBrush.Create;
end;

procedure TParShape.Register;
var cname: string;
begin
  if ParRegistry = nil then
    ParRegistry := TStringList.Create;
  cname := LowerCase(ClassName);
  if ParRegistry.IndexOf(cname) = -1 then
    ParRegistry.AddObject(cname, Self);
end;

destructor TParShape.Destroy;
begin
  FPen.Free;
  FBrush.Free;
  inherited Destroy;
end;

procedure TParShape.SetBrush(Value: TBrush);
begin
  if Value <> nil then FBrush.Assign(Value);
end;

procedure TParShape.SetPen(Value: TPen);
begin
  if Value <> nil then FPen.Assign(Value);
end;

procedure TParShape.RegError(const Msg: string);
begin
  if FParModel <> nil then
     FParModel.RegError(Msg, Self);
end;

function TParShape.GetVal(const name: string): double;
var msg: string;
begin
  result := 0;
  msg := 'Unknown parameter '''+name+'''';
  RegError(msg);
end;

procedure TParShape.SetString(const Value: string);
var p, w: integer; st, lcase: string;
begin
  lcase := LowerCase(Value);
  p := pos('solid=', lcase);
  if p <> 0 then begin
    st := copy(lcase, p+6, 255);
    st := copy(st, 1, pos(';', st) - 1);
    if st[1] = 'y'
    then FSolid := true
    else FSolid := false;
  end;
  { Scan pen info }
  p := pos('pclr=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+5, 255);
    st := copy(st, 1, pos(';', st) - 1);
    try Pen.Color := StringToColor('cl'+st);
    except
      st := 'Bad pen color '''+st+'''';
      RegError(st);
    end;
  end;
  p := pos('pstl=', lcase);
  if p <> 0 then begin
    st := copy(lcase, p+5, 255);
    st := copy(st, 1, pos(';', st) - 1);
         if st = 'dash'       then FPen.Style := psDash
    else if st = 'dot'        then FPen.Style := psDot
    else if st = 'dashdot'    then FPen.Style := psDashDot
    else if st = 'dashdotdot' then FPen.Style := psDashDotDot
    else if st = 'clear'      then FPen.Style := psClear
    else if st = 'solid'      then FPen.Style := psSolid
    else begin
      st := 'Bad pen style '''+st+'''';
      RegError(st);
    end;
  end;
  p := pos('pwdh=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+5, 255);
    st := copy(st, 1, pos(';', st) - 1);
    try w := StrToInt(st);
    except
      st := 'Bad pen width '''+st+'''';
      RegError(st);
      w := 1;
    end;
    if w < 1 then w := 1;
    Pen.Width := w;
  end;

  { Scan brush info }
  p := pos('bclr=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+5, 255);
    st := copy(st, 1, pos(';', st) - 1);
    try Brush.Color := StringToColor('cl'+st);
    except
      st := 'Bad brush color '''+st+'''';
      RegError(st);
    end;
  end;
  p := pos('bstl=', lcase);
  if p <> 0 then begin
    st := copy(lcase, p+5, 255);
    st := copy(st, 1, pos(';', st) - 1);
         if st = 'solid'      then FBrush.Style := bsSolid
    else if st = 'clear'      then FBrush.Style := bsClear
    else if st = 'horizontal' then FBrush.Style := bsHorizontal
    else if st = 'vertical'   then FBrush.Style := bsVertical
    else if st = 'fdiagonal'  then FBrush.Style := bsFDiagonal
    else if st = 'bdiagonal'  then FBrush.Style := bsBDiagonal
    else if st = 'cross'      then FBrush.Style := bsCross
    else if st = 'diagcross'  then FBrush.Style := bsDiagCross
    else begin
      st := 'Bad brush style '''+st+'''';
      RegError(st);
    end;
  end;
end;

function TParShape.GetString: string;
var st: string;
begin
  result := '';
  if FSolid then result := result + 'Solid=Y;';
  if FPen.Color <> clBlack then begin
    st := ColorToString(FPen.Color); Delete(st, 1, 2);
    result := result + 'PClr=' + st + ';';
  end;
  if FPen.Width <> 1 then
    result := result + 'PWdh=' + IntToStr(FPen.Width) + ';';
  if FPen.Style <> psSolid then begin
    result := result + 'PStl=';
         if FPen.Style = psDash       then result := result + 'Dash'
    else if FPen.Style = psDot        then result := result + 'Dot'
    else if FPen.Style = psDashDot    then result := result + 'DashDot'
    else if FPen.Style = psDashDotDot then result := result + 'DashDotDot'
    else if FPen.Style = psClear      then result := result + 'Clear';
    result := result + ';';
  end;

  if FBrush.Color <> clWhite then begin
    st := ColorToString(FBrush.Color); Delete(st, 1, 2);
    result := result + 'BClr=' + st + ';';
  end;
  if FBrush.Style <> bsSolid then begin
    result := result + 'BStl=';
         if FBrush.Style = bsClear      then result := result + 'Clear'
    else if FBrush.Style = bsHorizontal then result := result + 'Horizontal'
    else if FBrush.Style = bsVertical   then result := result + 'Vertical'
    else if FBrush.Style = bsFDiagonal  then result := result + 'FDiagonal'
    else if FBrush.Style = bsBDiagonal  then result := result + 'BDiagonal'
    else if FBrush.Style = bsCross      then result := result + 'Cross'
    else if FBrush.Style = bsDiagCross  then result := result + 'DiagCross';
    result := result + ';';
  end;
end;

function TParShape.IsInside(const r: TRect; v: TParView): boolean;
var
  rl, rt, rr, rb: double;
  minx, miny, maxx, maxy: double;
begin
  result := false;
  if v <> nil then with v do begin
    Self.GetExt(minx, miny, maxx, maxy);
    rl := GetPaperX(r.Left);  rt := GetPaperY(r.Top);
    rr := GetPaperX(r.Right); rb := GetPaperY(r.Bottom);
    result := (rl < minx) and (rr > maxx)
          and (rt > maxy) and (rb < miny);
  end;
end;

function TParShape.IsCrossing(const r: TRect; v: TParView): boolean;
begin result := false; end;

{ TParPoint }

constructor TParPoint.Create;
begin
  inherited Create;
  FX := '0'; FY := '0'; FStl := '+';
end;

function TParPoint.CreateInstance: TParShape;
begin
  result := TParPoint.Create;
end;

procedure TParPoint.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+2, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX := st;
  end;
  p := pos('y=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+2, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY := st;
  end;
  p := pos('stl=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+4, 255);
    st := copy(st, 1, pos(';', st) - 1);
    if (pos(st[1], '+ox*') = 0) and (FParModel <> nil) then begin
      st := 'Unknown style '''+st[1]+'''';
      RegError(st);
    end else FStl := st[1];
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX); GetVal(FY);
  end;
end;

function TParPoint.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X=%s;Y=%s;Stl=%s;', [FX, FY, FStl]);
end;

function TParPoint.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X' then result := FParModel.GetVal(FX)
    else if st = 'Y' then result := FParModel.GetVal(FY)
    else result := inherited GetVal(name);
  end;
end;

procedure TParPoint.Draw(v: TParView);
var dx, dy: double; x, y: integer;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX);
    dy := FParModel.GetVal(FY);
    x := v.GetViewX(dx);
    y := v.GetViewY(dy);
    with v.Canvas do begin
      if FStl = '+' then begin
        MoveTo(x-2, y); LineTo(x+3, y);
        MoveTo(x, y-2); LineTo(x, y+3);
      end else
      if FStl = 'o' then begin
        Ellipse(x-2, y-2, x+3, y+3);
      end else
      if FStl = 'x' then begin
        MoveTo(x-2, y+2); LineTo(x+3, y-3);
        MoveTo(x-2, y-2); LineTo(x+3, y+3);
      end else
      if FStl = '*' then begin
        MoveTo(x-2, y);   LineTo(x+3, y);
        MoveTo(x, y-2);   LineTo(x, y+3);
        MoveTo(x-2, y+2); LineTo(x+3, y-3);
        MoveTo(x-2, y-2); LineTo(x+3, y+3);
      end;
    end;
  end;
end;

function TParPoint.IsCrossing(const r: TRect; v: TParView): boolean;
begin
  result := IsInside(r, v);
end;

procedure TParPoint.GetExt(var minx, miny, maxx, maxy: double);
begin
  minx := 0; miny := 0; maxx := 0; maxy := 0;
  if FParModel <> nil then begin
    minx := FParModel.GetVal(FX);
    miny := FParModel.GetVal(FY);
    maxx := minx;
    maxy := miny;
  end;
end;

{ TParLine }

constructor TParLine.Create;
begin
  inherited Create;
  FX1 := '0'; FY1 := '0'; FX2 := '0'; FY2 := '0';
end;

function TParLine.CreateInstance: TParShape;
begin
  result := TParLine.Create;
end;

procedure TParLine.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x1=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX1 := st;
  end;
  p := pos('y1=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY1 := st;
  end;
  p := pos('x2=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX2 := st;
  end;
  p := pos('y2=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY2 := st;
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX1); GetVal(FY1);
    GetVal(FX2); GetVal(FY2);
  end;
end;

function TParLine.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X1=%s;Y1=%s;X2=%s;Y2=%s;',
    [FX1, FY1, FX2, FY2]);
end;

function TParLine.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X1' then result := FParModel.GetVal(FX1)
    else if st = 'Y1' then result := FParModel.GetVal(FY1)
    else if st = 'X2' then result := FParModel.GetVal(FX2)
    else if st = 'Y2' then result := FParModel.GetVal(FY2)
    else if st = 'CEN.X' then {center of line, rect, ellipse, ...}
      result := (FParModel.GetVal(FX1) + FParModel.GetVal(FX2))/2
    else if st = 'CEN.Y' then
      result := (FParModel.GetVal(FY1) + FParModel.GetVal(FY2))/2
    else inherited GetVal(name);
  end;
end;

procedure TParLine.Draw(v: TParView);
var dx, dy: double; x, y: integer;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x := v.GetViewX(dx);
    y := v.GetViewY(dy);
    v.Canvas.MoveTo(x, y);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x := v.GetViewX(dx);
    y := v.GetViewY(dy);
    v.Canvas.LineTo(x, y);
  end;
end;

function TParLine.IsCrossing(const r: TRect; v: TParView): boolean;
var LCrsEval: TLineCrossEval;
begin
  result := false;
  LCrsEval := TLineCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) then with LCrsEval do begin
    xa := FParModel.GetVal(FX1);
    ya := FParModel.GetVal(FY1);
    xb := FParModel.GetVal(FX2);
    yb := FParModel.GetVal(FY2);
    rl := v.GetPaperX(r.Left);
    rt := v.GetPaperY(r.Top);
    rr := v.GetPaperX(r.Right);
    rb := v.GetPaperY(r.Bottom);
    result := CrossRect;
  end;
  LCrsEval.Free;
end;

procedure TParLine.GetExt(var minx, miny, maxx, maxy: double);
var d: double;
begin
  minx :=  32767; miny :=  32767;
  maxx := -32767; maxy := -32767;
  if FParModel <> nil then begin
    minx := FParModel.GetVal(FX1);
    miny := FParModel.GetVal(FY1);
    maxx := FParModel.GetVal(FX2);
    maxy := FParModel.GetVal(FY2);
  end;
  if (minx > maxx) then begin
    d := maxx; maxx := minx; minx := d;
  end;
  if (miny > maxy) then begin
    d := maxy; maxy := miny; miny := d;
  end;
end;

{ TParRect }

function TParRect.CreateInstance: TParShape;
begin
  result := TParRect.Create;
end;

procedure TParRect.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2: integer;
  bs: TBrushStyle; bc: TColor;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    if FSolid then with v.Canvas do begin
      bs := Brush.Style; bc := Brush.Color;
      Brush.Style := bsSolid;
      Brush.Color := v.Color;
      Rectangle(x1, y1, x2, y2);
      Brush.Style := bs; Brush.Color := bc;
    end;
    v.Canvas.Rectangle(x1, y1, x2, y2);
  end;
end;

function TParRect.IsCrossing(const r: TRect; v: TParView): boolean;
var
  sl, st, sr, sb: double;
  LCrsEval: TLineCrossEval;
begin
  result := false;
  LCrsEval := TLineCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) then begin
    sl := FParModel.GetVal(FX1);
    st := FParModel.GetVal(FY1);
    sr := FParModel.GetVal(FX2);
    sb := FParModel.GetVal(FY2);
    with LCrsEval do begin {1}
      rl := v.GetPaperX(r.Left);
      rt := v.GetPaperY(r.Top);
      rr := v.GetPaperX(r.Right);
      rb := v.GetPaperY(r.Bottom);
      xa := sl; ya := st; xb := sr; yb := st;
      if CrossRect then result := true
      else begin {2}
        xa := sl; ya := st; xb := sl; yb := sb;
        if CrossRect then result := true
        else begin {3}
          xa := sl; ya := sb; xb := sr; yb := sb;
          if CrossRect then result := true
          else begin {4}
            xa := sr; ya := st; xb := sr; yb := sb;
            if CrossRect then result := true;
          end; {4}
        end; {3}
      end; {2}
    end; {1}
  end;
  LCrsEval.Free;
end;

{ TParRoundRect }

constructor TParRoundRect.Create;
begin
  inherited Create;
  FX3 := '0'; FY3 := '0';
end;

function TParRoundRect.CreateInstance: TParShape;
begin
  result := TParRoundRect.Create;
end;

procedure TParRoundRect.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x3=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX3 := st;
  end;
  p := pos('y3=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY3 := st;
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX3); GetVal(FY3);
  end;
end;

function TParRoundRect.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X3=%s;Y3=%s;', [FX3, FY3]);
end;

function TParRoundRect.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X3' then result := FParModel.GetVal(FX3)
    else if st = 'Y3' then result := FParModel.GetVal(FY3)
    else result := inherited GetVal(name);
  end;
end;

procedure TParRoundRect.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2, x3, y3: integer;
  bs: TBrushStyle; bc: TColor;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX3);
    dy := FParModel.GetVal(FY3);
    x3 := round(dx*v.Zoom);
    y3 := round(dy*v.Zoom);
    if FSolid then with v.Canvas do begin
      bs := Brush.Style; bc := Brush.Color;
      Brush.Style := bsSolid;
      Brush.Color := v.Color;
      RoundRect(x1, y1, x2, y2, x3, y3);
      Brush.Style := bs; Brush.Color := bc;
    end;
    v.Canvas.RoundRect(x1, y1, x2, y2, x3, y3);
  end;
end;

function TParRoundRect.IsCrossing(const r: TRect; v: TParView): boolean;
begin {NIY}
  result := inherited IsCrossing(r, v);
end;

{ TParEllipse }

function TParEllipse.CreateInstance: TParShape;
begin
  result := TParEllipse.Create;
end;

procedure TParEllipse.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2: integer;
  bs: TBrushStyle; bc: TColor;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    if FSolid then with v.Canvas do begin
      bs := Brush.Style; bc := Brush.Color;
      Brush.Style := bsSolid;
      Brush.Color := v.Color;
      Ellipse(x1, y1, x2, y2);
      Brush.Style := bs; Brush.Color := bc;
    end;
    v.Canvas.Ellipse(x1, y1, x2, y2);
  end;
end;

function TParEllipse.IsCrossing(const r: TRect; v: TParView): boolean;
var
  x1, y1, x2, y2: double;
  ECrsEval: TEllipseCrossEval;
begin
  result := false;
  ECrsEval := TEllipseCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) then with ECrsEval do begin
    x1 := FParModel.GetVal(FX1);
    y1 := FParModel.GetVal(FY1);
    x2 := FParModel.GetVal(FX2);
    y2 := FParModel.GetVal(FY2);
    a  := abs(x1-x2)/2;
    b  := abs(y1-y2)/2;
    xc := (x1+x2)/2;
    yc := (y1+y2)/2;
    rl := v.GetPaperX(r.Left);
    rt := v.GetPaperY(r.Top);
    rr := v.GetPaperX(r.Right);
    rb := v.GetPaperY(r.Bottom);
    result := CrossRect;
  end;
  ECrsEval.Free;
end;

{ TParArc }

constructor TParArc.Create;
begin
  inherited Create;
  FX3 := '0'; FY3 := '0'; FX4 := '0'; FY4 := '0';
end;

function TParArc.CreateInstance: TParShape;
begin
  result := TParArc.Create;
end;

procedure TParArc.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x3=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX3 := st;
  end;
  p := pos('y3=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY3 := st;
  end;
  p := pos('x4=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX4 := st;
  end;
  p := pos('y4=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY4 := st;
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX3); GetVal(FY3);
    GetVal(FX4); GetVal(FY4);
  end;
end;

function TParArc.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X3=%s;Y3=%s;X4=%s;Y4=%s;',
    [FX3, FY3, FX4, FY4]);
end;

function TParArc.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X3' then result := FParModel.GetVal(FX3)
    else if st = 'Y3' then result := FParModel.GetVal(FY3)
    else if st = 'X4' then result := FParModel.GetVal(FX4)
    else if st = 'Y4' then result := FParModel.GetVal(FY4)
    else result := inherited GetVal(name);
  end;
end;

procedure TParArc.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2, x3, y3, x4, y4: integer;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX3);
    dy := FParModel.GetVal(FY3);
    x3 := v.GetViewX(dx);
    y3 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX4);
    dy := FParModel.GetVal(FY4);
    x4 := v.GetViewX(dx);
    y4 := v.GetViewY(dy);
    v.Canvas.Arc(x1, y1, x2, y2, x3, y3, x4, y4);
  end;
end;

function TParArc.IsCrossing(const r: TRect; v: TParView): boolean;
var
  x1, y1, x2, y2: double;
  ACrsEval: TArcCrossEval;
begin
  result := false;
  ACrsEval := TArcCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) then with ACrsEval do begin
    x1 := FParModel.GetVal(FX1);
    y1 := FParModel.GetVal(FY1);
    x2 := FParModel.GetVal(FX2);
    y2 := FParModel.GetVal(FY2);
    x3 := FParModel.GetVal(FX3);
    y3 := FParModel.GetVal(FY3);
    x4 := FParModel.GetVal(FX4);
    y4 := FParModel.GetVal(FY4);
    a  := abs(x1-x2)/2;
    b  := abs(y1-y2)/2;
    xc := (x1+x2)/2;
    yc := (y1+y2)/2;
    rl := v.GetPaperX(r.Left);
    rt := v.GetPaperY(r.Top);
    rr := v.GetPaperX(r.Right);
    rb := v.GetPaperY(r.Bottom);
    result := CrossRect;
  end;
  ACrsEval.Free;
end;

procedure TParArc.GetExt(var minx, miny, maxx, maxy: double);
begin {NIY}
  inherited GetExt(minx, miny, maxx, maxy);
end;

{ TParChord }

function TParChord.CreateInstance: TParShape;
begin
  result := TParChord.Create;
end;

procedure TParChord.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2, x3, y3, x4, y4: integer;
  bs: TBrushStyle; bc: TColor;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX3);
    dy := FParModel.GetVal(FY3);
    x3 := v.GetViewX(dx);
    y3 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX4);
    dy := FParModel.GetVal(FY4);
    x4 := v.GetViewX(dx);
    y4 := v.GetViewY(dy);
    if FSolid then with v.Canvas do begin
      bs := Brush.Style; bc := Brush.Color;
      Brush.Style := bsSolid;
      Brush.Color := v.Color;
      v.Canvas.Chord(x1, y1, x2, y2, x3, y3, x4, y4);
      Brush.Style := bs; Brush.Color := bc;
    end;
    v.Canvas.Chord(x1, y1, x2, y2, x3, y3, x4, y4);
  end;
end;

function TParChord.IsCrossing(const r: TRect; v: TParView): boolean;
var
  x1, y1, x2, y2: double;
  x3, y3, x4, y4: double;
  LCrsEval: TLineCrossEval;
  ECrsEval: TEllipseCrossEval;
begin
  result   := false;
  LCrsEval := TLineCrossEval.Create;
  ECrsEval := TEllipseCrossEval.Create;
  if inherited IsCrossing(r, v) then result := true
  else if (FParModel <> nil) and (v <> nil) then with ECrsEval do begin
    x1 := FParModel.GetVal(FX1);
    y1 := FParModel.GetVal(FY1);
    x2 := FParModel.GetVal(FX2);
    y2 := FParModel.GetVal(FY2);
    x3 := FParModel.GetVal(FX3);
    y3 := FParModel.GetVal(FY3);
    x4 := FParModel.GetVal(FX4);
    y4 := FParModel.GetVal(FY4);
    a  := abs(x1-x2)/2;
    b  := abs(y1-y2)/2;
    xc := (x1+x2)/2;
    yc := (y1+y2)/2;
    CrossLine(xc, yc, x3, y3);
    LCrsEval.xa := X;
    LCrsEval.ya := Y;
    CrossLine(xc, yc, x4, y4);
    with LCrsEval do begin
      xb := ECrsEval.X;
      yb := ECrsEval.Y;
      rl := v.GetPaperX(r.Left);
      rt := v.GetPaperY(r.Top);
      rr := v.GetPaperX(r.Right);
      rb := v.GetPaperY(r.Bottom);
      result := CrossRect;
    end;
  end;
  LCrsEval.Free;
  ECrsEval.Free;
end;

{ TParPie }

function TParPie.CreateInstance: TParShape;
begin
  result := TParPie.Create;
end;

procedure TParPie.Draw(v: TParView);
var
  dx, dy: double;
  x1, y1, x2, y2, x3, y3, x4, y4: integer;
  bs: TBrushStyle; bc: TColor;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX1);
    dy := FParModel.GetVal(FY1);
    x1 := v.GetViewX(dx);
    y1 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX2);
    dy := FParModel.GetVal(FY2);
    x2 := v.GetViewX(dx);
    y2 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX3);
    dy := FParModel.GetVal(FY3);
    x3 := v.GetViewX(dx);
    y3 := v.GetViewY(dy);
    dx := FParModel.GetVal(FX4);
    dy := FParModel.GetVal(FY4);
    x4 := v.GetViewX(dx);
    y4 := v.GetViewY(dy);
    if FSolid then with v.Canvas do begin
      bs := Brush.Style; bc := Brush.Color;
      Brush.Style := bsSolid;
      Brush.Color := v.Color;
      v.Canvas.Pie(x1, y1, x2, y2, x3, y3, x4, y4);
      Brush.Style := bs; Brush.Color := bc;
    end;
    v.Canvas.Pie(x1, y1, x2, y2, x3, y3, x4, y4);
  end;
end;

function TParPie.IsCrossing(const r: TRect; v: TParView): boolean;
var
  x1, y1, x2, y2: double;
  x3, y3, x4, y4: double;
  LCrsEval: TLineCrossEval;
  ECrsEval: TEllipseCrossEval;
begin
  result   := false;
  LCrsEval := TLineCrossEval.Create;
  ECrsEval := TEllipseCrossEval.Create;
  if inherited IsCrossing(r, v) then result := true
  else if (FParModel <> nil) and (v <> nil) then with ECrsEval do begin
    x1 := FParModel.GetVal(FX1);
    y1 := FParModel.GetVal(FY1);
    x2 := FParModel.GetVal(FX2);
    y2 := FParModel.GetVal(FY2);
    x3 := FParModel.GetVal(FX3);
    y3 := FParModel.GetVal(FY3);
    x4 := FParModel.GetVal(FX4);
    y4 := FParModel.GetVal(FY4);
    a  := abs(x1-x2)/2;
    b  := abs(y1-y2)/2;
    xc := (x1+x2)/2;
    yc := (y1+y2)/2;
    LCrsEval.xa := xc;
    LCrsEval.ya := yc;
    CrossLine(xc, yc, x3, y3);
    LCrsEval.xb := X;
    LCrsEval.yb := Y;
    LCrsEval.rl := v.GetPaperX(r.Left);
    LCrsEval.rt := v.GetPaperY(r.Top);
    LCrsEval.rr := v.GetPaperX(r.Right);
    LCrsEval.rb := v.GetPaperY(r.Bottom);
    if LCrsEval.CrossRect then result := true
    else begin
      CrossLine(xc, yc, x4, y4);
      LCrsEval.xb := X;
      LCrsEval.yb := Y;
      result := LCrsEval.CrossRect;
    end;
  end;
  LCrsEval.Free;
  ECrsEval.Free;
end;

{ TParPolyline }

constructor TParPolyline.Create;
begin
  inherited Create;
  FX := TStringList.Create;
  FY := TStringList.Create;
end;

function TParPolyline.CreateInstance: TParShape;
begin
  result := TParPolyline.Create;
end;

destructor TParPolyline.Destroy;
begin
  FX.Free;
  FY.Free;
  inherited Destroy;
end;

procedure TParPolyline.SetString(const Value: string);
var
  i, p: integer;
  st, mask, lcase: string; num: string[2];
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  i := 0;
  while (i <= FX.Count) do begin
    str(i+1, num);
    mask := 'x'+num+'=';
    p := pos(mask, lcase);
    if p <> 0 then begin
      st := copy(Value, p+length(mask), 255);
      st := copy(st, 1, pos(';', st) - 1);
      if i = FX.Count then FX.Add(st)
      else FX.Strings[i] := st;
    end;
    mask := 'y'+num+'=';
    p := pos(mask, lcase);
    if p <> 0 then begin
      st := copy(Value, p+length(mask), 255);
      st := copy(st, 1, pos(';', st) - 1);
      if i = FY.Count then FY.Add(st)
      else FY.Strings[i] := st;
    end;
    inc(i);
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    for i := 0 to FX.Count-1 do begin
      GetVal(FX.Strings[i]); GetVal(FY.Strings[i]);
    end;
  end;
end;

function TParPolyline.GetString: string;
var i: integer;
begin
  result := inherited GetString;
  for i := 0 to FX.Count-1 do begin
    result := result + Format('X%d=%s;Y%d=%s;',
      [i+1, FX.Strings[i], i+1, FY.Strings[i]]);
  end;
end;

function TParPolyline.GetVal(const name: string): double;
var st: string; idx, code: integer; c: char;
begin
  result := 0;
  st := UpperCase(name);
  c  := st[1];
  Delete(st, 1, 1);
  val(st, idx, code); dec(idx);
  if (FParModel <> nil) and (code=0) and (idx < FX.Count) then begin
         if c = 'X' then result := FParModel.GetVal(FX.Strings[idx])
    else if c = 'Y' then result := FParModel.GetVal(FY.Strings[idx])
    else result := inherited GetVal(name);
  end else begin
    if (code=0) and (idx >= FX.Count)
      then RegError('Bad point index '''+name+'''');
  end;
end;

procedure TParPolyline.Draw(v: TParView);
var
  dx, dy: double;
  Points, pts: ^TPoint;
  i, count, size: integer;
begin
  count := FX.Count;
  size  := sizeof(TPoint)*count;
  if (size <> 0) and (v <> nil) and (FParModel <> nil) then begin
    GetMem(Points, size); pts := Points;
    FillChar(Points^, size, 0);
    for i := 0 to count-1 do begin
      dx := FParModel.GetVal(FX.Strings[i]);
      dy := FParModel.GetVal(FY.Strings[i]);
      pts^.x := v.GetViewX(dx);
      pts^.y := v.GetViewY(dy);
      inc(pts);
    end;
    Polyline(v.Canvas.Handle, Points^, count);
    FreeMem(Points, size);
  end;
end;

function TParPolyline.IsCrossing(const r: TRect; v: TParView): boolean;
var i: integer; LCrsEval: TLineCrossEval;
begin
  result := false;
  LCrsEval := TLineCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) and (FX.Count > 1) then begin
    with LCrsEval do begin
      rl := v.GetPaperX(r.Left);
      rt := v.GetPaperY(r.Top);
      rr := v.GetPaperX(r.Right);
      rb := v.GetPaperY(r.Bottom);
    end;
    for i := 0 to FX.Count-2 do with LCrsEval do begin
      xa := FParModel.GetVal(FX.Strings[i]);
      ya := FParModel.GetVal(FY.Strings[i]);
      xb := FParModel.GetVal(FX.Strings[i+1]);
      yb := FParModel.GetVal(FY.Strings[i+1]);
      result := CrossRect;
      if result then break;
    end;
  end;
  LCrsEval.Free;
end;

procedure TParPolyline.GetExt(var minx, miny, maxx, maxy: double);
var i: integer; dx, dy: double;
begin
  minx :=  32767; miny :=  32767;
  maxx := -32767; maxy := -32767;
  if FParModel <> nil then for i := 0 to FX.Count-1 do begin
    dx := FParModel.GetVal(FX.Strings[i]);
    dy := FParModel.GetVal(FY.Strings[i]);
    if dx > maxx then maxx := dx;
    if dx < minx then minx := dx;
    if dy > maxy then maxy := dy;
    if dy < miny then miny := dy;
  end;
end;

{ TParPolygon }

function TParPolygon.CreateInstance: TParShape;
begin
  result := TParPolygon.Create;
end;

procedure TParPolygon.Draw(v: TParView);
var
  dx, dy: double;
  Points, pts: ^TPoint;
  i, count, size: integer;
  bs: TBrushStyle; bc: TColor;
begin
  count := FX.Count;
  size  := sizeof(TPoint)*count;
  if (size <> 0) and (v <> nil) and (FParModel <> nil) then begin
    GetMem(Points, size); pts := Points;
    FillChar(Points^, size, 0);
    for i := 0 to count-1 do begin
      dx := FParModel.GetVal(FX.Strings[i]);
      dy := FParModel.GetVal(FY.Strings[i]);
      pts^.x := v.GetViewX(dx);
      pts^.y := v.GetViewY(dy);
      inc(pts);
    end;
    if FSolid then begin
      bs := v.Canvas.Brush.Style;
      bc := v.Canvas.Brush.Color;
      v.Canvas.Brush.Style := bsSolid;
      v.Canvas.Brush.Color := v.Color;
      Polygon(v.Canvas.Handle, Points^, count);
      v.Canvas.Brush.Style := bs;
      v.Canvas.Brush.Color := bc;
    end;
    Polygon(v.Canvas.Handle, Points^, count);
    FreeMem(Points, size);
  end;
end;

function TParPolygon.IsCrossing(const r: TRect; v: TParView): boolean;
var LCrsEval: TLineCrossEval;
begin
  result := inherited IsCrossing(r, v);
  if (not result) and (FParModel <> nil) and (v <> nil)
    and (FX.Count > 1) then begin
    LCrsEval := TLineCrossEval.Create;
    with LCrsEval do begin
      rl := v.GetPaperX(r.Left);
      rt := v.GetPaperY(r.Top);
      rr := v.GetPaperX(r.Right);
      rb := v.GetPaperY(r.Bottom);
      xa := FParModel.GetVal(FX.Strings[0]);
      ya := FParModel.GetVal(FY.Strings[0]);
      xb := FParModel.GetVal(FX.Strings[FX.Count-1]);
      yb := FParModel.GetVal(FY.Strings[FX.Count-1]);
      result := CrossRect;
    end;
    LCrsEval.Free;
  end;
end;

{ TParText }

constructor TParText.Create;
begin
  inherited Create;
  FAl := 'LB';
  FX := '0'; FY := '0'; FFmt := ''; FVal := '';
end;

function TParText.CreateInstance: TParShape;
begin
  result := TParText.Create;
end;

procedure TParText.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+2, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX := st;
  end;
  p := pos('y=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+2, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY := st;
  end;
  p := pos('fmt=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+4, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FFmt := st;
  end;
  p := pos('val=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+4, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FVal := st;
  end;
  p := pos('al=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FAl := UpperCase(st);
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX); GetVal(FY);
  end;
end;

function TParText.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X=%s;Y=%s;Al=%s;Fmt=%s;Val=%s;',
    [FX, FY, FAl, FFmt, FVal]);
end;

function TParText.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X' then result := FParModel.GetVal(FX)
    else if st = 'Y' then result := FParModel.GetVal(FY)
    else result := inherited GetVal(name);
  end;
end;

procedure TParText.Draw(v: TParView);
var dx, dy: double; x, y, w, h: integer; st: string;
begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx := FParModel.GetVal(FX);
    dy := FParModel.GetVal(FY);
    x := v.GetViewX(dx);
    y := v.GetViewY(dy);
    with v.Canvas do begin
      try st := Format(FFmt, [ FParModel.GetVal(FVal)]);
      except st := FFmt; end;
      w  := TextWidth(st);
      h  := TextHeight(st);
      if FAl[1] = 'R' then x := x - w;
      if FAl[1] = 'C' then x := x - w div 2;
      if FAl[2] = 'B' then y := y - h;
      if FAl[2] = 'C' then y := y - h div 2;
      TextOut(x, y, st);
    end;
  end;
end;

function TParText.IsCrossing(const r: TRect; v: TParView): boolean;
begin {NIY}
  result := IsInside(r, v);
end;

procedure TParText.GetExt(var minx, miny, maxx, maxy: double);
begin {NIY}
  minx := 0; miny := 0; maxx := 0; maxy := 0;
  if FParModel <> nil then begin
    minx := FParModel.GetVal(FX);
    miny := FParModel.GetVal(FY);
    maxx := minx;
    maxy := miny;
  end;
end;

{ TParDim }

constructor TParDim.Create;
begin
  inherited Create;
  FX1 := '0'; FY1 := '0'; FX2 := '0'; FY2 := '0';
  FGL := '0'; FA  := 'v'; FFmt := '%f';
end;

function TParDim.CreateInstance: TParShape;
begin
  result := TParDim.Create;
end;

procedure TParDim.SetString(const Value: string);
var p: integer; st, lcase: string;
begin
  inherited SetString(Value);
  lcase := LowerCase(Value);
  p := pos('x1=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX1 := st;
  end;
  p := pos('y1=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY1 := st;
  end;
  p := pos('x2=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FX2 := st;
  end;
  p := pos('y2=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FY2 := st;
  end;
  p := pos('gl=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+3, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FGL := st;
  end;
  p := pos('fmt=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+4, 255);
    st := copy(st, 1, pos(';', st) - 1);
    FFmt := st;
  end;
  p := pos('a=', lcase);
  if p <> 0 then begin
    st := copy(Value, p+2, 255);
    st := copy(st, 1, pos(';', st) - 1);
    if pos(st[1], 'vhar') = 0 then
      RegError(Format('Bad dim align ''%s''', [st]))
    else FA := st[1];
  end;
  {Check for error within the Update call}
  if FParModel <> nil then with FParModel do begin
    GetVal(FX1); GetVal(FY1);
    GetVal(FX2); GetVal(FY2);
    GetVal(FGL);
  end;
end;

function TParDim.GetString: string;
begin
  result := inherited GetString;
  result := result + Format('X1=%s;Y1=%s;X2=%s;Y2=%s;Gl=%s;A=%s;Fmt=%s;',
    [FX1, FY1, FX2, FY2, FGL, FA, FFmt]);
end;

function TParDim.GetVal(const name: string): double;
var st: string;
begin
  result := 0;
  st := UpperCase(name);
  if FParModel <> nil then begin
         if st = 'X1' then result := FParModel.GetVal(FX1)
    else if st = 'Y1' then result := FParModel.GetVal(FY1)
    else if st = 'X2' then result := FParModel.GetVal(FX2)
    else if st = 'Y2' then result := FParModel.GetVal(FY2)
    else if st = 'GL' then result := FParModel.GetVal(FGL)
    else inherited GetVal(name);
  end;
end;

procedure TParDim.Draw(v: TParView);
var
  cap: string;
  LogFont: TLogFont;
  Head: array [0..2] of TPoint;
  d, dx1, dy1, dx2, dy2, sina, cosa, ang: double;
  Escapement, x, y, x1, y1, x2, y2, L, w, h: integer;

const
  hw = 8; hh = 2;

function sign(const i: integer): integer;
begin if i>=0 then result := +1 else result := -1 end;

procedure rothead;
var i: integer; dx, dy, nx, ny: double;
begin
  for i := 1 to 2 do begin
    dx := Head[i].x - Head[0].x;
    dy := Head[i].y - Head[0].y;
    nx := dx*cosa + dy*sina;
    ny := dy*cosa - dx*sina;
    Head[i].x := Head[0].x + round(nx);
    Head[i].y := Head[0].y + round(ny);
  end;
end;

begin
  if (v <> nil) and (FParModel <> nil) then begin
    dx1 := FParModel.GetVal(FX1);
    dy1 := FParModel.GetVal(FY1);
    x1  := v.GetViewX(dx1);
    y1  := v.GetViewY(dy1);
    dx2 := FParModel.GetVal(FX2);
    dy2 := FParModel.GetVal(FY2);
    x2  := v.GetViewX(dx2);
    y2  := v.GetViewY(dy2);
    L   := round(FParModel.GetVal(FGL)*v.Zoom);
    case FA of
    'h': with v.Canvas do begin
        d := abs(dx2-dx1);
        try cap := Format(FFmt, [d]);
        except cap := FFmt end;
        w := TextWidth(cap); m_dw := (w+hw)/v.Zoom;
        h := TextHeight(cap);
        {|--|}
        MoveTo(x1, y1);
        LineTo(x1, y1+L+hh*sign(L));
        LineTo(x1, y1+L);
        if (cap[1] = '-') then LineTo(x1-w-hw, y1+L)
        else if (cap[1] = '+') then LineTo(x2+w+hw, y1+L);
        LineTo(x2, y1+L);
        LineTo(x2, y1+L+hh*sign(L));
        LineTo(x2, y2);
        {<-} if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x1,    y1+L);
          Head[1] := Point(x1+hw, y1+L+hh);
          Head[2] := Point(x1+hw, y1+L-hh);
        end else begin
          Head[0] := Point(x1,    y1+L);
          Head[1] := Point(x1-hw, y1+L+hh);
          Head[2] := Point(x1-hw, y1+L-hh);
        end;
        Polygon(Head);
        {->} if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x2,    y1+L);
          Head[1] := Point(x2-hw, y1+L+hh);
          Head[2] := Point(x2-hw, y1+L-hh);
        end else begin
          Head[0] := Point(x2,    y1+L);
          Head[1] := Point(x2+hw, y1+L+hh);
          Head[2] := Point(x2+hw, y1+L-hh);
        end;
        Polygon(Head);
        {Caption}
        if cap[1] = '-' then begin
          delete(cap, 1, 1);
          TextOut(x1-w-hw, y1+L-h-hh, cap);
        end else
        if cap[1] = '+' then begin
          delete(cap, 1, 1);
          TextOut(x2+hw, y1+L-h-hh, cap);
        end else
        TextOut((x1+x2-w) div 2, y1+L-h-hh, cap);
      end; {h}
    'v': with v.Canvas do begin
        d := abs(dy2-dy1);
        try cap := Format(FFmt, [d]);
        except cap := FFmt end;
        w := TextWidth(cap); m_dw := (w+hw)/v.Zoom;
        h := TextHeight(cap);
        {|--|}
        MoveTo(x1,              y1);
        LineTo(x1+L+hh*sign(L), y1);
        LineTo(x1+L,            y1);
        if (cap[1] = '-') then LineTo(x1+L, y1+w+hw)
        else if (cap[1] = '+') then LineTo(x1+L, y2-w-hw);
        LineTo(x1+L,            y2);
        LineTo(x1+L+hh*sign(L), y2);
        LineTo(x2,              y2);
        {V} if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x1+L,    y1);
          Head[1] := Point(x1+L+hh, y1-hw);
          Head[2] := Point(x1+L-hh, y1-hw);
        end else begin
          Head[0] := Point(x1+L,    y1);
          Head[1] := Point(x1+L+hh, y1+hw);
          Head[2] := Point(x1+L-hh, y1+hw);
        end;
        Polygon(Head);
        {^} if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x1+L,    y2);
          Head[1] := Point(x1+L+hh, y2+hw);
          Head[2] := Point(x1+L-hh, y2+hw);
        end else begin
          Head[0] := Point(x1+L,    y2);
          Head[1] := Point(x1+L+hh, y2-hw);
          Head[2] := Point(x1+L-hh, y2-hw);
        end;
        Polygon(Head);
        {Caption}
        GetObject(Font.Handle, SizeOf(LogFont), Addr(LogFont));
        Escapement := LogFont.lfEscapement;
        LogFont.lfEscapement := (90 * 10);
        LogFont.lfOutPrecision := OUT_TT_ONLY_PRECIS;
        Font.Handle := CreateFontIndirect(LogFont);
        if cap[1] = '-' then begin
          delete(cap, 1, 1);
          TextOut(x1+L-h-hh, y1+w+hw, cap);
        end else
        if cap[1] = '+' then begin
          delete(cap, 1, 1);
          TextOut(x1+L-h-hh, y2-hw, cap);
        end else
        TextOut(x1+L-h-hh, (y1+y2+w) div 2, cap);
        LogFont.lfEscapement := Escapement;
        Font.Handle := CreateFontIndirect(LogFont);
      end; {v}
    'a': with v.Canvas do begin
        d := sqrt(sqr(dx2-dx1)+sqr(dy2-dy1));
        sina := (dy2-dy1)/d;
        cosa := (dx2-dx1)/d;
        try cap := Format(FFmt, [d]);
        except cap := FFmt end;
        w := TextWidth(cap); m_dw := (w+hw)/v.Zoom;
        h := TextHeight(cap);
        {|--|}
        MoveTo(x1, y1);
        LineTo(x1+round((L+hh)*sina), y1+round((L+hh)*cosa));
        LineTo(x1+round(L*sina), y1+round(L*cosa));
        if (cap[1] = '-') then LineTo(x1+round(L*sina-(w+hw)*cosa),
                                      y1+round(L*cosa+(w+hw)*sina))
        else if (cap[1] = '+') then LineTo(x2+round(L*sina+(w+hw)*cosa),
                                           y2+round(L*cosa-(w+hw)*sina));
        LineTo(x2+round(L*sina), y2+round(L*cosa));
        LineTo(x2+round((L+hh)*sina), y2+round((L+hh)*cosa));
        LineTo(x2, y2);
        {<-}
        x := round( x1+L*sina );
        y := round( y1+L*cosa );
        if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x,    y);
          Head[1] := Point(x+hw, y+hh);
          Head[2] := Point(x+hw, y-hh);
        end else begin
          Head[0] := Point(x,    y);
          Head[1] := Point(x-hw, y+hh);
          Head[2] := Point(x-hw, y-hh);
        end;
        rothead; Polygon(Head);
        {->}
        x := round( x2+L*sina );
        y := round( y2+L*cosa );
        if (cap[1] <> '-') and (cap[1] <> '+') then begin
          Head[0] := Point(x,    y);
          Head[1] := Point(x-hw, y+hh);
          Head[2] := Point(x-hw, y-hh);
        end else begin
          Head[0] := Point(x,    y);
          Head[1] := Point(x+hw, y+hh);
          Head[2] := Point(x+hw, y-hh);
        end;
        rothead; Polygon(Head);
        {Caption}
        GetObject(Font.Handle, SizeOf(LogFont), Addr(LogFont));
        Escapement := LogFont.lfEscapement;
        if cosa <> 0 then ang := ArcTan(sina/cosa)*180/pi
        else ang := 90;
        LogFont.lfEscapement := round(ang * 10);
        LogFont.lfOutPrecision := OUT_TT_ONLY_PRECIS;
        Font.Handle := CreateFontIndirect(LogFont);
        if cap[1] = '-' then begin
          delete(cap, 1, 1);
          x := round( (x1+L*sina) - (h*sina+w*cosa) );
          y := round( (y1+L*cosa) - (h*cosa-w*sina) );
        end else
        if cap[1] = '+' then begin
          delete(cap, 1, 1);
          x := round( (x2+L*sina) - (h*sina-hw*cosa) );
          y := round( (y2+L*cosa) - (h*cosa+hw*sina) );
        end else begin
          x := round( (x1+x2+2*L*sina)/2 - (h*sina+0.5*w*cosa) );
          y := round( (y1+y2+2*L*cosa)/2 - (h*cosa-0.5*w*sina) );
        end;
        TextOut(x, y, cap);
        LogFont.lfEscapement := Escapement;
        Font.Handle := CreateFontIndirect(LogFont);
      end; {a}
    'r': with v.Canvas do begin
        d := sqrt(sqr(dx2-dx1)+sqr(dy2-dy1));
        sina := (dy2-dy1)/d;
        cosa := (dx2-dx1)/d;
        try cap := Format(FFmt, [d]);
        except cap := FFmt end;
        w := TextWidth(cap); m_dw := (w+hw)/v.Zoom;
        h := TextHeight(cap);
        {|--|}
        MoveTo(x1,   y1);
        LineTo(x2,   y2);
        LineTo(x2+L, y2);
        {<-}
        Head[0] := Point(x1,    y1);
        Head[1] := Point(x1+hw, y1+hh);
        Head[2] := Point(x1+hw, y1-hh);
        rothead; Polygon(Head);
        {Caption}
        if L > 0 then x := x2+L-w else x := x2+L;
        y := y2-h-1;
        TextOut(x, y, cap);
      end; {r}
    end; {case}
  end;
end;

function TParDim.IsCrossing(const r: TRect; v: TParView): boolean;
var
  LCrsEval: TLineCrossEval;
  minx, miny, maxx, maxy: double;
begin
  result := false;
  LCrsEval := TLineCrossEval.Create;
  if (FParModel <> nil) and (v <> nil) then with LCrsEval do begin
    rl := v.GetPaperX(r.Left);
    rt := v.GetPaperY(r.Top);
    rr := v.GetPaperX(r.Right);
    rb := v.GetPaperY(r.Bottom);

    GetExt(minx, miny, maxx, maxy);
    xa := m_dx[0];
    ya := m_dy[0];
    xb := m_dx[2];
    yb := m_dy[2];
    if (FA <> 'r') then result := CrossRect;
    if not result then begin
      xa := m_dx[1];
      ya := m_dy[1];
      xb := m_dx[3];
      yb := m_dy[3];
      result := CrossRect;
      if not result then begin
        xa := m_dx[2];
        ya := m_dy[2];
        xb := m_dx[3];
        yb := m_dy[3];
        result := CrossRect;
        if not result then begin
          xa := m_dx[2];
          ya := m_dy[2];
          xb := m_dx[4];
          yb := m_dy[4];
          result := CrossRect;
          if not result and (FA = 'r') then begin
            xa := m_dx[0];
            ya := m_dy[0];
            xb := m_dx[1];
            yb := m_dy[1];
            result := CrossRect;
          end;
        end;
      end;
    end;
  end;
  LCrsEval.Free;
end;

procedure TParDim.GetExt(var minx, miny, maxx, maxy: double);
var i: integer; d, dL, sina, cosa: double;
begin
  minx :=  32767; miny :=  32767;
  maxx := -32767; maxy := -32767;
  if FParModel <> nil then begin
         dL := FParModel.GetVal(FGL);
    m_dx[0] := FParModel.GetVal(FX1);
    m_dy[0] := FParModel.GetVal(FY1);
    m_dx[1] := FParModel.GetVal(FX2);
    m_dy[1] := FParModel.GetVal(FY2);

    case FA of
    'h': begin
        m_dx[2] := m_dx[0];
        m_dx[3] := m_dx[1];
        m_dy[2] := m_dy[0] - dL;
        m_dy[3] := m_dy[0] - dL;
        if FFmt[1] = '-' then begin
          m_dx[4] := m_dx[2] - m_dw;
          m_dy[4] := m_dy[2];
        end else if FFmt[1] = '+' then begin
          m_dx[4] := m_dx[3] + m_dw;
          m_dy[4] := m_dy[2];
        end else begin
          m_dx[4] := m_dx[2];
          m_dy[4] := m_dy[2];
        end;
      end; {h}
    'v': begin
        m_dy[2] := m_dy[0];
        m_dy[3] := m_dy[1];
        m_dx[2] := m_dx[0] + dL;
        m_dx[3] := m_dx[0] + dL;
        if FFmt[1] = '-' then begin
          m_dx[4] := m_dx[2];
          m_dy[4] := m_dy[2] - m_dw;
        end else if FFmt[1] = '+' then begin
          m_dx[4] := m_dx[2];
          m_dy[4] := m_dy[3] + m_dw;
        end else begin
          m_dx[4] := m_dx[2];
          m_dy[4] := m_dy[2];
        end;
      end; {v}
    'a': begin
        d := sqrt(sqr(m_dx[1]-m_dx[0])+sqr(m_dy[1]-m_dy[0]));
        sina := (m_dy[1]-m_dy[0])/d;
        cosa := (m_dx[1]-m_dx[0])/d;
        m_dx[2] := m_dx[0] + dL*sina;
        m_dy[2] := m_dy[0] - dL*cosa;
        m_dx[3] := m_dx[1] + dL*sina;
        m_dy[3] := m_dy[1] - dL*cosa;
        if FFmt[1] = '-' then begin
          m_dx[4] := m_dx[2] - m_dw * cosa;
          m_dy[4] := m_dy[2] - m_dw * sina;
        end else if FFmt[1] = '+' then begin
          m_dx[4] := m_dx[3] + m_dw * cosa;
          m_dy[4] := m_dy[3] + m_dw * sina;
        end else begin
          m_dx[4] := m_dx[2];
          m_dy[4] := m_dy[2];
        end;
      end; {a}
    'r': begin
        m_dx[2] := m_dx[1] + dL;
        m_dy[2] := m_dy[1];
        m_dx[3] := m_dx[2];
        m_dy[3] := m_dy[2];
        m_dx[4] := m_dx[2];
        m_dy[4] := m_dy[2];
      end; {r}
    end; {case}

    for i := 0 to 4 do begin
      if m_dx[i] < minx then minx := m_dx[i];
      if m_dx[i] > maxx then maxx := m_dx[i];
      if m_dy[i] < miny then miny := m_dy[i];
      if m_dy[i] > maxy then maxy := m_dy[i];
    end;
  end;
end;

{ TParModel }

constructor TParModel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVars := TStringList.Create;
  FDesc := TStringList.Create;
  FParShapes := TStringList.Create;
  FErrors := TStringList.Create;
  FSelection := TStringList.Create;

  (TParPoint.Create).Register;
  (TParLine.Create).Register;
  (TParRect.Create).Register;
  (TParRoundRect.Create).Register;
  (TParEllipse.Create).Register;
  (TParArc.Create).Register;
  (TParChord.Create).Register;
  (TParPie.Create).Register;
  (TParPolyline.Create).Register;
  (TParPolygon.Create).Register;
  (TParText.Create).Register;
  (TParDim.Create).Register;
end;

destructor TParModel.Destroy;
begin
  Clear;
  FVars.Free;
  FDesc.Free;
  FErrors.Free;
  FSelection.Free;
  FParShapes.Free;
  inherited Destroy;
end;

procedure TParModel.Clear;
var i: integer; ob: TObject;
begin
  FVars.Clear;
  FDesc.Clear;
  FErrors.Clear;
  FSelection.Clear;

  for i := 0 to FParShapes.Count-1 do begin
    ob := FParShapes.Objects[i] as TObject;
    if ob <> nil then ob.Free;
  end;
  FParShapes.Clear;
end;

function TParModel.GetStrings: TStrings;
var i: integer; sh: TParShape; st: string;
begin
  {make this check not to lose data on storing}
  if FParShapes.Count > 0 then FDesc.Clear;
  for i := 0 to FParShapes.Count-1 do begin
    sh := FParShapes.Objects[i] as TParShape;
    if sh <> nil then begin
      st := FParShapes.Strings[i] + ':';
      st := st + sh.Desc;
      FDesc.Add(st);
    end;
  end;
  result := FDesc;
end;

procedure TParModel.SetVars(Value: TStrings);
begin
  if Value <> nil then begin
    FVars.Assign(Value);
    Update; {no need do recreate the shapes}
  end;
end;

procedure TParModel.SetStrings(Value: TStrings);
var i: integer; ob: TObject;
begin
  if Value <> nil then begin
    {Will recreate the shapes in Update:}
    for i := 0 to FParShapes.Count-1 do begin
      ob := FParShapes.Objects[i] as TObject;
      if ob <> nil then ob.Free;
    end;
    FParShapes.Clear;
    FSelection.Clear;

    FDesc.Assign(Value);
    Update;
  end;
end;

procedure TParModel.RegError(const Msg: string; Sender: TParShape);
var st: string; idx: integer;
begin
  idx := FParShapes.IndexOfObject(Sender);
  if idx <> -1 then begin
    st := FParShapes.Strings[idx] + ': ' + Msg;
    FErrors.Add(st);
  end else if Msg <> '' then FErrors.Add(Msg);
end;

function TParModel.GetVal(const name: string): double;
var code: integer; st: string;
begin
  val(name, result, code);
  if code <> 0 then begin
    st := FVars.Values[name];
    if (st <> '') and (LowerCase(st) <> LowerCase(name))
      then result := GetVal(st)
    else
    if Assigned(FOnParse) then begin
      result := FOnParse(Self, name, code);
      if (code <> 0) then
        FErrors.Add(Format('Error %d parsing ''%s''', [code, name]));
    end
    else FErrors.Add('Undefined parameter '''+name+'''');
  end;
end;

procedure TParModel.Update;
var
  Form: TForm; Memo: TMemo;
  sh: TParShape; ob: TObject;
  i, ShapeIdx, TypeIdx: integer;
  ShapeName, ShapeType, ShapeDesc, st, msg: string;
begin
  FErrors.Clear;
  for i := 0 to FDesc.Count-1 do begin
    sh := nil;
    {  string format as folows:
       ShapeType.ShapeID: Field1=Value1; Field2=Value2;
    }
    st := FDesc.Strings[i]; if st = '' then continue;
    ShapeName := copy(st,1, pos(':', st)-1);
    ShapeDesc := copy(st,pos(':', st)+1, 255);

    ShapeIdx := FParShapes.IndexOf(LowerCase(ShapeName));
    if ShapeIdx <> -1 then begin
      ob := FParShapes.Objects[ShapeIdx];
      sh := TParShape(ob);
    end else begin
      ShapeType := copy(ShapeName,1, pos('.', ShapeName)-1);
      ShapeType := 'TPar' + ShapeType;
      TypeIdx := ParRegistry.IndexOf(LowerCase(ShapeType));
      if TypeIdx <> -1 then begin
        ob := ParRegistry.Objects[TypeIdx];
        sh := TParShape(ob);
        sh := sh.CreateInstance;
        FParShapes.AddObject(ShapeName, sh);
      end else begin
        msg := Format('Unregistered shape type: ''%s'' at line %d',
          [ShapeType, i+1]);
        FErrors.Add(msg);
      end; {TypeIdx = -1}
    end; {ShapeIdx = -1}
    if sh <> nil then with sh do begin
      Model := Self;
      Desc := ShapeDesc;
    end;
  end; {for i}
  if FErrors.Count > 0 then begin
    if not Assigned(FOnError) then begin {while csDesigning too}
      Form := TForm.Create(Application);
      Memo := TMemo.Create(Form);
      with Memo do begin
        Parent := Form;      WordWrap   := false;
        Align  := alClient;  ReadOnly   := true;
        Lines  := FErrors;   ScrollBars := ssBoth;
      end;
      Form.Caption := Format('Errors in model ''%s''', [Name]);
      with Form do begin
        Position := poScreenCenter;
        ShowModal;
      end;
      Form.Free;
    end else FOnError(Self);
  end; {FErrors.Count > 0}
end;

procedure TParModel.Draw(v: TParView);
var i: integer; sh: TParShape;
begin
  if v <> nil then begin
    for i := 0 to FParShapes.Count-1 do begin
      sh := FParShapes.Objects[i] as TParShape;
      if sh <> nil then with v.Canvas do begin
        Pen := sh.Pen;
        Brush := sh.Brush;
        Font.Color := sh.Pen.Color;
        if FSelection.IndexOfObject(sh) <> -1 then begin
          Pen := v.SelPen;
          Brush.Color := v.SelPen.Color;
          Font.Color := v.SelPen.Color;
          if Brush.Style = bsSolid then Brush.Style := bsClear;
        end;
        sh.Draw(v);
      end; {sh <> nil}
    end; {for i}
  end; {v <> nil}
end;

procedure TParModel.SelWindow(const r: TRect; v: TParView);
var i: integer; sh: TParShape; name: string;
begin
  if v <> nil then begin
    for i := 0 to FParShapes.Count-1 do begin
      name := FParShapes.Strings[i];
      sh := FParShapes.Objects[i] as TParShape;
      if sh <> nil then
        if sh.IsInside(r, v) and (FSelection.IndexOf(name) = -1) then
          FSelection.AddObject(name, sh);
    end;
  end;
end;

procedure TParModel.SelCross(const r: TRect; v: TParView);
var i: integer; sh: TParShape; name: string;
begin
  if v <> nil then begin
    for i := 0 to FParShapes.Count-1 do begin
      name := FParShapes.Strings[i];
      sh := FParShapes.Objects[i] as TParShape;
      if sh <> nil then
        if (sh.IsInside(r, v) or sh.IsCrossing(r, v))
          and (FSelection.IndexOf(name) = -1) then
            FSelection.AddObject(name, sh);
    end;
  end;
end;

function TParModel.SelSingle(const r: TRect; v: TParView): string;
var i: integer; sh: TParShape;
begin
  if v <> nil then begin {taking account for Z order}
    for i := FParShapes.Count-1 downto 0 do begin
      result := FParShapes.Strings[i];
      sh := FParShapes.Objects[i] as TParShape;
      if sh <> nil then
        if sh.IsInside(r, v) or sh.IsCrossing(r, v) then exit;
    end;
  end;
  result := '';
end;

procedure TParModel.GetExt(var minx, miny, maxx, maxy: double);
var i: integer; sh: TParShape; inx, iny, axx, axy: double;
begin
  minx :=  32767; miny :=  32767;
  maxx := -32767; maxy := -32767;
  for i := 0 to FParShapes.Count-1 do begin
    sh := FParShapes.Objects[i] as TParShape;
    if sh <> nil then begin
      sh.GetExt(inx, iny, axx, axy);
      if inx < minx then minx := inx;
      if iny < miny then miny := iny;
      if axx > maxx then maxx := axx;
      if axy > maxy then maxy := axy;
    end;
  end;
end;

procedure TParModel.SaveToFile(const FileName: string);
var Strings: TStrings;
begin
  Strings := TStringList.Create;
  try
    Strings.Add('[Parametric Graphic Model]');
    Strings.Add('Version=1.0');
    Strings.Add('[Vars]');
    Strings.AddStrings(FVars);
    Strings.Add('[Desc]');
    GetStrings; {generate compact}
    Strings.AddStrings(FDesc);
    Strings.SaveToFile(FileName);
  finally
    Strings.Free;
  end;
end;

procedure TParModel.LoadFromFile(const FileName: string);
var i: integer; Strings: TStrings; st: string;
begin
  Strings := TStringList.Create;
  try
    Strings.LoadFromFile(FileName);
    Clear;
    i := 0;
    while i < Strings.Count-1 do begin
      st := Strings.Strings[i];
      if st = '[Vars]' then begin
        while i < Strings.Count-1 do begin
          i := i+1;
          st := Strings.Strings[i];
          if st[1] = '[' then break;
          if (pos('=', st) <> 0) then FVars.Add(st);
        end;
      end;
      if st = '[Desc]' then begin
        while i < Strings.Count-1 do begin
          i := i+1;
          st := Strings.Strings[i];
          if st[1] = '[' then break;
          if (pos(':', st) <> 0) then FDesc.Add(st);
        end;
      end;
      i := i+1;
    end;
  finally
    Strings.Free;
    Update;
  end;
end;

{ TParView }

constructor TParView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FZoom := 1;
  FGridMajor := 10;
  FGridMinor := 1;
  Color := clWhite;
  FGridColor := clGray;
  Width := 200; Height := 100;
  FSelPen := TPen.Create;
  FSelPen.Style := psDot;
  FSelPen.Color := clLime;
end;

destructor TParView.Destroy;
begin
  FSelPen.Free;
  inherited Destroy;
end;

procedure TParView.SetSelPen(Value: TPen);
begin
  if Value <> nil then FSelPen.Assign(Value);
end;

procedure TParView.SetModel(Value: TParModel);
begin
  FParModel := Value;
  if FParModel <> nil then FParModel.Update;
  ZoomAll;
end;

procedure TParView.SetOrg(x, y: double);
begin
  if (x <> FOrgX) or (y <> FOrgY) then begin
    FOrgX := x; FOrgY := y;
    Paint;
  end;
end;

procedure TParView.SetOrgX(Value: double);
begin
  SetOrg(Value, FOrgY);
end;

procedure TParView.SetOrgY(Value: double);
begin
  SetOrg(FOrgX, Value);
end;

procedure TParView.SetZoom(Value: double);
begin
  if (Value <> FZoom) and (Value > 1e-17)  and (Value < 1e+6)
  then begin
    FZoom := Value;
    Paint;
  end;
end;

procedure TParView.SetGridMajor(Value: double);
begin
  if (Value <> FGridMajor) and (Value >= 0)  and (Value < 1e+7)
  then begin
    FGridMajor := Value;
    if FGridMajor < FGridMinor then FGridMinor := FGridMajor;
    Paint;
  end;
end;

procedure TParView.SetGridMinor(Value: double);
begin
  if (Value <> FGridMinor) and (Value >= 0)  and (Value < 1e+7)
  then begin
    FGridMinor := Value;
    if FGridMinor > FGridMajor then FGridMajor := FGridMinor;
    Paint;
  end;
end;

procedure TParView.SetGridColor(Value: TColor);
begin
  if (Value <> FGridColor) then begin
    FGridColor := Value;
    Paint;
  end;
end;

function TParView.GetViewX(dx: double): integer;
begin
  result := round((dx-FOrgX)*FZoom);
end;

function TParView.GetViewY(dy: double): integer;
begin
  result := Height - round((dy-FOrgY)*FZoom);
end;

function TParView.GetPaperX(x: integer): double;
begin
  result := x/FZoom + FOrgX;
end;

function TParView.GetPaperY(y: integer): double;
begin
  result := (Height - y)/FZoom + FOrgY;
end;

procedure TParView.SelWindow(const r: TRect);
begin
  if FParModel <> nil then FParModel.SelWindow(r, Self);
end;

procedure TParView.SelCross(const r: TRect);
begin
  if FParModel <> nil then FParModel.SelCross(r, Self);
end;

function TParView.SelSingle(const p: TPoint): string;
var r: TRect;
begin
  result := '';
  r.Left := p.X-2; r.Top := p.Y-2;
  r.Right := p.X+2; r.Bottom := p.Y+2;
  if FParModel <> nil then result := FParModel.SelSingle(r, Self);
end;

procedure TParView.SelPoint(const p: TPoint);
var r: TRect;
begin
  r.Left := p.X-2; r.Top := p.Y-2;
  r.Right := p.X+2; r.Bottom := p.Y+2;
  SelCross(r);
end;

procedure TParView.ZoomAll;
var dx, dy, minx, miny, maxx, maxy: double;
begin
  if FParModel <> nil then begin
    FParModel.GetExt(minx, miny, maxx, maxy);
    dx := maxx - minx;
    if (dx <= 0) then dx := 1 else dx := Width/dx;
    dy := maxy - miny;
    if (dy <= 0) then dy := 1 else dy := Height/dy;
    if dx < dy then FZoom := dx else FZoom := dy;
    FZoom := FZoom*0.9;
    FOrgX := minx;
    FOrgY := miny;
    FOrgX := FOrgX - ((Width/FZoom)  - maxx + minx)/2;
    FOrgY := FOrgY - ((Height/FZoom) - maxy + miny)/2;
    Paint;
  end;
end;

procedure TParView.DrawCS;
var cx, cy: integer;
begin
  cx := GetViewX(0);
  cy := GetViewY(0);
  with Canvas do begin
    Pen.Mode := pmCopy; Pen.Style := psSolid;
    Pen.Color := clRed; Pen.Width := 1;
    MoveTo(cx, cy);
    LineTo(cx+10, cy); LineTo(cx+5, cy+2);
    MoveTo(cx+10, cy); LineTo(cx+5, cy-2);
    MoveTo(cx, cy);
    LineTo(cx, cy-10); LineTo(cx-2, cy-5);
    MoveTo(cx, cy-10); LineTo(cx+2, cy-5);
  end;
end;

procedure TParView.DrawGrid;
var
  r: TRect;
  gleft, gtop: double;
  i, j, x, y, step, icount, jcount: integer;
begin
  r.Left := 0; r.Top := 0;
  r.Right := Width; r.Bottom := Height;

  with Canvas do begin
    Brush.Style := bsSolid; Brush.Color := Color;
    FillRect(r);
    Pen.Color := FGridColor; Pen.Width := 1;
    Pen.Style := psSolid; Pen.Mode := pmCopy;
  end;

  {Drawing Minor Grid}
  step := round(FGridMinor*FZoom);
  if (step) > 2 then begin
    icount := round(Width  / FZoom / FGridMinor);
    jcount := round(Height / FZoom / FGridMinor);
    gleft  := round(GetPaperX(0)/FGridMinor)*FGridMinor;
    gtop   := round(GetPaperY(Height)/FGridMinor)*FGridMinor;
    for i := 0 to icount do with Canvas do
      for j := 0 to jcount do with Canvas do begin
        x := GetViewX(gleft+i*FGridMinor);
        y := GetViewY(gtop +j*FGridMinor);
        Pixels[x, y] := FGridColor;
      end;
  end; {Minor Grid}

  {Drawing Major Grid}
  step := round(FGridMajor*FZoom);
  if (step) > 2 then begin
    {drawing vertical grid}
    icount := round(Width  / FZoom / FGridMinor);
    gleft := round(GetPaperX(0)/FGridMajor)*FGridMajor;
    for i := 0 to icount do with Canvas do begin
      x := GetViewX(gleft+i*FGridMajor);
      MoveTo(x, 0);
      LineTo(x, Height);
    end;
    {drawing horizontal grid}
    jcount := round(Height / FZoom / FGridMinor);
    gtop  := round(GetPaperY(Height)/FGridMajor)*FGridMajor;
    for j := 0 to jcount do with Canvas do begin
      y := GetViewY(gtop+j*FGridMajor);
      MoveTo(0, y);
      LineTo(Width, y);
    end;
  end; {Major Grid}

end;

procedure TParView.Paint;
begin
  Canvas.Font := Font;
  DrawFast;
end;

procedure TParView.Draw;
begin
  DrawGrid;
  if FParModel <> nil then FParModel.Draw(Self);
  DrawCS;
end;

procedure TParView.DrawFast;
var destdc, memdc: HDC; bm, sbm: HBITMAP;
begin
  destdc := Canvas.Handle;
  memdc := CreateCompatibleDC(destdc);
  bm := CreateCompatibleBitmap(destdc, Width, Height);
  sbm := SelectObject(memdc, bm);
  Canvas.Handle := memdc;
  Draw;
  BitBlt(destdc,0,0,  Width,Height, memdc,0,0, SRCCOPY);
  SelectObject(memdc, sbm);
  DeleteObject(bm);
  Canvas.Handle := destdc;
  DeleteDC(memdc);
end;

end.
