unit Input;

{  ******
   *
   * Module:    Input
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    The TInput class describes a single logical game input (such as BART_FIRE),
   *    and can notify all connected objects when the input occurs.  This is
   *    usually triggered by an input object such as TRocksKeyboard.
   *
   ****** }

interface
uses Message, Classes;

type TInput = class(TObject)
    public
        { Class constructor and destructor. }
        constructor Create(iInputCode: Integer);
        destructor Destroy;

        { Method to return this input's unique identifier. }
        function iGetInputCode: Integer;

        { Methods to get and set the input's selection state. }
        procedure SetSelectionState(bSelected: Boolean);
        function bIsSelected: Boolean;

        { Methods to managr the notification lists. }
        procedure NotifyOnChange(msgObject: TMessageAware);
        procedure RemoveNotification(msgObject: TMessageAware);

    private

        m_iInputCode: Integer;  { Unique input identifier. }
        m_lstNotify: TList;     { Of objects to be notified when state changes. }
        m_bSelected: Boolean;   { Current selection state of the input. }
end;

implementation

constructor TInput.Create(iInputCode: Integer);
begin
    { Perform the default construction. }
    inherited Create;

    { Initialize the object. }
    m_iInputCode := iInputCode;
    m_lstNotify := TList.Create;
    m_bSelected := False;
end;

destructor TInput.Destroy;
begin
    { Free our notification list. }
    m_lstNotify.Free;

    { Perform default cleanup. }
    inherited Destroy;
end;

function TInput.iGetInputCode: Integer;
begin
    Result := m_iInputCode;
end;

procedure TInput.SetSelectionState(bSelected: Boolean);
var
    iIndex: Integer;
    msgObject: TMessageAware;
begin
    { If the selection state has not changed then don't do anything else. }
    if (m_bSelected = bSelected) then
        Exit;

    { Store the new selection state. }
    m_bSelected := bSelected;

    { Send out notifications of the state change. }
    for iIndex := 0 to (m_lstNotify.Count - 1) do
    begin
        msgObject := m_lstNotify.Items[iIndex];
        msgObject.ProcessMessage(INPUT_DEVICE, m_iInputCode, m_bSelected, 0, 0);
    end;
end;

function TInput.bIsSelected: Boolean;
begin
    Result := m_bSelected;
end;

procedure TInput.NotifyOnChange(msgObject: TMessageAware);
begin
    m_lstNotify.Add(msgObject);
end;

procedure TInput.RemoveNotification(msgObject: TMessageAware);
begin
    m_lstNotify.Remove(msgObject);
end;

end.
