/** \file CON_3D.h
    3D Rendering interfaces: Screen3D, Camera, Mesh, Light

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/

class Screen3D;
class Camera;
class Light;

#ifndef H_CON_3D
#define H_CON_3D

#include <CON_Mesh.h>
#include <CON_Resource.h>

/**
3D Rendering control interface.  Must be created to enable 3D Rendering.
*/
class Screen3D : public SystemInterface
{
public:
  /** Return the D3D Device pointer for creating special effects.
      Cast the return value to LPDIRECT3DDEVICE3
  */
  virtual void* getD3DDevice() = 0;
  /** Returns a non zero value if an error has occured. */
  virtual long status() = 0;
  /** Must be called before any 3D Rendering */
  virtual long beginScene() = 0;
  /** Renders an object */
  virtual long renderObject(Mesh* O) = 0;
  /** Must be called after all rendering is done, and before flipping. */
  virtual long endScene() = 0;
  /** Can be called to override the default ambient lighting. */
  virtual long setAmbientLight(unsigned long Color) = 0;
  /** Set the 3D Scene fog settings. */
  virtual long setFog(ulong Color=0xFFFFFFFF, float Start=0.5f, float End=1.0f, int Enable=1) = 0;
  /** Flushes video memory textures.  May be used when scene changes completely. */
  virtual long flushTextures() = 0;
};

/** Generic light */
class Light : public Interface, public Transformable
{
public:
  /** Returns non-zero if an error has occurred. */
  virtual long status() = 0;

  /** Update the light state, after changing some settings. */
  virtual long update() = 0;

  /** Set the light's color.  
      A call to update() must be made to make the changes active. */
  virtual long setColor(float Red, float Green, float Blue) = 0;

  /** Set the light's range.  Valid for spot/point lights.
      A negative range indicates maximum range.
      A call to update() must be made to make the changes active. */
  virtual long setRange(float Range) = 0;

  /** Set the spot light's angles.  
      Theta is the inner cone, and Phi is the outer cone.
      They must conform to:  0 <= Theta <= Phi <= PI
      A call to update() must be made to make the changes active. */
  virtual long setAngles(float Theta, float Phi) = 0;
};

/**
A camera that gives a view point of the 3D world.
Only one camera can be active at a time, due to a tie with
a virtual microphone, giving the 3D sounds of the world.
*/
class Camera : public Interface
{
public:
  /** Clear the view screen, should be called before rendering a scene.
      Color default of -1 uses the background color as set by setBackgroundColor. */
  virtual long clear(long Color=-1, int ClearView=1, int ClearZBuf=1) = 0;

  /** Called prior to rendering for internal special effects. */
  virtual long preRender() = 0;

  /** Called after rendering for internal special effects. */
  virtual long postRender() = 0;

  /** Set the viewable area position and dimensions in 2D screen coordinates */
  virtual long setViewable(int x, int y, int width, int height) = 0;

  /** Set the position of the camera in the 3D World */
  virtual long setPosition(float x, float y, float z) = 0;

  /** Set the direction that the camera is pointed to. */
  virtual long setDirection(float tox, float toy, float toz,
                            float upx, float upy, float upz) = 0;

  /** Set the camera velocity.  This currently is used for audio doppler effects */
  virtual long setVelocity(float v) = 0;

  /** Add stars to a space scene, to be rendered in the preRender stage. */
  virtual long setStars(int Number=1000) = 0;

  /** Add a light to the scene.  Light objects are not maintained by
      the camera, and should be allocated and freed by the application */
  virtual long addLight(Light* L) = 0;

  /** remove a light from the scene. */
  virtual long removeLight(Light* L) = 0;

  /** Set the camera to show what the object 'sees' */
  virtual long follow(const Transformable& t) = 0;

  /** Activate the camera. */
  virtual long activate() = 0;

  /** Returns non zero if an error has occured. */
  virtual long status() = 0;

  /** Gets the camera's position in the 3D World  */
  virtual Vector3D getPosition() = 0;

  /** Gets the position of an object relative to the camera */
  virtual Vector3D getObjectPosition(Mesh* O) = 0;

  /** Gets the color used when the frame buffer is cleared. */
  virtual TColor   getBackgroundColor() = 0;

  /** Sets the color used when the frame buffer is cleared. */
  virtual void     setBackgroundColor(TColor Color) = 0;
};


////////////////////////////////////////////////////////////
// 3D Functions
////////////////////////////////////////////////////////////
#ifndef FUNC_3D
#define FUNC_3D

/** Color type, containing 4 floating point components (R,G,B,A)
    ranging from 0.0f to 1.0f */
class fColor 
{ 
public: 
  fColor(float r_=0.0f, float g_=0.0f, float b_=0.0f, float a_=0.0f) :
    r(r_),g(g_),b(b_),a(b_) {}
  float r,g,b,a; 
};

/** \relates Screen3D
    Initialization of 3D subsystem.  
    Must be performed prior to using any other 3D functions.
    Will be released automatically. 
*/
DLLExport Screen3D*  initScreen3D();

/** \relates Camera
    Create a new camera object.
*/
DLLExport Camera*    newCamera();


///////////  Lights  ///////////

/** \relates Light
    Create a new light, which is a parallel point (like Sun Light) with a specified color.
    This is similar to a point light, but requires less calculations by assuming great distance.
    Coordinates specify the initial position for the light.
*/
DLLExport Light*     newSunLight(float x, float y, float z, float Red=0.7f, float Green=0.9f, float Blue=0.7f);

/** \relates Light
    Create a new light, which is a parallel point (like Sun Light) with a specified color.
    This is similar to a point light, but requires less calculations by assuming great distance.
    Coordinates specify the initial position for the light.
*/
DLLExport Light*     newSunLight(const Vector3D& Position, 
                                 const fColor& Color=fColor(0.7f,0.9f,0.7f));

/** \relates Light
    Create a new light, which is a spot.  It has direction, range and angles
    for cones of effect.
*/
DLLExport Light*     newSpotLight(const Vector3D& Position, const Vector3D& Direction, 
                                  const fColor& Color=fColor(1.0f,1.0f,1.0f),
                                  const float Range=-1,
                                  const float Theta=PI/4, const float Phi=PI/2);

/** \relates Light
    Create a new light, which is a point source of light.  It radiates light
    equally in all directions but may have a limited range. (-1 => infinite range)
*/
DLLExport Light*     newPointLight(const Vector3D& Position,
                                   const fColor& Color=fColor(1.0f,1.0f,1.0f),
                                   const float Range=-1);

/** \relates Light
    Create a new light, which radiates in a specific direction always.
    It doesn't have a specific source position and has unlimited range.
*/
DLLExport Light*     newDirectionLight(const Vector3D& Direction,
                                       const fColor& Color=fColor(1.0f,1.0f,1.0f));


/** \relates Texture
    Create a new texture.
*/
DLLExport Texture*   newTexture(const int Width, const int Height);

/** \relates TextureCache
    Initializes an internal texture cache.  
    If RS is NULL, actual files will be used to load textures. 
    Setting Destroy to non zero will release the resource stream
    when the texture cache is released.
*/
DLLExport void          initTextureCache(ResourceStream* RS=NULL, int Destroy=0);

/** \relates TextureCache
    Creates a user TextureCache.  
    It must be release()d when it is no longer needed.
    Setting Destroy to non zero will release the resource stream
    when the texture cache is released.
*/
DLLExport TextureCache* newTextureCache(ResourceStream* RS=NULL, int Destroy=0);

/** \relates Mesh
    Loads the object from a .X file. name specifies file name */
//DLLExport long loadXFile(const char* Name, Mesh*& O, TextureCache* TC=NULL, float Scale=1.0f);



#endif // FUNC_3D


#endif // H_CON_3D