/* AddressBook - AddressBookFrame 
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package addressbook;

import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

import javax.swing.*;
import javax.swing.border.BevelBorder;

import rogatkin.*;

public class AddressBookFrame extends JFrame implements ActionListener, Program {
    public static final String PROGRAMNAME = "R-AddressBook";
    public static final String VERSION = "version 1.0 beta";

    final static int MENU = 1;
    final static int TOOL = 2;
    final static int STATUS = 4;
    final static int SEARCH = 8;
    final static int FOLDER = 16;
    int view = MENU+TOOL+STATUS+SEARCH+FOLDER;
    boolean standalone;

    BaseController controller;
    public AddressBookFrame(BaseController controller) {
	super(PROGRAMNAME);
	this.controller = controller;
	if (this.controller == null)
	    this.controller = new BaseController(this) {
		{
		    standalone=true;
		}
	    Image mainicon;
	    public Image getMainIcon() {
		if (mainicon == null)
		    mainicon = getResourceIcon(AddressBookResources.IMG_PROGRAM).getImage();
		return mainicon;
	    }
	};
        this.controller.load();
	setIconImage(this.controller.getMainIcon());
	Container c = getContentPane();
	if((view&MENU) != 0)
	    setJMenuBar(createMenu());
	if((view&TOOL) != 0)
	    c.add(createToolBar(JToolBar.HORIZONTAL), "North");
	c.add(createBook(), "Center");
	if((view&STATUS) != 0)
	    c.add(createStatusBar(), "South");
        addWindowListener( new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                actionPerformed(
                    new ActionEvent(this, 0, standalone?AddressBookResources.MENU_EXIT:AddressBookResources.MENU_CLOSE));
            }
        });
	pack();
	setVisible(true);
    }

    public String getName() {
	return PROGRAMNAME;
    }

    public String getVersion() {
	return VERSION;
    }

    public void save() {
	Serializer s = controller.getSerializer();
    }

    public void load() {
	Serializer s = controller.getSerializer();
    }

    JMenuBar createMenu() {
	JMenuBar menubar = new JMenuBar();
	JMenu menu, menu2;
	JMenuItem item;
	// file
	menubar.add(menu = new JMenu(AddressBookResources.MENU_FILE));
	createNewMenu(menu, this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_PROPERTIES));
        item.addActionListener(this);
	menu.add(item = new JMenuItem(AddressBookResources.MENU_DELETE));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(menu2 = new JMenu(AddressBookResources.MENU_IMPORT));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_ADDRBOOK));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_BUSCARD));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_OTHERADDRBOOK));
	menu.add(menu2 = new JMenu(AddressBookResources.MENU_EXPORT));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_ADDRBOOK));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_BUSCARD));
	menu2.add(item = new JMenuItem(AddressBookResources.MENU_OTHERADDRBOOK));
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_PRINT));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(standalone?AddressBookResources.MENU_EXIT:AddressBookResources.MENU_CLOSE));
        item.addActionListener(this);
	// edit
	menubar.add(menu = new JMenu(AddressBookResources.MENU_EDIT));
	menu.add(item = new JMenuItem(AddressBookResources.MENU_COPY));
	item.setAccelerator(KeyStroke.getKeyStroke("control C"));
        item.addActionListener(this);
	menu.add(item = new JMenuItem(AddressBookResources.MENU_PASTE));
	item.setAccelerator(KeyStroke.getKeyStroke("control V"));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_SELECTALL));
	item.setAccelerator(KeyStroke.getKeyStroke("control A"));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_PROFILE));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_FINDPEOPLE));
	item.setAccelerator(KeyStroke.getKeyStroke("control F"));
        item.addActionListener(this);
	// view
	menubar.add(menu = new JMenu(AddressBookResources.MENU_VIEW));
	menu.add(item = new JCheckBoxMenuItem(AddressBookResources.MENU_TOOLBAR));
        item.addActionListener(this);
	menu.add(item = new JCheckBoxMenuItem(AddressBookResources.MENU_STATUSBAR));
        item.addActionListener(this);
	menu.add(item = new JCheckBoxMenuItem(AddressBookResources.MENU_FOLDERGROUP));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(menu2 = new JMenu(AddressBookResources.MENU_SORTBY));
	ButtonGroup bg = new ButtonGroup();
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_NAME)));
	bg.add(item);
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_EMAILADDR)));
	bg.add(item);
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_BUSPHONE)));
	bg.add(item);
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_HOMEPHONE)));
	bg.add(item);
	menu2.addSeparator();
	bg = new ButtonGroup();
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_FIRSTNAME)));
	bg.add(item);
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_LASTNAME)));
	bg.add(item);
	menu2.addSeparator();
	bg = new ButtonGroup();
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_ASC)));
	bg.add(item);
	menu2.add(item = new JRadioButtonMenuItemEx(new RadioAction(AddressBookResources.MENU_DESC)));
	bg.add(item);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_REFRESH));
	item.setAccelerator(KeyStroke.getKeyStroke("F5"));
        item.addActionListener(this);
        // tools
	menubar.add(menu = new JMenu(AddressBookResources.MENU_TOOLS));
	menu.add(item = new JMenuItem(AddressBookResources.MENU_ACCOUNTS));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_OPTIONS));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(menu2 = new JMenu(AddressBookResources.MENU_ACTION));
	createActionMenu(menu2, this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_SYNCHRONIZE));
        item.addActionListener(this);
        // help
	menubar.add(menu = new JMenu(AddressBookResources.MENU_HELP));
	menu.add(item = new JMenuItem(AddressBookResources.MENU_CONTENTS));
	item.setAccelerator(KeyStroke.getKeyStroke("F1"));
        item.addActionListener(this);
	menu.addSeparator();
	menu.add(item = new JMenuItem(AddressBookResources.MENU_ABOUT+AddressBookFrame.PROGRAMNAME));
        item.addActionListener(this);
	return menubar;
    }

    JComponent createNewMenu(JComponent wrapper, ActionListener listener) {
	JMenuItem item;
	wrapper.add(item = new JMenuItem(AddressBookResources.MENU_NEWCONTACT));
	item.setAccelerator(KeyStroke.getKeyStroke("control N"));
        item.addActionListener(listener);
	wrapper.add(item = new JMenuItem(AddressBookResources.MENU_NEWBOOKMARK));
	item.setAccelerator(KeyStroke.getKeyStroke("control B"));
        item.addActionListener(listener);
	wrapper.add(item = new JMenuItem(AddressBookResources.MENU_NEWCOOKIE));
	item.setAccelerator(KeyStroke.getKeyStroke("control O"));
        item.addActionListener(listener);
	wrapper.add(item = new JMenuItem(AddressBookResources.MENU_NEWGROUP));
	item.setAccelerator(KeyStroke.getKeyStroke("control G"));
        item.addActionListener(listener);
	wrapper.add(item = new JMenuItem(AddressBookResources.MENU_NEWFOLDER));
	item.setAccelerator(KeyStroke.getKeyStroke("control R"));
        item.addActionListener(listener);
	return wrapper;
    }

    JComponent createActionMenu(JComponent wrapper, ActionListener listener) {
	JMenuItem item;
        for (int i=0; i<AddressBookResources.MENUS_ACTION.length; i++) {
	    wrapper.add(item = new JMenuItem(AddressBookResources.MENUS_ACTION[i]));
            item.addActionListener(listener);
        }
	return wrapper;
    }

    JToolBar createToolBar(int orientation) {
	JToolBar toolbar = new JToolBar(orientation);
	    JButton btn;
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_NEW));
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_PROPERTIES));
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_DELETE));
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_FINDPEOPLE));
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_PRINT));
	    btn = toolbar.add(new ToolAction(AddressBookResources.IMG_ACTION));
	    return toolbar;
    }

    JComponent createBook() {
	JTable table;
	String[][] data = new String[][] {
	    {"Ivanov, Peter Alex", "Ivanov.P.A@kgb.gov.ru", "095-8948920", "09763438334"},
	    {"Gregoriev, Ivan Peter", "Greg@russianword.com.ru", "64723898129", "47299283746"}};
	JPanel abp = new JPanel();
	abp.setLayout(new BorderLayout());
	JPanel sp = new JPanel();
	sp.setLayout(new FlowLayout());
	sp.add(new JLabel(AddressBookResources.LABEL_TNAMEORLIST));
	sp.add(new JTextField(20));
	abp.add(sp, "North");
	table = new JTable(data, AddressBookResources.HEADER_ADDRBOOK);
	abp.add(new JScrollPane(table), "Center");
	if((view&FOLDER) != 0) {
	    JTree t = new JTree();
	    return new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
                true, 
		new JScrollPane(t),
		new JScrollPane(abp));
	} else {
	    return abp;
	}
    }

    public static JPanel createButtonPanel(ActionListener al) {
        JButton btn;
        JPanel result = new JPanel();
        result.setLayout(new FlowLayout(FlowLayout.RIGHT));
        result.add(btn = new JButton(AddressBookResources.CMD_OK));
        btn.addActionListener(al);
        result.add(btn = new JButton(AddressBookResources.CMD_APPLY));
        btn.addActionListener(al);
        result.add(btn = new JButton(AddressBookResources.CMD_CANCEL));
        btn.addActionListener(al);
        return result;
    }

    JPanel createStatusBar() {
	JPanel p = new JPanel();
	p.add(new JLabel("Status bar"));
	p.setBorder(new BevelBorder(BevelBorder.LOWERED));
	return p;
    }
    
    public void actionPerformed(ActionEvent a) {
        String cmd = a.getActionCommand();
        if (cmd.equals(AddressBookResources.MENU_FINDPEOPLE)) {
        } else if (cmd.equals(AddressBookResources.MENU_NEWCONTACT)) {
	    new NewContactFrame();
        } else if (cmd.equals(AddressBookResources.MENU_OPTIONS)) {
	    new OptionsFrame(controller);
	} else if (cmd.equals(AddressBookResources.MENU_EXIT)) {
	    dispose();
	    System.exit(0);
	} else if (cmd.equals(AddressBookResources.MENU_CLOSE)) {
	    dispose();
        } else if (cmd.equals(AddressBookResources.MENU_ABOUT)) {
        } else if (cmd.equals(AddressBookResources.MENU_CONTENTS)) {
            controller.showUrl(AddressBookResources.URL_HELP);
	}
    }

    class ToolAction extends AbstractAction {
	Icon im;
	ToolAction(String name) {
	    super(name);
	    im = controller.getResourceIcon(name+AddressBookResources.EXT_GIF);
	}

	public Object getValue(String key) {
	    if (key == SMALL_ICON) // we can use == instead of equals here
		return im;
	    return super.getValue(key);
	}
	
	public void actionPerformed(ActionEvent a) {
	    if (AddressBookResources.IMG_NEW.equals(getValue(NAME))) {
		Rectangle r = ((Component)a.getSource()).getBounds();
		Point p = new Point(0, r.height);
		SwingUtilities.convertPointToScreen(p, (Component)a.getSource());
		((JPopupMenu)createNewMenu(new JPopupMenu(), AddressBookFrame.this))
		.show(AddressBookFrame.this, p.x, p.y);

	    } else if (AddressBookResources.IMG_ACTION.equals(getValue(NAME))) {
		Rectangle r = ((Component)a.getSource()).getBounds();
		Point p = new Point(0, r.height);
		SwingUtilities.convertPointToScreen(p, (Component)a.getSource());
		((JPopupMenu)createActionMenu(new JPopupMenu(), AddressBookFrame.this))
		.show(AddressBookFrame.this, p.x, p.y);
	    } else
		AddressBookFrame.this.actionPerformed(a);
	}
    }

    class RadioAction extends AbstractAction {
	RadioAction(String name) {
	    super(name);
	}
	
	public void actionPerformed(ActionEvent a) {
	    
	    AddressBookFrame.this.actionPerformed(a);
	}
    }

    class JRadioButtonMenuItemEx extends JRadioButtonMenuItem {
	Action action;
	PropertyChangeListener actionPropertyChangeListener;
	
	JRadioButtonMenuItemEx(Action action) {
	    setAction(action);
	}

	public Action getAction() {
	    return action;
	}

	public void setAction(Action new_action) {
	    Action prev_action = getAction();
	    if(action == null || !action.equals(new_action)) {
	    	action = new_action;
		if(prev_action != null) {
		    removeActionListener(prev_action);
		    prev_action.removePropertyChangeListener(actionPropertyChangeListener);
		    actionPropertyChangeListener = null;
		}
		configurePropertiesFromAction(action);
		if(action != null) {
		    addActionListener(action);
		    actionPropertyChangeListener = createActionPropertyChangeListener(action);
		    action.addPropertyChangeListener(actionPropertyChangeListener);
		}
		firePropertyChange("action", prev_action, action);
		revalidate();
		repaint();
	    }
	}

	protected void configurePropertiesFromAction(Action action) {
	    setEnabled(action.isEnabled());
	    setText((String)action.getValue(Action.NAME));
	}

	protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
	    return new PropertyChangeListener() {
		public void propertyChange(PropertyChangeEvent evt) {
		}
	    };
	}
    }

    static public void main(String[] args) {
	new AddressBookFrame(null);
    }
}