/* Quat - A 3D fractal generation program */ 
/* Copyright (C) 1997,98 Dirk Meyer */ 
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */ 
/* mail:  Dirk Meyer */ 
/*        Marbacher Weg 29 */ 
/*        D-71334 Waiblingen */ 
/*        Germany */ 
/* */ 
/* This program is free software; you can redistribute it and/or */ 
/* modify it under the terms of the GNU General Public License */ 
/* as published by the Free Software Foundation; either version 2 */ 
/* of the License, or (at your option) any later version. */ 
/* */ 
/* This program is distributed in the hope that it will be useful, */ 
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */ 
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */ 
/* GNU General Public License for more details. */ 
/* */ 
/* You should have received a copy of the GNU General Public License */ 
/* along with this program; if not, write to the Free Software */ 
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */ 
 
#include <math.h>
#include <time.h>
#include <qpainter.h>
//#include "JuliaPreview.h"
#include "JuliaPreview.moc"
#include "common.h"
#include "qmath.h"
#include "main.h"

//#include <stdio.h>

extern VER_ReturnVideoInfo ReturnVideoInfo;
extern VER_SetColors SetColors;
extern VER_Initialize Initialize;
extern VER_Done Done;
extern VER_update_bitmap update_bitmap;
extern VER_getline getline;
extern VER_putline putline;
extern VER_check_event check_event;
extern VER_Change_Name Change_Name;
extern VER_Debug Debug;
extern VER_eol eol;

extern time_t calc_time;
extern time_t old_time;

extern QCmds *cmds;

JuliaPreview *p_this;

JuliaPreview::JuliaPreview(QWidget *parent, const char *name) : QWidget(parent, name)
{
   len_x = 80; len_y = 60; julia_x = -2; julia_y = -1.5;
   julia_lx = 4; julia_ly = 3;
   Pixmap = new QPixmap((int)len_x, (int)len_y, -1);
   CHECK_PTR(Pixmap);
   p_this = this;
}

void JuliaPreview::Coo2C(int x, int y, double *cx, double *cy)
{
   *cx = julia_x + (double)x/len_x * julia_lx;
   *cy = julia_y + (double)y/len_y * julia_ly;
}

void JuliaPreview::setCx(QString s)
{
   cx = s.toDouble(NULL);
   CheckUpdate();
}

void JuliaPreview::setCy(QString s)
{
   cy = s.toDouble(NULL);
   CheckUpdate();
}

void JuliaPreview::setCk(QString s)
{
   ck = s.toDouble(NULL);
   CheckUpdate();
}

void JuliaPreview::setCl(QString s)
{
   cl = s.toDouble(NULL);
   CheckUpdate();
}

void JuliaPreview::setBailout(QString s)
{
   Bailout = s.toDouble(NULL);
   Bailout *= Bailout;
   CheckUpdate();
}

void JuliaPreview::setLvalue(QString s)
{
   LValue = s.toDouble(NULL);
   CheckUpdate();
}

void JuliaPreview::setMaxiter(QString s)
{
   Maxiter = s.toInt(NULL);
   CheckUpdate();
}

void JuliaPreview::setFormula(int i)
{
   Formula = i;
   CheckUpdate();
}

void JuliaPreview::setKvalue(int i)
{
   KValue = (double)i/100;
   CheckUpdate();
}

void JuliaPreview::paintEvent(QPaintEvent *QP)
{
   QPainter painter(this);
   painter.setClipRect(QP->rect());
   painter.drawPixmap(0, 0, *Pixmap, 0, 0, -1, -1);
   painter.setPen(QColor(255,0,0));
   if (!updated) painter.drawRect(0, 0, (int)len_x, (int)len_y);
   painter.end();
   painter.flush();
}

int JuliaPreview::CalcJPixel(int x, int y) 
{ 
   double tmp, p[4], q[4];  
   point x1, x2, x3, x4, x5, c; 
   point one = { 1, 0, 0, 0 }; 
   int i, it; 
    
   if (Formula != 0 && Formula != 1) return(0); 
   Coo2C(x, y, &p[0], &p[1]);
   p[2] = KValue; p[3] = LValue;   
   it = 0; 
   if (Formula == 0) forall(i,4) q[i] = p[i]*p[i]; 
   else if (Formula == 1)  
   { 
      forall(i,4) x1[i] = p[i]; 
      c[0] = cx; c[1] = cy; c[2] = ck; c[3] = cl; 
      forall(i,4) q[i] = p[i]*p[i]; 
   } 
   while (it<Maxiter && q[0]+q[1]+q[2]+q[3]<Bailout) 
   { 
      if (Formula == 0) 
      { 
         tmp = p[0]*p[1]; p[1] = tmp + tmp - cy; 
         tmp = p[0]*p[2]; p[2] = tmp + tmp - ck; 
         tmp = p[0]*p[3]; p[3] = tmp + tmp - cl; 
         p[0] = q[0] - q[1] - q[2] - q[3] - cx; 
         forall(i,4) q[i] = p[i]*p[i]; 
      } 
      else if (Formula == 1) 
      { 
         forall(i,4) x3[i] = x1[i]; 
         q_sub(x2, one, x1);      /* 1 - x[n] */ 
         q_mul(x4, c, x3);        /* c*x[n] */ 
         q_mul(x5, x4, x2);       /* c*x[n]*(1-x[n]) */ 
         forall(i,4) x1[i] = x5[i]; 
         forall(i,4) q[i] = x1[i]*x1[i]; 
      } 
      it++; 
   } 
   if (it<Maxiter) return(0); else return(1); 
} 

void JuliaPreview::CalcImage2D()
{
   int x, y;
   QPainter paint;
   QWidget *w;

   w = topLevelWidget();
   w->setCursor(waitCursor);
   Pixmap->fill(QColor(255,255,255));
   paint.begin(Pixmap);
   for (y=0; y<(int)len_y; y++)
      for (x=0; x<(int)len_x; x++)
         if (CalcJPixel(x, y)) paint.drawPoint(x, y);
   pic_cx = cx; pic_cy = cy;
   pic_ck = ck; pic_cl = cl; pic_Maxiter = Maxiter; pic_Bailout = Bailout;
   pic_LValue = LValue; pic_KValue = KValue;
   pic_Formula = Formula;
   w->setCursor(arrowCursor);
   updated = TRUE;
   is3D = FALSE;
   paint.end();
   paint.flush();
   repaint();
}

int JuliaPreview::QT_putline2(long x1, long x2, long xres, int y, unsigned char *Buf, int whichbuf)
{
   uint *p;
   int i;

   p = (uint *)p_this->Image->scanLine(y);
   for (i=x1; i<=x2; i++) 
      *(p+i) = qRgb(Buf[i*3], Buf[i*3+1], Buf[i*3+2]);
   return(0);
}

int JuliaPreview::QT_update_bitmap2(long x1, long x2, long xres, int y, unsigned char *Buf, int which)
{
   long i;
   QPainter painter;

   painter.begin(p_this->Pixmap);
   for (i=x1; i<=x2; i++)
   {
      painter.setPen(QColor(Buf[3*i], Buf[3*i+1], Buf[3*i+2]));
      painter.drawPoint(i, y);
   }
   painter.end();
   return(0);
}

void JuliaPreview::QT_eol2(int line)
{
   p_this->repaint(FALSE);
}

void JuliaPreview::CalcImage3D()
{
   QWidget *w;
   int xs = 0, ys = 0;
   char Error[stringlength];

   Image = new QImage(80, 60, 32, 0, QImage::IgnoreEndian);
   ReturnVideoInfo = (VER_ReturnVideoInfo)QCmds::QT_ReturnVideoInfo;
   SetColors = (VER_SetColors)QCmds::QT_SetColors;
   Initialize = DUMMY_Initialize;
   Done = DUMMY_Done;
   update_bitmap = (VER_update_bitmap)QT_update_bitmap2;
   getline = DUMMY_getline;
   putline = (VER_putline)QT_putline2;
   check_event = (VER_check_event)QCmds::QT_check_event;
   Change_Name = DUMMY_Change_Name;
   Debug = (VER_Debug)QCmds::QT_Debug;
   eol = (VER_eol)QT_eol2;
   SetDefaults(&cmds->_f, &cmds->_v, &cmds->_r, cmds->_col, cmds->_cut);
   cmds->_v.LXR = 4;
   cmds->_r.cols[0].col1[0] = 1;
   cmds->_r.cols[0].col1[1] = 1;
   cmds->_r.cols[0].col1[2] = 0;
   if (ownView)
   {
      cmds->_v = cmds->view;
      strncpy(cmds->_col, cmds->colscheme, 250);
      memcpy(cmds->_cut, cmds->cutbuf, sizeof(double)*140);
      cmds->_r = cmds->realpal;
   }
   cmds->_f.bailout = Bailout;
   cmds->_f.maxiter = Maxiter;
   cmds->_f.lvalue = LValue;
   cmds->_f.formula = Formula;
   cmds->_v.interocular = 0;
   cmds->_f.c[0] = cx; cmds->_f.c[1] = cy; cmds->_f.c[2] = ck;
   cmds->_f.c[3] = cl;
   cmds->_v.xres = 80; cmds->_v.yres = 60; cmds->_v.zres = 60;
   cmds->_v.antialiasing = 1;

   w = topLevelWidget();
   w->setCursor(waitCursor);
//   Pixmap->fill(QColor(255,255,255));

   old_time = calc_time;
   CreateImage(Error, &xs, &ys, &(cmds->_f), &(cmds->_v), &(cmds->_r),
      cmds->_col, cmds->_cut, 40, 0);
   calc_time = old_time;
   Pixmap->convertFromImage(*Image, 0);   

   is3D = TRUE;
   pic_ownView = ownView;
   pic_cx = cx; pic_cy = cy;
   pic_ck = ck; pic_cl = cl; pic_Maxiter = Maxiter; pic_Bailout = Bailout;
   pic_LValue = LValue; pic_KValue = KValue;
   pic_Formula = Formula;
   w->setCursor(arrowCursor);
   updated = TRUE;
   repaint();
   delete Image;
}

void JuliaPreview::setOwnView(bool b)
{
   ownView = b;
}

void JuliaPreview::CheckUpdate()
{
   bool old;

   old = updated;
   if (pic_cx != cx || pic_cy != cy || 
      pic_ck != ck || pic_cl != cl || pic_Maxiter != Maxiter
      || pic_Bailout != Bailout || pic_Formula != Formula
      || pic_LValue != LValue || pic_KValue != KValue
      || (is3D && ownView != pic_ownView))
   updated = FALSE; else updated = TRUE;
   if (old != updated) repaint();
}

