//	<><><><><><><><><><><><><>  Plot3DP.h  <><><><><><><><><><><><><><> 
//
// -----------------------------------------------------------------
//
// Plot3DP is an abstract C++ class used to
// plot a 3D view of a function defined parametrically
//
// Highlights:
//   1) Plots a 3D surface defined parametrically
//   2) Plot to any type of device;
//      MoveTo(), DrawTo() pure virtual methods must be
//      supplied by the derived class to do the actual plotting.
//   3) User selectable 3D view point.
//   4) Supports hidden line removal.
//   5) Provides for scaling.
//   6) Video resolution is settable.
//   7) Allows for user supplied (optional) call-back functions
//      that can be used to monitor plotting progress.
//   8) Has the ability to produce color gradient plots.
//
// Author:
//   Clark Dailey
//   9037 Cambridge Road
//   Woodbury, MN 55125
//   ccdailey@imation.com
//   (651)-714-9036
//
// This source code may be modified or used as is,
// as long as credit is given to the author.
//
// -----------------------------------------------------------------

#ifndef	_PLOT3DP_H_	 	// avoid repeated including
#define	_PLOT3DP_H_	1	// has been included

#include "..\common\plot3d.h"

// ----------------
//   point class
// ----------------
class Point3DP
{
public:
	Point3DP() { m_is_visible=-1; }
	~Point3DP() { }
	void Load(unsigned i,unsigned j,double x,double y,double z,double r,double g,double b)
	{
		m_i=i;
		m_j=j;
		m_xf=x;
		m_yf=y;
		m_zf=z;
		m_red=r;
		m_green=g;
		m_blue=b;
	}
	void Project(double xp,double yp,double zp)
	{
		m_xp=xp;
		m_yp=yp,
		m_zp=zp;
	}
public: // data
	unsigned m_i,m_j;		// mesh index
	double m_xf,m_yf,m_zf;	// function coordinates
	double m_xn,m_yn,m_zn;	// normal surface vector
	double m_xp,m_yp,m_zp;	// projection coordinates
	double m_red, m_green, m_blue;	// color of point
	int m_is_visible;		// -1=unknown, 0=no, 1=yes
};

// ----------------
//  triangle class
// ----------------
class Triangle3DP
{
public:
	Triangle3DP() { m_pPnt[0] = m_pPnt[1] = m_pPnt[2] = 0; m_color=RGB(255,255,255); }
	~Triangle3DP() { }
	// get coordinates of triangle
	double XPnt(unsigned i) { return(m_pPnt[i]->m_xp); }
	double YPnt(unsigned i) { return(m_pPnt[i]->m_yp); }
	double ZPnt(unsigned i) { return(m_pPnt[i]->m_zp); }
	// get color of points
	double RedPnt  (unsigned i) { return(m_pPnt[i]->m_red);   }
	double GreenPnt(unsigned i) { return(m_pPnt[i]->m_green); }
	double BluePnt (unsigned i) { return(m_pPnt[i]->m_blue);  }
	double RedTri  () { return(m_red);   }
	double GreenTri() { return(m_green); }
	double BlueTri () { return(m_blue);  }

	// returns: 0=no, 1=point is one of defining points of triangle
	int IsDefiningPoint(Point3DP* pPnt)
	{
		unsigned i,j;
		i = pPnt->m_i;
		j = pPnt->m_j;
		if (m_pPnt[0]->m_i == i && m_pPnt[0]->m_j == j) return(1);
		if (m_pPnt[1]->m_i == i && m_pPnt[1]->m_j == j) return(1);
		if (m_pPnt[2]->m_i == i && m_pPnt[2]->m_j == j) return(1);
		return(0);
	}
	// is a given xp,yp point within the triangle
	// returns: 0=no, 1=point is within triangle
	int Triangle3DP::IsInscribed(double xp,double yp)
	{
		double dir;
	
		// first check bounding rectangle
		if (xp<m_xpmin || xp>m_xpmax || yp<m_ypmin || yp>m_ypmax )
			return(0);
	
		// check directions for exclusion
		dir = m_dxp[0]*(yp-YPnt(1)) - (xp-XPnt(1))*m_dyp[0];
		if (dir*m_dir<0.0)
			return(0);
		dir = m_dxp[1]*(yp-YPnt(2)) - (xp-XPnt(2))*m_dyp[1];
		if (dir*m_dir<0.0)
			return(0);
		dir = m_dxp[2]*(yp-YPnt(0)) - (xp-XPnt(0))*m_dyp[2];
		if (dir*m_dir<0.0)
			return(0);
		return(1);
	}
	// get color of triangle
	long GetColor() { return(m_color); }

	// calculate zp intercept at a given xp,yp point
	double ZIntercept(double xp,double yp)
	{
		double zp,divisor;
		divisor = m_C;
		if (divisor==0.0) divisor=1.e-6; // avoid division by zero
		zp = -(m_A*xp + m_B*yp + m_D)/divisor;
		return(zp);
	}
	void Load(unsigned i,unsigned j,unsigned k,Point3DP* pP1,Point3DP* pP2,Point3DP* pP3);
	void SetColor(double r,double g,double b)
	{
		int ir,ig,ib;
		// r,g,b (1->100)
		m_red   = r;
		m_green = g;
		m_blue  = b;
		// convert to 0-255 scale
		ir = PCOLOR_TO_WINCOLOR(r);
		ig = PCOLOR_TO_WINCOLOR(g);
		ib = PCOLOR_TO_WINCOLOR(b);
		if (ir<0) ir=0;
		if (ig<0) ig=0;
		if (ib<0) ib=0;
		if (ir>255) ir=255;
		if (ig>255) ig=255;
		if (ib>255) ib=255;
		m_color = RGB(ir,ig,ib); // convert to RGB
	}
public: // data
	Point3DP* m_pPnt[3]; // pointer to three points (we don't own the memory)
	unsigned m_i,m_j,m_k;	// indexes of triangle
	double m_xpmin,m_xpmax,m_ypmin,m_ypmax,m_zpmin,m_zpmax; // min and max projections
	double m_dxp[3],m_dyp[3],m_dzp[3];	// point-to-point deltas
	double m_A, m_B, m_C, m_D;	// plane constants (A*xp+B*yp+C*zp+D=0)
	double m_area; // area of triangle
	double m_dir;  // direction of vertices (-1 or +1)
	long   m_color;	// triangle color (Windows)
	double m_red; 	// 0-100 (PLOT3D_MIN_COLOR -> PLOT3D_MAX_COLOR
	double m_green;	// 0-100 (PLOT3D_MIN_COLOR -> PLOT3D_MAX_COLOR
	double m_blue; 	// 0-100 (PLOT3D_MIN_COLOR -> PLOT3D_MAX_COLOR
};


// ------------------------------
//  3D parametric plotting class
// ------------------------------
class Plot3DP : public Plot3D // abstract class - must be derived from to use
{
public:
	Plot3DP() { ZeroValues(); }	// constructor
	virtual ~Plot3DP() { FreeMemory(); }	// destructor

	// Parameteric surface equations
	virtual double XParametric(double u,double v)=0;
	virtual double YParametric(double u,double v)=0;
	virtual double ZParametric(double u,double v)=0;
	virtual double RedColor   (double u,double v)=0;
	virtual double GreenColor (double u,double v)=0;
	virtual double BlueColor  (double u,double v)=0;

	// settings
	int SetParameters(unsigned nulines=100,unsigned nvlines=100,
		unsigned draw_ulines=1,unsigned draw_vlines=1,unsigned remove_hidden_lines=1)
	{
		if (nulines<2) nulines=2;
		if (nvlines<2) nvlines=2;
		m_nulines = nulines;
		m_nvlines = nvlines;
		m_remove_hidden_lines = remove_hidden_lines;
		m_draw_ulines = draw_ulines;
		m_draw_vlines = draw_vlines;
		return(0);
	}

	// returns: 0=ok, 1=invalid value
	int SetRanges(double umin=-10,double umax=10,double vmin=-10,double vmax=10)
	{
		if (umin>=umax || vmin>=vmax) return(1);
		m_umin = umin;
		m_umax = umax;
		m_vmin = vmin;
		m_vmax = vmax;
		return(0);
	}

	// allow user to load their own mesh for plotting
	int LoadMesh(double** x,double** y,double** z,unsigned m,unsigned n);

	// plotting
	int Plot(void);
	void Reset(void)
	{
		Init();
		FreeMemory();
		ZeroValues();
	}

	// retrieve triangle info
	unsigned TriangleCount() { return(m_nTriangles); }
	Triangle3DP* GetTriangle(unsigned index) { return( (index<m_nTriangles) ? &m_pTriangles[index] : (Triangle3DP*)0 ); }

	// calculate coordinates for export to Persistence-of-Vision (POV)
	void GetCenter(double* x,double* y,double* z) { *x = m_xmid; *y = m_ymid; *z = m_zmid; }
	void GetPOVCamera(double* xc,double* yc,double* zc)
	{
		double xdiam,ydiam,diam;
		double zp, scale;
		xdiam = fabs(m_xpmax-m_xpmin);
		ydiam = fabs(m_ypmax-m_ypmin);
		diam  = max(xdiam,ydiam);
		scale = fabs(m_scale);
		if (scale<1.e-6) scale=1.e-6;
		zp  = (diam/scale) + m_zpmid;
		*xc = xof(m_xpmid,m_ypmid,zp);
		*yc = yof(m_xpmid,m_ypmid,zp);
		*zc = zof(m_xpmid,m_ypmid,zp);
	}

protected:	// routines
	int  InitPlot(void);
	int  CalcPoints(void);
	int  CalcTriangles(void);
	int  PlotColorPixels(void);
	int  DrawULines(void);
	int  DrawVLines(void);
	void DrawAxis(void);
	void ZeroValues(void)
	{
		m_ExternalMesh = 0;
		m_nPoints = 0;
		m_nTriangles = 0;
		m_pPoints = 0;
		m_pTriangles = 0;
		m_pHashCounts = 0;
		m_pHashPointers = 0;
		m_nHashItems = 0;
		m_xpmin = m_ypmin = m_zpmin =  PLOT3D_DBL_MAX; // a very large number
		m_xpmax = m_ypmax = m_zpmax = -PLOT3D_DBL_MAX; // a very small number
		m_maxdxp = m_maxdyp = m_maxdzp = 0.0;
		m_background_color = RGB(255,255,255); // white
		m_is_color = 0;
		// use default values
		SetParameters();
		SetRanges();
	}
	void FreeMemory(void)
	{
		unsigned i,count;
		char* pList;
		delete [] m_pPoints;
		delete [] m_pTriangles;
		for (i=0;i<m_nHashItems;i++)
		{
			count = m_pHashCounts[i];
			if (count==0) continue;
			pList = HashListPtr(i);
			delete [] pList;
		} // for i
		delete [] m_pHashCounts;
		delete [] m_pHashPointers;
		m_pPoints = 0;
		m_pTriangles = 0;
		m_pHashCounts = 0;
		m_pHashPointers = 0;
		m_nHashItems = 0;
	}
	int  PlotLine(int FirstLine,Point3DP* pPnt1,Point3DP* pPnt2)
	{
		double	xp1,yp1,zp1,xp2,yp2,zp2;
		double	dxp,dyp,dzp,ddv;
		int is_vis2, is_vis1;
		int xv1,yv1,xv2,yv2;
		unsigned i,dxv,dyv,dv;
	
	  	xp1 = pPnt1->m_xp;
	  	yp1 = pPnt1->m_yp;
	  	zp1 = pPnt1->m_zp;
	
	  	xp2 = pPnt2->m_xp;
	  	yp2 = pPnt2->m_yp;
	  	zp2 = pPnt2->m_zp;
	
	  	xv1 = xvf(xp1);
	  	yv1 = yvf(yp1);
	  	xv2 = xvf(xp2);
	  	yv2 = yvf(yp2);
	
		dxv = abs(xv2-xv1);
		dyv = abs(yv2-yv1);
		dv  = max(dxv,dyv);
		if (dv<2) dv = 2;
		ddv = double(dv-1);
		dxp = (xp2-xp1)/ddv;
		dyp = (yp2-yp1)/ddv;
		dzp = (zp2-zp1)/ddv;
	
		// do a move for first line
	  	if (FirstLine)
	  		FilterMoveTo(xv1,yv1);
	
	  	is_vis1 = IsPointVisible(pPnt1);
		for (i=0; i<dv; i++)
		{
			// calculate new point along line segment
			xp2 = xp1 + dxp;
			yp2 = yp1 + dyp;
			zp2 = zp1 + dzp;
		  	xv2 = xvf(xp2);
		  	yv2 = yvf(yp2);
	
		  	is_vis2 = IsPointVisible(xp2,yp2,zp2);
		
		 	if (is_vis1 && is_vis2)
		 		FilterDrawTo(xv2,yv2);
		 	else
		 		FilterMoveTo(xv2,yv2);
	
			// update for next pass
			xp1 = xp2;
			yp1 = yp2;
			zp1 = zp2;
			is_vis1 = is_vis2;
		} // for i
	
		return(0);
	}
	long CalcPixelColor(int xv,int yv)
	{
		long color;
		double xp,yp,zp,zpmax=-PLOT3D_DBL_MAX;
		Triangle3DP* pTri;
		Triangle3DP* pTriMax=0;
		unsigned nt,index,count;
		char* pList;
		char* pPtr;
	
		// convert to plot space
		xp = xv2xp(xv);
		yp = yv2yp(yv);
	
		// search triangles
		index = HashIndex(xp,yp);
		count = m_pHashCounts[index];
		pList = HashListPtr(index);
		for (nt=0; nt<count; nt++)
		{
			pPtr = HashListValuePtr(pList,nt);
			pTri = (Triangle3DP*)pPtr; // get pointer to triangle
	
			// check if point is inscribed in triangle
			if (!pTri->IsInscribed(xp,yp)) continue;
	
			// find closest point
			zp = pTri->ZIntercept(xp,yp);
			if (zp<=zpmax) continue;
	
			// point is closer
			zpmax = zp;
			pTriMax = pTri;
	
		} // for nt
		if (pTriMax)
			color = pTriMax->GetColor();
		else
			color = m_background_color;
		return(color);
	}
	int IsPointVisible(double xp,double yp,double zp)
	{
		Triangle3DP* pTri;
		unsigned nt;
	
		if (!m_remove_hidden_lines) return(1);
	
		// search triangles
		unsigned index = HashIndex(xp,yp);
		unsigned count = m_pHashCounts[index];
		char* pList = HashListPtr(index);
		char* pPtr;
		for (nt=0; nt<count; nt++)
		{
			pPtr = HashListValuePtr(pList,nt);
			pTri = (Triangle3DP*)pPtr; // get pointer to triangle
	
			// check if point is inscribed in triangle
			if (!pTri->IsInscribed(xp,yp)) continue;
	
			// check zp intercept to see if it is in front or behind
			if ((pTri->ZIntercept(xp,yp)-m_zptol)<zp) continue;
	
			// triangle is in front; it's hidden
			return(0);
		} // for nt
		return(1);
	}
	int  IsPointVisible(Point3DP* pPnt)
	{
		if (pPnt->m_is_visible != -1)
			return(pPnt->m_is_visible);
		pPnt->m_is_visible = IsPointVisible(pPnt->m_xp,pPnt->m_yp,pPnt->m_zp);
		return(pPnt->m_is_visible);
	}
	Point3DP* Point(unsigned i,unsigned j)
	{
		unsigned index = (i*m_nvlines) + (j);
		return(&m_pPoints[index]);
	}
	unsigned HashIndex(double xp,double yp)
	{
		unsigned index;
		int	ix, iy;
		if (xp<m_xpmin)
			xp = m_xpmin;
		if (yp<m_ypmin)
			yp = m_ypmin;
		if (xp>m_xpmax)
			xp = m_xpmax;
		if (yp>m_ypmax)
			yp = m_ypmax;
		ix = (int)((xp-m_xpmin)/m_dxbin);
		iy = (int)((yp-m_ypmin)/m_dybin);
		index = (iy*m_nxbins) + ix;
		if (index>=m_nHashItems)
			index = m_nHashItems-1;
		return(index);
	}
	char* HashListPtr(unsigned index)
	{
		char* pList;
		if (index>=m_nHashItems)
			return(NULL);
		pList = m_pHashPointers[index];
		return(pList);
	}
	void SetHashListPtr(unsigned index,char *pPtr)
	{
		if (index>=m_nHashItems)
			return;
		m_pHashPointers[index] = pPtr;
	}
	char* HashListValuePtr(char* pList,unsigned value)
	{
		char** pPtrPtr = (char**)pList;
		pList = pPtrPtr[value];
		return(pList);
	}
	void SetHashListValuePtr(char* pList,unsigned value,char *pPtr)
	{
		char** pPtrPtr = (char**)pList;
		pPtrPtr[value] = pPtr;
	}

	// call back routines
	virtual void cbBegULine(double u) { }	// called at beginning of plotting of constant u line
	virtual void cbEndULine(double u) { }	// called when plotting of u line is complete
	virtual void cbBegVLine(double v) { }	// called at beginning of plotting of constant v line
	virtual void cbEndVLine(double v) { }	// called when plotting of v line is complete

protected:	// user input data
	// flags
	unsigned m_remove_hidden_lines;	// 0=no, 1=remove hidden lines
	unsigned m_draw_ulines;			// 0=no, 1=draw U lines
	unsigned m_draw_vlines;			// 0=no, 1=draw V lines

	// parametric parameters
	unsigned m_nulines, m_nvlines;	// number of u,v contour lines to draw
	double	m_umin, m_umax;			// range of u parameter
	double	m_vmin, m_vmax;			// range of v parameter


protected:	// internal data
	double	m_xpmin,m_xpmax,m_ypmin,m_ypmax,m_zpmin,m_zpmax; // min and max projections
	double	m_xstart[2],m_ystart[2], m_zstart[2];	// starting x,y,z points: 0=even,1=odd
	double	m_maxdxp, m_maxdyp, m_maxdzp, m_zptol;	// max triangle deltas
	unsigned m_nPoints;		// number of points in m_pPoints
	unsigned m_nTriangles;	// number of triangles in m_pTriangles
	unsigned m_ExternalMesh;	// 0=internal, 1=external
	unsigned m_nutriangles, m_nvtriangles;
	Point3DP*	 m_pPoints;		// pointer to an array of Point3DP
	Triangle3DP* m_pTriangles;	// pointer to an array of Triiangle3DP

	// hash look-up table for speed
	unsigned	m_nHashItems;	// number of items in m_pHashCounts and m_pHashPointers;
	unsigned*	m_pHashCounts;	// pointer to an array of counts
	char**		m_pHashPointers;// pointer to an array of pointers
	unsigned	m_nxbins;		// number of bins along x axis
	unsigned	m_nybins;		// number of bins along y axis
	double		m_dxbin;		// 1/(width of each bin)
	double		m_dybin;		// 1/(height of each bin)
};


#endif	/* if _PLOT3DP_H_ included */

/*	<><><><><><><><><><><><><>  Plot3DP.h  <><><><><><><><><><><><><><> */
