//	<><><><><><><><><><><><><>  Plot3DZ.cpp  <><><><><><><><><><><><><><> */
//		
// ----------------------------------------------------
//
// Implementation of Plot3DZ C++ class
//
// Author:
//   Clark Dailey
//   9037 Cambridge Road
//   Woodbury, MN 55125
//   ccdailey@imation.com
//   (651)-714-9036
//
// (C) Copyright Clark Dailey, Inc. 1999
// This source code may be modified or used as is,
// as long as credit is given to the author.
//
// ----------------------------------------------------

#include "stdafx.h"	// precompiled header
#include <stdio.h>
#include "Plot3DZ.h"


// ------------------------------------------------------------
//                         Constructor
// ------------------------------------------------------------
Plot3DZ::Plot3DZ()
{
	// use default values
	SetParameters();

	// initialize dynamic memory
	m_mask_size = 0;
	m_pLoYMask = 0;
	m_pHiYMask = 0;

	// clear
	m_region_last=0;
	m_xv_last=0;
	m_yv_last=0;
	m_color_last=0;
	m_front_edge_drawn=0;
}

// ------------------------------------------------------------
//                          Destructor
// ------------------------------------------------------------
Plot3DZ::~Plot3DZ()
{
	DeleteMaskArrays();
}

// ------------------------------------------------------------
int Plot3DZ::SetParameters(unsigned nxlines,unsigned nylines,
				  unsigned show_xlines, unsigned show_ylines,
				  unsigned remove_hidden_lines)
{
	if (nxlines<2 || !show_xlines) nxlines=2;
	if (nylines<2 || !show_ylines) nylines=2;
	m_nxlines = nxlines;
	m_nylines = nylines;
	m_show_xlines = show_xlines;
	m_show_ylines = show_ylines;
	m_remove_hidden_lines = remove_hidden_lines;
	return(0);
}

// ------------------------------------------------------------
//                   Hidden Line Masking Routines
// ------------------------------------------------------------
// returns: 0=ok, 1=error
int Plot3DZ::AllocateMaskArrays()
{
	int rc=0;
	unsigned int nsize;
	nsize = m_xvmax - m_xvmin + 10;
	if (nsize==0) nsize=1;
	m_pLoYMask = new int[nsize];
	m_pHiYMask = new int[nsize];
	if (m_pLoYMask==0 || m_pHiYMask==0)
	{
		DeleteMaskArrays();
		rc = 1;
	}
	else
	{
		m_mask_size = nsize;
	}
	return(rc);
}

// ------------------------------------------------------------
// delete mask arrays
void Plot3DZ::DeleteMaskArrays()
{
	delete [] m_pHiYMask;
	delete [] m_pLoYMask;
	m_pLoYMask = 0;
	m_pHiYMask = 0;
	m_mask_size = 0;
}

// ------------------------------------------------------------
void Plot3DZ::InitMaskArrays()
{
	// initialize mask arrays
	if (m_pLoYMask==0 || m_pHiYMask==0) return;
	for (unsigned i=0;i<m_mask_size;i++)
	{
		m_pLoYMask[i] =  PLOT_LO_MASK;
		m_pHiYMask[i] =  PLOT_HI_MASK;
	}
}

// ------------------------------------------------------------
// masking access functions
inline int  Plot3DZ::HiYMask(int xv)
{
	int	i;
	i = xv - m_xvmin;
	assert((unsigned)i<m_mask_size);
	return(m_pHiYMask[i]);
}

inline int  Plot3DZ::LoYMask(int xv)
{
	int	i;
	i = xv - m_xvmin;
	assert((unsigned)i<m_mask_size);
	return(m_pLoYMask[i]);
}

inline void Plot3DZ::SetHiYMask(int xv,int yv)
{
	int	i;
	i = xv - m_xvmin;
	assert((unsigned)i<m_mask_size);
	m_pHiYMask[i] = yv;
}

inline void Plot3DZ::SetLoYMask(int xv,int yv)
{
	int	i;
	i = xv - m_xvmin;
	assert((unsigned)i<m_mask_size);
	m_pLoYMask[i] = yv;
}

// ------------------------------------------------------------
int Plot3DZ::InitPlot()
{
	PlotOutputOn();
	CalcTM();		// calculate transformation matrix
	CalcCorners();	// calculate corners of x,y plate
	CalcDeltas();
	AllocateMaskArrays();
	m_front_edge_drawn = 0;
	return(0);
}

// ------------------------------------------------------------
#define	DBGPLOT	TRACE	// TRACE for Windows95, printf for DOS
void Plot3DZ::Dump()
{
	DBGPLOT("------------------ Plot3DZ Dump Begin -----------------------\n");
	DBGPLOT("Inputs:\n");
	DBGPLOT("  nXlines=%6u nYlines=%6u\n", m_nxlines,m_nylines);
	DBGPLOT("  Show_Xlines=%u Show_Ylines=%u Remove_HiddenLines=%u\n", m_show_xlines,m_show_ylines,m_remove_hidden_lines);

	DBGPLOT("Deltas:\n");
	DBGPLOT("  dxline=%10.4f  dyline=%10.4f\n",m_dxline,m_dyline);
	DBGPLOT("  nxincs=%6u  nyincs=%5u\n", m_nxincs, m_nyincs);
	DBGPLOT("  nxedge=%6u  nyedge=%5u\n", m_nxedge, m_nyedge);
	DBGPLOT("  dxedge=%10.4f  dyedge=%10.4f\n", m_dxedge, m_dyedge);
	DBGPLOT("  dxinc [0]=%10.4f dxinc [1]=%10.4f dyinc [0]=%10.4f dyinc [1]=%10.4f\n", m_dxinc[0],m_dxinc[1],m_dyinc[0],m_dyinc[1]);
	DBGPLOT("  xstart[0]=%10.4f xstart[1]=%10.4f ystart[0]=%10.4f ystart[1]=%10.4f\n", m_xstart[0],m_xstart[1],m_ystart[0],m_ystart[1]);

	DBGPLOT("Hidden Line Removal\n");
	DBGPLOT("  region_last=%d  xv_last=%d  yv_last=%d\n", m_region_last,m_xv_last,m_yv_last);
	DBGPLOT("  color_last=%ld  front_drawn=%d\n", m_color_last,m_front_edge_drawn);

	DBGPLOT("Mask Array: nsize=%u\n", m_mask_size);
	DBGPLOT("  *hi: %p\n", m_pHiYMask);
	DBGPLOT("  *lo: %p\n", m_pLoYMask);
	DBGPLOT("------------------- Plot3DZ Dump End ------------------------\n");
}

// ------------------------------------------------------------
void Plot3DZ::CalcDeltas()
{
	double	dxinc, dyinc, dxedge, dyedge;
	int xv0, xv1, xv2;

	// calc deltas between constant x,y lines
	m_dxline =(m_xmax - m_xmin)/double(m_nxlines-1);
	m_dyline =(m_ymax - m_ymin)/double(m_nylines-1);

	// calc plot to video scaling
	m_p2v = (m_scale * (double)(m_xvmax - m_xvmin)) / (double)(m_xpmax - m_xpmin);

	// calc plotting x,y increments
	xv0 = xvf(m_xpcorner[0]);
	xv1 = xvf(m_xpcorner[1]);
	xv2 = xvf(m_xpcorner[2]);
	m_nxincs = abs(xv1-xv0);
	m_nyincs = abs(xv2-xv0);
	if (m_nxincs<2) m_nxincs = 2;
	if (m_nyincs<2) m_nyincs = 2;
	m_nxedge = (unsigned)(((double)m_nxincs/(double)(m_nxlines-1)));
	m_nyedge = (unsigned)(((double)m_nyincs/(double)(m_nylines-1)));
	if (m_nxedge<2) m_nxedge = 2;
	if (m_nyedge<2) m_nyedge = 2;
	dxinc = (m_xmax-m_xmin)/double(m_nxincs-1);
	dyinc = (m_ymax-m_ymin)/double(m_nyincs-1);
	dxedge = m_dxline/double(m_nxedge-1);
	dyedge = m_dyline/double(m_nyedge-1);

	// plotting must start from outmost of the screen inward for proper hidden line removal
	if (m_zpcorner[1]>=m_zpcorner[0])
	{
		m_xstart[0] =  m_xmax;
		m_xstart[1] =  m_xmin;
		m_dxinc[0]  = -dxinc;
		m_dxinc[1]  =  dxinc;
		m_dxline    = -m_dxline;
		m_dxedge    = -dxedge;
	}
	else
	{
		m_xstart[0] =  m_xmin;
		m_xstart[1] =  m_xmax;
		m_dxinc[0]  =  dxinc;
		m_dxinc[1]  = -dxinc;
		m_dxedge    =  dxedge;
	}
	if (m_zpcorner[2]>m_zpcorner[0])
	{
		m_ystart[0] =  m_ymax;
		m_ystart[1] =  m_ymin;
		m_dyinc[0]  = -dyinc;
		m_dyinc[1]  =  dyinc;
		m_dyline    = -m_dyline;
		m_dyedge    = -dyedge;
	}
	else
	{
		m_ystart[0] =  m_ymin;
		m_ystart[1] =  m_ymax;
		m_dyinc[0]  =  dyinc;
		m_dyinc[1]  = -dyinc;
		m_dyedge    =  dyedge;
	}
	if (m_ypcorner[0]<m_ypcorner[4])
	{
		m_zstart[0] = m_zmin;
		m_zstart[1] = m_zmax;
	}
	else
	{
		m_zstart[0] = m_zmax;
		m_zstart[1] = m_zmin;
	}
}

// ------------------------------------------------------------
int Plot3DZ::InitHiddenLine(int xv,int yv)
{
	int	region;

	region = PLOT_REGION_HIDDEN;	// assume hidden
	PLOT_ASSERT((unsigned)xv < m_mask_size);
	if (yv >= HiYMask(xv)) region = PLOT_REGION_ABOVE;
	if (yv <= LoYMask(xv)) region = PLOT_REGION_BELOW;

	// update hidden line removal masking
	if (yv > HiYMask(xv)) SetHiYMask(xv,yv);
	if (yv < LoYMask(xv)) SetLoYMask(xv,yv);

	// update last value
	m_region_last = region;
	if (region != PLOT_REGION_HIDDEN) FilterMoveTo(xv,yv);
	return(region);
}

// ------------------------------------------------------------
// does clipping for hidden line removal
// returns: region value for xv,yv
int Plot3DZ::CalcHiddenLine(int xv,int yv,long color)
{
	int region;
	int	draw_mask,yvt,dxv,adxv;

	region = PLOT_REGION_HIDDEN;	// assume hidden
	PLOT_ASSERT((unsigned)xv < m_mask_size);
	if (yv >= HiYMask(xv)) region = PLOT_REGION_ABOVE;	// above
	if (yv <= LoYMask(xv)) region = PLOT_REGION_BELOW;	// below

	// masking requires no skipped points
	dxv = xv - m_xv_last;
	adxv = abs(dxv);
	if (adxv>1)
	{
		int	xvt, yvt;
		long colort;

		// use linear interpolation to generate missing point
		xvt = (dxv>0) ? m_xv_last+1 : m_xv_last-1;
		yvt = (int)(((double)(yv - m_yv_last)*(xvt - m_xv_last))/(double)dxv);
		yvt += m_yv_last;
		colort = m_color_last;
		CalcHiddenLine(xvt,yvt,colort);
	}

	// handle clipping
	draw_mask = (m_region_last<<2) | region;
	switch(draw_mask)
	{
				//  from to
	case 0:		//    00 00     hidden to hidden
		// OK
		break;		  
	case 1:		//    00 01     hidden to above
		// OK
		yvt = HiYMask(m_xv_last);
 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
 		FilterMoveTo(m_xv_last,yvt);
 		FilterColorTo(color);
 		FilterDrawTo(xv,yv);
		break;		  
	case 2:		//    00 10     hidden to below
		// OK
		yvt = LoYMask(m_xv_last);
 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
 		FilterMoveTo(m_xv_last,yvt);
 		FilterColorTo(color);
 		FilterDrawTo(xv,yv);
		break;		  
	case 3:		//    00 11     invalid
		// OK
		break;		  
	case 4:		//    01 00     above  to hidden
 		FilterColorTo(color);
		yvt  = HiYMask(xv);
 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
 		FilterDrawTo(xv,yvt);
		break;		  
	case 5:		//    01 01     above  to above 
		// OK
		FilterColorTo(color);
		FilterDrawTo(xv,yv);
		break;		  
	case 6:		//    01 10     above  to below 
		if (HiYMask(m_xv_last)!=LoYMask(m_xv_last) &&
		    HiYMask(  xv     )!=LoYMask(  xv     ) )
		{
	 		yvt = HiYMask(m_xv_last);
	 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
	 		FilterDrawTo(m_xv_last,yvt);
	 		yvt = LoYMask(xv);
	 		if (IfPlotMaskNotSet(yvt))
	 			yvt = LoYMask(m_xv_last);
	 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
	 		FilterMoveTo(xv,yvt);
		}
 		FilterColorTo(color);
 		FilterDrawTo(xv,yv);
		break;		  
	case 7:		//    01 11     invalid
		// OK
		break;		  
	case 8:		//    10 00     below  to hidden
 		FilterColorTo(color);
		yvt  = LoYMask(xv);
 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
 		FilterDrawTo(xv,yvt);
		break;		  
	case 9:		//    10 01     below  to above 
		if (HiYMask(m_xv_last)!=LoYMask(m_xv_last) &&
		    HiYMask(  xv     )!=LoYMask(  xv     ) )
		{
	  		yvt = LoYMask(m_xv_last);
	 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
	  		FilterDrawTo(m_xv_last,yvt);
	  		yvt = HiYMask(xv);
	  		if (IfPlotMaskNotSet(yvt))
	  			yvt = HiYMask(m_xv_last);
	 		PLOT_ASSERT(!IfPlotMaskNotSet(yvt));
	  		FilterMoveTo(xv,yvt);
		}
  		FilterColorTo(color);
  		FilterDrawTo(xv,yv);
		break;		  
	case 10:	//    10 10     below  to below 
		// OK
		FilterColorTo(color);
		FilterDrawTo(xv,yv);
		break;
	}

	// update hidden line removal masking
	if (yv > HiYMask(xv)) SetHiYMask(xv,yv);
	if (yv < LoYMask(xv)) SetLoYMask(xv,yv);
	m_region_last = region;

  	return(region);
}

// ------------------------------------------------------------
void Plot3DZ::PlotPoint(unsigned nPoint,double x,double y)
{
	double z;
	double xp, yp;
	int	xv,yv;
	long color;

	// calculate z
	z = Zfunc(x,y); // invert to account for inverted y axis

	// calculate xv,yv video points
	xp = xpf(x,y,z);
	yp = ypf(x,y,z);
	xv = xvf(xp);
	yv = yvf(yp);
	color = Color(x,y,z);

	// plot the point
	if (m_remove_hidden_lines)
	{
		// handle hidden line removal
	   	if (nPoint==0)
   			InitHiddenLine(xv,yv);
	   	else
	   		CalcHiddenLine(xv,yv,color);
	}
	else
	{
		// no hidden line removal
	   	if (nPoint == 0)
   			FilterMoveTo(xv,yv);
	   	else
   		{
	   		FilterColorTo(color);
   			FilterDrawTo(xv,yv);
	   	}
	}

	// save as last
	m_xv_last = xv;
	m_yv_last = yv;
	m_color_last = color;
}

// ------------------------------------------------------------
// does the actual work of plotting
int Plot3DZ::Plot()
{
	InitPlot();
	cbBegPlot();

	// plot constant x lines if requested
	if (m_show_xlines)
	{
		InitMaskArrays();
		PlotFrontEdge(0);
		PlotXLines();
	}

	// plot constant y lines if requested
	if (m_show_ylines)
	{
		InitMaskArrays();
		PlotFrontEdge(1);
		PlotYLines();
	}

	DrawAxis();

	cbEndPlot();
	return(0);
}

// ------------------------------------------------------------
inline void Plot3DZ::PlotXLine(unsigned nPoints,double x,double ystart,double yend,double dy,int issueCB)
{
	unsigned i;
	double y;

	if (m_abort_plotting) return;
	if (issueCB) cbBegXLine(x);
	for (i=0,y=ystart; i<nPoints; i++,y+=dy)
	{
		if (i+1==nPoints) y = yend;
		PlotPoint(i,x,y);
	}
	if (issueCB) cbEndXLine(x);
	cbCheckUserAbort();
}

// ------------------------------------------------------------
inline void Plot3DZ::PlotYLine(unsigned nPoints,double y,double xstart,double xend,double dx,int issueCB)
{
	unsigned i;
	double x;

	if (m_abort_plotting) return;
	if (issueCB) cbBegYLine(y);
	for (i=0,x=xstart; i<nPoints; i++,x+=dx)
	{
		if (i+1==nPoints) x = xend;
		PlotPoint(i,x,y);
	}
	if (issueCB) cbEndYLine(y);
	cbCheckUserAbort();
}

// ------------------------------------------------------------
// plot front edge lines
void Plot3DZ::PlotFrontEdge(int yDir)
{
	if (m_front_edge_drawn) PlotOutputOff();	// only draw once

	if (yDir)
	{
		PlotXLine(m_nyincs,m_xstart[0],m_ystart[1],m_ystart[0],m_dyinc[1],1);
		PlotYLine(m_nxincs,m_ystart[0],m_xstart[0],m_xstart[1],m_dxinc[0],1);
	}
	else
	{
		PlotYLine(m_nxincs,m_ystart[0],m_xstart[1],m_xstart[0],m_dxinc[1],1);
		PlotXLine(m_nyincs,m_xstart[0],m_ystart[0],m_ystart[1],m_dyinc[0],1);
	}

	PlotOutputOn();	// plotting back on
	m_front_edge_drawn = 1;
}

// ------------------------------------------------------------
// plot constant x lines
int Plot3DZ::PlotXLines()
{
	unsigned int ix,nedge,nxlines,lastline,nxedge2;
	double x,xstart,xend;

	// Note: assumes front edge has already been drawn and masked
	nxedge2 = m_nxedge * 2;
	lastline = m_nxlines-1;
	nxlines = (m_nxlines+1)&(~1);	// make it even number
	for (ix=1,x=m_xstart[0]+m_dxline; ix<nxlines; )
	{
		// draw edge tick
 		xstart = x - m_dxline - m_dxline;
		xend = x;
 		nedge = nxedge2;
 		if (ix==1)	// first line
 		{
	 		nedge = m_nxedge;
	 		xstart = x - m_dxline;
 		}
 		else if (ix>=lastline && (m_nxlines & 1))	// last line is odd
		{
	 		nedge = m_nxedge;
			xend = x - m_dxline;
		}
		PlotYLine(nedge,m_ystart[1],xstart,xend,m_dxedge,0);

		// down
		if (ix > lastline) break;
		if (m_show_ylines && ix>=lastline) PlotOutputOff();	// just mask it
		PlotXLine(m_nyincs,x,m_ystart[1],m_ystart[0],m_dyinc[1],1);
		ix++;
		x += m_dxline;

		// and back
		if (ix > lastline) break;
		if (m_show_ylines && ix>=lastline) PlotOutputOff();	// just mask it
		PlotXLine(m_nyincs,x,m_ystart[0],m_ystart[1],m_dyinc[0],1);
		ix++;
		x += m_dxline;

	} // for ix
	PlotOutputOn();
	return(0);
}

// ------------------------------------------------------------
// plot constant y lines
int Plot3DZ::PlotYLines()
{
	unsigned int iy,nedge,nylines,lastline,nyedge2;
	double y,ystart,yend;

	// Note: assumes front edge has already been drawn and masked
	nyedge2 = m_nyedge * 2;
	lastline = m_nylines-1;
	nylines = (m_nylines+1)&(~1);	// make it even number
	for (iy=1,y=m_ystart[0]+m_dyline; iy<nylines; )
	{
		// draw edge tick
 		ystart = y - m_dyline - m_dyline;
		yend = y;
 		nedge = nyedge2;
 		if (iy==1)	// first line
 		{
	 		nedge = m_nyedge;
	 		ystart = y - m_dyline;
 		}
 		else if (iy>=lastline && (m_nylines & 1))	// last line is odd
		{
	 		nedge = m_nyedge;
			yend = y - m_dyline;
		}
		PlotXLine(nedge,m_xstart[1],ystart,yend,m_dyedge,0);

		// down
		if (iy > lastline) break;
		if (m_show_xlines && iy>=lastline) PlotOutputOff();	// just mask it
		PlotYLine(m_nxincs,y,m_xstart[1],m_xstart[0],m_dxinc[1],1);
		iy++;
		y += m_dyline;

		// and back
		if (iy > lastline) break;
		if (m_show_xlines && iy>=lastline) PlotOutputOff();	// just mask it
		PlotYLine(m_nxincs,y,m_xstart[0],m_xstart[1],m_dxinc[0],1);
		iy++;
		y += m_dyline;

	} // for iy
	PlotOutputOn();
	return(0);
}

// ------------------------------------------------------------
void Plot3DZ::DrawAxis(void)
{
	CString str;
	double dx,dy,dz,x1,y1,z1,x2,y2,xp,yp;
	double degrees,gap_length,tick_length;
	int xv1,yv1,xv2,yv2;
	unsigned n;

	if (!m_show_axis) return;

	FilterColorTo(m_axis.axis_color);
	cbBegDrawAxis();

	// Constant Y Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],m_zstart[0]);

	// Constant X Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[1],m_zstart[0]);

	// Vertical Front Edges
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[0],
  		     Zfunc(m_xstart[0],m_ystart[0]));

	MapAndDrawLine(m_xstart[1],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],
  		     Zfunc(m_xstart[1],m_ystart[0]));

	MapAndDrawLine(m_xstart[0],m_ystart[1],m_zstart[0],
				   m_xstart[0],m_ystart[1],
  		     Zfunc(m_xstart[0],m_ystart[1]));

	// -------------
	// X tick marks
	// -------------
	if (m_axis.nxticks > 0)
	{
		n = m_axis.nxticks-1;
		if (n==0) n=1; // avoid divions by zero
		dx =  m_xstart[0] - m_xstart[1];
		dy = (m_ystart[1] - m_ystart[0])/((double)n);
		gap_length  = dx * (double)m_axis.tick_spacing / 500.;
		tick_length = dx * (double)m_axis.tick_length  / 500.;
	
		x1 = m_xstart[0]+gap_length;  y1 = m_ystart[0];	z1 = m_zstart[0];
		x2 = x1 + tick_length;
		MapPoint(m_xstart[1],m_ystart[0],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nxticks;n++,y1+=dy)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x2,y1,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("y=%-8.4lf",y1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Y tick marks
	// -------------
	if (m_axis.nyticks > 0)
	{
		n = m_axis.nyticks-1;
		if (n==0) n=1; // avoid divions by zero
		dy =  m_ystart[0] - m_ystart[1];
		dx = (m_xstart[1] - m_xstart[0])/((double)n);
		gap_length  = dy * (double)m_axis.tick_spacing / 500.;
		tick_length = dy * (double)m_axis.tick_length  / 500.;
	
		y1 = m_ystart[0]+gap_length;  x1 = m_xstart[0];	z1 = m_zstart[0];
		y2 = y1 + tick_length;
		MapPoint(m_xstart[0],m_ystart[1],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nyticks;n++,x1+=dx)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x1,y2,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("x=%-8.4lf",x1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Z tick marks
	// -------------
	if (m_axis.nzticks > 0)
	{
		n = m_axis.nzticks-1;
		if (n==0) n=1; // avoid divions by zero
		dz =  m_zstart[1] - m_zstart[0];
		gap_length  = dz * (double)m_axis.tick_spacing / 500.;
		tick_length = dz * (double)m_axis.tick_length  / 500.;
		dz = dz/(double)n;

		x1 = m_xcorner[0];  y1 = m_ycorner[0];  z1=m_xpcorner[0];
		if (m_xpcorner[1]>z1)
		{	x1 = m_xcorner[1];  y1 = m_ycorner[1];  z1=m_xpcorner[1];	}
		if (m_xpcorner[2]>z1)
		{	x1 = m_xcorner[2];  y1 = m_ycorner[2];  z1=m_xpcorner[2];	}
		if (m_xpcorner[3]>z1)
		{	x1 = m_xcorner[3];  y1 = m_ycorner[3];  z1=m_xpcorner[3];	}
		MapAndDrawLine(x1,y1,m_zstart[0],
					   x1,y1,m_zstart[1]);
		z1 = m_zstart[0];
		for (n=0;n<m_axis.nzticks;n++,z1+=dz)
		{
			xp = xpf(x1,y1,z1)+gap_length;
			yp = ypf(x1,y1,z1);
			xv1 = xvf(xp);
			yv1 = yvf(yp);
			xp += tick_length;
			xv2 = xvf(xp);
			yv2 = yvf(yp);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("z=%-8.4lf",z1);
			PlotText(&m_axis,0.,xv2,yv2+m_axis.font_height,str);
		}
	}

	cbEndDrawAxis();
}

/*	<><><><><><><><><><><><><>  Plot3DZ.cpp  <><><><><><><><><><><><><><> */














