//	<><><><><><><><><><><><><>  Plot3DP.cpp  <><><><><><><><><><><><><><> */
//		
// ----------------------------------------------------
//
// Implementation of Plot3DP C++ class
//
// --------------------------------------------------------------------------
//

#include "stdafx.h"	// precompiled header
#include <stdio.h>
#include <float.h>
#include "Plot3DP.h"

// ------------------------------------------------------------
void Triangle3DP::Load(unsigned i,unsigned j,unsigned k,Point3DP* pP1,Point3DP* pP2,Point3DP* pP3)
{
 	if (!pP1 || !pP2 || !pP3) return;

	// store input values
 	m_i = i;
 	m_j = j;
 	m_k = k;
 	m_pPnt[0] = pP1;
 	m_pPnt[1] = pP2;
 	m_pPnt[2] = pP3;

	// default min and max projections
 	m_xpmin = m_xpmax = XPnt(0);
 	m_ypmin = m_ypmax = YPnt(0);
 	m_zpmin = m_zpmax = ZPnt(0);

 	// check for max projection
 	if (XPnt(1) > m_xpmax) m_xpmax = XPnt(1);
 	if (YPnt(1) > m_ypmax) m_ypmax = YPnt(1);
 	if (ZPnt(1) > m_zpmax) m_zpmax = ZPnt(1);
 	if (XPnt(2) > m_xpmax) m_xpmax = XPnt(2);
 	if (YPnt(2) > m_ypmax) m_ypmax = YPnt(2);
 	if (ZPnt(2) > m_zpmax) m_zpmax = ZPnt(2);

 	// check for min projection
 	if (XPnt(1) < m_xpmin) m_xpmin = XPnt(1);
 	if (YPnt(1) < m_ypmin) m_ypmin = YPnt(1);
 	if (ZPnt(1) < m_zpmin) m_zpmin = ZPnt(1);
 	if (XPnt(2) < m_xpmin) m_xpmin = XPnt(2);
 	if (YPnt(2) < m_ypmin) m_ypmin = YPnt(2);
 	if (ZPnt(2) < m_zpmin) m_zpmin = ZPnt(2);

 	// calculate point-to-point deltas
 	m_dxp[0] = XPnt(1) - XPnt(0);
 	m_dyp[0] = YPnt(1) - YPnt(0);
 	m_dzp[0] = ZPnt(1) - ZPnt(0);
 	m_dxp[1] = XPnt(2) - XPnt(1);
 	m_dyp[1] = YPnt(2) - YPnt(1);
 	m_dzp[1] = ZPnt(2) - ZPnt(1);
 	m_dxp[2] = XPnt(0) - XPnt(2);
 	m_dyp[2] = YPnt(0) - YPnt(2);
 	m_dzp[2] = ZPnt(0) - ZPnt(2);

 	// calculate plane constants (A*xp + B*yp + C*zp + D = 0)
 	m_A = m_dyp[2]*m_dzp[0] - m_dyp[0]*m_dzp[2];
 	m_B = m_dxp[0]*m_dzp[2] - m_dxp[2]*m_dzp[0];
 	m_C = m_dxp[2]*m_dyp[0] - m_dxp[0]*m_dyp[2];
 	m_D = -(XPnt(0)*m_A + YPnt(0)*m_B + ZPnt(0)*m_C);
	m_area = (m_dxp[0]*m_dyp[1] - m_dxp[1]*m_dyp[0])/2.;
	m_dir = m_area<0.0 ? -1. : 1.;
}

// ------------------------------------------------------------
// returns: 0=ok, 1=not enough memory
// ### redo this; LoadMeshPoint
int Plot3DP::LoadMesh(double** x,double** y,double** z,unsigned m,unsigned n)
{
	Point3DP* pPnt;
	delete [] m_pPoints; // delete any previously allocated memory
	if (m<2) m=2; // avoid division by zero
	if (n<2) n=2; // avoid division by zero
	m_nulines = m;
	m_nvlines = n;
	m_nPoints = m_nulines * m_nvlines;
	m_pPoints = new Point3DP [m_nPoints] ;
	if (!m_pPoints) return(1);
	unsigned i,j;
	for (i=0; i<m_nulines; i++)
	{
		for (j=0; j<m_nvlines; i++)
		{
			pPnt = Point(i,j);
			pPnt->Load(i,j,x[i][j],y[i][j],z[i][j],
				PLOT3D_MAX_COLOR, PLOT3D_MAX_COLOR, PLOT3D_MAX_COLOR);
		} // for j
	} // for i
	m_ExternalMesh = 1;	// using external mesh
	return(0);
}

// ------------------------------------------------------------
#define	NORMAL_VECTOR_DELTA		(1000)	// delta for partial derivative calculation

// returns: 0=ok, 1=not enough memory
int Plot3DP::CalcPoints()
{
	Point3DP* pPnt;
	double xp,yp,zp;
	double x,y,z;
	double x1,y1,z1;
	double x2,y2,z2;
	double dx1,dy1,dz1;
	double dx2,dy2,dz2;
	double nx,ny,nz, len;
	double du, dv, un, vn;
	double u,v;
	double udelta, vdelta;
	double red=PLOT3D_MAX_COLOR, green=PLOT3D_MAX_COLOR, blue=PLOT3D_MAX_COLOR;
	unsigned i,j;

	// range check to avoid division by zero
	if (m_nulines<2) m_nulines=2;
	if (m_nvlines<2) m_nvlines=2;

	// generate points using parametric equations
	du = (m_umax - m_umin)/double(NORMAL_VECTOR_DELTA);
	dv = (m_vmax - m_vmin)/double(NORMAL_VECTOR_DELTA);
	if (!m_ExternalMesh)
	{
		// user supplied parametric equations
		m_nPoints = m_nulines * m_nvlines;
		m_pPoints = new Point3DP [m_nPoints];
		if (!m_pPoints) return(1);
		udelta = (m_umax - m_umin)/double(m_nulines-1);
		vdelta = (m_vmax - m_vmin)/double(m_nvlines-1);
		for (i=0, u=m_umin; i<m_nulines; i++,u+=udelta)
		{
			for (j=0, v=m_vmin; j<m_nvlines; j++,v+=vdelta)
			{
				// calculate coordinates of point
				pPnt = Point(i,j);
				x  = XParametric(u, v);
				y  = YParametric(u, v);
				z  = ZParametric(u, v);
				red   = RedColor(u,v);
				green = GreenColor(u,v);
				blue  = BlueColor (u,v);
				pPnt->Load(i,j,x,y,z,red,green,blue);
				// calculate normal surface vector at the point
				un = u+du;
				vn = v+dv;
				x1 = XParametric(un,v);
				y1 = YParametric(un,v);
				z1 = ZParametric(un,v);
				x2 = XParametric(u, vn);
				y2 = YParametric(u, vn);
				z2 = ZParametric(u, vn);
				dx1 = x1 - x;
				dy1 = y1 - y;
				dz1 = z1 - z;
				dx2 = x2 - x;
				dy2 = y2 - y;
				dz2 = z2 - z;
				nx =  dy1*dz2 - dy2*dz1;
				ny = -dx1*dz2 + dx2*dz1;
				nz =  dx1*dy2 - dx2*dy1;
				len = sqrt(nx*nx+ny*ny+nz*nz);
				if (len == 0.0)
				{
					nx = ny = nz = 0.;
				}
				else
				{
					// unit normal vector
					nx = nx/len;
					ny = ny/len;
					nz = nz/len;
				}
				// store the unit normal vector
				pPnt->m_xn = nx;
				pPnt->m_yn = ny;
				pPnt->m_zn = nz;
			} // for j
		} // for i
	}

	// project points and calculate max and min projections
	// default min and max to extreme values
	m_xpmin = m_ypmin = m_zpmin =  DBL_MAX; // a very large number
	m_xpmax = m_ypmax = m_zpmax = -DBL_MAX; // a very small (negative) number
	m_xmin  = m_ymin  = m_zmin  =  DBL_MAX; // a very large number
	m_xmax  = m_ymax  = m_zmax  = -DBL_MAX; // a very small (negative) number
 	for (i=0; i<m_nulines; i++)
 	{
 		for (j=0; j<m_nvlines; j++)
 		{
			pPnt = Point(i,j);
			x = pPnt->m_xf;
			y = pPnt->m_yf;
			z = pPnt->m_zf;
			// adjust min and max surface values
			if (x<m_xmin) m_xmin = x;
			if (y<m_ymin) m_ymin = y;
			if (z<m_zmin) m_zmin = z;
			if (x>m_xmax) m_xmax = x;
			if (y>m_ymax) m_ymax = y;
			if (z>m_zmax) m_zmax = z;
			// calculate projection
			xp = xpf(x,y,z);	
			yp = ypf(x,y,z);
			zp = zpf(x,y,z);
 			pPnt->Project(xp,yp,zp);
			// adjust min and max projections
			if (xp<m_xpmin) m_xpmin = xp;
			if (yp<m_ypmin) m_ypmin = yp;
			if (zp<m_zpmin) m_zpmin = zp;
			if (xp>m_xpmax) m_xpmax = xp;
			if (yp>m_ypmax) m_ypmax = yp;
			if (zp>m_zpmax) m_zpmax = zp;
 		} // for j
 	} // for i

	// calculate mid points
	m_xmid = (m_xmin + m_xmax)/2.;
	m_ymid = (m_ymin + m_ymax)/2.;
	m_zmid = (m_zmin + m_zmax)/2.;

	m_xpmid = (m_xpmin + m_xpmax)/2.;
	m_ypmid = (m_ypmin + m_ypmax)/2.;
	m_zpmid = (m_zpmin + m_zpmax)/2.;

	return(0);
}

// ------------------------------------------------------------
#define	PLOT3DP_DISTANCE_TOLERANCE	(0.5)	// percentage of total delta zp
#define	BIN_MULT_FACTOR	(1.1)	// this must be greater than 1.0
#define	EMPTY_HASH	((unsigned)(~0))

// returns: 0=ok, 1=not enough memory
int Plot3DP::CalcTriangles()
{
	Triangle3DP* pTrngl;
	Point3DP* pPnts[3];
	unsigned i,j,k,it,count,nbytes,index;
	unsigned hindex[4];
	double dx,dy,dz;
	double r0,g0,b0,r1,g1,b1,r2,g2,b2,rt,gt,bt;
	char* pList;

 	// allocate memory for triangles
	m_nutriangles = m_nulines-1;
	m_nvtriangles = m_nvlines-1;
 	m_nTriangles = m_nutriangles * m_nvtriangles * 2;
 	m_pTriangles = new Triangle3DP [m_nTriangles];
 	if (!m_pTriangles) return(1);

	// associate points to triangles and calculate maximums
	m_maxdxp = m_maxdyp = m_maxdzp = 0.0;
	for (i=0,it=0; i<m_nutriangles; i++)
	{
		for (j=0; j<m_nvtriangles; j++)
		{
			// lower triangle
			k = 0;
			pTrngl = &m_pTriangles[it++];
			pPnts[0] = Point(i  ,j+1);
			pPnts[1] = Point(i  ,j  );
			pPnts[2] = Point(i+1,j  );
			pTrngl->Load(i,j,k,pPnts[0],pPnts[1],pPnts[2]);
			dx = pTrngl->m_xpmax - pTrngl->m_xpmin;
			dy = pTrngl->m_ypmax - pTrngl->m_ypmin;
			dz = pTrngl->m_zpmax - pTrngl->m_zpmin;
			if (dx > m_maxdxp) m_maxdxp = dx;
			if (dy > m_maxdyp) m_maxdyp = dy;
			if (dz > m_maxdzp) m_maxdzp = dz;

			// upper triangle
			k = 1;
			pTrngl = &m_pTriangles[it++];
			pPnts[0] = Point(i+1,j  );
			pPnts[1] = Point(i+1,j+1);
			pPnts[2] = Point(i  ,j+1);
			pTrngl->Load(i,j,k,pPnts[0],pPnts[1],pPnts[2]);
			dx = pTrngl->m_xpmax - pTrngl->m_xpmin;
			dy = pTrngl->m_ypmax - pTrngl->m_ypmin;
			dz = pTrngl->m_zpmax - pTrngl->m_zpmin;
			if (dx > m_maxdxp) m_maxdxp = dx;
			if (dy > m_maxdyp) m_maxdyp = dy;
			if (dz > m_maxdzp) m_maxdzp = dz;
		} // for j
	} // for i

	// calculate maximum delta of the three directions
	m_zptol = ((m_zpmax-m_zpmin) * PLOT3DP_DISTANCE_TOLERANCE )/100.;

	// calculate hash look-up-table values
	// put triangles into bins based upon location
	dx = m_xpmax - m_xpmin;
	dy = m_ypmax - m_ypmin;
	if (dx==0.0) dx = 1.e-6;	// avoid division by zero
	if (dy==0.0) dx = 1.e-6;	// avoid division by zero
	if (m_maxdxp==0.0) m_maxdxp=1.e-6; // avoid division by zero
	if (m_maxdyp==0.0) m_maxdyp=1.e-6; // avoid division by zero
	m_nxbins = (unsigned)(dx/(m_maxdxp*BIN_MULT_FACTOR) + 1.0);
	m_nybins = (unsigned)(dy/(m_maxdyp*BIN_MULT_FACTOR) + 1.0);
	if (m_nxbins<1) m_nxbins=1;
	if (m_nybins<1) m_nybins=1;
	m_nHashItems = m_nxbins * m_nybins;
	m_dxbin = dx/(double)m_nxbins;
	m_dybin = dy/(double)m_nybins;

	// allocate memory for hash tables
 	m_pHashCounts = new unsigned [m_nHashItems];
	if (!m_pHashCounts) return(1);
	memset(m_pHashCounts,0,sizeof(unsigned)*m_nHashItems);
 	m_pHashPointers = new char * [m_nHashItems];
	if (!m_pHashPointers) return(1);
	memset(m_pHashPointers,0,m_nHashItems*sizeof(char*));

	// determine counts for each hash item
	for (i=0; i<m_nTriangles; i++)
	{
		pTrngl = &m_pTriangles[i];

		// calculate color of triangle from average of points
		r0=pTrngl->RedPnt(0); g0=pTrngl->GreenPnt(0); b0=pTrngl->BluePnt(0);
		r1=pTrngl->RedPnt(1); g1=pTrngl->GreenPnt(1); b1=pTrngl->BluePnt(1);
		r2=pTrngl->RedPnt(2); g2=pTrngl->GreenPnt(2); b2=pTrngl->BluePnt(2);
		// average the three points
		rt = (r0+r1+r2)/3.;
		gt = (g0+g1+g2)/3.;
		bt = (b0+b1+b2)/3.;
		// set the color of the triangle
		pTrngl->SetColor(rt,gt,bt);

		// test four corners of bounding rectangle
		hindex[0] = HashIndex(pTrngl->m_xpmin,pTrngl->m_ypmin);
		hindex[1] = HashIndex(pTrngl->m_xpmax,pTrngl->m_ypmin);
		hindex[2] = HashIndex(pTrngl->m_xpmin,pTrngl->m_ypmax);
		hindex[3] = HashIndex(pTrngl->m_xpmax,pTrngl->m_ypmax);
		// eliminate duplicates
		if (hindex[0] == hindex[1]) hindex[1] = EMPTY_HASH;
		if (hindex[0] == hindex[2]) hindex[2] = EMPTY_HASH;
		if (hindex[0] == hindex[3]) hindex[3] = EMPTY_HASH;
		if (hindex[1] == hindex[2]) hindex[2] = EMPTY_HASH;
		if (hindex[1] == hindex[3]) hindex[3] = EMPTY_HASH;
		if (hindex[2] == hindex[3]) hindex[3] = EMPTY_HASH;
		// increment counts
	 	if (hindex[0] != EMPTY_HASH) m_pHashCounts[hindex[0]]++;
		if (hindex[1] != EMPTY_HASH) m_pHashCounts[hindex[1]]++;
		if (hindex[2] != EMPTY_HASH) m_pHashCounts[hindex[2]]++;
		if (hindex[3] != EMPTY_HASH) m_pHashCounts[hindex[3]]++;
	} // for i

	// allocate memory for hash lists
	for (i=0; i<m_nHashItems; i++)
	{
		count = m_pHashCounts[i];
		if (!count) continue;
		nbytes = count*sizeof(char*);
		pList = new char [nbytes];
		if (!pList) return(1);
		memset(pList,0,nbytes);
		SetHashListPtr(i,pList);
	} // for i

	// generate hash lists
	memset(m_pHashCounts,0,sizeof(unsigned)*m_nHashItems); // clear counts
	for (i=0; i<m_nTriangles; i++)
	{
		pTrngl = &m_pTriangles[i];
		// test four corners of bounding rectangle
		hindex[0] = HashIndex(pTrngl->m_xpmin,pTrngl->m_ypmin);
		hindex[1] = HashIndex(pTrngl->m_xpmax,pTrngl->m_ypmin);
		hindex[2] = HashIndex(pTrngl->m_xpmin,pTrngl->m_ypmax);
		hindex[3] = HashIndex(pTrngl->m_xpmax,pTrngl->m_ypmax);
		// eliminate duplicates
		if (hindex[0] == hindex[1]) hindex[1] = EMPTY_HASH;
		if (hindex[0] == hindex[2]) hindex[2] = EMPTY_HASH;
		if (hindex[0] == hindex[3]) hindex[3] = EMPTY_HASH;
		if (hindex[1] == hindex[2]) hindex[2] = EMPTY_HASH;
		if (hindex[1] == hindex[3]) hindex[3] = EMPTY_HASH;
		if (hindex[2] == hindex[3]) hindex[3] = EMPTY_HASH;
		// store the triangle in the list
	 	if (hindex[0] != EMPTY_HASH)
		{
			index = hindex[0];
			pList = HashListPtr(index);
			count = m_pHashCounts[index];
			SetHashListValuePtr(pList,count,(char*)pTrngl);
			m_pHashCounts[index]++;
		}
	 	if (hindex[1] != EMPTY_HASH)
		{
			index = hindex[1];
			pList = HashListPtr(index);
			count = m_pHashCounts[index];
			SetHashListValuePtr(pList,count,(char*)pTrngl);
			m_pHashCounts[index]++;
		}
	 	if (hindex[2] != EMPTY_HASH)
		{
			index = hindex[2];
			pList = HashListPtr(index);
			count = m_pHashCounts[index];
			SetHashListValuePtr(pList,count,(char*)pTrngl);
			m_pHashCounts[index]++;
		}
	 	if (hindex[3] != EMPTY_HASH)
		{
			index = hindex[3];
			pList = HashListPtr(index);
			count = m_pHashCounts[index];
			SetHashListValuePtr(pList,count,(char*)pTrngl);
			m_pHashCounts[index]++;
		}
	} // for i
	return(0);
}

// ------------------------------------------------------------
// returns: 0=ok, 1=not enough memory
int Plot3DP::InitPlot()
{
	PlotOutputOn();
	CalcTM();		// calculate transformation matrix
	if (CalcPoints()) return(1);	// calculate mesh points
	CalcCorners();	// calculate corners of x,y plate

	// calculate mid points
	m_xpmid = (m_xpmin + m_xpmax)/2.;
	m_ypmid = (m_ypmin + m_ypmax)/2.;
	m_zpmid = (m_zpmin + m_zpmax)/2.;

	// calc plot to video scaling
	double d;
	m_p2v = (m_scale * (double)(m_xvmax - m_xvmin)) / (double)(m_xpmax - m_xpmin);
	d     = (m_scale * (double)(m_yvmax - m_yvmin)) / (double)(m_ypmax - m_ypmin);
	if (d < m_p2v)
		m_p2v = d;

	// calculate triangles
	// triangles are needed for hidden line removal and exporting
	if (CalcTriangles()) return(1);
	return(0);
}

// ------------------------------------------------------------
// does the actual work of plotting
// returns: 0=ok, 1=user abort, 2=not enough memory to plot
int Plot3DP::Plot()
{
	if (InitPlot()) return(2);
	cbBegPlot();

	// plot color pixels
	PlotColorPixels();

	// Draw U Lines
	DrawULines();

	// Draw V Lines
	DrawVLines();

	// draw axis
	DrawAxis();

	cbEndPlot();

	return(0);
}

// ------------------------------------------------------------
int Plot3DP::PlotColorPixels()
{
	long color;
	int xv,yv;

	if (!m_is_color) return(0);
	for (yv=m_yvmin; yv<m_yvmax; yv++)
	{
		if (m_abort_plotting) break;
		for (xv=m_xvmin; xv<m_xvmax; xv++)
		{
			color = CalcPixelColor(xv,yv);
			SetPixelColor(xv,yv,color);
		} // for xv
		cbCheckUserAbort();
	} // for yv
	return(0);
}

// ------------------------------------------------------------
int Plot3DP::DrawULines()
{
	Point3DP* pPnt1;
	Point3DP* pPnt2;
	unsigned ix,iy;

	if (!m_draw_ulines) return(0);
 	for (ix=0; ix<m_nulines; ix++)
 	{
		if (m_abort_plotting) break;
	 	cbCheckUserAbort();
 		cbBegULine(ix);
		pPnt1 = Point(ix,0);
 		for (iy=1; iy<m_nvlines; iy++)
 		{
			pPnt2 = Point(ix,iy);
			PlotLine(1==iy,pPnt1,pPnt2);
			pPnt1 = pPnt2;
 		} // for iy
 		cbEndULine(ix);
 	} // for ix
	return(0);
}

// ------------------------------------------------------------
int Plot3DP::DrawVLines()
{
	Point3DP* pPnt1;
	Point3DP* pPnt2;
	unsigned ix,iy;

	if (!m_draw_vlines) return(0);
 	for (iy=0; iy<m_nvlines; iy++)
 	{
		if (m_abort_plotting) break;
	 	cbCheckUserAbort();
 		cbBegVLine(iy);
		pPnt1 = Point(0,iy);
 		for (ix=1; ix<m_nulines; ix++)
 		{
			pPnt2 = Point(ix,iy);
			PlotLine(1==ix,pPnt1,pPnt2);
			pPnt1 = pPnt2;
 		} // for ix
 		cbEndVLine(iy);
 	} // for iy
	return(0);
}

// ------------------------------------------------------------
void Plot3DP::DrawAxis(void)
{
	CString str;
	double dx,dy,dz,x1,y1,z1,x2,y2,xp,yp;
	double degrees,gap_length,tick_length;
	int xv1,yv1,xv2,yv2;
	unsigned n;

	return;	// ### needs work

	if (!m_show_axis) return;

	FilterColorTo(m_axis.axis_color);
	cbBegDrawAxis();

	// Constant Y Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],m_zstart[0]);

	// Constant X Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[1],m_zstart[0]);

	// Vertical Front Edges
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[0],m_zstart[1]);

	MapAndDrawLine(m_xstart[1],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],m_zstart[1]);

	MapAndDrawLine(m_xstart[0],m_ystart[1],m_zstart[0],
				   m_xstart[0],m_ystart[1],m_zstart[1]);

	// -------------
	// X tick marks
	// -------------
	if (m_axis.nxticks > 0)
	{
		n = m_axis.nxticks-1;
		if (n==0) n=1; // avoid divions by zero
		dx =  m_xstart[0] - m_xstart[1];
		dy = (m_ystart[1] - m_ystart[0])/((double)n);
		gap_length  = dx * (double)m_axis.tick_spacing / 500.;
		tick_length = dx * (double)m_axis.tick_length  / 500.;
	
		x1 = m_xstart[0]+gap_length;  y1 = m_ystart[0];	z1 = m_zstart[0];
		x2 = x1 + tick_length;
		MapPoint(m_xstart[1],m_ystart[0],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nxticks;n++,y1+=dy)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x2,y1,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("y=%-8.4lf",y1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Y tick marks
	// -------------
	if (m_axis.nyticks > 0)
	{
		n = m_axis.nyticks-1;
		if (n==0) n=1; // avoid divions by zero
		dy =  m_ystart[0] - m_ystart[1];
		dx = (m_xstart[1] - m_xstart[0])/((double)n);
		gap_length  = dy * (double)m_axis.tick_spacing / 500.;
		tick_length = dy * (double)m_axis.tick_length  / 500.;
	
		y1 = m_ystart[0]+gap_length;  x1 = m_xstart[0];	z1 = m_zstart[0];
		y2 = y1 + tick_length;
		MapPoint(m_xstart[0],m_ystart[1],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nyticks;n++,x1+=dx)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x1,y2,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("x=%-8.4lf",x1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Z tick marks
	// -------------
	if (m_axis.nzticks > 0)
	{
		n = m_axis.nzticks-1;
		if (n==0) n=1; // avoid divions by zero
		dz =  m_zstart[1] - m_zstart[0];
		gap_length  = dz * (double)m_axis.tick_spacing / 500.;
		tick_length = dz * (double)m_axis.tick_length  / 500.;
		dz = dz/(double)n;

		x1 = m_xcorner[0];  y1 = m_ycorner[0];  z1=m_xpcorner[0];
		if (m_xpcorner[1]>z1)
		{	x1 = m_xcorner[1];  y1 = m_ycorner[1];  z1=m_xpcorner[1];	}
		if (m_xpcorner[2]>z1)
		{	x1 = m_xcorner[2];  y1 = m_ycorner[2];  z1=m_xpcorner[2];	}
		if (m_xpcorner[3]>z1)
		{	x1 = m_xcorner[3];  y1 = m_ycorner[3];  z1=m_xpcorner[3];	}
		MapAndDrawLine(x1,y1,m_zstart[0],
					   x1,y1,m_zstart[1]);
		z1 = m_zstart[0];
		for (n=0;n<m_axis.nzticks;n++,z1+=dz)
		{
			xp = xpf(x1,y1,z1)+gap_length;
			yp = ypf(x1,y1,z1);
			xv1 = xvf(xp);
			yv1 = yvf(yp);
			xp += tick_length;
			xv2 = xvf(xp);
			yv2 = yvf(yp);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("z=%-8.4lf",z1);
			PlotText(&m_axis,0.,xv2,yv2+m_axis.font_height,str);
		}
	}

	cbEndDrawAxis();
}

/*	<><><><><><><><><><><><><>  Plot3DP.cpp  <><><><><><><><><><><><><><> */














