package JProjects.eab.data;

import java.sql.*;
import java.io.*;

/**<p>This class is used for
handling Large Object (LOB) input and output by using streams. </p>
*/
public class DAIOStream implements Cloneable, Serializable {
    private transient InputStream _in;
    private transient  InputStream _savedStream;
    private transient  OutputStream _out;
    private String _fileName;
    private int _readLength;
    private int _bytesWritten;
    private int _blockSize;
    private transient DAPropertySupport _changes;

    private static final int DEF_BLOCK_SIZE = 1024;

    /**This constructor creates a DAIOStream object that writes to the specified
     *file. Length of the data to read (in bytes) and read block size also specified.
     *@param fileName name of the file.
     *@param readL an integer indicating the length (in bytes) to be read from the object into the 
     *output stream. If you pass in 0 (zero), then the entire LOB is extracted to the output stream.
     *@param blockS read block size in bytes.
    */
    public DAIOStream(String fileName,int readL,int blockS) {
	_fileName = fileName;
	_readLength = readL;
	_blockSize = blockS;
    }

    /**This constructor creates a DAIOStream object with the specified
     * attributes.
     *@param in input data stream.
     *@param out output data stream.
     */
    public DAIOStream(InputStream in,OutputStream out) {
	_in = in;
	_out = out;
	_readLength  = 0;
	_blockSize = DEF_BLOCK_SIZE;
    }

    /**This constructor creates a DAIOStream object that writes to the specified
     *file.
     *@param fileName name of the file.
    */
    public DAIOStream(String fileName) {
	this(fileName,0,DEF_BLOCK_SIZE);
    }

    /**This constructor creates a DAIOStream object with input and output streams
        set to <samp>null</samp>.
    */
    public DAIOStream() {
	this("",0,DEF_BLOCK_SIZE);
    }

    public void addPropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.addPropertyChangeListener(l);
    }

    public void removePropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.removePropertyChangeListener(l);
    }

    /**Use this method to create an Object with the same attributes as the current
     * DAIOStream object.
     */
    public Object clone() {
	DAIOStream obj = new DAIOStream(_fileName,_readLength,_blockSize);
	obj._in = _in;
	obj._out = _out;
	obj._savedStream = _savedStream;
	return obj;
    }

    /**Use this method to get the size, in bytes, of the block of information that
     * will be read from the stream at a time.
     *@return an integer indicating the current read block size.
    */
    public int getBlockSize() {
	return _blockSize;
    }

    
    /**Use this method to set the size, in bytes, of the block of information to
     * read from the stream at a time.
     *@param blSz the size, in bytes, of the block of information to
     * read from the stream at a time.
     */
    public void setBlockSize(int blSz) {
	_blockSize = blSz;
    }

    /**Use this method to get the number of bytes read from the current DAIOStream
     * object into the output stream.
     *@return an integer indicating the number of bytes directed to the
     * output stream.
     */
    public int getBytesWritten() {
	return _bytesWritten;
    }

    /**Use this method to get the input stream that was saved when <samp>getFromInputStream</samp>
     * was called (if no output stream was specified and <samp>save</samp>
     * was set to <samp>true</samp>).
    */
    public InputStream getDatabaseInputStream() {
	return _savedStream;
    }

    /**When a manager method fetches a DAIOStream object from a row, it puts
     *the associated stream into the internal sequence. Use this method to direct that stream into the 
     *current output stream. If the current output is <samp>null</samp> and
     * the save parameter has the value <samp>true</samp>, the
     * method saves a reference to the input stream so that you can extract it later.
     *@param in the InputStream for the LOB in the current fetched row.
     *@param save Pass in <samp>true</samp> if you want to save the input stream reference if there 
     *is no output stream specified. You can direct the stream later. If you do not want to save the
     *input stream reference, pass in <samp>false</samp>.
     *@return an integer indicating the number of bytes read into the LOB. This will return 0 (zero) 
     *if <samp>in</samp> is <samp>null</samp>, or if the stream is saved to a reference rather than 
     *directed to output.
     */
    public int getFromLob(InputStream in, boolean save) {
	int wrCnt=0,lim=(_readLength<=0?Integer.MAX_VALUE:_readLength);
	if (save) {
	    _savedStream = in;
	} else {
	    OutputStream out = getOutputStreamOrFile();
	    if (out!=null ) try {
		byte[] buf = new byte[_blockSize];
		for (int rC=in.read(buf);rC>=0 && wrCnt<lim;wrCnt+=rC,rC=in.read(buf))
		    out.write(buf,0,rC);
		out.close();
		_out = null;
	    } catch(IOException ex) {
		ex.printStackTrace();
	    }
	}
	_bytesWritten = wrCnt;
	return wrCnt;
    }

    /**Use this method to  return the input stream that is associated with the
     * current LOB.
     *@return the InputStream object associated with the current DAIOStream object.
     */
    public InputStream getInputStream() {
	return _in;
    }

    /**Use this method to  set the input stream that is associated with the
     * current LOB.
     *@param in the InputStream object to associate with the current DAIOStream object.
     */
    public void setInputStream(InputStream in) {
	_in = in;
    }

    /**Manager uses this method to get the InputStream object associated with the current DAIOStream 
     *object. If that object is <samp>null</samp> and input file name is not <samp>null</samp>,
     *then FileInputStream is created.
     *@return the InputStream object associated with the current DAIOStream object.
     */
    public InputStream getInputStreamOrFile() {
	if (_in!=null)
	    return _in;
	if (_savedStream != null)
	    return _savedStream;
	if (_fileName != null && _fileName.trim().length()>0)
	    try {
		File f = new File(_fileName);
		setReadLength((int)f.length());
		_in = new FileInputStream(_fileName);
		return _in;
	    } catch (IOException ex) {
		ex.printStackTrace();
	    }
	return null;
    }

    /**Use this method to return the name of the output file.It returns an empty
     * string if the current file setting is <samp>null</samp>.
     *@return a string indicating the name of the current output file.
    */
    public String getFileName() {
	return _fileName;
    }

    /**Use this method to set the name of the current output file.
     *@param fileName a string indicating the name of the  output file.
     */
    public void setFileName(String fileName) {
	_fileName = fileName;
    }


    /**Use this method to get OutputStream object associated with this DAIOStream object.
     *@return an OutputStream object assosiated with this DAIOStream object.
     */
    public OutputStream getOutputStream() {
	return _out;
    }

    /**Use this method to set OutputStream object associated with this DAIOStream object.
     *@param out an OutputStream object that gets associated with this DAIOStream object.
     */
    public void setOutputStream(OutputStream out) {
	_out = out;
    }


    /**Use this method to get the length if data, in bytes, that will be read from
     * the current DAIOStream object into the output stream.
     *@return an integer indicating the current setting of the read length.
     */
    public int getReadLength() {
	return _readLength;
    }

    /**Use this method to specify the length of data, in bytes, that you want to
     * read from the current DAIOStream object into the output stream. If you set the read length to 0 
     *(zero), all the data in the DAIOStream object will be directed to the output stream.
     *@param readLen an integer indicating the desired read length, in bytes.
     */
    public void setReadLength(int readLen) {
	_readLength = readLen;
    }

    /**Manager uses this method to get OutputStream object associated with this DAIOStream object.
     *If that object is <samp>null</samp> and output file name is not <samp>null</samp>, 
     *then FileOutputStream is created.
     *@return an OutputStream object associated with this DAIOStream object.
     */
    public OutputStream getOutputStreamOrFile() {
	OutputStream res = _out;
	if (_out==null) try {
	    if (_fileName!=null && _fileName.trim().length()>0)
		_out= res = new FileOutputStream(_fileName);
	} catch (IOException ex) {
	    ex.printStackTrace();
	}
	return res;
    }


}
