/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.registry;

import com.jniwrapper.FunctionExecutionException;
import com.jniwrapper.Parameter;
import com.jniwrapper.UInt32;
import com.jniwrapper.WideString;
import com.jniwrapper.win32.Handle;

/**
 * Win32 specific code for reading/writing registry data.
 *
 * @author Alexei Vidmich
 *
 * @deprecated Use RegistryKey instead
 */
public class RegistryItem
{
    public static int HKEY_CLASSES_ROOT = 0x80000000;
    public static int HKEY_CURRENT_USER = 0x80000001;
    public static int HKEY_LOCAL_MACHINE = 0x80000002;
    public static int HKEY_USERS = 0x80000003;
    public static int HKEY_PERFORMANCE_DATA = 0x80000004;
    public static int HKEY_CURRENT_CONFIG = 0x80000005;
    public static int HKEY_DYN_DATA = 0x80000006;

    /*
     #define KEY_READ                ((STANDARD_RIGHTS_READ       |\     0x00020000L
     KEY_QUERY_VALUE            |\         0x0001
     KEY_ENUMERATE_SUB_KEYS     |\         0x0008
     KEY_NOTIFY)                 \         0x0010
     &                           \
     (~SYNCHRONIZE))                    0x00100000L
     */
    protected static long KEY_READ = 0x00020019L;

    /*
     #define KEY_WRITE               ((STANDARD_RIGHTS_WRITE      |\     0x00020000L
     KEY_SET_VALUE              |\         0x0002
     KEY_CREATE_SUB_KEY)         \         0x0004
     &                           \
     (~SYNCHRONIZE))                    0x00100000L
     */
    protected static long KEY_WRITE = 0x00020006L;

    /*
     *  #define ERROR_SUCCESS                    0L
     */
    protected static long ERROR_SUCCESS = 0L;

    /*
     *  #define REG_OPTION_NON_VOLATILE     (0x00000000L)
     */
    protected static long REG_OPTION_NON_VOLATILE = 0x00000000L;

    /*
     *   #define REG_NONE                    ( 0 )   // No value type
     *   #define REG_SZ                      ( 1 )   // Unicode nul terminated string
     *   #define REG_EXPAND_SZ               ( 2 )   // Unicode nul terminated string
     *                                               // (with environment variable references)
     *   #define REG_BINARY                  ( 3 )   // Free form binary
     *   #define REG_DWORD                   ( 4 )   // 32-bit number
     *   #define REG_DWORD_LITTLE_ENDIAN     ( 4 )   // 32-bit number (same as REG_DWORD)
     *   #define REG_DWORD_BIG_ENDIAN        ( 5 )   // 32-bit number
     *   #define REG_LINK                    ( 6 )   // Symbolic Link (unicode)
     *   #define REG_MULTI_SZ                ( 7 )   // Multiple Unicode strings
     *   #define REG_RESOURCE_LIST           ( 8 )   // Resource list in the resource map
     *   #define REG_FULL_RESOURCE_DESCRIPTOR ( 9 )  // Resource list in the hardware description
     *   #define REG_RESOURCE_REQUIREMENTS_LIST ( 10 )
     */
    public static final int REG_SZ = 1;
    public static final int REG_BINARY = 3;
    public static final int REG_DWORD = 4;
    public static final int REG_MULTI_SZ = 7;

    public static final int REG_CREATED_NEW_KEY = 0x00000001;   // New Registry Key created
    public static final int REG_OPENED_EXISTING_KEY = 0x00000002;   // Existing Key opened

    private int _hKey;
    private String _path = null;
    private String _name = null;
    private int _type = REG_SZ;      // default is String
    private Handle _handle = new Handle();
    private Parameter _value = null;

    /**
     * Constructor with maximum parameters to initialize
     * This constructor is used for reading/wriiting values to Registry
     */
    public RegistryItem(int hKey, String path, String name, Parameter value, int type)
    {
        setHKey(hKey);
        setPath(path);
        setName(name);
        setValue(value);
        setType(type);
    }

    /**
     * Constructor for writing Strings to Registry
     */
    public RegistryItem(int hKey, String path, String name, String value)
    {
        this(hKey, path, name, new WideString(value), REG_SZ);
    }

    /**
     * Constructor for reading Strings
     */
    public RegistryItem(int hKey, String path, String name, int maxValueLength)
    {
        this(hKey, path, name, new WideString(maxValueLength), REG_SZ);
    }

    /**
     * read value of the Registry item
     */
    public void read()
    {
        try
        {
            openKey(new UInt32(KEY_READ));
            queryKey();
        }
        finally
        {
            closeKey();
        }
    }

    /**
     * write value of the Registry item
     */
    public void write()
    {
        try
        {
            openKey(new UInt32(KEY_WRITE));
            saveKey();
        }
        finally
        {
            closeKey();
        }
    }

    /**
     * removes the item from the Registry
     */
    public void delete()
    {
        try
        {
            openKey(new UInt32(KEY_WRITE));
            deleteKey();
        }
        finally
        {
            closeKey();
        }
    }

    /**
     * Opens registry key handle for reading or writing using this API function
     *   RegCreateKeyEx(...);
     * @param accessType constant that specifies type of access
     */
    private void openKey(UInt32 accessType)
    {
        long resultValue = WinRegistry.createKey(
                getHKey(),
                getPath(),
                REG_OPTION_NON_VOLATILE,
                accessType.getValue(),
                _handle);

        if (ERROR_SUCCESS != resultValue)
        {
            throw new RegistryException("Can't open registry key: " + toString() + ". Error number: " + resultValue);
        }
    }

    /**
     * closes registry key handle
     */
    private void closeKey()
    {
        long resultValue;
        try
        {
            resultValue = WinRegistry.closeKey(_handle);
        }
        catch (FunctionExecutionException e)
        {
            throw new RegistryException(e);
        }

        if (ERROR_SUCCESS != resultValue)
        {
            throw new RegistryException("Error while closing the registry key for " + toString() + " with error code:" + resultValue);
        }
    }

    /**
     * retrieves Value string from the registry for this Registry Item
     */
    private void queryKey()
    {
        long resultValue;
        try
        {
            resultValue = WinRegistry.getValue(_handle, getName(), getType(), getValue());
        }
        catch (FunctionExecutionException e)
        {
            throw new RegistryException(e);
        }

        if (ERROR_SUCCESS != resultValue)
        {
            throw new RegistryException("Error while reading the registry key for " + toString() + " with error code:" + resultValue);
        }
    }

    private void saveKey()
    {
        long resultValue;
        try
        {
            resultValue = WinRegistry.setValue(_handle, getName(), getType(), getValue(), getValueSize());
        }
        catch (FunctionExecutionException e)
        {
            throw new RegistryException(e);
        }
        if (ERROR_SUCCESS != resultValue)
        {
            throw new RegistryException("Error while setting the registry key for " + toString() + " with error code:" + resultValue);
        }
    }

    private void deleteKey()
    {
        long resultValue;
        try
        {
            resultValue = WinRegistry.deleteKey(_handle, getName());
        }
        catch (FunctionExecutionException e)
        {
            throw new RegistryException(e);
        }
        if (ERROR_SUCCESS != resultValue)
        {
            throw new RegistryException("Error while deleting the registry key for " + toString() + " with error code:" + resultValue);
        }
    }

    /**
     * count data size
     * this implementation works for Strings only
     */
    private int getValueSize()
    {
        return getValue().getLength();
    }

    public int getHKey()
    {
        return _hKey;
    }

    public void setHKey(int hKey)
    {
        _hKey = hKey;
    }

    public String getPath()
    {
        return _path;
    }

    public void setPath(String path)
    {
        _path = path;
    }

    public String getName()
    {
        return _name;
    }

    public void setName(String name)
    {
        _name = name;
    }

    public Parameter getValue()
    {
        return _value;
    }

    public void setValue(Parameter value)
    {
        _value = value;
    }

    public int getType()
    {
        return _type;
    }

    public void setType(int type)
    {
        switch (type)
        {
            case REG_BINARY:
            case REG_DWORD:
            case REG_MULTI_SZ:
            case REG_SZ:
                {
                    _type = type;
                    break;
                }
            default:
                {
                    throw new IllegalArgumentException("The type: " + type + " isn't supported.");
                }
        }
    }

    public String toString()
    {
        String result = getPath() + "->" + getName();
        return result;
    }
}
