/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.ui.dialogs;

import com.jniwrapper.*;
import com.jniwrapper.util.FlagSet;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.shell.Shell32;
import com.jniwrapper.win32.shell.ShellFolder;
import com.jniwrapper.win32.VersionInfo;
import com.jniwrapper.win32.ui.WindowTools;
import com.jniwrapper.win32.ui.Wnd;

import java.awt.*;
import java.io.File;

/**
 * SelectFolderDialog is a wrapper to the native BrowseForFolder dialog
 *
 * @author Serge Piletsky
 *
 * todo: implement initial folder selection (COM support is required)
 */
public class SelectFolderDialog
{
    static final FunctionName FUNCTION_BROWSE_FOR_FOLDER = new FunctionName("SHBrowseForFolder");

    private ShellFolder _startFolder = ShellFolder.DESKTOP;
    private Window _owner = null;
    private String _title = "";
    private String _folder = "";
    private SelectFolderDialogOptions _flags = new SelectFolderDialogOptions();

    public SelectFolderDialog()
    {
        reset();
    }

    public SelectFolderDialog(String caption)
    {
        this();
        setTitle(caption);
    }

    public SelectFolderDialog(Window owner)
    {
        setOwner(owner);
        reset();
    }

    public SelectFolderDialog(Window owner, String caption)
    {
        this(owner);
        setTitle(caption);
    }

    protected void reset()
    {
        _flags.clear();
    }

    public String getTitle()
    {
        return _title;
    }

    /**
     * Sets captions that will be displayed on SelectFolderDialog
     * @param title
     */
    public void setTitle(String title)
    {
        _title = title;
    }

    /**
     * @return selected folder
     */
    public String getFolder()
    {
        return _folder;
    }

    /**
     * Full path to the folder selected by the user
     * @param folder
     */
    public void setFolder(String folder)
    {
        _folder = folder;
    }

    public Window getOwner()
    {
        return _owner;
    }

    public void setOwner(Window owner)
    {
        _owner = owner;
    }

    /**
     * @return {@link FlagSet} contains dialog options.
     */
    public SelectFolderDialogOptions getOptions()
    {
        return _flags;
    }

    /**
     * Location of the root folder from which to start browsing. Only the specified
     * folder and any folders beneath it in the namespace hierarchy  appear
     * in the dialog box.
     * @return start location folder id
     */
    public ShellFolder getStartFolder()
    {
        return _startFolder;
    }

    /**
     * Location of the root folder from which to start browsing. Only the specified
     * folder and any folders beneath it in the namespace hierarchy  appear
     * in the dialog box.
     * @param startFolder is folder id
     */
    public void setStartFolder(ShellFolder startFolder)
    {
        _startFolder = startFolder;
    }

    /**
     * Callback for SelectFolderFialog
     */
    protected Callback getDialogCallback()
    {
        return null;
    }

    /**
     * Opens SelectFolderDialog
     * @return true if folder selected
     */
    public boolean execute()
    {
        Wnd owner = _owner != null? new Wnd(WindowTools.getWindowHandle(_owner)): new Wnd();
        final int folderID = _startFolder.getFolderID();
        Handle root = ShellFolder.getFolderIDList(folderID);
        if (root.isNull())
            return false;
        BIStructure browseInfoStructure = new BIStructure(getDialogCallback(), root);
        browseInfoStructure.setOwner(owner);
        browseInfoStructure.setTitle(getTitle());
        browseInfoStructure.setFlags(_flags.getFlags());
        File directory = new File(_folder);
        if (!directory.exists())
        {
            _folder = "";
        }
        if (_folder.length() > 0)
        {
            final ZeroTerminatedString dir = Shell32.getInstance().stringParam(_folder);
            browseInfoStructure.setParam(dir);
        }
        final Function browseForFolder = Shell32.getInstance().getFunction(FUNCTION_BROWSE_FOR_FOLDER.toString());
        Handle returnValue = new Handle();
        if (_owner != null)
            DialogHelper.invokeDialog(_owner, browseForFolder, returnValue, new Parameter[] {new Pointer(browseInfoStructure)});
        else
            browseForFolder.invoke(returnValue, new Pointer(browseInfoStructure));
        if (returnValue.isNull())
            return false;
        boolean result = !returnValue.isNull();
        if (result)
        {
            String selectedFolder = ShellFolder.getPathFromIDList(returnValue);
            if (selectedFolder.length() == 0 && _flags.contains(SelectFolderDialogOptions.BROWSEFORCOMPUTER))
            {
                _folder = "\\\\" + browseInfoStructure.getDisplayName();
            }
            else
            {
                _folder = selectedFolder;
            }
        }
        return result;
    }

    /**
     * Factory method that creates dialog, customized to browse computers.
     */
    public static SelectFolderDialog createBrowseForComputersDialog()
    {
        SelectFolderDialog dialog = new SelectFolderDialog();
        final SelectFolderDialogOptions flags = dialog.getOptions();
        flags.clear();
        flags.add(SelectFolderDialogOptions.NEWDIALOGSTYLE);
        flags.add(SelectFolderDialogOptions.NONEWFOLDERBUTTON);
        flags.add(SelectFolderDialogOptions.BROWSEFORCOMPUTER);
        dialog.setStartFolder(ShellFolder.NETWORK);
        return dialog;
    }

    /**
     * Factory method that creates dialog, customized to browse printers.
     */
    public static SelectFolderDialog createBrowseForPrintersDialog()
    {
        SelectFolderDialog dialog = new SelectFolderDialog();
        final SelectFolderDialogOptions flags = dialog.getOptions();
        flags.clear();
        // this hack allows to select printer in Win2k properly
        VersionInfo versionInfo = new VersionInfo();
        if (!versionInfo.isWin2k())
            flags.add(SelectFolderDialogOptions.NEWDIALOGSTYLE);
        flags.add(SelectFolderDialogOptions.NONEWFOLDERBUTTON | SelectFolderDialogOptions.BROWSEFORPRINTER);
        dialog.setStartFolder(ShellFolder.NETWORK);
        return dialog;
    }

    public class SelectFolderDialogOptions extends FlagSet
    {
        public static final int RETURNONLYFSDIRS   = 0x0001;  // For finding a folder to start document searching
        public static final int DONTGOBELOWDOMAIN  = 0x0002;  // For starting the Find Computer
        public static final int STATUSTEXT         = 0x0004;
        public static final int RETURNFSANCESTORS  = 0x0008;
        public static final int EDITBOX            = 0x0010;
        public static final int VALIDATE           = 0x0020;  // insist on valid result (or CANCEL)
        public static final int NEWDIALOGSTYLE     = 0x0040;
        public static final int BROWSEINCLUDEURLS  = 0x0080;
        public static final int BROWSEFORCOMPUTER  = 0x1000;  // Browsing for Computers
        public static final int BROWSEFORPRINTER   = 0x2000;  // Browsing for Printers
        public static final int BROWSEINCLUDEFILES = 0x4000;  // Browsing for Everything
        public static final int SHAREABLE          = 0x8000;
        public static final int NONEWFOLDERBUTTON  = 0x0200;

        public SelectFolderDialogOptions()
        {
            super();
            reset();
        }

        public void reset()
        {
            clear();
            add(RETURNONLYFSDIRS | DONTGOBELOWDOMAIN | NEWDIALOGSTYLE);
        }

        /**
         * Only return file system directories. If the user selects folders
         * that are not part of the file system, the OK button is unavailable.
         * @return
         */
        public boolean isOnlyFilesystem()
        {
            return contains(RETURNONLYFSDIRS);
        }

        public void setOnlyFileSystem(boolean onlyFileSystem)
        {
            if (onlyFileSystem)
                add(RETURNONLYFSDIRS);
            else
                remove(RETURNONLYFSDIRS);
        }

        /**
         *
         * @return if dialog set to use new dialog style
         */
        public boolean isNewDialogStyle()
        {
            return contains(NEWDIALOGSTYLE);
        }

        /**
         * Use the new user interface.
         * Setting this flag provides the user with a larger dialog box that can be resized
         * @param newDialogStyle
         */
        public void setNewDialogStyle(boolean newDialogStyle)
        {
            if (newDialogStyle)
                add(NEWDIALOGSTYLE);
            else
                remove(NEWDIALOGSTYLE);
        }

        /**
         *
         * @return true if dialog set to display files as well as folders
         */
        public boolean isSelectFiles()
        {
            return contains(BROWSEINCLUDEFILES);
        }

        /**
         * The browse dialog box will display files as well as folders.
         * @param selectFiles
         */
        public void setSelectFiles(boolean selectFiles)
        {
            if (selectFiles)
                add(BROWSEINCLUDEFILES);
            else
                remove(BROWSEINCLUDEFILES);
        }

        /**
         * @return true if dialog set to include text control
         */
        public boolean isShowTextBox()
        {
            return contains(EDITBOX);
        }

        /**
         * Include an edit control in the browse dialog box that allows the user to type the name of an item
         * @param showTextBox
         */
        public void setShowTextBox(boolean showTextBox)
        {
            if (showTextBox)
                add(EDITBOX);
            else
                remove(EDITBOX);
        }

        /**
         *
         * @return true if dialog set to exclude new folder button
         */
        public boolean isShowNewFolderButton()
        {
            return !contains(NONEWFOLDERBUTTON);
        }

        /**
         * Do not include the New Folder button in the browse dialog box
         * @param showNewFolderButton
         */
        public void setShowNewFolderButton(boolean showNewFolderButton)
        {
            if (showNewFolderButton)
                remove(NONEWFOLDERBUTTON);
            else
                add(NONEWFOLDERBUTTON);
        }

        /**
         * @return true if dialog set to show network folder
         */
        public boolean isShowNetworkFolders()
        {
            return !contains(DONTGOBELOWDOMAIN);
        }

        /**
         * Show network folders
         * @param showNetworkFolders
         */
        public void setShowNetworkFolders(boolean showNetworkFolders)
        {
            if(showNetworkFolders)
                remove(DONTGOBELOWDOMAIN);
            else
                add(DONTGOBELOWDOMAIN);
        }

        /**
         * @return true if dialog specified to show files
         */
        public boolean isShowFiles()
        {
            return contains(BROWSEINCLUDEFILES);
        }

        /**
         * Show files
         * @param showFiles
         */
        public void setShowFiles(boolean showFiles)
        {
            if (showFiles)
                add(BROWSEINCLUDEFILES);
            else
                remove(BROWSEINCLUDEFILES);
        }
    }
}
