package cern.lhcias.csgui.Services;

import cern.lhcias.csgui.interfaces.*;
import cern.lhcias.csgui.Utils.*;
import cern.lhcias.csgui.WinMgr.*;
import cern.lhcias.csgui.Events.*;
import java.util.*;
import java.awt.*;
import java.net.*;
import java.io.*;

/**
*   This class handles all the configurations of the TagsView and the windows which
*   contains them.
*   A "window configuration" contains the description of the window itself, the type of
*   the TagsView to create, a "TagsView configuration" and informations on how to
*   instanciate the TagsView.
*   It can also be a set of "window configurations", thus enabling to load multiple 
*   windows with a single action.
*   A list of window configurations is maintened on the server. To each window
*   configuration is associated a name. This list may be retrieved with the
*   method loadWindowConfigurationList .
*   It is possible to add a new window configuration to this list using 
*   the publishWindowConfiguration method.
*   A TagsView configuration is, depending on the TagsView, either a String or a
*   pointer (such as an URL) to a remote configuration file. 
*   This TagsView configuration will be passed to the TagsView upon creation.
*   If it is a pointer to a remote configuration file, it is up to the TagsView to
*   load the configuration file. This can be done using the loadConfigurationFile method.
*   The TagsViews can use saveConfigurationFile to save their configurations.
*   If the TagsView uses the loadInitConfigurationFile method to load a configuration
*   file, then the initializations of the tags in the real-time database and the 
*   initializations of the dataservers will be done automatically before returning. This
*   implies that the configuration file uses the standard way of defining tags and
*   dataservers.
*   
*   This implementation uses a Web server and a set of remote programs 
*   to offer the service.
*/
public class configurationMgr {
    /** String used in configuration files to define lines which contain
    *   some tags' configuration.
    */
    public final static String TAGDEF      = "#TagDef";
    
    /** String used as the ViewType name of DataBase configuration files.
    */
    public final static String DBCONFIG    = "DBConfig";
    
    /** String used in configuration files to define lines which contain
    *   some DataServers' configuration.
    */
    public final static String DATASERVER  = "#DataServer";
    
    /** statement used in configuration files to include other configuration
    *   files.
    */
    public final static String INCLUDE     = "#include";

    String  remoteCGI;
    String  publishURL;
    String  configurationList;
    TagArrays TA;
    CentralManager gl;

	// ------------- Event Handling -------------------------------------
    Vector myListeners  = new Vector();

    public synchronized void addOpenViewListener( OpenViewListener listener) {
        myListeners.addElement(listener);
    }
    
    public synchronized void removeOpenViewListener(OpenViewListener listener) {
        myListeners.removeElement(listener);
    }
    
    void fireOpenViewEvent(String viewType,boolean newWindowFlag, String config, int w,int h)
    {
        OpenViewEvent openvevt = new OpenViewEvent(this);
        openvevt.setViewType(viewType);
        openvevt.setNewWindowFlag(newWindowFlag);
        openvevt.setConfig(config);
        openvevt.setWidth(w);
        openvevt.setHeight(h);
        for (int i=0; i<myListeners.size(); i++) {
            ((OpenViewListener) myListeners.elementAt(i)).receiveOpenView(openvevt);
        }
    }

    /**
    * @param postCGIPg the URL of the pg needed to save the configuration file.
    * @param PublishURL the URL of the pg needed to add a window configuration.
    * @param ConfigurationList the URL of the list of window configurations.
    * @param ta the TagArrays used to store the Tags.
    */
    public configurationMgr(String postCGIPg, String PublishURL,
                            String ConfigurationList, TagArrays ta, CentralManager cm) {
        publishURL = PublishURL;
        remoteCGI = postCGIPg;
        configurationList = ConfigurationList;
        TA = ta;
        gl = cm;
    }

    /**
    * Loads the configuration file named configName. Here, it is assumed that the 
    * configName is a URL.
    * So it reads the file located at the URL configName and returns the content 
    * in a Vector.
    */
    public Vector loadConfigurationFile(String configName) {
        return MyUtils.ReadFile(configName);
    }

    /** Loads a configuration file and initialize all Tags and DataServers found
    *   in it before returning the configuration.
    *   It calls initDataServers and initDataBase
    */
    public Vector loadInitConfigurationFile(String configName) {
        initDataServers(configName);
        initDataBase(configName);
        return loadConfigurationFile(configName);
    }

    /** 
    * Loads the configuration file located at "adresse" and returns a Vector of
    *      Strings representing the content of the file. If the vector Values
    *      is not null, it will substitue parts of the file according to what is
    *      described in the vector (see method substitutes).
    *      If recurse is set to true, then recursive reading is done for the lines 
    *      which starts with recurseHeader. The following syntax is assumed:
    *          recurseHeader original_substring1 substringToReplace1 original_substring2 substringToReplace2 ...
    *      If toEvaluate is true then the method will try to evaluate encountered
    *      expressions.
    *      It also calls initDataServers and initDataBase
    */
    public Vector loadIncSubstConfigurationFile(String adresse, Vector Values, 
                boolean recurse, boolean toEvaluate) {
       initDataServers(adresse);
       initDataBase(adresse);
       return MyUtils.recurseReadSubstituteFile(adresse, Values,recurse, 
                INCLUDE, toEvaluate);
    }


	/** 
	* Saves a configuration file on a Web server.
	* If configName is null and is "", then a name will be created for this configuration.
	* The correct configName is returned to the calling method.
	*/
	public String saveConfigurationFile(String configName, String[] configLines) {
        if (remoteCGI != null)
    	    return MyUtils.postStringArrays(remoteCGI, configName, configLines);
    	else return null;
	}

    /**
    * Adds the configuration of a TagsView in the list of known configurations
    * together with windowing information (current size of tv).
    */
    public void publishWindowConfiguration(TagsView tv, String confName) {
			Rectangle rec = tv.getBounds();
			String config = confName+"%20width="+rec.width+"%20height="+rec.height+
			                "%20type="+tv.getType()+"%20config="+tv.toString();
			MyUtils.execute_URL(publishURL+"?"+config); 
    }
    
    
    /**
	*      This method reads the configuration file config and search for lines starting
	*      with TAGDEF. When it encounters such a line, it sends
	*      the remaining part of the line to TagArrays.addConfig()
    */
	public void initDataBase(String config) {
        URL exec_URL;
        MyUtils MyUtils;
        
		try {
			exec_URL = symantec.itools.net.RelativeURL.getURL(config);
		} catch (MalformedURLException e) {
			System.out.println("BAD URL:[" + config + "]\n");
			return;
		}
		try {
			InputStream IS = exec_URL.openStream();
			BufferedReader DIS = new BufferedReader(new InputStreamReader(IS));
			// DataInputStream DIS = new DataInputStream(IS);
			boolean flag = true;
			String ligne;

			while (flag) {
				try {
					ligne = DIS.readLine();
					MyUtils = new MyUtils(ligne);
					if (ligne == null)
						flag=false;
					else
						if (ligne.startsWith(TAGDEF)) {
						   try {
                               String config2=ligne.substring(TAGDEF.length()+1);
                               config2.trim();
                               TA.addConfig(config2);
                           } catch (Exception e) {}
						}
				}catch (IOException e) {
					flag = false;
				}
			}
			IS.close();
 	    }catch (IOException e) {
			System.out.println("ERROR URL:[" + config + "]\n");
		}
	}

    /**
	*      This method reads the configuration file config and search for lines starting
	*      with DATASERVER. When it encounters such a line, it sends
	*      the remaining part of the line to CentralManager.setDataServer()
    */
	public void initDataServers(String config) {
        URL exec_URL;
        MyUtils MyUtils;
        
		try {
			exec_URL = symantec.itools.net.RelativeURL.getURL(config);
		} catch (MalformedURLException e) {
			System.out.println("BAD URL:[" + config + "]\n");
			return;
		}
		try {
			InputStream IS = exec_URL.openStream();
			BufferedReader DIS = new BufferedReader(new InputStreamReader(IS));
			// DataInputStream DIS = new DataInputStream(IS);
			boolean flag = true;
			String ligne;

			while (flag) {
				try {
					ligne = DIS.readLine();
					MyUtils = new MyUtils(ligne);
					if (ligne == null)
						flag=false;
					else
						if (ligne.startsWith(DATASERVER)) {
						   try {
                               String config2=ligne.substring(DATASERVER.length());
                               config2.trim();
                               gl.setDataServer(config2);
                           } catch (Exception e) {}
						}
				}catch (IOException e) {
					flag = false;
				}
			}
			IS.close();
 	    }catch (IOException e) {
			System.out.println("ERROR URL:[" + config + "]\n");
		}
	}

    /**
    *   Given a configuration description, it opens the correct TagsView(s).
    *   If config is a configuration of type "MultipleWindows", then the
    *   different TagsViews are created.
    *   @param config the configuration description
    *   @param configurations the list of configurations (names + descriptions) as
    *       loaded by loadConfigurationList()
    *   @param newWindow if true, then TagsView is created in a new window.
    */
    public void openMultipleViews(String config, Vector configurations, 
                                  boolean newWindow) {
        int index = config.indexOf("MultipleWindows");
        if (index > -1) {
                config = config.substring(index+16);
                int i=1;
                String window = MyUtils.get_token(config, i);
                while (window != null) {
                    int index2 = configurations.indexOf(window);
                    if (index2 > -1) {
                        String conf = (String) configurations.elementAt(index2+1);
                        openView(conf, newWindow);
                    }
                    i++;
                    window = MyUtils.get_token(config, i);
                }
                
            }
        else
			    openView(config, newWindow);
	}

    /**
    *   Given a configuration description, it opens the correct TagsView.
    *   @param config the configuration description
    *   @param newWindow if true, then TagsView is created in a new window.
    */
	public void openView(String config, boolean newWindow) {
	        int index;
            int w=0, h=0;
            String viewType = "";
            String configPart;

			Vector configParameters = MyUtils.getPropertiesinConfig(config,' ');
            // -------- Do we have a simple config ? -----------
			index = configParameters.indexOf("config");
			if (index > -1)
			    configPart = (String) configParameters.elementAt(index+1);
			else {
			    configPart = config;
    			// ---------- It is a file -------------------------
    			String adresse;
                if (configPart.indexOf('|')>-1)
                    adresse = config.substring(0,config.indexOf('|'));
                else 
                    adresse = config;
	    		Vector ViewType = MyUtils.getFileParam("#ViewType", adresse);
                if (ViewType == null) return;
                if (ViewType.size() == 0) return;
                viewType = (String) ViewType.elementAt(0);
                Vector Size = MyUtils.getFileParam("#Size", adresse);
                if (Size != null) {
                    if (Size.size()>0) w = (new Integer((String) Size.elementAt(0))).intValue();
                    if (Size.size()>1) h = (new Integer((String) Size.elementAt(1))).intValue();
                }
            }
	
			index = configParameters.indexOf("type");
			if (index > -1)
			  viewType = (String) configParameters.elementAt(index+1);
			index = configParameters.indexOf("width");
			if (index > -1)
			  w = (new Integer((String) configParameters.elementAt(index+1))).intValue();
			index = configParameters.indexOf("height");
			if (index > -1)
			  h = (new Integer((String) configParameters.elementAt(index+1))).intValue();
			
            fireOpenViewEvent(viewType,newWindow, configPart, w, h);
	}
	
    /** 
    *   loads the list of window configurations in a Vector, in the following way:
    *       name1,window conf1,name2,window conf2,...
    */
    public Vector loadWindowConfigurationList(){
        Vector configurations = new Vector();
        URL exec_URL;
        MyUtils MyUtils;
        
		try {
			exec_URL = symantec.itools.net.RelativeURL.getURL(configurationList);
		} catch (MalformedURLException e) {
			System.out.println("BAD URL:[" + configurationList + "]\n");
			return configurations;
		}
		try {
			InputStream IS = exec_URL.openStream();
			DataInputStream DIS = new DataInputStream(IS);
			boolean flag = true;
			String ligne;

			while (flag) {
				try {
					ligne = DIS.readLine();
					MyUtils = new MyUtils(ligne);
					if (ligne == null)
						flag=false;
					else {
						if (!ligne.startsWith("#")) {
						     String name = MyUtils.get_token(1);
						     String adresse = ligne.substring(name.length()+1);
						     if (name!=null && adresse!=null) {
	                         	 configurations.addElement(name);
	                         	 configurations.addElement(adresse);
	                         }
						}
					}
				}catch (IOException e) {
					flag = false;
				}
			}
			IS.close();
 	    }catch (IOException e) {
			System.out.println("ERROR URL:[" + configurationList + "]\n");
		}
        return configurations;
	}
}