
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * An include directive is used to include text from another file. It has 
  * two forms: #include and #parse. #include reads in the supplied file 
  * as and includes it as a complete string, without parsing it. #parse
  * reads in the supplied file and parses it, as though its contents had
  * appeared here directly. 
  * <p>
  * Note that a variable can be used to determine which file to read in,
  * so that in practice the named file may not be known until run time.
  * <p>
  * Here are some examples:<pre>
  *
  *     #include "unparsed.txt"
  *     #include "$variableUnparsedFile"
  *     #parse   "includeMacros.wm"
  *     #parse   "some$VariableMacro"
  *
  * </pre>
  */
class IncludeDirective implements Macro
{

   /**
     * The filename we want to include--may be a macro
     */
   final private Object fileName;

   /**
     * True if we were told to parse the included text
     */
   final private boolean parse;

   /**
     * Create an include directive. Note that the supplied variable
     * may be a macro, in which case it is evaluated to determine the
     * actual filename (it will be evaluated at run time, once 
     * given a context).
     * <p>
     * @param fileName an object representing the file to read in
     * @param parseFile true if we are to parse the included text
     */
   IncludeDirective(Object fileName, boolean parseFile) {
      this.fileName = fileName;
      parse = parseFile;
   }

   /**
     * Parse the directive. We assume the "#" has already been eaten.
     * <p>
     * @exception ParseException if an unrecoverable parse error occurred
     * @exception IOException if we could not successfully write to out
     */
   static public final Object parse(ParseTool in) 
      throws IOException, ParseException
   {
 
      /**
        * Whether we will construct a parsing include directive or not
        */
      boolean parse;

      /**
        * What filename to give to the include directive we construct
        */
      Object fileName;

      if (in.ttype != in.TT_WORD) {
         if (Engine.debug) {
            Engine.log.debug("Include: next token is not a word");
         }
         return null;
      }

      if (in.sval.equals("include")) { // read file in as a string string 
         if (Engine.debug) {
            Engine.log.debug("Include: including unparsed file");
         }
         parse = false;
      } else if (in.sval.equals("parse")) { // parse file contents as a block
         if (Engine.debug) {
            Engine.log.debug("Include: including parsed file");
         }
         parse = true;
      } else {
         if (Engine.debug) {
            Engine.log.debug("Include: " + in.sval + " unrecognized.");
         }
         return null;
      }

      in.nextToken(); // eat the word "include" or "parse"
      if (! in.parseSpaces()) {
         throw new ParseException(in,
             "Parsing include, expected spaces after keyword, but got:" 
             + (char) in.ttype + " (spaces should separate directive from filename)");
      }
      
      fileName = Term.parse(in);
      if (fileName == null) {
         throw new ParseException(in,
               "Parsing include directive, expected filename term"
               + " but what followed was not a term....");
      }
      in.parseSpaces(); // spaces after term mean nothing

      if (in.ttype != in.TT_EOL && in.ttype != in.TT_EOF) {
         throw new ParseException(in,
               "Parsing include directive. Got junk after directive:" 
               + (char) in.ttype + " (nothing should follow on the same line)");
      }
      return new IncludeDirective(fileName, parse);
   }

   /**
     * Apply the values in the supplied context to evaluate the 
     * include filename; then read that file in and return it as 
     * a string. If we are a parsing directive then the file will
     * be parsed before it is returned as a string.
     * @exception InvalidContextException is required data is missing
     */ 
   final public Object evaluate(Object context)
      throws InvalidContextException
   {
      try {
         StringWriter sw = new SizedStringWriter(512);
         write(sw,context);
         return sw.toString();
      } catch(IOException e) {
         Engine.log.exception(e);
         Engine.log.warning(
              "Include: evaluate got IO exception on write to StringWriter");
         return "";
      }
   }  

   /**
     * Apply the values in the supplied context to evaluate the 
     * include filename; then read that file in and write it out
     * to the supplied Writer. If we are a parsing directive the
     * file will be parsed before it is written.
     * <p>
     * @exception IOException if an error occurred with out
     * @exception InvalidContextException if required data was missing
     */
   final public void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {

      if (fileName == null) {
         Engine.log.error("Include: attempt to write with null filename");
         return;
      }

      // evaluate or toString to find the actual filename to use
      String fname;
      if (fileName instanceof Macro) {
         fname = ((Macro) fileName).evaluate(context).toString();
      } else {
         fname = fileName.toString();
      }
      if (Engine.debug) {
         Engine.log.debug("Include fname: " + fname);
      }

      try {
         Reader fstream = new FileReader(fname);
        
         if (parse) {
            ParseTool in = new ParseTool(fname,fstream); 
            try {
               in.nextToken(); // must start out with current token on top
               Block b = (Block) Block.parse(in);
               b.write(out,context);
            } catch (InvalidContextException e) {
               String emsg = "Block.parse did not return a Block!";
               Engine.log.exception(e);
               Engine.log.error("Include: " + emsg);
               out.write("<!--\nERROR: " + emsg + " \n-->");
            }
         } else {
            int num;
            char buf[] = new char[2048]; // 2048 arbitrary buffer size
            while ( (num = fstream.read(buf)) > 0 ) {
               out.write(buf,0,num);
            }
         }
      } catch (ParseException e) {
         Engine.log.exception(e);
         String warning = "Include: file " + fileName + " did not parse!";
         Engine.log.warning(warning);
         out.write("<!--\nWARNING: " + warning + " \n-->");
      } catch (IOException e) {
         Engine.log.exception(e);
         String warning = "Include: Error reading file " + fname;
         Engine.log.warning(warning);
         out.write("<!--\nWARNING: " + warning + " \n-->");
      } 

   }

   /**
     * Test harness
     */
   final public static void main(String arg[]) {


      System.out.println("Testing IncludeDirective:");

      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);


      Map m = new HashMap();
      
      m.put("file","include.txt");
      m.put("fsfile","examples/include.txt");
      m.put("hello","contents of hello variable");


      String tests[] = { 
            "include \"examples/include.txt\"",
            "parse \"examples/$file\"",
            "parse $fsfile",
            "include examples/ttword"
         };


      System.out.println("- - - - - - - - - - - - - - - - -");
      for (int i = 0; i < tests.length; i++) {
         System.out.println("TESTING: " + tests[i] + " :");
         try {
            ParseTool iin = new ParseTool("string",new StringReader(tests[i]));
            iin.nextToken();
            IncludeDirective id = (IncludeDirective) IncludeDirective.parse(iin);
            if (id == null) {
               System.out.println("FAILED--null returned from parse().");
            } else {
               String result = id.evaluate(m).toString();
               System.out.println(result);
            }
         } catch (Exception e) {
            System.out.println("FAILED--threw exception:");
            e.printStackTrace();
            System.out.println();
         }
         System.out.println("- - - - - - - - - - - - - - - - -");
      }
      System.out.println();
      System.out.println("Done.");

   }
}

