
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import java.util.*;
import java.io.*;
import org.webmacro.util.*;
import java.lang.reflect.*;

/**
  * A list is a sequence of terms. It's used in two common cases:
  * the items in an array initializer; and the arguments to a 
  * method call. 
  */
public class List implements Macro
{

   final private Object[] _content; // the list data


   /**
     * Create a Macro representing a List. These evaluate to
     * type Objects[], after recursively resolving macros in the list.
     */
   final public static Object parse(ParseTool in) 
      throws ParseException, IOException
   {

      // find the start char, identify end char
      int endChar;
      switch (in.ttype) {
         case '(': endChar = ')'; break;
         case '[': endChar = ']'; break;
         default : return null;
      }

      // create the list object
      Vector newList = new Vector();
      in.nextToken(); // we ate it

      // eat spaces at the start of the list
      in.parseSpaces();

      Object term;
      boolean isMacro = false; // not a macro yet
      while ((term = Term.parse(in)) != null) {
         if (term instanceof Macro) {
            isMacro = true; // at least one Macro term
         }
         newList.addElement(term);
         in.parseSpaces(); 
         if (in.parseChar(',')) { in.parseSpaces(); }
      }

      // find the end character
      if (! in.parseChar((char) endChar) ) {
         throw new ParseException(in, "Expected end of list, instead got " + (char) in.ttype);
      }
      Object[] alist = new Object[ newList.size() ];
      newList.copyInto(alist);
      if (isMacro) {
         return new List(alist);
      } else {
         return alist;
      }
   }

   /**
     * create a new list
     */
   List(Object[] content) {
      _content = content;
   }

   public void write(Writer out, Object context)
      throws InvalidContextException, IOException
   {
      out.write(evaluate(context).toString());
   }

   public String toString()
   {
      StringBuffer sb = new StringBuffer();
      sb.append("(");
      for(int i =0; i<_content.length; i++) {
         if (i != 0) {
            sb.append(", ");
         }
         sb.append(_content[i].toString());
      }
      sb.append(")");
      return sb.toString();
   }
   public Object evaluate(Object context) 
      throws InvalidContextException
   {
      Object[] ret = new Object[ _content.length ];
      for (int i = 0; i < _content.length; i++) {
         Object m = _content[i];
         if (m instanceof Macro) {
            m = ((Macro) m).evaluate(context);
         }
         ret[i] = m;
      }
      return ret;
   }

   public static void main(String arg[]) {
  
      try {
         String testCase[] = { 
            "( one, two, \"three has text\", $four.has.fields, \'five has single quotes\' )",
            "[one two three]",
            "( )",
            "( one, two, three ]"
         };

         Hashtable context = new Hashtable();
         context.put("test","this is the value of test");
         for (int i = 0; i < testCase.length; i++) {
            System.out.println("Test case: " + testCase[i]);
            ParseTool in = new ParseTool(testCase[i],new StringReader(testCase[i]));
            in.nextToken();

            try {
               Object olist = List.parse(in);
               if (olist instanceof Macro) {
                  olist = ((Macro) olist).evaluate(context);
               }
               System.out.println(olist.getClass());
               Object[] list = (Object[]) olist;

               for (int j = 0; j < list.length; j++) {
                  System.out.println("  term: " + list[j]);
               }
            } catch (Exception e) {
               e.printStackTrace();
            }
         }
      } catch (Exception em) {
         System.out.println("ERROR: ");
         em.printStackTrace();
      }
   }
}
