/*
 * Copyright (C) 2001 Advanced Software Engineering Limited.
 *
 * This file is part of the ChartDirector software. Usage of this file is
 * subjected to the ChartDirector license agreement. See the LICENSE.TXT
 * file that comes with the ChartDirector software release, or visit
 * Advanced Software Engineering's web site at www.advsofteng.com or 
 * email to support@advsofteng.com for information.
 *
 */
 
#ifndef LIBGRAPHICS_HDR
#define LIBGRAPHICS_HDR


#include <wchar.h>
#include "memblock.h"


///////////////////////////////////////////////////////////////////////////////////////
//	Forward declarations
///////////////////////////////////////////////////////////////////////////////////////
class DrawArea;


///////////////////////////////////////////////////////////////////////////////////////
//	constants
///////////////////////////////////////////////////////////////////////////////////////
enum
{
	Transparent = 0xff000000
};

enum Alignment 
{ 
	TopLeft = 7,	TopCenter = 8,		TopRight = 9,
	Left = 4,		Center = 5,			Right = 6,
	BottomLeft = 1,	BottomCenter = 2,	BottomRight = 3,
	Top = TopCenter,
	Bottom = BottomCenter
};


///////////////////////////////////////////////////////////////////////////////////////
//	True Type text abstract class
///////////////////////////////////////////////////////////////////////////////////////
class TTFText
{
public :
	enum AntiAliasMode { NoAntiAlias, AntiAlias, AutoAntiAlias };

	static TTFText *create(DrawArea *d, const char *text, const char *font, int fontIndex, 
		double fontHeight, double fontWidth, double angle, bool vertical, AntiAliasMode antiAlias);
	static TTFText *create(DrawArea *d, const wchar_t *text, const char *font, int fontIndex,
		double fontHeight, double fontWidth, double angle, bool vertical, AntiAliasMode antiAlias);
	static void destroy(TTFText *t);
	virtual ~TTFText() {}
	virtual void destroy() = 0;

	virtual int getWidth() const = 0;
	virtual int getHeight() const = 0;
	virtual int getLineHeight() const = 0;
	virtual int getLineDistance() const = 0;
	virtual void draw(int x, int y, int color, Alignment alignment) const = 0;
};


///////////////////////////////////////////////////////////////////////////////////////
//	main DrawArea abstract class
///////////////////////////////////////////////////////////////////////////////////////
class DrawArea
{
public :

	enum PaletteMode { TryPalette, ForcePalette, NoPalette };
	enum DitherMethod { Quantize, OrderedDither, ErrorDiffusion };
	
	__declspec(dllexport) static DrawArea* create();
	virtual void destroy() = 0;
	static bool setLicenseFile(const char *filename);
	virtual ~DrawArea() {}

	virtual void setSize(int width, int height, int bgColor = 0xffffff) = 0;
	virtual int getWidth() const = 0;
	virtual int getHeight() const = 0;
	
	virtual void setBgColor(int c) = 0;

	virtual void pixel(int x, int y, int c) = 0;
	virtual int getPixel(int x, int y) const = 0;

	virtual void hline(int x1, int x2, int y, int c) = 0;
	virtual void vline(int y1, int y2, int x, int c) = 0;
	virtual void line(int x1, int y1, int x2, int y2, int c, int lineWidth = 1) = 0;
	virtual void arc(int cx, int cy, int rx, int ry, double a1, double a2, int c) = 0;

	virtual void rect(int x1, int y1, int x2, int y2, int edgeColor, int fillColor) = 0;
	virtual void polygon(const int *x, const int *y, int noOfPoints, int edgeColor, int fillColor) = 0;
	virtual void surface(int x1, int y1, int x2, int y2, int depthX, int depthY, 
		int edgeColor, int fillColor) = 0;
	virtual void sector(int cx, int cy, int rx, int ry, double a1, double a2, int edgeColor, int fillColor) = 0;
	virtual void cylinder(int cx, int cy, int rx, int ry, double a1, double a2, 
		int depthX, int depthY, int edgeColor, int fillColor) = 0;
	virtual void circle(int cx, int cy, int rx, int ry, int edgeColor, int fillColor) = 0;

	virtual void fill(int x, int y, int color, int borderColor) = 0;
	virtual void fill(int x, int y, int color) = 0;

	virtual void text(const char *str, const char *font, int fontIndex, double fontHeight, double fontWidth, 
		double angle, bool vertical, int x, int y, int color, Alignment alignment = TopLeft) = 0;
	virtual void text(const wchar_t *str, const char *font, int fontIndex, double fontHeight, double fontWidth, 
		double angle, bool vertical, int x, int y, int color, Alignment alignment = TopLeft) = 0;
	virtual void text(const char *str, const char *font, double fontSize,	
		int x, int y, int color) = 0;
	virtual void text(const wchar_t *str, const char *font, double fontSize, 
		int x, int y, int color) = 0;
	virtual TTFText* text(const wchar_t *text, const char *font, int fontIndex, 
		double fontHeight, double fontWidth, double angle, bool vertical) = 0;
	virtual TTFText* text(const char *text, const char *font, int fontIndex, 
		double fontHeight, double fontWidth, double angle, bool vertical) = 0;
	virtual TTFText* text(const char *str, const char *font, double fontSize) = 0;
	virtual TTFText* text(const wchar_t *str, const char *font, double fontSize) = 0;
	virtual void close(TTFText *text) = 0;

	virtual void merge(const DrawArea *d, int x, int y, Alignment align, int transparency) = 0;
	virtual void tile(const DrawArea *d, int transparency) = 0;

	virtual bool loadGIF(const char *filename) = 0;
	virtual bool loadPNG(const char *filename) = 0;
	virtual bool loadJPG(const char *filename) = 0;
	virtual bool loadWMP(const char *filename) = 0;
	virtual bool load(const char *filename) = 0;

	virtual bool out(const char *filename) = 0;

	virtual bool outGIF(const char *filename) = 0;
	virtual bool outPNG(const char *filename) = 0;
	virtual bool outJPG(const char *filename, int quality = 80) = 0;
	virtual bool outWMP(const char *filename) = 0;

	virtual bool outGIF(const char **data, int *len) = 0;
	virtual bool outPNG(const char **data, int *len) = 0;
	virtual bool outJPG(const char **data, int *len, int quality = 80) = 0;
	virtual bool outWMP(const char **data, int *len) = 0;

	virtual MemBlock outGIF() = 0;
	virtual MemBlock outPNG() = 0;
	virtual MemBlock outJPG(int quality = 80) = 0;
	virtual MemBlock outWMP() = 0;

	virtual void setPaletteMode(PaletteMode p) = 0;
	virtual void setDitherMethod(DitherMethod m) = 0;
	virtual void setTransparentColor(int c) = 0;
	virtual void setAntiAliasText(TTFText::AntiAliasMode a) = 0;
	virtual void setInterlace(bool i) = 0;

	virtual void setColorTable(const int *colors, int noOfColors, int offset) = 0;
	virtual int getARGBColor(int c) = 0;
};


#endif