//
//                  ooo        ooooo
//                  `88.       .888'
//                   888b     d'888   .oooo.   oooo    ooo
//                   8 Y88. .P  888  `P  )88b   `88b..8P'
//                   8  `888'   888   .oP"888     Y888'
//                   8    Y     888  d8(  888   .o8"'88b
//                  o8o        o888o `Y888""8o o88'   888o
//
//                          Across the event horizon...
//
//                       Microsoft Windows 95/98/NT Version
//
//  Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Dan Higdon
//
// maxvmap.cpp
//
// Value Mapper
//
//  This module provides facilities to convert linear values into
// non-linear ranges.  These ranges are defined by tables, which may
// be generated by utility members or some user-interface element.
//
//  Why would you want a nonlinearization feature?  It helps to tailor
// the response of a joystick or other linear input device, making the
// response more "realistic".  Why is this not in the MaxJoystick code?
// because you may want to use it elsewhere.
//
//  Each response "curve" is described by a table of three or more values
// representing the shape of the desired response curve.  This table is
// effectively a function mapping input values to output values.
// Values not present in the table are linearly interpolated between the
// two closest values present.  For instance, linear response is described
// by MaxValueMap (3, [0, 32767, 65535]), and inverse linear by
// MaxValueMap (3, [65535, 32767, 0]) (in pseudo-C, naturally).
// The appearance of nonlinearity can be introduced to the mapping
// by adding additional sample points between the ends.  The exact number
// of samples must be determined by the application, but MUST HAVE A SIZE
// EQUAL TO (2^X + 1), WHERE X IS ANY INTEGER [1..15].
//
//  The mapper works by partitioning the input range into a number of
// "bins" equal to the number of spaces between values in the table.
// For instance, a 5 value table has 4 bins.  The remainder of the input
// value is used to interpolate between the two table values surrounding
// the bin.
//
//  Two versions of the mapper are supplied, one that operates on
// unsigned short values, and one that works on signed short values.
//
//

//
//
//                                Includes
//
//

#include <portable.h>
#include <assert.h>

#include "maxvmap.hpp"

//
//
//                                 Code
//
//


//Ŀ
//  MaxValueBase - set_shift                                                
//      Set the table size and bit shift mask.                              
//
void MaxValueBase::set_shift (ushort bits, ushort table_size)
{
    // Calculate the table shift (divide) value
    table_size_shift = 0;
    while ((table_size >>= 1) != 0)
        table_size_shift++;

    // Normalize shift to work with ushorts
    table_size_shift = ushort (bits - table_size_shift);
    table_size_mask  = ushort ((1 << table_size_shift) - 1);
}


//Ŀ
//  MaxValueMap - operator()                                                
//      Map the parameter according to the interpolation table.             
//  Input values are expected to be [-32768..32767]                         
//
short MaxValueMap::operator() (short v) const
{
    // If no table has been selected, assume this is an identity map
    if (table == 0)
        return v;

    // Calculate the interpolation
    const short  av  = (v < 0) ? -v: v;                 // abs (v)
    const long   bin = av >> table_size_shift;          // "value bin"
    const short  tv  = table[bin];                      // table value
    const long   dif = table[bin+1] - tv;               // Value difference
    const long   nv  = av & table_size_mask;            // normalized value
    const long   iv  = (nv * dif) >> table_size_shift;  // interpolated value

    // Return the table base value plus the interpolated value
    if (v < 0)
        return -short (tv + iv);
    else
        return short (tv + iv);
}


//Ŀ
//  MaxUValueMap - operator()                                               
//      Map the parameter according to the interpolation table.             
//  Input values are expected to be [0..65535]                              
//
ushort MaxUValueMap::operator() (ushort v) const
{
    // If no table has been selected, assume this is an identity map
    if (table == 0)
        return v;

    // Calculate the interpolation
    const long   bin = v >> table_size_shift;           // "value bin"
    const ushort tv  = table[bin];                      // table value
    const long   dif = table[bin+1] - tv;               // Value difference
    const long   nv  = v & table_size_mask;             // normalized value
    const long   iv  = (nv * dif) >> table_size_shift;  // interpolated value

    // Return the table base value plus the interpolated value
    return ushort (tv + iv);
}


//Ŀ
//  MaxValueBase - make_linear                                              
//
template <class T>
static void linear (T *table, ushort table_size, T start, T end)
{
    if (start < end)
    {
        const long delta = end - start;

        for (long i = 0; i < table_size; ++i)
            *table++ = T (i * delta / table_size);
    }
    else
    {
        const long delta = start - end;

        for (long i = table_size - 1; i >= 0; --i)
            *table++ = T (i * delta / table_size);
    }
}

void MaxValueBase::make_linear (short *table, ushort table_size, short start, short end)
{
    linear (table, table_size, start, end);
}

void MaxValueBase::make_linear (ushort *table, ushort table_size, ushort start, ushort end)
{
    linear (table, table_size, start, end);
}


//Ŀ
//  MaxValueBase - make_bez                                                 
//

// A simple inline utility
template <class T>
static void bezier_recurse (T *table,
                            ushort pt0x, T pt0y,
                            ushort pt1x, T pt1y,
                            ushort pt2x, T pt2y)
{
    if (   pt0x == pt1x
        && pt0y == pt1y
        && pt1x == pt2x
        && pt1y == pt2y)
    {
        table[pt0x] = pt0y;
    }
    else // Otherwise, draw curve segs
    {
        // Calculate midpoints
        const ushort m1x = (pt0x + pt1x + 1) / 2;
        const T      m1y = (pt0y + pt1y + 1) / 2;
        const ushort m3x = (pt1x + pt2x + 1) / 2;
        const T      m3y = (pt1y + pt2y + 1) / 2;
        const ushort m2x = (m1x  + m3x  + 1) / 2;
        const T      m2y = (m1y  + m3y  + 1) / 2;

        bezier_recurse (table, pt0x, pt0y, m1x, m1y, m2x, m2y);
        bezier_recurse (table, m2x, m2y, m3x, m3y, pt2x, pt2y);
    }
}

void MaxValueBase::make_bez (ushort *table, ushort table_size,
                             ushort start,  ushort end,
                             ushort pointx, ushort pointy)
{
    bezier_recurse (table, ushort (0), start, pointx, pointy, table_size, end);
}


void MaxValueBase::make_bez (short  *table, ushort table_size,
                             short  start,  short  end,
                             ushort pointx, short  pointy)
{
    bezier_recurse (table, ushort (0), start, pointx, pointy, table_size, end);
}


// End of module - expon.cpp 
