/*
**  FROM.C
**
**  POP3 client, uses ASYNC functions.
**
**  Note that this program reads FROM.INI.
**
*/

#include <windows.h>
#include <winsock.h>

#include "wil.h"
#include "message.h"
#include "paint.h"
#include "about.h"
#include "async.h"
#include "readini.h"
#include "str.h"

#ifdef WIN32
#define USE_INS HINSTANCE
#define USE_PTR PSTR
#else
#define USE_INS HANDLE
#define USE_PTR LPSTR
#endif

LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

/* globals */

HWND hMainWnd;            /* main window handle */

#define BS            8
#define LF           10
#define CR           13
#define ESC          27

#define MAX_BUF     512
#define STR_SIZE     50

#define POP3_PORT   110

#define FROM_CONNECT   101
#define FROM_CONN_OK   102
#define FROM_USER_OK   103
#define FROM_PASS_OK   104
#define FROM_STAT_OK   105
#define FROM_LOOP      106
#define FROM_TOP_OK    107
#define FROM_READ_OK   108
#define FROM_QUIT      109
#define FROM_QUIT_OK   110
#define FROM_FAILURE   111

static USE_INS hInstance;
static int WinWidth = 8 * NCOLS;  /* window width */
static int WinHeight = 15 * NROWS;/* window height */
static char Temp[MAX_BUF+8];      /* temprary buffer */
static SOCKET Socket = 0;         /* socket */
static int EchoFlag = 1;          /* echo commands of TRUE */
static HCURSOR ArrowCursor;       /* arrow cursor */
static HCURSOR WaitCursor;        /* hour glass cursor */
static LPSTR BufferPtr;           /* pointer to buffer in ASYNC */
static int MsgCount;
static int MsgIndex;

static char Pop3String[STR_SIZE] = "";  /* POP3 server */
static char UserString[STR_SIZE] = "";  /* POP3 user */
static char PassString[STR_SIZE] = "";  /* POP3 password */

/* display error message */

static void DisplayError(int Code, LPSTR Msg)
{DisplayString("ERROR: ");
 if(Msg) DisplayString(Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,50);
    DisplayLine((LPSTR)Temp);
   }
 /* restore arrow cursor */
 SetCursor(ArrowCursor);
}

/* WinMain */

#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(USE_INS hInst, USE_INS hPrevInstance,
        USE_PTR szCmdLine,  int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "HostIcon");
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HostMenu";
    wc.lpszClassName = "HostWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "HostWClass",   "From",    WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,  CW_USEDEFAULT,
        WinWidth,       WinHeight,
        NULL,           NULL,
        hInstance,      NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT iMsg,WPARAM wParam,LPARAM lParam)
{int n, Code;
 HDC hDC;
 PAINTSTRUCT ps;
 static char LastChar;
#ifdef WIN32
#else
 static FARPROC lpfnAboutDlgProc;
#endif
 hMainWnd = hWindow;
 switch (iMsg)
    {case WM_CREATE:
      /* create cursors */
      ArrowCursor = LoadCursor(NULL, IDC_ARROW);
      WaitCursor = LoadCursor(NULL, IDC_WAIT);
      SetCursor(ArrowCursor);
#ifdef WIN32
#else
       /* create thunk for Win16 */
       lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc,hInstance);
#endif
      /* initialize paint module */
      PaintInit();
      /* attach WINSOCK */
      DisplayString("Attaching WINSOCK...");
      Code = wilAttach();
      DisplayLine("OK");
      if(Code<0) DisplayError(Code,"wilAttach fails:");
      else
        {DisplayString(" Description: ");
         wilGetDescription((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Name: ");
         wilGetMyHostName((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Addr: ");
         wilGetMyHostDotted(0,(LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
        }
      /* open FROM.INI file */
      *Temp = '\0';
      if(!IniOpen("FROM.INI")) break;
      DisplayLine("FROM.INI opened");
      while(1)
        {/* read next line from FROM.INI */
         if(IniRead((LPSTR)Temp)<0) break;
            /*DisplayLine((LPSTR)Temp);*/
         /* test for all legal keywords */
         IniExtract((LPSTR)Temp,"POP3",(LPSTR)Pop3String);
         IniExtract((LPSTR)Temp,"USER",(LPSTR)UserString);
         IniExtract((LPSTR)Temp,"PASS",(LPSTR)PassString);
        }
      /* verify that we have all strings read in */
      if(lstrlen((LPSTR)Pop3String)==0) DisplayLine("Missing POP3 server name.");
      if(lstrlen((LPSTR)UserString)==0) DisplayLine("Missing POP3 USER name");
      if(lstrlen((LPSTR)PassString)==0) DisplayLine("Missing POP3 Password");
      break;

     case WM_COMMAND:
         switch(wParam)
           {
            case MSG_ABOUT :
#ifdef WIN32
               DialogBox(hInstance, "AboutBox", hMainWnd, AboutDlgProc);
#else
               DialogBox(hInstance, "AboutBox", hMainWnd, lpfnAboutDlgProc);
#endif
               return 0;

            case MSG_DEBUG:
              wsprintf((LPSTR)Temp,"DEBUG: Pop3String=[%s]", (LPSTR)Pop3String);
              DisplayLine((LPSTR)Temp);
              wsprintf((LPSTR)Temp,"DEBUG: UserString=[%s]", (LPSTR)UserString);
              DisplayLine((LPSTR)Temp);
              wsprintf((LPSTR)Temp,"DEBUG: PassString=[%s]", (LPSTR)PassString);
              DisplayLine((LPSTR)Temp);
              break;

            case MSG_ECHO:
              EchoFlag = 1 - EchoFlag;
              DisplayString("Echo flag is now ");
              if(EchoFlag) DisplayLine("ON");
              else DisplayLine("OFF");
              break;

            case MSG_BREAK:
              SetCursor(ArrowCursor);
              wilCancelBlocking();
              wilCloseSocket(Socket,500);
              break;

            case MSG_EXIT:
              wilRelease();
              DestroyWindow(hMainWnd);
              break;

            case MSG_MAIL_CHK:
              POST_MSG(FROM_CONNECT);
              break;

           }
         break;

    case WM_USER: /* posted by WIL */
      AsyncProcessMsg(lParam);
      break;

    case WM_USER+1:  /* posted by Async functions */
      /* test response code [+OK maps to 100, -ERR maps to 500] */
      if(lParam>=500)
        {/* POP3/SMTP server returns fatal error code */
         POST_MSG(FROM_FAILURE);
         break;
        }
      /* execute case */
      switch(wParam)
        {
         case FROM_CONNECT:
           /* log onto POP3 server */
           SetCursor(WaitCursor);
           AsyncSetEcho(FALSE);
           Socket = AsyncConnect(hMainWnd,"POP3",(LPSTR)Pop3String, POP3_PORT,
                       FROM_CONN_OK, FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_CONN_OK:
           /* send USER command */
           wsprintf((LPSTR)Temp,"USER %s", (LPSTR)UserString);
           AsyncCommand((LPSTR)Temp, FROM_USER_OK,
                       FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_USER_OK:
           /* send PASS (password) command */
           wsprintf((LPSTR)Temp,"PASS %s", (LPSTR)PassString);
           AsyncCommand((LPSTR)Temp, FROM_PASS_OK,
                       FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_PASS_OK:
           /* send STAT (statistics) command */
           AsyncCommand("STAT", FROM_STAT_OK,
                  FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_STAT_OK:
           /* get the message count */
           BufferPtr = AsyncGetBufPtr();
           MsgCount = (int) wilParseDecimal(BufferPtr+4);
           wsprintf((LPSTR)Temp,"  %d messages waiting",MsgCount);
           DisplayLine((LPSTR)Temp);
           /* look at each message in turn */
           MsgIndex = 0;
           POST_MSG(FROM_LOOP);
           break;

         case FROM_LOOP:
           /* is there another header ? */
           MsgIndex++;
           if(MsgIndex>MsgCount)
             {/* no more headers, so sign off POP server */
              POST_MSG(FROM_QUIT);
              break;
             }
           /* get header for message <MsgIndex> */
           wsprintf((LPSTR)Temp,"TOP %d 0", MsgIndex);
           AsyncCommand((LPSTR)Temp, FROM_TOP_OK,
                       FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_TOP_OK:
           /* display header info */
           wsprintf((LPSTR)Temp,"Msg %d: ",MsgIndex);
           DisplayString((LPSTR)Temp);
           /* read 1st line of header info */
           LastChar = '\n';
           AsyncRead(FROM_READ_OK, FROM_FAILURE, ASYNC_SINGLE_LINE);
           break;

         case FROM_READ_OK:
           /* read was successfull */
           BufferPtr = AsyncGetBufPtr();
           /* check if this is the last line */
           if((LastChar=='\n')&&(*BufferPtr=='.')&&(*(BufferPtr+1)=='\r') )
             {/* end of this header message */
              DisplayLine("");
              POST_MSG(FROM_LOOP);
              break;
             }
           /* got good header line */
           n = lstrlen((LPSTR)BufferPtr);
           LastChar = *(BufferPtr+n-1);
           /* look for "From" or "Subject" */
           Code = IsLeftString(BufferPtr,(LPSTR)"From");
           if(!Code) Code = IsLeftString(BufferPtr,(LPSTR)"Subject");
           if(Code)
             {StripCRLF(BufferPtr);
              wsprintf((LPSTR)Temp,"%s ", BufferPtr);
              DisplayString((LPSTR)Temp);
             }
           /* read next header line */
           AsyncRead(FROM_READ_OK, FROM_FAILURE, ASYNC_SINGLE_LINE);
           break;

         case FROM_QUIT:
           /* send QUIT command */
           AsyncCommand("QUIT", FROM_QUIT_OK,
                  FROM_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case FROM_FAILURE:
           DisplayLine("POP3 command fails.");
           /* fall thru... */

         case FROM_QUIT_OK:
           /* all done */
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;

        } /* end-switch(wParam) */

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(PaintGetColPos(),PaintGetRowPos());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    default:
      return (DefWindowProc(hMainWnd, iMsg, wParam, lParam));
   }
  return 0;
 }


