{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}

unit DSContainer;

interface

{$I FIBPlus.inc}
uses
 {$IFDEF MSWINDOWS}
  Windows, Messages, SysUtils, Classes, DB;
 {$ENDIF}
 {$IFDEF LINUX}
  Types, SysUtils, Classes, DB;
 {$ENDIF}

type

  TKindDataSetEvent = (   deBeforeOpen ,   deAfterOpen,deBeforeClose     ,
      deAfterClose,deBeforeInsert,deAfterInsert,deBeforeEdit,deAfterEdit,
      deBeforePost,deAfterPost,deBeforeCancel,deAfterCancel,deBeforeDelete,
      deAfterDelete,deBeforeScroll,deAfterScroll,deOnNewRecord,deOnCalcFields,
      deBeforeRefresh,deAfterRefresh
     );

  TKindDataSetError=(
   deOnEditError,deOnPostError,deOnDeleteError
  );

  TOnDataSetEvent     =procedure (DataSet:TDataSet;Event:TKindDataSetEvent) of object;
  TOnDataSetError     =procedure (DataSet:TDataSet;Event:TKindDataSetError;
      E: EDatabaseError; var Action: TDataAction
  ) of object;

  TUserEvent      =procedure (Sender:TObject;Receiver:TDataSet;const EventName:string;
     var Info :string
  ) of object;


  TDataSetsContainer = class;

  TDataSetsContainer = class(TComponent)
  private
   FActive       :boolean;
   FMasterContainer:TDataSetsContainer;
   FOnDataSetEvent:TOnDataSetEvent;
   FOnDataSetError:TOnDataSetError;
   FOnUserEvent :TUserEvent;
   vDataSetsList :TList;
   procedure     SetMasterContainer(Value:TDataSetsContainer);
   procedure     CheckCircularMaster(Value:TDataSetsContainer);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
   constructor Create(AOwner:Tcomponent);override;
   destructor  Destroy;override;

   procedure AddDataSet(Value:TDataSet);
   procedure RemoveDataSet(Value:TDataSet);
   procedure NotifyDataSets(Sender:TObject;const UDA,UDE:string;var Info :string);  dynamic;
                                                                   //  (UDA - user defined address)
                                                                  //  (UDE - user defined Event)
   procedure DataSetEvent(DataSet:TDataSet;Event:TKindDataSetEvent);
   procedure DataSetError(DataSet:TDataSet;Event:TKindDataSetError;
    E: EDatabaseError; var Action: TDataAction
   );
   procedure UserEvent(Sender:TObject;Receiver:TDataSet;const EventName:string; var Info :string);
   function  DataSetCount:integer;
   function  DataSet(Index:integer):TDataSet;
  published
   property Active    :boolean read FActive write FActive default true;
   property OnDataSetEvent  :TOnDataSetEvent read FOnDataSetEvent  write FOnDataSetEvent;
   property OnDataSetError  :TOnDataSetError read FOnDataSetError  write FOnDataSetError;
   property OnUserEvent :TUserEvent  read FOnUserEvent write FOnUserEvent;
   property MasterContainer:TDataSetsContainer read FMasterContainer write SetMasterContainer;
  end;


implementation

uses StrUtil, pFIBDataSet, FIBConsts;

constructor TDataSetsContainer.Create(AOwner:Tcomponent);//override;
begin
 inherited Create(AOwner);
 FActive         :=true;
 FMasterContainer :=nil;
 vDataSetsList    :=TList.Create;
end;

destructor  TDataSetsContainer.Destroy;//override;
begin
 vDataSetsList.Free;
 inherited Destroy;
end;

procedure TDataSetsContainer.Notification(AComponent: TComponent; Operation: TOperation); //override;
begin
 if (Operation=opRemove) then begin
  if AComponent<>FMasterContainer then vDataSetsList.Remove(AComponent);
  if AComponent=FMasterContainer then FMasterContainer:=nil;
 end;
 inherited Notification(AComponent,Operation)
end;

procedure TDataSetsContainer.DataSetEvent(DataSet:TDataSet;Event:TKindDataSetEvent);
begin
 if Assigned(FMasterContainer) then FMasterContainer.DataSetEvent(DataSet,Event);
 if Active then begin
  if Assigned(FOnDataSetEvent) then FOnDataSetEvent(DataSet,Event)
 end;
end;

procedure TDataSetsContainer.DataSetError(DataSet:TDataSet;Event:TKindDataSetError;
 E: EDatabaseError; var Action: TDataAction);
begin
 if Assigned(FMasterContainer) then FMasterContainer.DataSetError(DataSet,Event,E,Action);
 if Active then begin
   if Assigned(FOnDataSetError) then FOnDataSetError(DataSet,Event,E,Action)
 end;
end;

procedure TDataSetsContainer.UserEvent(Sender:TObject;Receiver:TDataSet;const EventName:string;var Info:string);
begin
 if Assigned(FMasterContainer) then FMasterContainer.UserEvent(Sender,Receiver,EventName,Info);
 if Active then begin
   if Assigned(FOnUserEvent) then FOnUserEvent(Sender,Receiver,EventName,Info)
 end;
end;

procedure TDataSetsContainer.SetMasterContainer(Value:TDataSetsContainer);
var i:integer;
begin
  CheckCircularMaster(Value);
  if FMasterContainer<>nil then
    for i:=0 to Pred(vDataSetsList.Count) do
     FMasterContainer.RemoveDataSet(TDataSet(vDataSetsList[i]));
  FMasterContainer:=Value;
  if Value<>nil then begin
    Value.FreeNotification(Self);
    for i:=0 to Pred(vDataSetsList.Count) do
     Value.AddDataSet(TDataSet(vDataSetsList[i]))
  end;
end;

procedure TDataSetsContainer.CheckCircularMaster(Value:TDataSetsContainer);
var CurContainer:TDataSetsContainer;
begin
  if Value = Self then raise Exception.Create(SFIBErrorCircularLinks);
  CurContainer:=Value;
  while CurContainer<>nil do
   if   CurContainer.MasterContainer=Self then raise Exception.Create(SFIBErrorCircularLinks)
   else CurContainer:=CurContainer.MasterContainer;
end;


procedure TDataSetsContainer.AddDataSet(Value:TDataSet);
begin
 if Value<>nil then begin
  if vDataSetsList.IndexOf(Value)=-1 then vDataSetsList.Add(Value);
  if MasterContainer<>nil then MasterContainer.AddDataSet(Value);
  Value.FreeNotification(Self)
 end;
end;

procedure TDataSetsContainer.RemoveDataSet(Value:TDataSet);
begin
 if vDataSetsList.IndexOf(Value)<>-1 then vDataSetsList.Remove(Value);
 if MasterContainer<>nil then MasterContainer.RemoveDataSet(Value);
end;

procedure TDataSetsContainer.NotifyDataSets(Sender:TObject;const UDA,UDE:string;var Info :string);
var i:integer;
begin
 for i:=0 to Pred(vDataSetsList.Count) do begin
   if (UDA='') or
    WildStringCompare(
     UpperCase(TDataSet(vDataSetsList[i]).Owner.Name+'.'+TDataSet(vDataSetsList[i]).Name),
     UpperCase(UDA)
    )
   then
   if (TDataSet(vDataSetsList[i]) is TpFibDataSet) then
    with TpFibDataSet(vDataSetsList[i]) do
     if (ReceiveEvents.Count=0) or (WldIndexOf(ReceiveEvents,UDE,false)<>-1 )then
      DoUserEvent(Sender,UDE,Info)
 end;
end;


function  TDataSetsContainer.DataSetCount:integer;
begin
 Result   :=vDataSetsList.Count
end;

function  TDataSetsContainer.DataSet(Index:integer):TDataSet;
begin
  if (Index<vDataSetsList.Count) and (Index>-1) then
   Result :=TDataSet(vDataSetsList[Index])
  else
   Result :=nil
end;

initialization

end.

