VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "VarTrace"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
Attribute VB_Ext_KEY = "SavedWithClassBuilder" ,"Yes"
Attribute VB_Ext_KEY = "Top_Level" ,"Yes"
'**
' An event source designed to allow users to trace reads and writes on variables.
'
' @describe     This object is designed to allow external programs to receive
'   events whenever a variable is read from, written to, or destroyed.
'   The following code properly dimensions a VarTrace object to receive events:
'   <blockquote>
'   <code><font color="blue">Dim WithEvents</font> vt <font color="blue">As</font> VarTrace</code>
'   </blockquote>
'   After reserving space for the VarTrace object itself, events may be
'   captured using the standard event capture mechanism of other Visual
'   Basic events sources such as Textboxes, Buttons and other Controls.
'   Note: The "WithEvents" keyword may only be used in class type VB
'   modules including Forms, Classes and Controls.
'
' @author       Andrew Friedl
' @copyright    1997.03.10, BlackBox Software & Consulting
'*
Option Explicit

Public Event TraceRead(Value As Variant)
Public Event TraceWrite(OldValue As Variant, NewValue As Variant)
Public Event TraceDestroy(Value As Variant)

Private m_Var As Variable
Private m_ID As String

'**
' Causes the VarTrace object to disconnect itself from the variable being traced.
'
' @describe After a call to the Quite subroutine, no more events will be triggered for reads,
'           writes, or deletes upon the variable being traced.  An instance of VarTrace is
'           rendered useless after to this subroutine.
'           <b>Note:</b> Quite is always called prior to the destruction of a VarTrace object,
'           and it is generally a good ides to set them to Nothing when you're done with them.
'*
Public Sub Quit()
    If Not (m_Var Is Nothing) Then
        Call m_Var.EndTrace(Me)
    End If
    Set m_Var = Nothing
End Sub

'**
' Returns a reference to the variable being traced
'
' @describe When a VarTrace is monitoring a Variable for reads, writes and
'   destruction, it maintains a reference to the Variable object being traced.
'   This reference is needed by the tracer object so that tracing may be
'   terminated when it is desirable to do so.
'
' @returns  a reference to the Variable object being traced.
'*
Friend Property Get Variable() As Variable
    Set Variable = m_Var
End Property

'**
' Sets a reference to the variable being traced
'
' @describe When a VarTrace is monitoring a Variable for reads, writes and
'   destruction, it maintains a reference to the Variable object being traced.
'   This reference is needed by the tracer object so that tracing may be
'   terminated when it is desirable to do so.
'
' param     V a reference to the Variable object to be traced.
'*
Friend Property Set Variable(V As Variable)
    Set m_Var = V
End Property

'**
' Retrieves the globally unique identifier for the VarTrace object.
'
' @describe     When a VarTrace is created, it is assigned a globally unique
'   ID that allows differentiation between VarTrace objects.<i>A hold over from
'   previous version, this property may be depracated in future releases.</i>
'
' @returns      A string identifier.
'*
Friend Property Get ID() As String
    ID = m_ID
End Property

'**
' Sets the globally unique identifier for the VarTrace object.
'
' @describe     When a VarTrace is created, it is assigned a globally unique
'   ID that allows differentiation between VarTrace objects.<i>A hold over from
'   previous version, this property may be depracated in future releases.</i>
'
' @param      S A string identifier.
'*
Friend Property Let ID(S As String)
    m_ID = S
End Property

'**
' Subroutine to be called when the Variable being traced is Read.
'
' @describe     When a traced Variable is read, the Variable object indicates this
'   to all appropriate VarTraces assigned to it by calling this member subroutine
'   on each one of them.  This subroutine Raises the TraceRead event.  User code then
'   has direct control over the read process if code has been setup to capture the
'   event.
'   <b>Note:</b> Because the TraceRead event occurs prior to the users retrieval of
'   data, the value can be altered within the event, allowing users to create dynamic
'   variables such as auto-incrementing values.
'
' @param      Value The current (Variant) value of the Variable.  This value is read/write
'   and may be altered within the event procedure.
'*
Friend Sub TraceRead(Value)
    RaiseEvent TraceRead(Value)
End Sub

'**
' Subroutine to be called when the Variable being traced is Written.
'
' @describe     When a traced Variable is written, the Variable object indicates this
'   to all appropriate VarTraces assigned to it by calling this member subroutine
'   on each one of them.  This subroutine Raises the TraceWrite event.  User code then
'   has direct control over the write process if code has been setup to capture the
'   event.
'   <b>Note:</b> Because the Tracewrite event occurs prior to the storage of the users
'   data, the new value can be altered within the event, allowing users to create dynamic
'   or read only.  This also allows dynamic updates of external conditions based upon changes
'   of variables stored within the VarStack.
'
' @param      OldValue The current value assigned to the Variable.
' @param      NewValue The value that is about to be assigned to the Variable. Altering this
'   value alters the values that will be stored in the Variable.
'   <b>Note:</b> If this value is altered within the TraceWrite event, subsequent VarTraces
'   receive the altered value instead of the original value.
'*
Friend Sub TraceWrite(OldValue As Variant, NewValue As Variant)
    RaiseEvent TraceWrite(OldValue, NewValue)
End Sub

'**
' Subroutine to be called when the Variable being traced is about ti be destroyed.
'
' @describe     When a Variable is about to be deleted, it informs all VarTraces
'   currently assigned to it by calling this subroutine on each of them.  User code
'   (if assigned) is informed via the TraceDestroy event.
'
' @param        Value a variant type that is the current value of the variable.
'   Altering the Value has no effect as the variable will be destroyed immediately
'   after all traces have been notified.
'*
Friend Sub TraceDestroy(Value As Variant)
    RaiseEvent TraceDestroy(Value)
End Sub
