VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "Variable"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
Attribute VB_Ext_KEY = "SavedWithClassBuilder" ,"Yes"
Attribute VB_Ext_KEY = "Top_Level" ,"Yes"
'**
' Storage wrapper for variants that provides tracing and readonly protection.
'
' @describe
'   This class provides a useful wrapper around Visual Basic's Variant
'   data type.  Using this class, variables may be created with a ReadOnly
'   property and provide for Read/Write/Destroy tracing capabilities.
'
' @author       Andrew F. Friedl
' @copyright    1997.04.10, BlackBox Software & Consulting
'*
Option Explicit

Public Name As String
Public ReadOnly As Boolean
Private m_Value As Variant
Private m_Traces As Collection
Private m_IsTracing As Boolean
Private mTraceID As Long

'**
' Sets the value of the Variable.
'
' @describe     This property attempts to set the value of the Variable.
'   The value may not be an object type.  This function will generate an error
'   if the Variable has been marked ReadOnly. If the variable has traces
'   assigned, they will be invoked.
'
' @param    Data The new value of the Variable.
'*
Public Property Let Value(Data As Variant)
    If Not SetValue(Data) Then
        Err.Raise vbObjectError, "Variable", Name & " is ReadOnly"
    End If
End Property

'**
' Sets the value of the Variable.
'
' @describe     This property attempts to set the value of the Variable.
'   The value must be an object type.  This function will generate an error
'   if the Variable has been marked ReadOnly.  If the variable has traces
'   assigned, they will be invoked.
'
' @param    Data The new value of the Variable.
'*
Public Property Set Value(Data As Variant)
    If Not SetValue(Data) Then
        Err.Raise vbObjectError, "Variable", Name & " is ReadOnly"
    End If
End Property

'**
' Retrieves the value of the Variable.
'
' @describe     This property retrieves the current value of the variable.
'   The value is placed in the Value parameter.  If the value of this variable
'   is an alias, or another variable, then the value of that variable is returned.
'
' @returns      The current value of the Variable.
'*
Public Property Get Value() As Variant
    Dim Temp As Variant
    Call GetValue(Temp)
    If IsObject(Temp) Then
        Set Value = Temp
    Else
        Let Value = Temp
    End If
End Property

'**
' Sets the current value of the Variable.
'
' @describe     This function attempts to set the value of the Variable.
'   The value may be set regardless of type.  This function will fail if
'   the Variable has been marked ReadOnly.  This function is independent
'   of the data type being retrieved.
'
' @param    Data  The new value of the Variable.
' @returns  True, if the value was set, False otherwise.
'*
Public Function SetValue(Data As Variant) As Boolean
    On Error Resume Next
    
    '
    ' Check for ReadOnly Values
    '
    If ReadOnly Then
        SetValue = False
        Exit Function
    End If
    
    ' handle traces
    If Not m_IsTracing Then
        m_IsTracing = True
        Call TraceWrite(Data)
        m_IsTracing = False
    End If

    If IsObject(Value) Then
        If TypeName(m_Value) = "Variable" Then
            SetValue = m_Value.SetValue(Data)
        Else
            Set m_Value = Value
            SetValue = True
        End If
    Else
        m_Value = Data
        SetValue = True
    End If
End Function

'**
' Informs the Variable it is about to be deleted.
'
' @describe     This subroutine informs tha variable that it is about to
'   be deleted.  It is here that the variable then informs all traces about
'    impending demise through the TraceDestroy event.
'*
Friend Sub UnsetValue()
    If Not m_IsTracing Then
        m_IsTracing = True
        Call TraceDestroy
        m_IsTracing = False
    End If
End Sub

'**
' Retrieves the current value of the Variable.
'
' @describe     This subrounting retrieves the current value of the variable.
'   The value is placed in the Value parameter.  If the value of this variable
'   is an alias, or another variable, then the value of that variable is returned.
'
' @param    Data The variant to store the value.
'*
Public Sub GetValue(Data As Variant)
    ' handle traces
    If Not m_IsTracing Then
        m_IsTracing = True
        Call TraceRead
        m_IsTracing = False
    End If

    If IsObject(m_Value) Then
        If TypeName(m_Value) = "Variable" Then
        m_Value.GetValue Value
        Else
            Set Data = m_Value
        End If
    Else
        Let Data = m_Value
    End If
End Sub

'**
' Adds a trace to the variable.
'
' @describe
'   This function adds a new trace to the variable. The newtrace is placed
'   at the end of the trace list, and assuming a suitable definition was
'   made, will provide Read/Write/Destroy event notifications.
'
' @returns  A reference to a new VarTrace object.
'*
Friend Function AddTrace() As VarTrace
    Dim Trace As VarTrace
    
    ' add a new trace for this object
    If m_Traces Is Nothing Then
        Set m_Traces = New Collection
    End If

    Set Trace = New VarTrace
    Set Trace.Variable = Me
    Trace.ID = "T" & TraceID
    m_Traces.Add Trace, Trace.ID
    Set AddTrace = Trace

End Function

'**
' Removes a trace from the trace list.
'
' @describe
'   This subroutine removes the trace from the variables trace list
'   provided that it is a member of this objects trace list.
'
' @param    Trace   A reference to the Trace that is to be removed from the list.
'*
Friend Sub EndTrace(Trace As VarTrace)
    Dim B As Boolean
    
    If m_Traces Is Nothing Then Exit Sub
    
    ' remove the trace
    On Error Resume Next
    Call m_Traces.Remove(Trace.ID)
    Set Trace.Variable = Nothing
End Sub

'**
' Terminates all tracing on this variable.
'*
Friend Sub EndAllTraces()
    Dim Trace As VarTrace
    
    If Not (m_Traces Is Nothing) Then
        For Each Trace In m_Traces
            Set Trace.Variable = Nothing
        Next
    End If
    Set m_Traces = Nothing
End Sub

'**
' Initiates a TraceRead event all traces within this variable.
'
' @describe This subroutine causes the variable to cycle through its trace
'   list and initiate TraceRead events.  Each VarTrace object is passed a
'   reference to the variable data in succession.  Modification of the data
'   by any of the traces, causes a data change to be passed to all subsequent
'   traces, and to the actual data itself.
'*
Private Sub TraceRead()
    Dim Trace As VarTrace
    On Error Resume Next
    If Not (m_Traces Is Nothing) Then
        For Each Trace In m_Traces
            Call Trace.TraceRead(m_Value)
            Err.Clear
        Next Trace
    End If
End Sub

'**
' Initiates a TraceWrite event all traces within this variable.
'
' @describe This subroutine causes the variable to cycle through its trace
'   list and initiate TraceWrite events.  Each VarTrace object is passed a
'   reference to the variable data in succession.  Modification of the data
'   by any of the traces, causes a data change to be passed to all subsequent
'   traces, and to the actual data itself.
'
' @param    Data    The current value of this variables data.
'*
Private Sub TraceWrite(Data)
    Dim Trace As VarTrace
    On Error Resume Next
    If Not (m_Traces Is Nothing) Then
        For Each Trace In m_Traces
            Call Trace.TraceWrite(m_Value, Data)
            Err.Clear
        Next Trace
    End If
End Sub

'**
' Initiates a TraceDestroy event all traces within this variable.
'
' @describe This subroutine causes the variable to cycle through its trace
'   list and initiate TraceDestroy events.
'*
Private Sub TraceDestroy()
    Dim Trace As VarTrace
    On Error Resume Next
    If Not (m_Traces Is Nothing) Then
        For Each Trace In m_Traces
            Call Trace.TraceDestroy(m_Value)
            Err.Clear
        Next Trace
    End If
End Sub

Friend Function IsAnObject() As Boolean
    IsAnObject = IsObject(m_Value)
End Function

Private Sub Class_Initialize()
    mTraceID = 0
End Sub

Private Property Get TraceID() As Long
    mTraceID = mTraceID + 1
    TraceID = mTraceID
End Property
