		PAGE 63, 136
COMMENT @
*******************************************************************************
Microsoft QuickAssembler functions by Bruce Terry
Written 01/11/1990
Functions to provide support for the enhanced sound capabilities of the
Tandy 1000 TL and Tandy 1000 SL.
*******************************************************************************
@
		.MODEL  LARGE

		NAME    TLSLSnd
		TITLE   Tandy 1000 TL & SL Sound Functions

		.DATA
; *****   Sound status flags   *****
Not_Busy        equ     0
Playing         equ     1
Recording       equ     2

; *****   Parameter structs   *****
params1         STRUC                   ; Parameter struct for "setsound"
bp_reg1         dw      ?               ; Saved BP
IF @CodeSize                            ; If memory model is medium, large,
  ret_seg1      dw      ?               ;   or huge, saved return segment
ENDIF
ret_off1        dw      ?               ; Return offset
flag_ptr1       dd      ?               ; Pointer to flag
params1         ENDS

params2         STRUC                   ; Parameter struct for "record"
bp_reg2         dw      ?
IF @CodeSize
  ret_seg2      dw      ?
ENDIF
ret_off2        dw      ?
rec_buff        dd      ?               ; Pointer to record buffer
rec_size        dd      ?               ; Record buffer size
rec_rate        dw      ?               ; Record transfer rate
params2         ENDS

params3         STRUC                   ; Parameter struct for "play"
bp_reg3         dw      ?
IF @CodeSize
  ret_seg3      dw      ?
ENDIF
ret_off3        dw      ?
play_buff       dd      ?               ; Pointer to play buffer
play_size       dd      ?               ; Play size
play_rate       dw      ?               ; Play transfer rate
play_vol        dw      ?               ; Play volume
params3         ENDS

; *****   ISR addresses   *****
p_isr15h        dd      ?               ; Storage for Int 15h handler

; *****   Miscellaneous data   *****
flag_ptr        dd      ?               ; Application's completion flag pointer
busy            dw      ?               ; Internal flag for sound status
event           dw      ?               ; Event watch flag
buffer          dd      ?               ; Record/play buffer
buff_seg        dw      ?               ; Current segment address of buffer
size_buff       dd      ?               ; Size of buffer
t_rate          dw      ?               ; Record transfer rate
volume          dw      ?               ; Play volume


		.CODE
COMMENT @
******************************************************************************
Function to stop all sound input and output to the speaker.
Usage:  void stopsound(void);
Returns:  Nothing
Register preserved:  All but ax
******************************************************************************
@
		PUBLIC  _stopsound
_stopsound      PROC
		push    ds
		mov     ax, @DATA
		mov     ds, ax
		mov     busy, Not_Busy
		mov     event, 0

stop010:        mov     ah, 84h
		int     1Ah

stop020:        mov     ah, 81h
		int     1Ah
		jb      stop020
		cmp     event, 0
		je      stop010
		pop     ds
		ret
_stopsound      ENDP


COMMENT @
******************************************************************************
Function to initialize the sound recording functions.  Mainly, Int 15h is
hooked so that completion of the DMA transfer after a sound record/play can
be detected.
Usage:  void initsound(flag_ptr)
	  int far *flag_ptr     /* Pointer to DMA completion flag */
Returns:  Nothing
Registers preserved:  BP, DS
The variable "flag_ptr" should be a pointer to an integer variable that will
be used by the calling application to determine when the sound record/play
operation is done.  Upon completion of a sound record/play, the variable
pointed to by "flag_ptr" will be given a value of 1.
******************************************************************************
@
		PUBLIC  _initsound
_initsound      PROC
		push	bp
		mov     bp, sp
		push    ds

; *****   Initialize data segment, flag pointer, and sound status   *****
		mov     ax, @DATA
		mov     ds, ax
		mov     ax, WORD PTR [bp].flag_ptr1[0]
		mov     WORD PTR flag_ptr[0], ax
		mov     ax, WORD PTR [bp].flag_ptr1[2]
		mov     WORD PTR flag_ptr[2], ax
		mov     busy, Not_Busy

; *****   Hook into Int 15h   *****
		mov     ax, 3515h
		int     21h
		mov     WORD PTR p_isr15h[0], bx
		mov     WORD PTR p_isr15h[2], es
		mov     dx, OFFSET isr15h
		mov     ax, SEG isr15h
		mov     ds, ax
		mov     ax, 2515h
		int     21h
		pop     ds
		call    _stopsound

; *****   Restore registers and return   *****
		pop	bp
		ret
_initsound      ENDP


COMMENT @
******************************************************************************
Function to restore all vectors hooked by the sound recording/playing
functions.
Usage:  void endsound(void);
Returns:  Nothing
Registers preserved:  BP, DS
******************************************************************************
@
		PUBLIC  _endsound
_endsound       PROC
		push    ds
		mov     ax, @DATA
		mov     ds, ax
		call    _stopsound
		lds     dx, p_isr15h
		mov     ax, 2515h
		int     21h
		pop     ds
		ret
_endsound       ENDP


COMMENT @
******************************************************************************
Function to use the Tandy 1000 TL or SL's enhanced sound BIOS support to
record sound entered through the microphone into a specified buffer.
Usage:  int record(buff, size, t_rate)
	  char far *buff;               /* Pointer to recording buffer */
	  long size;                    /* Size of buffer */
	  unsigned t_rate;              /* Transfer rate (1 - 4095) */
Returns:  - "Not_Busy" if successful recording attempt
	  - "Playing" if their is a buffer still being played.
	  - "Recording" if their is a buffer still being recorded into.
Registers preserved:  BP, DS
After this function is called, control immediately returns to the calling
program.  While the calling program is running, the buffer is filled with the
recorded sound until either the buffer is full or until function "stopsound"
is called.  When the sound recording stops, the address pointed to by the
flag pointer passed to function "initsound" is filled with a value of 1.
This is how the calling program can determine when the transfer is completed.
******************************************************************************
@
		PUBLIC  _record
_record         PROC
		push    bp
		mov     bp, sp
		push    ds

; *****   Check busy flag.  Exit if busy   *****
		mov     ax, @DATA
		mov     ds, ax
		mov     ax, busy
		cmp     busy, Not_Busy
		jne     rec020

; *****   Get and store parameters
		mov     ax, WORD PTR [bp].rec_buff[0]
		mov     WORD PTR buffer[0], ax
		mov     ax, WORD PTR [bp].rec_buff[2]
		mov     WORD PTR buffer[2], ax
		mov     buff_seg, ax
		mov     ax, WORD PTR [bp].rec_size[0]
		mov     WORD PTR size_buff[0], ax
		mov     ax, WORD PTR [bp].rec_size[2]
		mov     WORD PTR size_buff[2], ax
		mov     ax, [bp].rec_rate
		mov     t_rate, ax

; *****   Begin recording   *****
		call    _stopsound
		mov     busy, Recording
		mov     event, 0
		mov     ah, 82h
		les     bx, buffer
		mov     cx, 0FFFFh
		cmp     WORD PTR size_buff[2], 0
		jne     rec010
		mov     cx, WORD PTR size_buff[0]

rec010:         mov     dx, t_rate
		int     1Ah

rec020:         pop     ds
		pop     bp
		ret
_record         ENDP


COMMENT @
******************************************************************************
Function to play a pre-recorded sound through the Tandy 1000 TL & SL enhanced
sound BIOS support.
Usage:  int play(buffer, size, t_rate, volume)
	  char *buffer;                 /* Buffer containing recorded sound */
	  long size;                    /* Size of buffer */
	  unsigned t_rate,              /* Transfer rate (1 - 4095) */
		   volume;              /* Volume (0 - 7) */
Returns:  - "Not_Busy" if successful recording attempt
	  - "Playing" if their is a buffer still being played.
	  - "Recording" if their is a buffer still being recorded into.
Registers preserved:  BP, DS
NOTE:  The transfer rate for playing the sound should be 10.0 times as large
       in the input rate on a TL, and 11.5 times as large on an SL.
******************************************************************************
@
		PUBLIC  _play
_play           PROC
		push    bp
		mov     bp, sp
		push    ds

; *****   Check busy flag.  Exit if busy   *****
		mov     ax, @DATA
		mov     ds, ax
		mov     ax, busy
		cmp     busy, Not_Busy
		jne     play020

; *****   Get and store parameters   *****
		mov     ax, WORD PTR [bp].play_buff[0]
		mov     WORD PTR buffer[0], ax
		mov     ax, WORD PTR [bp].play_buff[2]
		mov     WORD PTR buffer[2], ax
		mov     buff_seg, ax
		mov     ax, WORD PTR [bp].play_size[0]
		mov     WORD PTR size_buff[0], ax
		mov     ax, WORD PTR [bp].play_size[2]
		mov     WORD PTR size_buff[2], ax
		mov     ax, [bp].play_rate
		mov     t_rate, ax
		mov     ax, [bp].play_vol
		mov     volume, ax

; *****   Begin playing   *****
		call    _stopsound
		mov     busy, Playing
		les     bx, buffer
		mov     cx, 0FFFFh
		cmp     WORD PTR size_buff[2], 0
		jne     play010
		mov     cx, WORD PTR size_buff[0]

play010:        mov     dx, t_rate
		mov     ax, volume
		mov     ah, 83h
		int     1Ah

play020:        pop     ds
		pop     bp
		ret
_play           ENDP


COMMENT @
******************************************************************************
Interrupt handler designed to hook into Int 15h and set certain flags when
the DMA transfer for sound input/output is completed.  In addition to setting
these flags, this handler will continue any sound input/output for buffers
larger than 64k.
******************************************************************************
@
isr15h          PROC    FAR

; *****   Store registers   *****
		push    ds
		pushf
		push    es
		push    dx
		push    cx
		push    bx
		push    ax

; *****   Set data segment and set appropriate flags if necessary   *****
		mov     ax, @DATA
		mov     ds, ax
		pop     ax
		push    ax
		cmp     ax, 91FBh
		je      isr15_010
		jmp     isr15_exit

isr15_010:      mov     event, 1
		cmp     busy, Not_Busy
		jne     isr15_020
		jmp     isr15_exit

; *****   Determine if still recording or playing   *****
isr15_020:      mov     ax, WORD PTR size_buff[0]
		mov     dx, WORD PTR size_buff[2]
		sub     ax, 0FFFFh
		je      isr15_030
		dec     ax              ; Set to segment beginning

isr15_030:      sbb     dx, 0
		mov     WORD PTR size_buff[0], ax
		mov     WORD PTR size_buff[2], dx
		cmp     dx, 0
		jge     isr15_040
		xor     dx, dx
		xor     ax, ax

isr15_040:      or      ax, dx
		cmp     ax, 0
		jne     isr15_050
		les     bx, flag_ptr
		mov     WORD PTR es:[bx], 1
		mov     busy, Not_Busy
		jmp     SHORT isr15_exit

isr15_050:      cmp     busy, Playing
		jne     isr15_070

; *****   Continue playing   *****
		mov     ax, buff_seg
		add     ax, 1000h       ; Point to next 64k boundary
		mov     buff_seg, ax
		mov     es, ax
		mov     bx, WORD PTR buffer[0]
		mov     cx, 0FFFFh
		cmp     dx, 0
		jne     isr15_060
		mov     cx, WORD PTR size_buff[0]

isr15_060:      mov     ax, volume
		mov     ah, 83h
		mov     dx, t_rate
		int     1Ah
		jmp     SHORT isr15_exit

; *****   Continue recording   *****
isr15_070:      mov     ax, buff_seg
		add     ax, 1000h       ; Point to next 64k boundary
		mov     buff_seg, ax
		mov     es, ax
		mov     bx, WORD PTR buffer[0]
		mov     cx, 0FFFFh
		cmp     dx, 0
		jne     isr15_080
		mov     cx, WORD PTR size_buff[0]

isr15_080:      mov     ah, 82h
		mov     dx, t_rate
		int     1Ah
		jmp     SHORT isr15_exit

; *****   Restore registers, chain to previous isr, and exit   *****
isr15_exit:     pop     ax
		pop     bx
		pop     cx
		pop     dx
		pop     es
		cli
		call    p_isr15h
		pop     ds
		sti
		ret     2
isr15h          ENDP
		END
