; PLAYWAV.ASM
;
; Version 1.4 - September 30, 1996.  This is a complete rewrite of the
; venerable Playwav program to play .wav files on the Tandy DAC chip.
;
; This version is .com rather than .exe; it manages its own memory.  It
; uses autoinitialize DMA for playback, bypassing the BIOS to avoid the
; clicks at buffer switch time that earlier versions were prone to.  The
; DMA buffer is 64k, divided into 8 8k segments.  There is also a 32k
; buffer for file I/O.  The program will terminate if unable to allocate
; the needed buffers.
;
; This program is based on information contained in the _Tandy 1000TL
; Technical Reference Manual_; the TLSLSND package by Bruce M. Terry, Jr.;
; the RIFF WAVE file format specification excerpted by Rob Ryan from the
; official Microsoft "Multimedia Programming Interface and Data Specification,
; v1.0"; and Frank Durda's PSSJ Digital Sound Toolkit.
;
; Syntax:
;   PLAYWAV <filename>
;
; There are no command-line options.
;

JMP	START

;
; Local data.
;
; Return code from the program (ERRORLEVEL for batch files).  Possible
; return codes are:
;    0      normal termination (including halting via keystroke)
;    1      insufficient memory
;    2      no Tandy DAC present
;    3      no .wav file specified on the command line
;    4      open failed on .wav file
;    5      invalid .wav file or unsupported type
;    6      read error on .wav file
;    7      file truncated (whatever's there is played)
;    8      output underflow
;
RETCODE		DB	0	; assume no problems
		;
		; Messages to be displayed to the user, one for each
		; possible return code.
		;
MSGNORMAL	DB	"Done.",0Dh,0Ah,"$"
MSGNOMEM	DB	"Insufficient memory.",0Dh,0Ah,"$"
MSGNODAC	DB	"Tandy DAC not found.",0Dh,0Ah,"$"
MSGSYNTAX	DB	"Must specify input .wav file.",0Dh,0Ah,"$"
MSGOPENFAIL	DB	"Open failed on .wav file.",0Dh,0Ah,"$"
MSGINVALID	DB	"Invalid .wav file or unsupported type.",0Dh,0Ah,"$"
MSGFILEERR	DB	"File or disk error reading .wav file.",0Dh,0Ah,"$"
MSGTRUNC	DB	".wav file is truncated.",0Dh,0Ah,"$"
MSGUNDERFLOW	DB	"Output underflow - unable to maintain sampling rate."
		DB	0Dh,0Ah
		DB	"Copy .wav to hard disk or RAM disk and try again,"
		DB	0Dh,0Ah
		DB	"convert to 8-bit mono, or convert to reduce sampling"
		DB	0Dh,0Ah,"rate.",0Dh,0Ah,"$"
		;
		; Extra messages unrelated to the return code.
		;
MSGLOADING	DB	"Playing - press a key to abort.",0Dh,0Ah,"$"
		;
		; Pointers to the termination messages above, conveniently
		; placed in an array for easy access.
		;
MSGS		DW	OFFSET MSGNORMAL
		DW	OFFSET MSGNOMEM
		DW	OFFSET MSGNODAC
		DW	OFFSET MSGSYNTAX
		DW	OFFSET MSGOPENFAIL
		DW	OFFSET MSGINVALID
		DW	OFFSET MSGFILEERR
		DW	OFFSET MSGTRUNC
		DW	OFFSET MSGUNDERFLOW
		;
		; Variables for memory allocation.
		;
FREESEG		DW	0		; paragraph address of unused RAM
ENDALLOC	EQU	WORD PTR [2]	; end of allocated RAM
DMASEG		DW	0		; segment address of 64k DMA buffer
FREESEG2	DW	0		; segment of RAM past DMA buffer
FILESEG		DW	0		; segment for file I/O
		;
		; Base I/O port address of the Tandy DAC.
		;
DACBASE		DW	0
		;
		; Filename from the command line.  Copied here so we can
		; append a .wav extension to it.
		;
FILENAME	DB	128 DUP ('*')
WAVSTR		DB	".WAV",0	; .wav extension, for appending
		;
		; File handle for the input .wav file.
		;
HANDLE		DW	0
		;
		; Default vectors for hooked interrupts.
		;
INT0FDEFAULT	DD	0
INT1BDEFAULT	DD	0
		;
		; Strings used in the .wav header.
		;
RIFFSTR		DB	"RIFF"		; RIFF signature
WAVESTR		DB	"WAVE"		; WAVE signature
FMTSTR		DB	"fmt "		; format chunk header
DATASTR		DB	"data"		; data chunk header
		;
		; Data from the .wav header.
		;
NCHANNELS	DW	0		; 1 = mono, 2 = stereo
SAMPRATE	DW	0		; sampling rate in Hz
BITSPER		DW	0		; bits per sample, 1-16
DATALEN		DD	0		; length of sound data
		;
		; Table of file reader functions.  These functions each
		; read one 8k segment of data into the DMA buffer, converting
		; as necessary to mono 8-bit.
		;
READERTBL	DW	OFFSET MONO8
		DW	OFFSET STEREO8
		DW	OFFSET MONO16
		DW	OFFSET STEREO16
		;
		; Stuff we compute from the .wav header.
		;
FMTFOUND	DB	0	; 1 if format chunk found
READER		DW	0	; pointer to file reader function
NSAMPLES	LABEL	DWORD	; number of (multichannel) samples
NOAUTOSIZE	DW	0	; number of samples left after autoinit
AUTOLOOPS	DW	0	; number of times to loop in autoinit mode
BUFFERSEGS	DD	0	; 8k buffer segments remaining to play
LASTBYTES	DW	0	; number of samples in last 8k segment
DMACLOCK	DD	3579545	; DMA clock rate, for computing divider
DMAPAGE		DB	0	; DMA page register value for our buffer
VOLUME		DB	0E0h	; DAC volume - maybe a future option?
AMPFREQ		DW	0	; value for DAC amplitude/frequency registers
		;
		; 8k buffer segment to fill with data from the file.
		; Naturally, we fill segments in order, 0-7 and back to 0.
		;
CURRENTSEG	DW	0
		;
		; 8k buffer segment that DMA was in when we started to load
		; the current segment.  This is used to determine whether
		; underflow has occurred.
		;
SEGPRE		DW	0
		;
		; Flag, set to 1 when all DMA is complete.  Also used when
		; finalizing the chip.
		;
DMADONE		DB	0
		;
		; Counter of the number of DMA interrupts that have occurred
		; since the last 8k buffer segment was loaded.  Used to
		; determine whether underflow has occurred.
		;
IRQCOUNT	DB	0
		;
		; Flag for whether we have a new-model PSSJ chip.  If so,
		; it needs to be programmed differently to ramp up to the
		; baseline or back down to the speaker idling position.
		;
NEWCHIP		DB	0	; 1 if new chip
		;
		; Values to write to the DAC to ramp it up, with
		; appropriate delays between, for the old DAC, for each
		; final volume 0-7.  These are records of the following
		; format:
		;    word           number of volume records
		;    3 bytes each   volume records
		; Volume records are in this format:
		;    byte           DAC volume
		;    byte           initial output sample
		;    byte           final volume sample
		; The table comes from the PSSJ Digital Sound Toolkit.
		;
OGAIN0		DW	1
		DB	0,80h,80h
OGAIN1		DW	1
		DB	20h,0,80h
OGAIN2		DW	2
		DB	20h,0,2Ch
		DB	40h,0,80h
OGAIN3		DW	2
		DB	20h,0,67h
		DB	60h,0,80h
OGAIN4		DW	2
		DB	20h,0,0BCh
		DB	80h,0,80h
OGAIN5		DW	3
		DB	20h,0,0BCh
		DB	80h,0,02Bh
		DB	0A0h,0,80h
OGAIN6		DW	3
		DB	20h,0,0BCh
		DB	80h,0,67h
		DB	0C0h,0,80h
OGAIN7		DW	3
		DB	20h,0,0BCh
		DB	80h,0,0BDh
		DB	0E0h,0,80h
		;
		; Values to write to the DAC to ramp it up, with
		; appropriate delays between, for the new DAC, for each
		; final volume 1-7.  See above for the format.  The table
		; comes from the PSSJ Digital Sound Toolkit.
		;
NGAIN1		DW	2
		DB	80h,0DCh,0
		DB	20h,0BDh,80h
NGAIN2		DW	2
		DB	80h,0DCh,0Dh
		DB	40h,80h,80h
NGAIN3		DW	2
		DB	80h,0DCh,3Dh
		DB	60h,80h,80h
NGAIN4		DW	1
		DB	80h,0DCh,80h
NGAIN5		DW	1
		DB	0A0h,7Dh,80h
NGAIN6		DW	1
		DB	0C0h,3Ah,80h
NGAIN7		DW	1
		DB	0E0h,0Ah,80h
		;
		; Table of pointers to instruction records to ramp up the
		; old DAC.  The desired output volume is an index into this
		; table.
		;
OGAINTBL	DW	OFFSET OGAIN0
		DW	OFFSET OGAIN1
		DW	OFFSET OGAIN2
		DW	OFFSET OGAIN3
		DW	OFFSET OGAIN4
		DW	OFFSET OGAIN5
		DW	OFFSET OGAIN6
		DW	OFFSET OGAIN7
		;
		; Table of pointers to instruction records to ramp up the
		; new DAC.  The desired output volume is an index into this
		; table.
		;
NGAINTBL	DW	OFFSET OGAIN0
		DW	OFFSET NGAIN1
		DW	OFFSET NGAIN2
		DW	OFFSET NGAIN3
		DW	OFFSET NGAIN4
		DW	OFFSET NGAIN5
		DW	OFFSET NGAIN6
		DW	OFFSET NGAIN7

;
; Routine to delay a little bit.  Used when ramping up and down.
;
DELAY:
	PUSH	CX
	MOV	CX,100
	LOOP	$
	POP	CX
	RET

;
; Routine to ramp the sound chip up from the speaker idling position to
; the baseline sample at the desired output volume.  Assumes that the DAC
; is in direct write mode, no DMA, and that the volume is initially set to
; zero.
;
RAMPUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	;
	; Get the DAC data port in DI, the volume port in DX.
	;
	MOV	DI,DACBASE
	INC	DI
	MOV	DX,DI
	INC	DX
	INC	DX
	;
	; DS:SI addresses the table of records for the chip.
	;
	MOV	SI,OFFSET OGAINTBL
	CMP	NEWCHIP,1
	JNE	RAMPUP_FINDGOAL
	MOV	SI,OFFSET NGAINTBL
	;
	; DS:SI addresses the correct record for the desired volume level.
	;
RAMPUP_FINDGOAL:
	MOV	AL,VOLUME
	MOV	AH,0
	MOV	CL,4
	SHR	AX,CL
	ADD	SI,AX
	MOV	SI,[SI]
	;
	; Clear the direction flag.  CX is the number of times we need to
	; change the volume.
	;
	CLD
	LODSW
	MOV	CX,AX
	;
	; Loop over the volume settings.
	;
RAMPUP_VOLLOOP:
	LODSW			; volume in AL, initial sample in AH
	;
	; Set the volume and the initial sample for that volume.
	;
	CLI			; do this as fast as possible
	OUT	DX,AL		; set the volume
	XCHG	DI,DX		; DX = data port, DI = volume port
	MOV	AL,AH		; initial sample in AL, copy in AH
	OUT	DX,AL		; write the initial sample
	STI
	;
	; Wait for a little bit.
	;
	CALL	DELAY
	;
	; Figure out whether we're going up or down.
	;
	LODSB			; final sample in AL
	XCHG	AL,AH		; current sample in AL, final sample in AH
	CMP	AL,AH
	JE	RAMPUP_VOLLPEND	; if current = final, go to next volume
	JA	RAMPUP_DOWNLOOP	; else if current > final, going down
	;
	; Here we slowly increment the sample value until we get to the
	; final one for this volume setting.
	;
RAMPUP_UPLOOP:
	INC	AL		; increment the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,AH		; go again if not yet at final value
	JNE	RAMPUP_UPLOOP
	JMP	RAMPUP_VOLLPEND	; if no more, go to next volume setting
	;
	; Here we slowly decrement the sample value until we get to the
	; final one for this volume setting.
	;
RAMPUP_DOWNLOOP:
	DEC	AL		; decrement the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,AH		; go again if not yet at final value
	JNE	RAMPUP_DOWNLOOP
	;
	; Get the volume port back in DX, data port in DI, and go to the
	; next volume setting.
	;
RAMPUP_VOLLPEND:
	XCHG	DI,DX		; DX = volume port, DI = data port
	LOOP	RAMPUP_VOLLOOP
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
	RET

;
; Routine to ramp the sound chip down from the baseline sample at the
; current output volume to the speaker idling position.  Assumes that the
; DAC is in direct write mode, no DMA.
;
RAMPDOWN:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	;
	; Get the DAC data port in DI, the volume port in DX.
	;
	MOV	DI,DACBASE
	INC	DI
	MOV	DX,DI
	INC	DX
	INC	DX
	;
	; DS:SI addresses the table of records for the chip.
	;
	MOV	SI,OFFSET OGAINTBL
	CMP	NEWCHIP,1
	JNE	RAMPDOWN_FINDGOAL
	MOV	SI,OFFSET NGAINTBL
	;
	; DS:SI addresses the correct record for the desired volume level.
	;
RAMPDOWN_FINDGOAL:
	MOV	AL,VOLUME
	MOV	AH,0
	MOV	CL,4
	SHR	AX,CL
	ADD	SI,AX
	MOV	SI,[SI]
	;
	; CX is the number of times we need to change the volume.  DS:SI
	; points to the last byte of the record for the output volume.
	;
	MOV	CX,[SI]
	ADD	SI,CX
	ADD	SI,CX
	ADD	SI,CX
	INC	SI
	;
	; Loop over the volume settings.  We do this with DF set since
	; we're going backwards.
	;
	STD
RAMPDOWN_VOLLOOP:
	LODSB			; initial sample in AL
	MOV	AH,AL		; initial sample in AH
	LODSB			; final sample in AL
	MOV	BL,AL		; final sample in BL
	LODSB			; volume in AL, initial sample in AH
	;
	; Set the volume and the initial sample for that volume.
	;
	CLI			; do this as fast as possible
	OUT	DX,AL		; set the volume
	XCHG	DI,DX		; DX = data port, DI = volume port
	MOV	AL,AH		; initial sample in AL
	OUT	DX,AL		; write the initial sample
	STI
	;
	; Wait for a little bit.
	;
	CALL	DELAY
	;
	; Figure out whether we're going up or down.
	;
	CMP	AL,BL
	JE	RAMPDOWN_VOLLPEND ; if current = final, go to next volume
	JA	RAMPDOWN_DOWNLOOP ; else if current > final, going down
	;
	; Here we slowly increment the sample value until we get to the
	; final one for this volume setting.
	;
RAMPDOWN_UPLOOP:
	INC	AL		; increment the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,BL		; go again if not yet at final value
	JNE	RAMPDOWN_UPLOOP
	JMP	RAMPDOWN_VOLLPEND ; if no more, go to next volume setting
	;
	; Here we slowly decrement the sample value until we get to the
	; final one for this volume setting.
	;
RAMPDOWN_DOWNLOOP:
	DEC	AL		; decrement the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,BL		; go again if not yet at final value
	JNE	RAMPDOWN_DOWNLOOP
	;
	; Get the volume port back in DX, data port in DI, and go to the
	; next volume setting.
	;
RAMPDOWN_VOLLPEND:
	XCHG	DI,DX		; DX = volume port, DI = data port
	LOOP	RAMPDOWN_VOLLOOP
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Handler for <control>-C and <control>-<break>.  Does nothing, disabling
; those keys.
;
INT1BHDLR:
INT23HDLR:
	IRET

;
; Critical error handler.  Fails the system call (DOS 3.1 or later).  This
; will abort the program under earlier DOSes, but if playing on a Tandy
; system, DOS 3.3 or later can be assumed, and PSSJ expansion cards are
; very rare.
;
INT24HDLR:
	MOV	AL,3
	IRET 

;
; Routine to check for a Tandy DAC.  If present, clears carry and returns
; with the DAC base port address in AX.  If not, returns with carry set.
;
CHKDAC:
	PUSH	CX
	PUSH	DX
	;
	; Check for PCMCIA Socket Services.
	;
	MOV	AX,8003h
	XOR	CX,CX
	INT	1Ah
	CMP	CX,5353h
	JE	CHKDAC_NODAC
	;
	; Check for Tandy DAC.
	;
	MOV	AX,8100h
	INT	1Ah
	CMP	AX,8000h
	JAE	CHKDAC_NODAC
	MOV	DX,AX		; DX = base DAC port + 2
	ADD	DX,2
	CLI
	IN	AL,DX		; get current value and save in CL
	MOV	CL,AL
	MOV	CH,1		; set CH=1 (assume no DAC)
	XOR	AL,AL		; clear all the bits
	OUT	DX,AL
	IN	AL,DX		; read them back
	OR	AL,AL			; all clear?
	JNZ	CHKDAC_DACCHKDONE	; if not, no DAC
	NOT	AL			; set all the bits
	OUT	DX,AL
	IN	AL,DX			; read them back
	NOT	AL			; all set?
	JNZ	CHKDAC_DACCHKDONE	; if not, no DAC
	MOV	CH,0		; CH=0 (definitely a DAC here)
CHKDAC_DACCHKDONE:
	MOV	AL,CL		; restore previous value at DAC base + 2
	OUT	DX,AL
	STI
	MOV	AX,DX		; get DAC base in AX
	SUB	AX,2
	RCR	CH,1		; clear carry if DAC was there, set if not
	JMP	CHKDAC_END
CHKDAC_NODAC:
	STC
CHKDAC_END:
	POP	DX
	POP	CX
	RET

;
; Routine to determine whether we have a new-model DAC with the double
; buffer or not.  Returns nothing, sets NEWCHIP.  This information is
; needed for ramping the chip up to the baseline and back down again.
;
CHKVER:
	PUSH	AX
	PUSH	DX
	MOV	DX,DACBASE	; address amplitude/frequency MSB register
	ADD	DX,3
	IN	AL,DX		; clear bit 4
	AND	AL,0EFh
	OUT	DX,AL
	OR	AL,10h		; set bit 4
	OUT	DX,AL
	IN	AL,DX		; is it set?
	TEST	AL,10h
	JZ	CHKVER_NEW	; if not, it's a new version
	AND	AL,0EFh		; clear bit 4
	OUT	DX,AL
	IN	AL,DX		; is it clear?
	TEST	AL,10h
	JNZ	CHKVER_NEW	; if not, it's a new version
	MOV	NEWCHIP,0
	JMP	CHKVER_DONE
CHKVER_NEW:
	MOV	NEWCHIP,1
CHKVER_DONE:
	POP	DX
	POP	AX
	RET

;
; Subroutine, takes pointer to string in DS:SI and length of string in CX,
; skips over blanks and tabs, returns pointer to first nonblank character
; in the string in DS:SI, length of remaining string in CX.  If end of
; string is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPBLANKS:
	PUSH	AX
SKIPBLANKS_LOOP:
	JCXZ	SKIPBLANKS_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPBLANKS_LOOP
	CMP	AL,20h
	JE	SKIPBLANKS_LOOP
	DEC	SI
	INC	CX
SKIPBLANKS_END:
	POP	AX
	RET

;
; Subroutine, takes pointer to string is DS:SI and length of string in CX,
; skips over nonblank characters, returns pointer to first blank or tab in
; the string in DS:SI, length of remaining string in CX.  If end of string
; is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPNONBLANK:
	PUSH	AX
SKIPNONBLANK_LOOP:
	JCXZ	SKIPNONBLANK_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPNONBLANK_LPEND
	CMP	AL,20h
	JNE	SKIPNONBLANK_LOOP
SKIPNONBLANK_LPEND:
	DEC	SI
	INC	CX
SKIPNONBLANK_END:
	POP	AX
	RET

;
; Routine to get the filename from the command line.  This routine assumes
; that DS addresses the PSP (i.e., .com program).  Returns the offset of
; the ASCIIZ pathname in SI.  Clears carry if successful, sets it if there
; is no filename specified.
;
GETNAME:
	PUSH	CX
	MOV	CL,[80h]	; CX = length of command line
	MOV	CH,0
	MOV	SI,81h		; SI -> command line
	CLD			; DF set for incrementing
	;
	; Skip over blanks and tabs.
	;
	CALL	SKIPBLANKS
	JCXZ	GETNAME_NONE
	PUSH	SI		; save pointer to the name
	;
	; Skip over nonblank characters to find the end of the name and
	; append a null.
	;
	CALL	SKIPNONBLANK
	MOV	BYTE PTR [SI],0
	POP	SI		; get back the name pointer
	;
	; Got the name.
	;
	CLC
	JMP	GETNAME_DONE
	;
	; No filename specified.
	;
GETNAME_NONE:
	STC
GETNAME_DONE:
	POP	CX
	RET

;
; Subroutine, copies the output filename from the command line into the
; local buffer FILENAME and appends the ".WAV" extension if the name doesn't
; already have an extension.  Returns nothing.  On entry, DS:SI addresses
; the ASCIIZ name from the command line.
;
WAVEXT:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	SI
	PUSH	DI
	;
	; Get the length of the name from the command line.
	;
	MOV	DI,SI		; ES:DI -> name from command line
	MOV	CX,128
	MOV	AL,0
	CLD
	REPNE	SCASB
	SUB	DI,SI		; DI is length of string
	MOV	CX,DI		; get length in CX for move (including null)
	DEC	DI		; go back to the terminating null
	MOV	BX,DI		; save length in BX for scan (no null)
	;
	; Move the name from the command line to the FILENAME buffer.
	;
	MOV	DI,OFFSET FILENAME
	REP	MOVSB		; copy to FILENAME buffer
	;
	; Scan for a period in the name.
	;
	MOV	DI,OFFSET FILENAME	; DI addresses the name
	MOV	AL,'.'			; scan for a period
	MOV	CX,BX			; get count for scan
	REPNE	SCASB
	JE	WAVEXT_EXTFOUND
	;
	; No extension is present.  Append one.
	;
	MOV	SI,OFFSET WAVSTR	; SI -> ".WAV" (ASCIIZ)
	MOV	CX,5			; move 5 bytes
	REP	MOVSB			; append the extension
	;
	; All done.
	;
WAVEXT_EXTFOUND:
	POP	DI
	POP	SI
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to read and process the .wav header.  Determines the sampling
; rate, number of channels, width of a sample, etc.  Leaves the file pointer
; at the start of the actual sound.  Clears carry if successful, sets it
; if not (not a .wav file, file read error, more than 16 bits per channel,
; more than 2 channels, unsupported format type (not Microsoft PCM)).
; Assumes that DS and ES both address local data.
;
DOHDR:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	DS
	;
	; Read RIFF and WAVE headers from file.
	;
	MOV	AH,3Fh		; read file or device
	MOV	BX,HANDLE	; BX always == HANDLE below
	MOV	CX,12		; 12 bytes
	MOV	DX,FILESEG	; DS:DX -> file I/O buffer
	MOV	DS,DX		; DS does not address local data! Use CS:
	XOR	DX,DX
	INT	21h
	JNC	>L1
	JMP	DOHDR_FILEERR	; CF set => file or disk error
L1:
	CMP	AX,CX		; check for premature EOF
	JE	>L2
	JMP	DOHDR_INVALID
	;
	; Verify the RIFF header.
	;
L2:
	MOV	SI,DX			; DS:SI -> RIFF header from file
	MOV	DI,OFFSET RIFFSTR	; ES:DI -> "RIFF"
	MOV	CX,4
	CLD
	REPE	CMPSB
	JE	>L3
	JMP	DOHDR_INVALID		; invalid if signature not found
L3:
	ADD	SI,4			; DS:SI -> "WAVE" string from file
	MOV	DI,OFFSET WAVESTR	; ES:DI -> "WAVE"
	MOV	CX,4
	REPE	CMPSB
	JE	DOHDR_CHUNKLP
	JMP	DOHDR_INVALID		; invalid if signature not found
	;
	; Loop over the chunks, looking for the format and data chunks,
	; skipping over the others.
	;
DOHDR_CHUNKLP:
	MOV	AH,3Fh		; read file or device (BX == HANDLE still)
	MOV	CX,8		; 8 bytes
	XOR	DX,DX		; DS:DX -> file buffer (DS == FILESEG still)
	INT	21h
	JNC	>L4
	JMP	DOHDR_FILEERR	; CF set => file or disk error
L4:
	CMP	AX,CX		; check for premature EOF
	JE	>L5
	JMP	DOHDR_INVALID
	;
	; Check for a format chunk.
	;
L5:
	MOV	SI,DX			; DS:SI -> chunk tag from file
	MOV	DI,OFFSET FMTSTR	; ES:DI -> "fmt "
	MOV	CX,4
	CLD
	REPE	CMPSB
	JNE	DOHDR_NOTFMT
	;
	; Format chunk, error if we found one before.
	;
	CMP	CS:FMTFOUND,0
	JE	>L6
	JMP	DOHDR_INVALID
	;
	; Verify the chunk length.
	;
L6:
	MOV	CX,[SI]		; CX = length of chunk, hopefully
	CMP	CX,16
	JE	>L7
	JMP	DOHDR_INVALID
L7:
	CMP	WORD PTR [SI+2],0
	JE	>L8
	JMP	DOHDR_INVALID
	;
	; Length verifies.  Read in the chunk.
	;
L8:
	MOV	AH,3Fh		; read file or device (BX == HANDLE still)
	XOR	DX,DX		; DS:DX -> file buffer (DS == FILESEG)
	INT	21h		; CX set to 16 above, reading 16 bytes
	JNC	>L9
	JMP	DOHDR_FILEERR	; CF set => file or disk error
L9:
	CMP	AX,CX		; check for premature EOF
	JNE	DOHDR_INVALID
	;
	; Get data from the format chunk.
	;
	MOV	SI,DX		; DS:SI -> format chunk from file
	CLD
	LODSW			; AX = format code
	CMP	AX,1
	JNE	DOHDR_INVALID	; unsupported if not Microsoft PCM
	LODSW			; AX = number of channels
	OR	AX,AX
	JZ	DOHDR_INVALID	; 0 channels is invalid
	CMP	AX,2
	JA	DOHDR_INVALID	; more than 2 channels is unsupported
	MOV	CS:NCHANNELS,AX	; save number of channels
	LODSW			; AX = sampling rate, hopefully
	MOV	DX,AX		; save in DX
	LODSW
	OR	AX,AX
	JNZ	DOHDR_INVALID	; rate > 65535 Hz is unsupported
	CMP	DX,875
	JB	DOHDR_INVALID	; so is rate < 875 Hz
	MOV	CS:SAMPRATE,DX	; save sampling rate
	ADD	SI,6		; skip over bytes/sec and block align
	LODSW			; AX = bits/sample
	OR	AX,AX
	JZ	DOHDR_INVALID	; 0-bit samples are invalid
	CMP	AX,16
	JA	DOHDR_INVALID	; more than 16 bits per sample is unsupported
	MOV	CS:BITSPER,AX	; save bits per sample
	MOV	CS:FMTFOUND,1	; we got a valid format chunk
	JMP	DOHDR_CHUNKLP	; get the next chunk
	;
	; Not a format chunk.  Check for a data chunk.
	;
DOHDR_NOTFMT:
	MOV	SI,DX			; DS:SI -> chunk tag from file
	MOV	DI,OFFSET DATASTR	; ES:DI -> "data"
	MOV	CX,4
	REPE	CMPSB			; DF still clear from above
	JNE	DOHDR_NOTDATA
	CMP	CS:FMTFOUND,1		; error if no format chunk
	JNE	DOHDR_INVALID
	MOV	DI,OFFSET DATALEN	; save the length of the sound data
	MOVSW
	MOVSW
	;
	; Valid .wav header processed successfully.
	;
	MOV	CS:RETCODE,0
	CLC
	JMP	DOHDR_DONE
	;
	; Not a format or data chunk.  Skip over it.
	;
DOHDR_NOTDATA:
	MOV	AX,4201h		; seek relative to current pointer
	MOV	DX,[4]
	MOV	CX,[6]
	INT	21h
	JC	DOHDR_FILEERR		; seek error
	JMP	DOHDR_CHUNKLP
	;
	; Invalid or truncated .wav file, or unsupported type.
	;
DOHDR_INVALID:
	MOV	CS:RETCODE,5
	STC
	JMP	DOHDR_DONE
	;
	; File or disk error while reading from .wav file.
	;
DOHDR_FILEERR:
	MOV	CS:RETCODE,6
	STC
	JMP	DOHDR_DONE
	;
	; All done.
	;
DOHDR_DONE:
	POP	DS
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Main setup routine.  This routine computes various variables that will
; be needed during playback, based on the data read from the .wav header.
; It also checks for a truncated file.
;
SETUP:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	;
	; Check for a truncated file.  First, get the current file pointer
	; position and save in DI:SI.
	;
	MOV	AX,4201h	; seek relative to current position
	MOV	BX,HANDLE
	XOR	CX,CX		; offset 0
	MOV	DX,CX
	INT	21h
	JC	SETUP_FILEERR	; seek error
	MOV	DI,DX
	MOV	SI,AX
	;
	; Seek to the end of the file and get the position there.
	;
	MOV	AX,4202h	; seek relative to EOF
	MOV	DX,CX		; CX is still zero
	INT	21h		; BX == HANDLE still
	JC	SETUP_FILEERR	; seek error
	;
	; Subtract to determine the number of bytes from the start of the
	; data chunk to the end of the file.
	;
	SUB	AX,SI
	SBB	DX,DI
	JB	SETUP_FILEERR	; already past EOF?! should never be
	;
	; Save the length of the file from here to the end in DI:SI, and
	; get the start of the data chunk back in CX:DX.
	;
	MOV	CX,DI
	MOV	DI,DX
	MOV	DX,SI
	MOV	SI,AX
	;
	; Seek back to the start of the data chunk.
	;
	MOV	AX,4200h	; seek relative to beginning of file
	INT	21h		; BX == HANDLE still, CX:DX is where we began
	JNC	SETUP_CHKTRUNC	; seek error
	;
	; Seek error on the .wav file.
	;
SETUP_FILEERR:
	MOV	RETCODE,6
	STC
	JMP	SETUP_DONE
	;
	; DI:SI is now the length of the file past the start of the data
	; chunk.  Check whether it is less than the length of data from
	; the .wav header, indicating a truncated file.
	;
SETUP_CHKTRUNC:
	MOV	AX,SI		; copy to DX:AX
	MOV	DX,DI
	SUB	AX,WORD PTR DATALEN
	SBB	DX,WORD PTR DATALEN+2
	JNB	SETUP_NOTRUNC
	;
	; The file is truncated.  Adjust the data length to match the
	; actual size of the file.
	;
	MOV	WORD PTR DATALEN,SI
	MOV	WORD PTR DATALEN+2,DI
	MOV	RETCODE,7	; set return code for a truncated file
	;
	; If the file is truncated, we dealt with it.  Determine the number
	; of bytes per mono sample.
	;
SETUP_NOTRUNC:
	MOV	AX,BITSPER
	ADD	AX,7		; round number of bits up to multiple of 8
	SHR	AX,1		;   (see the .wav specification)
	SHR	AX,1
	SHR	AX,1
	;
	; Determine the sample type:  mono 8-bit (0), stereo 8-bit (2),
	; mono 16-bit (4), or stereo 16-bit (6).  The type is actually an
	; offset into a table of function pointers.  Also, compute the
	; number of bits to shift to convert bytes of sample data to number
	; of samples.
	;
	DEC	AX		; AX = 0 if 8-bit, 1 if 16-bit
	MOV	CX,AX		; CX = 0 if 8-bit, 1 if 16-bit
	MOV	BX,NCHANNELS	; BX = 1 if mono, 2 if stereo
	DEC	BX		; BX = 0 if mono, 1 if stereo
	ADD	CX,BX		; CX = shift to convert bytes to samples
	SHR	BX,1		; CF = 0 if mono, 1 if stereo
	RCL	AX,1		; AX = 0 if mono 8-bit, 1 if stereo 8-bit,
				;      2 if mono 16-bit, 3 if stereo 16-bit
	SHL	AX,1		; AX = 0 if mono 8-bit, 2 if stereo 8-bit,
				;      4 if mono 16-bit, 6 if stereo 16-bit
	;
	; AX is the type and CX is the number of shifts to convert bytes
	; to samples.  Set the file reader and the number of samples.
	;
	MOV	BX,AX			; set the file reader function
	MOV	AX,READERTBL[BX]
	MOV	READER,AX
	MOV	AX,WORD PTR DATALEN
	MOV	DX,WORD PTR DATALEN+2
	JCXZ	SETUP_NOSHIFT
	SHR	DX,1
	RCR	AX,1
	DEC	CX
	JZ	SETUP_NOSHIFT
	SHR	DX,1
	RCR	AX,1
SETUP_NOSHIFT:
	MOV	WORD PTR NSAMPLES,AX
	MOV	WORD PTR NSAMPLES+2,DX
	;
	; Compute the number of (whole) 8k buffer segments we have.
	;
	MOV	CX,13
SETUP_SEGLP1:
	SHR	DX,1
	RCR	AX,1
	LOOP	SETUP_SEGLP1
	;
	; See if there's a partial segment.  Adjust the number of buffer
	; segments to include the partial one, and set the number of samples
	; in the last (possibly partial) segment.
	;
	MOV	LASTBYTES,8192
	MOV	CX,WORD PTR NSAMPLES
	AND	CX,1FFFh
	JZ	SETUP_LASTOK
	MOV	LASTBYTES,CX
	ADD	AX,1
	ADC	DX,0
SETUP_LASTOK:
	MOV	WORD PTR BUFFERSEGS,AX
	MOV	WORD PTR BUFFERSEGS+2,DX
	;
	; Compute the DAC divider value for the sampling rate used, then
	; the word to program at ports DACBASE+2 and DACBASE+3 to play
	; at that rate.
	;
	MOV	AX,WORD PTR DMACLOCK
	MOV	DX,WORD PTR DMACLOCK+2
	MOV	BX,SAMPRATE
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	OR	AH,VOLUME
	MOV	AMPFREQ,AX
	;
	; Determine the value to be programmed in the DMA channel 1 page
	; register.
	;
	MOV	AX,DMASEG
	MOV	CL,4
	SHR	AH,CL
	MOV	DMAPAGE,AH
	;
	; All done.
	;
	CLC
SETUP_DONE:
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; File reader function for 8-bit mono files.  Loads samples into the
; buffer segment numbered CURRENTSEG (0-7).  Decrements BUFFERSEGS.  The
; number of samples loaded is 8k, unless this is the last segment, in which
; case LASTBYTES samples are loaded.  Returns carry clear if successful;
; otherwise, sets RETCODE to 6 indicating a read error on the input file
; and returns carry set.  Premature EOF is considered a read error since
; truncated files have already been dealt with in the SETUP procedure.
;     This reader has no conversion to do and so does not need the file
; I/O buffer; it reads directly into the DMA buffer.
;
MONO8:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DS
	;
	; Get the offset of CURRENTSEG in the DMA buffer in DX.
	;
	MOV	DX,CURRENTSEG
	XCHG	DL,DH
	MOV	CL,5
	SHL	DH,CL
	;
	; Get number of samples to load in CX.  Decrement BUFFERSEGS.
	;
	MOV	CX,8192
	MOV	AX,WORD PTR BUFFERSEGS
	MOV	BX,WORD PTR BUFFERSEGS+2
	SUB	AX,1
	SBB	BX,0
	MOV	WORD PTR BUFFERSEGS,AX
	MOV	WORD PTR BUFFERSEGS+2,BX
	OR	AX,BX
	JNZ	MONO8_NOTLAST
	MOV	CX,LASTBYTES
	; 
	; Read in some data.
	;
MONO8_NOTLAST:
	MOV	AH,3Fh
	MOV	BX,HANDLE
	MOV	DS,DMASEG	; DS not pointing at local data any more!
	INT	21h
	JC	MONO8_FILEERR
	CMP	AX,CX
	JNE	MONO8_FILEERR
	;
	; Seems fine.
	;
	CLC
	JMP	MONO8_DONE
	;
	; File or disk error.
	;
MONO8_FILEERR:
	MOV	CS:RETCODE,6
	STC
MONO8_DONE:
	POP	DS
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; File reader function for 8-bit stereo files.  Loads samples into the
; buffer segment numbered CURRENTSEG (0-7).  Decrements BUFFERSEGS.  The
; number of samples loaded is 8k, unless this is the last segment, in which
; case LASTBYTES samples are loaded.  Returns carry clear if successful;
; otherwise, sets RETCODE to 6 indicating a read error on the input file
; and returns carry set.  Premature EOF is considered a read error since
; truncated files have already been dealt with in the SETUP procedure.
;     This reader mixes stereo to mono.
;
STEREO8:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	DS
	PUSH	ES
	;
	; Get number of samples to load in CX.  Decrement BUFFERSEGS.
	;
	MOV	CX,8192
	MOV	AX,WORD PTR BUFFERSEGS
	MOV	BX,WORD PTR BUFFERSEGS+2
	SUB	AX,1
	SBB	BX,0
	MOV	WORD PTR BUFFERSEGS,AX
	MOV	WORD PTR BUFFERSEGS+2,BX
	OR	AX,BX
	JNZ	STEREO8_NOTLAST
	MOV	CX,LASTBYTES
	; 
	; Read in some data.
	;
STEREO8_NOTLAST:
	MOV	AH,3Fh
	MOV	BX,HANDLE
	SHL	CX,1		; n samples = 2*n bytes
	MOV	DS,FILESEG	; DS not pointing at local data any more!
	XOR	DX,DX
	INT	21h
	JC	STEREO8_FILEERR
	CMP	AX,CX
	JNE	STEREO8_FILEERR
	;
	; Get the offset of CURRENTSEG in the DMA buffer in DI.
	;
	MOV	DI,CS:CURRENTSEG
	MOV	CL,13
	SHL	DI,CL
	;
	; CX is the number of samples again.  DS:SI -> file buffer, ES:DI
	; -> DMA buffer.  DF clear for incrementing.
	;
	SHR	AX,1
	MOV	CX,AX
	XOR	SI,SI
	MOV	ES,CS:DMASEG
	CLD
	;
	; Mix the stereo samples to mono and store the results in the DMA
	; buffer.
	;
STEREO8_MIXLOOP:
	LODSW
	ADD	AL,AH
	RCR	AL,1
	STOSB
	LOOP	STEREO8_MIXLOOP
	;
	; Seems fine.
	;
	CLC
	JMP	STEREO8_DONE
	;
	; File or disk error.
	;
STEREO8_FILEERR:
	MOV	CS:RETCODE,6
	STC
STEREO8_DONE:
	POP	ES
	POP	DS
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; File reader function for 16-bit mono files.  Loads samples into the
; buffer segment numbered CURRENTSEG (0-7).  Decrements BUFFERSEGS.  The
; number of samples loaded is 8k, unless this is the last segment, in which
; case LASTBYTES samples are loaded.  Returns carry clear if successful;
; otherwise, sets RETCODE to 6 indicating a read error on the input file
; and returns carry set.  Premature EOF is considered a read error since
; truncated files have already been dealt with in the SETUP procedure.
;     This reader converts 16-bit to 8-bit.
;
MONO16:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	DS
	PUSH	ES
	;
	; Get number of samples to load in CX.  Decrement BUFFERSEGS.
	;
	MOV	CX,8192
	MOV	AX,WORD PTR BUFFERSEGS
	MOV	BX,WORD PTR BUFFERSEGS+2
	SUB	AX,1
	SBB	BX,0
	MOV	WORD PTR BUFFERSEGS,AX
	MOV	WORD PTR BUFFERSEGS+2,BX
	OR	AX,BX
	JNZ	MONO16_NOTLAST
	MOV	CX,LASTBYTES
	; 
	; Read in some data.
	;
MONO16_NOTLAST:
	MOV	AH,3Fh
	MOV	BX,HANDLE
	SHL	CX,1		; n samples = 2*n bytes
	MOV	DS,FILESEG	; DS not pointing at local data any more!
	XOR	DX,DX
	INT	21h
	JC	MONO16_FILEERR
	CMP	AX,CX
	JNE	MONO16_FILEERR
	;
	; Get the offset of CURRENTSEG in the DMA buffer in DI.
	;
	MOV	DI,CS:CURRENTSEG
	MOV	CL,13
	SHL	DI,CL
	;
	; CX is the number of samples again.  DS:SI -> file buffer, ES:DI
	; -> DMA buffer.  DF clear for incrementing.
	;
	SHR	AX,1
	MOV	CX,AX
	XOR	SI,SI
	MOV	ES,CS:DMASEG
	CLD
	;
	; Convert 16-bit signed samples to 8-bit unsigned and store the
	; results in the DMA buffer.
	;
MONO16_MIXLOOP:
	LODSW
	MOV	AL,AH
	ADD	AL,128
	STOSB
	LOOP	MONO16_MIXLOOP
	;
	; Seems fine.
	;
	CLC
	JMP	MONO16_DONE
	;
	; File or disk error.
	;
MONO16_FILEERR:
	MOV	CS:RETCODE,6
	STC
MONO16_DONE:
	POP	ES
	POP	DS
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; File reader function for 16-bit stereo files.  Loads samples into the
; buffer segment numbered CURRENTSEG (0-7).  Decrements BUFFERSEGS.  The
; number of samples loaded is 8k, unless this is the last segment, in which
; case LASTBYTES samples are loaded.  Returns carry clear if successful;
; otherwise, sets RETCODE to 6 indicating a read error on the input file
; and returns carry set.  Premature EOF is considered a read error since
; truncated files have already been dealt with in the SETUP procedure.
;     This reader converts 16-bit to 8-bit and mixes to mono.
;
STEREO16:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	DS
	PUSH	ES
	;
	; Get number of samples to load in CX.  Decrement BUFFERSEGS.
	;
	MOV	CX,8192
	MOV	AX,WORD PTR BUFFERSEGS
	MOV	BX,WORD PTR BUFFERSEGS+2
	SUB	AX,1
	SBB	BX,0
	MOV	WORD PTR BUFFERSEGS,AX
	MOV	WORD PTR BUFFERSEGS+2,BX
	OR	AX,BX
	JNZ	STEREO16_NOTLAST
	MOV	CX,LASTBYTES
	; 
	; Read in some data.
	;
STEREO16_NOTLAST:
	MOV	AH,3Fh
	MOV	BX,HANDLE
	SHL	CX,1		; n samples = 4*n bytes
	SHL	CX,1
	MOV	DS,FILESEG	; DS not pointing at local data any more!
	XOR	DX,DX
	INT	21h
	JC	STEREO16_FILEERR
	CMP	AX,CX
	JNE	STEREO16_FILEERR
	;
	; Get the offset of CURRENTSEG in the DMA buffer in DI.
	;
	MOV	DI,CS:CURRENTSEG
	MOV	CL,13
	SHL	DI,CL
	;
	; CX is the number of samples again.  DS:SI -> file buffer, ES:DI
	; -> DMA buffer.  DF clear for incrementing.
	;
	SHR	AX,1
	SHR	AX,1
	MOV	CX,AX
	XOR	SI,SI
	MOV	ES,CS:DMASEG
	CLD
	;
	; Mix the stereo samples to mono, converting them from 16-bit
	; signed to 8-bit unsigned, and store the results in the DMA buffer.
	;
STEREO16_MIXLOOP:
	LODSW			; get left channel and store in DX
	MOV	DX,AX
	LODSW			; get right channel in AX
	ADD	AH,128		; convert both channels to unsigned
	ADD	DH,128
	ADD	AX,DX		; add them
	RCR	AX,1		; divide by 2
	MOV	AL,AH		; get most significant byte in AL
	STOSB			; put it in the DMA buffer
	LOOP	STEREO16_MIXLOOP
	;
	; Seems fine.
	;
	CLC
	JMP	STEREO16_DONE
	;
	; File or disk error.
	;
STEREO16_FILEERR:
	MOV	CS:RETCODE,6
	STC
STEREO16_DONE:
	POP	ES
	POP	DS
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Interrupt handler for IRQ 7, the sound chip interrupt, for non-
; autoinitialize mode.
;
INT0FNOAUTO:
	CLI
	PUSH	AX
	PUSH	DX
        ;
        ; Read the interrupt controller's in-service register to see if an
        ; IRQ 7 has in fact occurred.  If not (electrical noise), just restore
        ; registers and return.
        ;
        MOV	AL,0Bh
        OUT	20h,AL
        JMP	$+2
        IN	AL,20h
        TEST	AL,80h
        JZ	INT0FNOAUTO_RESTORE
	;
	; Check if it was a DAC interrupt.  If not, just issue EOI and
	; return.
	;
	MOV	DX,CS:DACBASE
	IN	AL,DX
	TEST	AL,8
	JZ	INT0FNOAUTO_EOI
	;
	; It's our interrupt, all right.  Set the flag so that the main
	; program will know the interrupt occurred.  In non-autoinitialize
	; mode, we're all done playing sound at this point.  Clear the
	; DMA interrupt at the sound chip.
	;
	MOV	CS:DMADONE,1
	AND	AL,0F7h
	OUT	DX,AL
	OR	AL,8
	OUT	DX,AL
	;
	; Disable DMA channel 1, stopping playback.  SOUNDSTOP will unhook
	; the interrupt and put the sound chip back in joystick mode.
	;
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Issue EOI to the interrupt controller.
	;
INT0FNOAUTO_EOI:
	MOV	AL,20h
	OUT	20h,AL
INT0FNOAUTO_RESTORE:
	POP	DX
	POP	AX
	IRET			; will set the interrupt bit

;
; Routine to start non-autoinitialize mode playback, used when there are
; 64k or fewer samples in the sound file.  Programs the DMA controller,
; sound chip, and interrupt controller, and hooks Int 0Fh.
;
NOAUTOSTART:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	ES
	;
	; Disable IRQ 7 at the interrupt controller.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	;
	; Disable DMA channel 1.
	;
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Program the sound chip for playback mode, no DMA yet.
	;
	MOV	DX,DACBASE	; get DAC base port
	MOV	AL,10h		; enable DMA interrupt, still joystick mode
	OUT	DX,AL
	ADD	DX,3		; set sound volume to zero
	XOR	AL,AL
	OUT	DX,AL
	MOV	DX,DACBASE	; direct write to DAC, DMA disabled, DMA
	MOV	AL,13h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	STI
	;
	; Ramp up the chip to the baseline.  It takes a while to do it, so
	; it should be done with interrupts enabled.
	;
	CALL	RAMPUP
	;
	; Program the sound chip for playback with DMA.
	;
	CLI
	MOV	DX,DACBASE	; direct write to DAC, DMA enabled, DMA
	MOV	AL,17h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	MOV	AL,1Fh		; direct write to DAC, DMA enabled, DMA
	OUT	DX,AL		;   interrupt enabled, DMA interrupt allowed
	MOV	AX,AMPFREQ	; program amplitude/frequency
	ADD	DX,2
	OUT	DX,AL
	INC	DX
	MOV	AL,AH
	OUT	DX,AL
	;
	; Set up the DMA controller.
	;
	MOV	AL,5		; disable channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,49h		; select channel 1, read transfer from memory,
	OUT	0Bh,AL		;   autoinitialization disabled, address incre-
	JMP	$+2		;   ment, single mode
	MOV	AL,DMAPAGE	; set DMA channel 1 page register
	OUT	83h,AL
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	MOV	AX,NOAUTOSIZE	; get the number of samples to play
	DEC	AX		; program count = (# of bytes) - 1
	OUT	03h,AL		;   program low word
	JMP	$+2
	MOV	AL,AH		;   program high word
	OUT	03h,AL
	JMP	$+2
	MOV	AL,0		; set base address
	OUT	02h,AL		;   program low word
	JMP	$+2
	OUT	02h,AL		;   program high word
	;
	; Hook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT0FDEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT0FDEFAULT+2,AX
	MOV	ES:[BX],OFFSET INT0FNOAUTO
	MOV	ES:[BX+2],CS
	;
	; Enable IRQ7 at the interrupt controller.
	;
	IN	AL,21h
	JMP	$+2
	AND	AL,7Fh
	OUT	21h,AL
	;
	; Enable DMA channel 1, starting playback.
	;
	MOV	AL,1
	OUT	0Ah,AL
	STI
	POP	ES
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Interrupt handler for IRQ 7, the sound chip interrupt, for autoinitialize
; mode.  This interrupt handler has a bit more to do that the non-
; autoinitialize one so is more complicated.
;
INT0FAUTO:
	CLI
	PUSH	AX	; just save AX here, we might not need DX or DS
        ;
        ; Read the interrupt controller's in-service register to see if an
        ; IRQ 7 has in fact occurred.  If not (electrical noise), just restore
        ; registers and return.
        ;
        MOV	AL,0Bh
        OUT	20h,AL
        JMP	$+2
        IN	AL,20h
        TEST	AL,80h
        JZ	INT0FAUTO_RESTOREAX	; only register to restore is AX
	;
	; Check if it was a DAC interrupt.  If not, just issue EOI and
	; return.
	;
	PUSH	DX		; guess we do need 'em
	PUSH	DS
	MOV	AX,CS		; DS addresses local data
	MOV	DS,AX
	MOV	DX,DACBASE
	IN	AL,DX
	TEST	AL,8
	JZ	INT0FAUTO_EOI
	;
	; It was our interrupt.  Increment the count of DMA EOP interrupts
	; for the underflow checker.  Clear the DMA interrupt at the sound
	; chip so we will get another interrupt.
	;
	INC	IRQCOUNT
	AND	AL,0F7h
	OUT	DX,AL
	OR	AL,8
	OUT	DX,AL
	;
	; Decrement the number of autoinitialize loops remaining.  If more
	; remain, just issue EOI and return.
	;
	DEC	AUTOLOOPS
	JNZ	INT0FAUTO_EOI
	;
	; Disable DMA channel 1.  Either we're all done, or we need to
	; reprogram the DMA controller.
	;
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Do we have anything left to play?  It's remotely possible that
	; we had an exact multiple of 64k samples in the file.
	;
	CMP	NOAUTOSIZE,0
	JNE	INT0FAUTO_AUTOOFF
	;
	; We're all done.  Set the flag to indicate it is so.  DMA was
	; already stopped at the DMA controller above, so sound is no
	; longer playing.  SOUNDSTOP will take care of unhooking the
	; interrupt and putting the sound chip back in joystick mode.
	;
	MOV	DMADONE,1
	;
	; Go issue EOI.
	;
	JMP	INT0FAUTO_EOI
	;
	; Time to switch out of autoinitialize mode to play the remaining
	; samples.  Reprogram the DMA controller.
	;
INT0FAUTO_AUTOOFF:
	MOV	AL,49h		; select channel 1, read transfer from memory,
	OUT	0Bh,AL		;   autoinitialization disabled, address incre-
	JMP	$+2		;   ment, single mode
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL		; note - DMA page register already set
	JMP	$+2
	MOV	AX,NOAUTOSIZE	; get the number of samples to play
	DEC	AX		; program count = (# of bytes) - 1
	OUT	03h,AL		;   program low word
	JMP	$+2
	MOV	AL,AH		;   program high word
	OUT	03h,AL
	JMP	$+2
	MOV	AL,0		; set base address
	OUT	02h,AL		;   program low word
	JMP	$+2
	OUT	02h,AL		;   program high word
	;
	; Change the Int 0Fh vector so that when we get the next interrupt,
	; it will go to the non-autoinitialize handler.
	;
	XOR	AX,AX		; DS -> interrupt vector table
	MOV	DS,AX
	MOV	WORD PTR [3Ch],OFFSET INT0FNOAUTO
	MOV	[3Eh],CS
	;
	; Enable DMA channel 1, starting playback.
	;
	MOV	AL,1
	OUT	0Ah,AL
	;
	; Issue EOI to the interrupt controller.
	;
INT0FAUTO_EOI:
	MOV	AL,20h
	OUT	20h,AL
	POP	DS
	POP	DX
INT0FAUTO_RESTOREAX:
	POP	AX
	IRET			; will set the interrupt bit

;
; Routine to start autoinitialize mode playback, used when there are more
; than 64k samples in the sound file.  Programs the DMA controller, sound
; chip, and interrupt controller, and hooks Int 0Fh.
;
AUTOSTART:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	ES
	;
	; Disable IRQ 7 at the interrupt controller.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	;
	; Disable DMA channel 1.
	;
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Program the sound chip for playback mode, no DMA yet.
	;
	MOV	DX,DACBASE	; get DAC base port
	MOV	AL,10h		; enable DMA interrupt, still joystick mode
	OUT	DX,AL
	ADD	DX,3		; set sound volume to zero
	XOR	AL,AL
	OUT	DX,AL
	MOV	DX,DACBASE	; direct write to DAC, DMA disabled, DMA
	MOV	AL,13h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	STI
	;
	; Ramp up the chip to the baseline.  It takes a while to do it, so
	; it should be done with interrupts enabled.
	;
	CALL	RAMPUP
	;
	; Program the sound chip for playback with DMA.
	;
	CLI
	MOV	DX,DACBASE	; direct write to DAC, DMA enabled, DMA
	MOV	AL,17h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	MOV	AL,1Fh		; direct write to DAC, DMA enabled, DMA
	OUT	DX,AL		;   interrupt enabled, DMA interrupt allowed
	MOV	AX,AMPFREQ	; program amplitude/frequency
	ADD	DX,2
	OUT	DX,AL
	INC	DX
	MOV	AL,AH
	OUT	DX,AL
	;
	; Set up the DMA controller.
	;
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,59h		; select channel 1, read transfer from memory,
	OUT	0Bh,AL		;   autoinitialization enabled, address incre-
	JMP	$+2		;   ment, single mode
	MOV	AL,DMAPAGE	; set DMA channel 1 page register
	OUT	83h,AL
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	OUT	03h,AL		; program low word of count - count programmed
	JMP	$+2		;   is 0FFFFh for a 64k autotinitialize buffer
	OUT	03h,AL		; program high word
	JMP	$+2
	INC	AL		; set base address = 0
	OUT	02h,AL		;   program low word
	JMP	$+2
	OUT	02h,AL		;   program high word
	;
	; Hook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT0FDEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT0FDEFAULT+2,AX
	MOV	ES:[BX],OFFSET INT0FAUTO
	MOV	ES:[BX+2],CS
	;
	; Enable IRQ7 at the interrupt controller.
	;
	IN	AL,21h
	JMP	$+2
	AND	AL,7Fh
	OUT	21h,AL
	;
	; Enable DMA channel 1, starting playback.
	;
	MOV	AL,1
	OUT	0Ah,AL
	STI
	POP	ES
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Routine to get the DMA current address and convert it to an 8k buffer
; segment number.  Returns the segment number in AX.
;
GETSEG:
	PUSH	CX
	CLI
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	IN	AL,2		; read low byte of address (throw it away)
	JMP	$+2
	IN	AL,2		; read high byte of address
	STI
	MOV	CL,5
	SHR	AL,CL
	CBW
	POP	CX
	RET

;
; Routine to poll the DMA current address registers to see when it is
; safe to load sound into the current 8k buffer segment, CURRENTSEG.  It
; is safe when the current address is not in CURRENTSEG.  Saves the segment
; DMA is in in SEGPRE and sets IRQCOUNT to 0 for later underflow testing.
;
WAITSEG:
	PUSH	AX
WAITSEG_LOOP:
	CALL	GETSEG
	CMP	AX,CURRENTSEG
	JE	WAITSEG_LOOP
	MOV	SEGPRE,AX
	MOV	IRQCOUNT,AH	; segments are 0-7, so AH is 0
	POP	AX
	RET

;
; Routine to check for underflow during file reading.  Reads the DMA
; current address registers and the DMA interrupt count to determine
; whether underflow occurred during file reading.  Returns with carry clear
; if no underflow; if underflow, sets RETCODE to 8 ("output underflow") and
; returns with carry set.
;
CHKSEG:
	PUSH	AX
	PUSH	BX
	;
	; DMA all done?  Definitely underflow then.
	;
	CMP	DMADONE,1
	JE	CHKSEG_UNDERFLOW
	;
	; Get the current DMA segment (SEGPOST) in AX, segment we loaded
	; into in BX.
	;
	CALL	GETSEG
	MOV	BX,CURRENTSEG
	;
	; Case 1:  previous DMA segment < CURRENTSEG.
	;
	CMP	SEGPRE,BX
	JA	CHKSEG_ABOVE
	CMP	IRQCOUNT,0		; underflow if DMA EOP happened
	JNE	CHKSEG_UNDERFLOW
	CMP	AX,BX			; underflow if DMA not still below
	JAE	CHKSEG_UNDERFLOW	;   loaded segment
	JMP	CHKSEG_OK
	;
	; Case 2:  previous DMA segment > CURRENTSEG.
	;
CHKSEG_ABOVE:
	CMP	IRQCOUNT,1		; no underflow if no DMA EOP
	JB	CHKSEG_OK		; ... but definitely underflow if
	JA	CHKSEG_UNDERFLOW	;   2 or more DMA EOP's
	CMP	AX,BX			; underflow if 1 DMA EOP and DMA is
	JAE	CHKSEG_UNDERFLOW	;   not below loaded segment
	;
	; No underflow.
	;
CHKSEG_OK:
	CLC
	JMP	CHKSEG_DONE
	;
	; Underflow occurred.
	;
CHKSEG_UNDERFLOW:
	MOV	RETCODE,8
	STC
CHKSEG_DONE:
	POP	BX
	POP	AX
	RET

;
; Routine to clean up after completing sound playback.  Unhooks Int 0Fh
; and puts the sound chip back in joystick mode.
;
SOUNDSTOP:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	ES
	;
	; Disable DMA channel 1, if not disabled already.
	;
	CLI
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Stop DMA at the sound chip.
	;
	MOV	DX,DACBASE
	IN	AL,DX
	AND	AL,0F3h		; clear DMA enable and DMA interrupt clear
	OUT	DX,AL
	STI
	;
	; Ramp the chip down to avoid the click at the end.  It needs to
	; be done with interrupts enabled since it takes a while.
	;
	CALL	RAMPDOWN
	;
	; Okay, now we're going to clear the DMA interrupt enable bit.
	; That may cause an interrupt, which we need to wait for.  Use
	; the non-autoinitialize mode handler for this.
	;
	CLI
	XOR	AX,AX		; set sound chip for joystick mode, DMA
	OUT	DX,AL		;   interrupt enable bit clear
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	ES:[BX],OFFSET INT0FNOAUTO
	MOV	ES:[BX+2],CS
	INC	DX		; read a byte from the data port (probably
	IN	AL,DX		;   not needed for playing)
	STI
	;
	; Wait for the interrupt, but not forever.  There doesn't seem to
	; be one generated on the 1000TL; maybe there is on the new chip?
	;
	MOV	CX,2000
	LOOP	$
	;
	; The DAC is in joystick mode.  Disable further interrupts on IRQ 7.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	JMP	$+2
	;
	; Unhook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,WORD PTR INT0FDEFAULT
	MOV	ES:[BX],AX
	MOV	AX,WORD PTR INT0FDEFAULT+2
	MOV	ES:[BX+2],AX
	STI
	POP	ES
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to flush the DOS and BIOS keyboard buffers.
;
KEYFLUSH:
	PUSH	AX
	;
	; Flush the DOS typeahead buffer.
	;
	MOV	AX,0CFFh
	INT	21h
	;
	; Flush the BIOS typeahead buffer.
	;
KEYFLUSH_LOOP:
	MOV	AH,1
	INT	16h
	JZ	KEYFLUSH_DONE
	MOV	AH,0
	INT	16h
	JMP	KEYFLUSH_LOOP
KEYFLUSH_DONE:
	POP	AX
	RET

;
; Main program.
;
; Save the default Int 1Bh vector.  We don't need to save the 23h or 24h
; vectors since DOS will restore them for us.
;
START:
	MOV	AX,351Bh
	INT	21h
	MOV	WORD PTR INT1BDEFAULT,BX
	MOV	WORD PTR INT1BDEFAULT+2,ES
	MOV	AX,DS				; restore ES
	MOV	ES,AX
	;
	; Hook Int 1Bh to disable <control>-<break>.
	;
	MOV	AX,251Bh
	MOV	DX,OFFSET INT1BHDLR
	INT	21h
	;
	; Hook Int 23h to disable <control>-C.
	;
	MOV	AX,2523h
	MOV	DX,OFFSET INT23HDLR
	INT	21h
	;
	; Hook the critical error interrupt.  We don't want to get aborted
	; if there's a disk error since the DAC will need to be reset and
	; interrupts unhooked.
	;
	MOV	AX,2524h
	MOV	DX,OFFSET INT24HDLR
	INT	21h
	;
	; Shrink the stack to 2 kilobytes.
	;
	MOV	SP,OFFSET STACKEND
	;
	; Find the start of unused RAM.
	;
	MOV	AX,SP		; AX = paragraphs for data, code and stack
	ADD	AX,15
	MOV	CL,4
	SHR	AX,CL
	MOV	BX,CS		; add in paragraph address of program start
	ADD	AX,BX		; AX = paragraph address of unused RAM
	MOV	FREESEG,AX	; save it
	;
	; Determine a candidate for DMA buffer.
	;
	ADD	AX,0FFFh	; round AX up to a 64k boundary
	AND	AX,0F000h
	MOV	DMASEG,AX	; save it
	;
	; Verify the DMA buffer.
	;
	MOV	RETCODE,1	; "insufficient RAM" if we can't get buffers
	CMP	AX,ENDALLOC	; if past the end of RAM, skip it
	JB	>L1
	JMP	TERMINATE
L1:
	ADD	AX,1000h	; make sure we own the whole thing
	CMP	AX,ENDALLOC
	JBE	>L2
	JMP	TERMINATE
L2:
	MOV	FREESEG2,AX	; save the address of RAM past the DMA buffer
	;
	; Now get a buffer for reading from the file.
	;
	MOV	AX,FREESEG	; first try below the DMA buffer
	MOV	FILESEG,AX
	ADD	AX,800h
	CMP	AX,DMASEG	; will it fit?
	JBE	MEMDONE
	MOV	AX,FREESEG2	; didn't fit, try above the DMA buffer
	MOV	FILESEG,AX
	ADD	AX,800h
	CMP	AX,ENDALLOC	; will it fit?
	JBE	MEMDONE
	JMP	TERMINATE	; exit if not
	;
	; We have buffers allocated.  Check for the presence of a PSSJ
	; chip.
	;
MEMDONE:
	MOV	RETCODE,2	; "no DAC" if we don't find it
	CALL	CHKDAC
	JNC	>L3
	JMP	TERMINATE
L3:
	MOV	DACBASE,AX	; save DAC base I/O port address
	CALL	CHKVER		; get the DAC version, old or new
	;
	; Now get the .wav filename from the command line.
	;
	MOV	RETCODE,3	; "no filename" if we don't find it
	CALL	GETNAME
	JNC	>L4
	JMP	TERMINATE
	;
	; DS:SI addresses the filename.  Copy the filename to a local
	; buffer and append the .wav extension if it doesn't already have
	; one.
L4:
	CALL	WAVEXT
	;
	; Open the file for reading, deny writes to other processes.  DOS
	; will close the file for us when we terminate.
	;
	MOV	RETCODE,4	; "open failed" if unsuccessful
	MOV	AX,3D20h
	MOV	DX,OFFSET FILENAME
	INT	21h
	JNC	>L5
	JMP	TERMINATE
L5:
	MOV	HANDLE,AX
	;
	; Read and process the .wav header.  This sets a lot of variables.
	; DOHDR sets RETCODE (to 5, 6 or 0), so we don't need to set it
	; here.
	;
	CALL	DOHDR
	JNC	>L6
	JMP	TERMINATE
	;
	; Do whatever setup remains, apart from loading sound and programming
	; the DAC.  SETUP sets RETCODE as well (to 6, 7 or 0).  Note:  don't
	; change RETCODE from here on out unless you get another error; in
	; case of a truncated file, we will just play whatever's there, then
	; use the value of RETCODE (7) to display the error.  SETUP returns
	; with carry clear for a truncated file.
	;
L6:
	CALL	SETUP
	JNC	>L7
	JMP	TERMINATE
	;
	; Special (rare) case:  no sound to play.  Unlikely, but possible
	; if the file is truncated.
	;
L7:
	MOV	AX,WORD PTR NSAMPLES
	OR	AX,WORD PTR NSAMPLES+2
	JNZ	>L8
	JMP	TERMINATE
	;
	; Tell the user what's going on.
	;
L8:
	MOV	DX,OFFSET MSGLOADING
	MOV	AH,9
	INT	21h
	;
	; Read in some sound data (fill the DMA buffer if possible).
	;
	MOV	CURRENTSEG,0	; not really necessary, for clarity
	CALL	KEYFLUSH	; flush the keyboard buffer
LOADLOOP:
	MOV	AH,1		; halt if a key has been pressed
	INT	16h
	JZ	>L9
	JMP	TERMINATE
L9:
	CALL	WORD PTR READER	; load an 8k buffer segment
	JNC	>L10
	JMP	TERMINATE
L10:
	MOV	AX,WORD PTR BUFFERSEGS
	OR	AX,WORD PTR BUFFERSEGS+2
	JZ	NOAUTOINIT	; if no more sound data, exit the loop
	INC	CURRENTSEG
	AND	CURRENTSEG,7
	JNZ	LOADLOOP	; go again if DMA buffer is not full
	JMP	AUTOINIT
	;
	; Case 1:  There are 64k or less total sound samples in the file,
	; and all have been loaded.  We start DMA in non-autoinitialize
	; mode and play through.
	;
NOAUTOINIT:
	MOV	DMADONE,0	; not done playing, we haven't started
	CALL	NOAUTOSTART	; start DMA (hooks Int 0Fh)
NOAUTOWAIT:
	CMP	DMADONE,1	; was there an interrupt?
	JE	NOAUTOSTOP	; stop if so
	MOV	AH,1		; halt if a key has been pressed
	INT	16h
	JZ	NOAUTOWAIT	; otherwise, continue waiting
NOAUTOSTOP:
	CALL	SOUNDSTOP	; unhook Int 0Fh, finalize sound chip
	JMP	TERMINATE
	;
	; Case 2:  There are more than 64k total sound samples in the file.
	; We have loaded the DMA buffer with the first 64k.  We want to
	; start DMA in autoinitialize mode and continue to load samples as
	; space becomes available in the buffer.  The DMA EOP interrupt
	; handler for autoinitialize mode should switch out of that mode
	; when we get to the last buffer full.  DMADONE is only set when
	; all DMA has stopped, both autoinitialize looping and the final
	; non-autoinitialize bit.
	;
AUTOINIT:
	MOV	DMADONE,0	; not done playing, we haven't started
	CALL	AUTOSTART	; start DMA (hooks Int 0Fh)
AUTOWAIT:
	CMP	DMADONE,1	; all done playing sound?
	JE	AUTOSTOP	; stop if so
	MOV	AH,1		; has a key been pressed?
	INT	16h
	JNZ	AUTOSTOP	; stop if so
	MOV	AX,WORD PTR BUFFERSEGS		; more data to load?
	OR	AX,WORD PTR BUFFERSEGS+2
	JZ	AUTOWAIT	; if not, don't load any more :-)
	CALL	WAITSEG		; wait until there's room
	CALL	WORD PTR READER	; load an 8k buffer segment
	JC	AUTOSTOP	; stop if error loading
	CALL	CHKSEG		; check for underflow
	JC	AUTOSTOP
	INC	CURRENTSEG	; next buffer segment for the reader
	AND	CURRENTSEG,7
	JMP	AUTOWAIT	; go again (wrap around to the beginning)
AUTOSTOP:
	CALL	SOUNDSTOP	; unhook Int 0Fh, finalize sound chip
	;
	; Jump here when we're done.  RETCODE is the return code for the
	; program.
	;
TERMINATE:
	CALL	KEYFLUSH	; flush the keyboard buffer
	LDS	DX,INT1BDEFAULT	; unhook Int 1Bh
	MOV	AX,251Bh
	INT	21h
	MOV	AX,CS		; get DS back
	MOV	DS,AX
	MOV	BL,RETCODE	; display the message for the return code
	MOV	BH,0
	SHL	BX,1
	MOV	DX,MSGS[BX]
	MOV	AH,9
	INT	21h
	MOV	AL,RETCODE	; terminate with return code
	MOV	AH,4Ch
	INT	21h
	EVEN

;
; Reserve 2k for the stack.
;
STACKSTART	DW	1024 DUP (0)
STACKEND	EQU	$
