/* --------------------------------- grmswin.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Graphics driver for Microsoft Windows.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <windows.h>

#include "config.h"
#include "fly.h"
#include "colors.h"

HDC		hdc, texthdc;
RECT		textrect, rect;
static int	textx = 1, texty = 1, maxx, maxy;
static HPEN 	pens[256];
static COLORREF	colors[256];
TEXTMETRIC	textmetric;
HPALETTE 	hpal;
static HPALETTE oldhpal;
static LOGPALETTE * plgpl;
static PALETTEENTRY ape[256];

HWND		ghWndMain  = NULL;
HWND		ghWndText  = NULL;
extern HINSTANCE	Fly8Instance;
extern char	FAR Fly8AppName[10];
extern char	FAR Fly8Message[15];
extern char	FAR Fly8Text[10];

#define TEXTHEIGHT textmetric.tmHeight
#define TEXTWIDTH  textmetric.tmMaxCharWidth

void FAR
perror ( const char *p)
{
	MessageBox((HWND)NULL, (LPCSTR)p, (LPCSTR)"Fly8 Error", MB_OK);
}

static void FAR
SetActive (int page)
{
        page = page;
}

static void FAR
SetVisual (int page)
{
        page = page;
}

static int FAR
CursorMode (int mode)
{
        mode = mode;
	return (0);
}

static void FAR
TextColor (Uint fg, Uint bg)
{
	SetTextColor (texthdc, colors[bg]);
	SetBkColor (texthdc, colors[fg]);
}

static void FAR
FontSet (DEVICE *dev, char *fname)
{
        fname = fname;
	dev->FontWidth  = TEXTWIDTH;
	dev->FontHeight = TEXTHEIGHT;
}

static void textPos () {	/* make x and y for text window valid */
	if (textx > maxx) {
		textx = 1;
		texty += TEXTHEIGHT;
	}
	if (texty > maxy)
		texty = 1;
}

static char _Gbuf[2048];
static int bufptr = 0;

static void FAR 
WriteChar (int c)
{
        char _buf[2];

	if (c == '\n') {
		texty += TEXTWIDTH;
		textx = 1;
	} else if (c == '\r') {
		textx = 1;
	} else if (c == '\f') {
		textx = 1;
		texty = 1;
	} else {
	        _buf[0] = (char)c;
        	TextOut (texthdc, textx, texty, _buf, 1);
	        textx += TEXTWIDTH;
        }

        textPos ();        
}

static void FAR
TextChar (int c)
{        
        if (bufptr < 2048)		/* save char for WM_PAINT */
	        _Gbuf[bufptr++] = (char)c;
        
	WriteChar (c);
}

extern void
ReDrawText (void)
{
	int	i;
	char	*p;

       	GetClientRect (ghWndText, &textrect);
       	maxx = textrect.right;
       	maxy = textrect.bottom;
	textx = texty = 1;
	for (p = _Gbuf, i = 0; i < bufptr; ++p, ++i)
		WriteChar (*p);
}

static void FAR
TextPut (int c)
{
        TextChar (c);
}

static void FAR
SetTextPos (int row, int col)
{
	textx = 1  + (col-1)*TEXTWIDTH;
	texty = 1  + (row-1)*TEXTHEIGHT;
	textPos ();
}

static void FAR
PushTextPos (void)
{
}

static void FAR
PopTextPos (void)
{
}

static void FAR
TextClear (void)
{
}

static void FAR
WriteMode (int mode)
{
        mode = mode;
#if 0
	switch (mode) {
	default:
	case T_MSET:
		mode = GXcopy;
		break;
	case T_MOR:
		mode = GXor;
		break;
	case T_MXOR:
		mode = GXxor;
		break;
	}
	XSetFunction (TheDisplay, TheGC, mode);
#endif
}

static void FAR
SetPalette (int n, long c)
{
	Uint	r, g, b;

	r = 0x0ff & (Uint)(c);
	g = 0x0ff & (Uint)(c >> 8);
	b = 0x0ff & (Uint)(c >> 16);

#if 0
#define	COLOR_STEP	32

	if (r > b && r > g) {
		if ((r += (r+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			r = 0x0ff;
	} else if (g > r && g > b) {
		if ((g += (g+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			g = 0x0ff;
	} else if (b > r && b > g) {
		if ((b += (b+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			b = 0x0ff;
	} else if (r && (r == g && g == b)) {
		if ((r += (r+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			r = 0x0ff;
		if ((g += (g+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			g = 0x0ff;
		if ((b += (b+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
			b = 0x0ff;
	}
	c = r | (g<<8) | ((long)b<<16);	
#endif

     	ape[n].peRed = 	 LOBYTE(r);
     	ape[n].peGreen = LOBYTE(g);
    	ape[n].peBlue =  LOBYTE(b);
    	ape[n].peFlags = 0;

	SetPaletteEntries (hpal, 0, 256, ape);
	RealizePalette (hdc);

	if (pens[n])
	        DeleteObject (pens[n]);

	pens[n] = CreatePen (PS_SOLID, 0, 
				PALETTEINDEX(GetNearestPaletteIndex (hpal, c)));
	colors[n] = c;
}

static void
moveTo (int x, int y)
{
  	MoveTo (hdc, x, y);
}

static void 
set_color (Uint c)
{
	static Uint    last_color = 0xffff;

  	if (last_color != c) {
    		if (!pens[c])
			MessageBox((HWND)NULL, (LPCSTR)"No pen created for that colour",
				   (LPCSTR)"Fly8 Error", MB_OK);
		else
			SelectObject (hdc, pens[c]);
  	}
	last_color = c;
	SetROP2 (hdc, R2_COPYPEN);
}

static void
drawTo (int x, int y, Uint c)
{
        set_color (c);
  	LineTo (hdc, x, y);
}

static void FAR
OpenTextWindow (SCREEN *scr)
{
        ghWndText = CreateWindow(
                        (LPSTR)Fly8Text,
                        (LPSTR)Fly8Message,
                        WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
                        scr->minx,                  /*  x  */
                        scr->miny,                  /*  y  */
                        scr->sizex - scr->minx,     /* cx  */
                        scr->sizey - scr->miny,     /* cy  */
                        (HWND)NULL,             /* no parent */
                        (HMENU)NULL,            /* use class menu */
                        (HANDLE)Fly8Instance,   /* handle to window instance */
                        (LPSTR)NULL             /* no params to pass on */
                        );

        /* Make window visible */
        ShowWindow (ghWndText, SW_SHOW );
        UpdateWindow (ghWndText);

        texthdc = GetDC (ghWndText);
	GetClientRect (ghWndText, &textrect);
	maxx = textrect.right;
	maxy = textrect.bottom;
}

static void FAR
CloseTextWindow (SCREEN *scr)
{
        scr = scr;
        DestroyWindow ( ghWndText );
}

static void FAR
ellipse (register int x1, register int y1, int rx, int ry, register Uint color)
{
	int	ax, bx, cx, dx, ay, by, cy, dy;

	ax = fmul ( 3196, rx);		/* sin (pi/16) */
	ay = fmul ( 3196, ry);
	bx = fmul ( 9102, rx);		/* sin (3*pi/16) */
	by = fmul ( 9102, ry);
	cx = fmul (13623, rx);		/* sin (5*pi/16) */
	cy = fmul (13623, ry);
	dx = fmul (16069, rx);		/* sin (7*pi/16) */
	dy = fmul (16069, ry);

	moveTo (x1+dx, y1-ay);
	drawTo (x1+cx, y1-by, color);
	drawTo (x1+bx, y1-cy, color);
	drawTo (x1+ax, y1-dy, color);
	drawTo (x1-ax, y1-dy, color);
	drawTo (x1-bx, y1-cy, color);
	drawTo (x1-cx, y1-by, color);
	drawTo (x1-dx, y1-ay, color);
	drawTo (x1-dx, y1+ay, color);
	drawTo (x1-cx, y1+by, color);
	drawTo (x1-bx, y1+cy, color);
	drawTo (x1-ax, y1+dy, color);
	drawTo (x1+ax, y1+dy, color);
	drawTo (x1+bx, y1+cy, color);
	drawTo (x1+cx, y1+by, color);
	drawTo (x1+dx, y1+ay, color);
	drawTo (x1+dx, y1-ay, color);
}

static void FAR
Sync (void)
{
}

static void
ClearWin (void)
{
  	st.flags |= SF_CLEARED;
	FillRect (hdc, &rect, GetStockObject (BLACK_BRUSH));
}

static int FAR
Init (DEVICE *dev)
{
        int i, r, g, b;
        
        if (dev->sizex == 0 || dev->sizey == 0)
                return (1);

	for (i = 0; i < 256; ++i) {
	        colors[i] = 0;
	        pens[i] = 0;
	}
	
        ghWndMain = CreateWindow(
                        (LPSTR)Fly8AppName,
                        (LPSTR)Fly8Message,
                        WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
                        0,                      /*  x  */
                        0,                      /*  y  */
                        dev->sizex,             /* cx  */
                        dev->sizey,             /* cy  */
                        (HWND)NULL,             /* no parent */
                        (HMENU)NULL,            /* use class menu */
                        (HANDLE)Fly8Instance,   /* handle to window instance */
                        (LPSTR)NULL             /* no params to pass on */
                        );

        /* Make window visible */
        ShowWindow (ghWndMain, SW_SHOW );
        UpdateWindow (ghWndMain);

        hdc = GetDC (ghWndMain);
	GetClientRect (ghWndMain, &rect);
  	CS->sizex = rect.right;
  	CS->sizey = rect.bottom;
  	set_screen (rect.right, rect.bottom);

	dev->npages = 1;
	GetTextMetrics (hdc, &textmetric);
	dev->FontWidth  = TEXTWIDTH;
	dev->FontHeight = TEXTHEIGHT;
	
	SetROP2 (hdc, R2_COPYPEN);

	plgpl = (LOGPALETTE*) LocalAlloc(LPTR,
    		sizeof(LOGPALETTE) + 256 * sizeof(PALETTEENTRY));

	plgpl->palNumEntries = 256;
	plgpl->palVersion = 0x300;

	for (i = 0, r = 0, g = 127, b = 127; i < 256;
        	i++, r += 1, g += 1, b += 1) {
    		ape[i].peRed =
        		plgpl->palPalEntry[i].peRed = LOBYTE(r);
    		ape[i].peGreen =
        		plgpl->palPalEntry[i].peGreen = LOBYTE(g);
    		ape[i].peBlue =
        		plgpl->palPalEntry[i].peBlue = LOBYTE(b);
    		ape[i].peFlags =
        		plgpl->palPalEntry[i].peFlags = 0;
	}
	hpal = CreatePalette(plgpl);
	LocalFree((HLOCAL) plgpl);
	oldhpal = SelectPalette (hdc, hpal, FALSE);
	SetPaletteEntries (hpal, 0, 256, ape);
	RealizePalette (hdc);

	st.black   = 0;	/* cannot change */
	st.white   = 1;	/* cannot change */
	dev->colors = GetDeviceCaps (hdc, NUMCOLORS);

	if (dev->colors < 16) {
		SetPalette (st.black, C_BLACK);
		SetPalette (st.white, C_WHITE);
		st.red     =
		st.blue    =
		st.magenta = 
		st.green   =
		st.brown   =
		st.gray    =
		st.hudlow  =
		st.hudhigh =
		st.lblue   =
		st.lred    =
		st.lgray   =
		st.ground  = st.white; 
	} else {
		st.red     = 2;	/* do not change! */
		st.blue    = 4;	/* do not change! */
		st.magenta = 6;	/* do not change! */
		st.green   = 3;
		st.brown   = 5;
		st.gray    = 7;
		st.hudlow  = 8;
		st.hudhigh = 9;
		st.lblue   = 10;
		st.lred    = 11;
		st.lgray   = 12;
		st.ground  = 13;
/*		st.white   = 15;	keep 15 for OR'ed white */
		SetPalette (st.black,   C_BLACK);
		SetPalette (st.white,   C_WHITE);
		SetPalette (st.red,     C_RED);
		SetPalette (st.blue,    C_BLUE);
		SetPalette (st.magenta, C_MAGENTA);
		SetPalette (st.green,   C_GREEN);
		SetPalette (st.brown,   C_BROWN);
		SetPalette (st.gray,    C_GRAY);
		SetPalette (st.hudlow,  C_LYELLOW);
		SetPalette (st.hudhigh, C_YELLOW);
		SetPalette (st.lred,    C_LIGHTRED);
		SetPalette (st.lblue,   C_SKYBLUE);
		SetPalette (st.lgray,   C_LIGHTGRAY);
		SetPalette (st.ground,  C_GRAY);
	}

	SetVisual (0);
	SetActive (0);

        ClearWin ();

	return (0);
}

static void FAR
Term (DEVICE *dev)		/* done */
{
	int i;
	
	for (i = 0; i < 256; ++i)
	        if (pens[i])
	        	DeleteObject (pens[i]);	/* delete Pens */
	
        if (ghWndMain)
		ReleaseDC (ghWndMain, hdc);
        if (ghWndText)
		ReleaseDC (ghWndText, texthdc);

	SelectPalette (hdc, oldhpal, FALSE);
	DeleteObject(hpal);
	dev = dev;
}

static void
Flush (void)
{
}

static void FAR
Shutters (int eye)
{
	return;
}

struct GrDriver GrMSWin = {
	"GrMSWin",
	0,
	0,
	Init,
	Term,
	OpenTextWindow,
	CloseTextWindow,
	FontSet,
	TextPut,
	TextChar,
	TextColor,
	CursorMode,
	moveTo,
	drawTo,
	SetVisual,
	SetActive,
	0,
	SetTextPos,
	PushTextPos,
	PopTextPos,
	TextClear,
	WriteMode,
	SetPalette,
	ellipse,
	Sync,
	Flush,
	Shutters
};
