/* --------------------------------- piper.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Show the piper (aiming reticule) on the Head Up Display.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "fly.h"
#include "hud.h"

#undef	CP
#undef	CW
#undef	CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

static int	Csin[] = {8192, 14189}, Ccos[] = {14189, 8192};

extern void FAR
keep_inside (int *x, int *y, int xl, int xh, int yl, int yh, int orgx,
	int orgy, int clipx, int clipy, int shifty)
{
	int	t;

	if (xl > xh)
		(t = xl, xl = xh, xh = t);
	if (*x > (t = orgx+clipx-xh))
		*x = t;
	else if (*x < (t = orgx-clipx-xl))
		*x = t;

	if (yl > yh)
		(t = yl, yl = yh, yh = t);
	orgy += shifty;
	if (*y > (t = orgy+clipy-yh))
		*y = t;
	else if (*y < (t = orgy-clipy-yl))
		*y = t;
}

extern void FAR
show_piper (OBJECT *obj, OBJECT *target, int x, int y, int dx, int dy, int ds,
	int mode, int off_screen, int dist, int tti, int closure, int orgx,
	int orgy, int clipx, int clipy, int hbottom, int hleft, VECT TA,
	int ss, int shifty)
{
	int	datax, datay, nlines, detail, hud1, f156, fa18, knots;
	int	t, i, d, dd, ax, ay, bx, by, rx, ry, ratio;
	ANGLE	a;
	long	tc, td;
	char	*name;
	VECT	R;

	hud1 = EE(obj)->hud1;
	i = hud1 & HUD_TYPES;
	fa18 = i == HUD_FA18;
	f156  = i == HUD_F15 || i == HUD_F16;
	knots = hud1 & HUD_KNOTS;

	if (knots) {
		tc = closure + fmul (closure, 15465);	/* knots */
		td = 3L*dist + fmul (dist, 4601); /* feet */
	} else {
		tc = closure;
		td = dist;
	}

	dd = num_size (9L, ss);

	if (NEWTGT(obj))
		t = NEWTGT(obj)--&1;
	else
		t = 1;

	if (t) {
		ratio = f156 ? F15RAIMC : F18RAIMC;
		rx = fmul (dx, ratio);
		ry = fmul (dy, ratio);
		add_5op (T_ELLIPSE, x, y, rx, ry, st.hfg);
		if (ds > 1) {
			if (hud1 & HUD_THICK) {
				add_5op (T_ELLIPSE, x, y, rx+1, ry+1, st.hfg);
			}
/*
 * Target distance
*/
			add_line (x, y-ry, T_MOVE);	/* 12 */
			add_line (x, y-dy, st.hfg);
			add_line (x, y+ry, T_MOVE);	/* 6 */
			add_line (x, y+dy, st.hfg);
			add_line (x-rx, y, T_MOVE);	/* 9 */
			add_line (x-dx, y, st.hfg);
			add_line (x+rx, y, T_MOVE);	/* 3 */
			add_line (x+dx, y, st.hfg);
			for (i = 0; i < 2; ++i) {	/* ticks */
				ax = fmul(Csin[i], dx);
				ay = fmul(Ccos[i], dy);
				bx = fmul (ax, ratio);
				by = fmul (ay, ratio);
				add_line (x+bx, y+by, T_MOVE);
				add_line (x+ax, y+ay, st.hfg);
				add_line (x-bx, y+by, T_MOVE);
				add_line (x-ax, y+ay, st.hfg);
				add_line (x-bx, y-by, T_MOVE);
				add_line (x-ax, y-ay, st.hfg);
				add_line (x+bx, y-by, T_MOVE);
				add_line (x+ax, y-ay, st.hfg);
			}
			if (td > 11500L)		/* marker */
				i = 11500;
			else
				i = (int)td;
			a = muldiv (i, D90, 12000)*4;	/* 360Deg = 12k */
			ax = fmul(SIN (a), rx);
			ay = fmul(COS (a), ry);
			bx = fmul (ax, ratio);		/* use same ratio */
			by = fmul (ay, ratio);
			add_line (x+ax, y-ay, T_MOVE);
			add_line (x+bx, y-by, st.hfgi);

/*
 * Aspect angle(m61) or tti (mk82)
*/
			if (WE_MK82 == EE(obj)->weapon) {
				if (tti > 120)
					a = 0;
				else
					a = muldiv (D90/6, tti, 10)*2;
			} else {
				Mxpose (obj->T);
				VMmul (R, target->V, obj->T);
				Mxpose (obj->T);
				t = ihypot2d (R[X], R[Z]);
				a = ATAN (t, -R[Y]);
			}
			ax = fmul (dx, SIN (a));
			ay = fmul (dy, COS (a));
			bx = fmul (dx, (FONE-ratio)/2);
			by = fmul (dy, (FONE-ratio)/2);
			keep_inside (&x, &y, ax-bx, ax+bx, -ay-by, -ay+by,
				orgx, orgy, clipx, clipy, shifty);
			add_5op (T_ELLIPSE, x+ax, y-ay, bx, by, st.hfgi);

			if (WE_M61 == EE(obj)->weapon) {
/*
 * Target closure rate
*/
				if (fa18) {
					if (tc >= 0)	/* show 10s */
						tc = ((tc+5)/10)*10;
					else
						tc = ((tc-5)/10)*10;
					d = num_size (tc, ss);
					ax = x+dx;
					ay = y+dy+ss*5/2;
					keep_inside (&ax, &ay, -d, 2*dd,
						-ss-ss/2, 0, orgx, orgy,
						clipx, clipy, shifty);
					stroke_num (ax-d, ay-ss/2, tc, ss,
						st.hfg);
					stroke_str (ax,   ay, "Vc", ss, st.hfg);
				}
/*
 * Target acceleration vector
*/
				if (hud1 & HUD_ACCVECT) {
					Mxpose (obj->T);
					VMmul (R, TA, obj->T);
					Mxpose (obj->T);
					t = ihypot2d (R[X], R[Z]);
					i = muldiv (10*VONE, st.interval, 1000);
					if (t < i)
						t = i;
					if (t) {
						ax = muldiv (R[X], rx, t);
						ay = muldiv (R[Z], ry, t);
						add_line (x,    y,    T_MOVE);
						add_line (x+ax, y-ay, st.hfg);
					}
				}
			}
		}
		if (off_screen == 1) {			/* ahead */
			add_line (x,    y+ry, T_MOVE);
			add_line (x,    y-ry, st.hfg);
			add_line (x+rx, y,    T_MOVE);
			add_line (x-rx, y,    st.hfg);
		} else if (off_screen == 2){		/* behind */
			d = SIN (D90/2);
			ax = fmul(d,rx);
			ay = fmul(d,ry);
			add_line (x-ax, y+ay, T_MOVE);
			add_line (x+ax, y-ay, st.hfg);
			add_line (x-ax, y-ay, T_MOVE);
			add_line (x+ax, y+ay, st.hfg);
		} else {
			add_line (x, y, T_MOVE);	/* center dot */
			add_line (x, y, st.hfgi);
		}
	}

	if (EE(obj)->hud & HUD_DATA) {
		name = get_name (obj, target, mode);
		detail = (1 == ds) || (2&mode);
		nlines = (detail ? 3 : 0) + !!name;

		if (2&mode) {
			datax = hleft;
			datay = hbottom+shifty-(nlines-1)*ss;
		} else {
			if (y+dy+nlines*ss >= orgy+clipy+shifty)
				datay = y-dy-2-(nlines-1)*ss;
			else
				datay = y+dy+ss;
			datax = 4*dd;
			if (name) {
				t = stroke_size (name, ss);
				if (t < datax)
					t = datax;
			} else
				t = datax;

			t = orgx+clipx-t;
			datax = x-dx;
			if (datax > t)
				datax = t;
		}

		show_data (obj, datax, datay, detail, knots, dist, closure,
			name, tti, mode, ss);
	}
}
