/*
    setpanel.c - A user-space program to write to the ASUS DigiMatrix front panel.
    Copyright (c) 2005 Richard Taylor <richard@artaylor.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Some info gathered about the registers:
00 - 1st Digit
01 - 2nd
02 - 3rd
03 - 4th
04 - 5th
05 - 6th

06 - Colon 1, 2, 3 (bottom, middle, both)
07 - Not sure... 
08 - Show play symbol
09 - Show pause symbol
10 - 
11 - HD Mode (HD Symbol)
12 - CD Mode (CD Symbol)
13 - FM Mode (FM + Radio on)
14 - AM Mode (AM + Radio on)
15 - Turns off Radio / Current mode? (Doesn't clear icon)

30 - Hour (Hex e.g. 0x23 = 11pm)
31 - Minute

33 - Reads back the value written...
34 - Displays temperature (hex value written). 
35 - Go back to digit mode (resets to -- --:--)

36 - Volume control count (8-bit +/-)
37 - 
38 - 
39 - 
40 -

41 - Display time - how to set?!!

Don't go above 86 - the I2C bus freezes! 
*/


#include <sys/io.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>
#include <time.h>
#include "i2c-dev.h"

#define DM_VERSION "00.03"

// Address on the I2C bus of the AudioDL (OZ268) IC. 
#define AUDIODJ_I2C_ADDRESS 0x49

// Update intervals for temperature display and volume control in milli-seconds. 
#define TEMPERATURE_UPDATE_INTERVAL_MS 2000
#define VOLUME_UPDATE_INTERVAL_MS 50

#define I2C_TRANSACTION_DELAY_MS 1000

void print_i2c_busses(int);
void help(char *argv0) __attribute__ ((noreturn));

#ifdef FALSE
#define BOOL bool
#else
typedef enum _bool
{
    FALSE=0,
    TRUE
} BOOL;
#endif

    // First digit: 
    // 0x80 of 0x00 makes LED not light up for representative mode. 
    
    // All digits:
    // 0x0f = blank
    // 0x0e = 'C' (2nd Digit = 'D') CD MODE LED
    // 0x0d = 'P' (2nd Digit = '-') HD MODE LED
    // 0x0c = '-' (2nd Digit = 'M') Invalid MODE
    // 0x0b = 'A'                   AM MODE LED 
    // 0x0a = 'F'                   FM MODE LED
    // < 0x0a = Number.i

typedef enum _radio_mode
{
    OFF_MODE = 0,
    AM_RADIO,
    FM_RADIO,
    CD_MODE,
    HD_MODE
} RADIO_MODE;

typedef enum _leds
{
    LED_NONE = 0x8f,
    LED_AM = 0x0b,
    LED_FM = 0x0a,
    LED_CD = 0x0e,
    LED_HD = 0x0d,
} LED;


// Writing to Reg 0x07 does nothing, 0x08 = PLAY LED, 0x09 = PAUSE LED.
typedef enum _pp
{
    PP_NONE = 0x07,
    PP_PLAY = 0x08,
    PP_PAUSE = 0x09
} PP;


// Dots are in reg 0x06
    // 0x00 = No Dots.
    // 0x01 = middle dot
    // 0x02 = bottom dot 
    // 0x03 = both dots
typedef enum _dots
{
    DOTS_NONE = 0,
    DOTS_MIDDLE,
    DOTS_BOTTOM,
    DOTS_BOTH
} DOTS;


BOOL bI2CInitialised = FALSE;

static struct option longopts[] = {
    { "i2c",         1, 0, 'i' },
    { "radio",       0, 0, 'r' },
    { "display",     1, 0, 'd' },
    { "temp",        0, 0, 't' },
    { "update-temp", 0, 0, 'u' },
    { "volume",      0, 0, 'l' },
    { "manual",      0, 0, 'm' },
    { "time",        0, 0, 's' },
    { "version",     0, 0, 'v' },
    { "help",        0, 0, 'h' },
    { NULL,          0, 0,  0  }
};

void help(char *argv0)
{
        printf(
"\n"
"Usage: %s -i <I2C-BUS> [OPTIONS]\n"
"\n"
"  If you are using this on a system other than the ASUS DigiMatrix,\n"
"  then there is a chance that you may destroy an EEPROM.\n"
"\n"
"  -i, --i2c <n>              Sets the I2C Bus (Compulsary)\n"
"  -r, --radio [am | fm | off]  Set the radio to am, fm or off\n"
"\n"
"  -d, --display [\"xxxxxx\"]   Set the display to the numbers specified\n"
"                             Available are: [0-9, A, C, F, P]\n"
"                             Limited availability: [M, D, -] (try them :)\n"
"                             A Capital letter in digit[0] will also light up the mode led\n"
"                             A small letter will suppress the led\n"
"                             Anything else will be displayed as blank.\n"
"                             Adding: [am | fm | cd | hd]\n"
"                                     [play | pause]\n"
"                                     [both | middle | bottom]\n"
"                             will activate the relevant LED / dots.\n"
"  e.g.\n"
"  (1) -d 123456 will display 12 3456 \n"
"  (2) -d \"123456 fm pause both\" will display 12 34:56 and also light up the fm and pause leds.\n"
"  N.B. You cannot have multiple mode LEDs lit at the same time (e.g. am + fm, or play + pause).\n"
"\n"
"  -t, --temp                 Display the current temperature.\n"
"  -u, --update-temp          Updates the temperature every 2 seconds.\n"
"  -l, --volume               Displays any changes on the volume control.\n"
"  -m, --manual               Allows you to interactively change a register.\n"
"  -s, --time                 Sets and displays the current local time.\n"
"\n"
"  -v, --version              Print version\n"
"  -h, --help                 This help\n"
"\n", argv0);

  printf("\n\nFor available I2C busses, use i2cdetect.\nYou are looking for \"SiS96x SMBus adapter at 0xe600\"\n\n");
  exit(1);
}

int digimatrix_write_byte_data(int file, char reg, char data)
{
    int res;
    // TODO: Implement kernel specific stuff here. 
    res = i2c_smbus_write_byte_data(file, reg, data);
    usleep(I2C_TRANSACTION_DELAY_MS);
    return res;
}

int digimatrix_write_byte(int file, char data)
{
    int res;
    // TODO: Implement kernel specific stuff here. 
    res = i2c_smbus_write_byte(file, data);
    usleep(I2C_TRANSACTION_DELAY_MS);
    return res;
}

int digimatrix_read_byte(int file)
{
    int res;
    // TODO: Implement kernel specific stuff here. 
    res = i2c_smbus_read_byte(file);
    usleep(I2C_TRANSACTION_DELAY_MS);
    return res;
}

static void digimatrix_TurnOnRadio(int file, RADIO_MODE radio_mode)
{
    digimatrix_write_byte_data(file, 0x23, 0x02);

    switch (radio_mode)
    {
        case OFF_MODE:
            // Turns off the Radio
            digimatrix_write_byte_data(file, 0x0f, 0x00);
            break;
        
        case AM_RADIO:
            // Turns on AM Radio;
            digimatrix_write_byte_data(file, 0x0e, 0x00);
            break;
        
        case FM_RADIO:
            // Turns on FM Radio
            digimatrix_write_byte_data(file, 0x0d, 0x00);
            break;
        
        case CD_MODE:
            // Turns on CD mode!!!
            digimatrix_write_byte_data(file, 0x0c, 0x00);
            break;
        
        case HD_MODE:
            // Enables HD mode!!!
            digimatrix_write_byte_data(file, 0x0b, 0x00);
            break;

        default:
            // Turns off the Radio
            digimatrix_write_byte_data(file, 0x0f, 0x00);
            break;
        
    }
}

static void digimatrix_DisplayNumbers(int file, char digits[6], 
                                      LED led, PP pp, DOTS dots)
{
    int n;
    
    digimatrix_write_byte_data(file, 0x23, 0x02); // Set to digit mode (-- --:--).
    
    // Set up the Dots.
    digimatrix_write_byte_data(file, 0x06, (unsigned char)dots);

    // Set the Play / Pause LEDs
    digimatrix_write_byte_data(file, (unsigned char)pp, 0x01);

    // Now set the mode LEDs
    // If a letter is used as the first digit next, the mode LED will be overridden. 
    if (led != LED_NONE)
    {
        digimatrix_write_byte_data(file, 0x00, (unsigned char) led);
    }

    

    // Set the state of each digit.
    for (n=0; n<6; n++)
    {
        if (digits[n] <= '9' && digits[n] >= '0')
        {
            digimatrix_write_byte_data(file, n, digits[n] - '0');
        }
        else
        {
                switch (digits[n])
                {
                case 'a':
                    digimatrix_write_byte_data(file, n, 0x8b);
                    break;
                case 'A':
                    digimatrix_write_byte_data(file, n, 0x0b);
                    break;

                case 'f':
                    digimatrix_write_byte_data(file, n, 0x8a);
                    break;
                case 'F':
                    digimatrix_write_byte_data(file, n, 0x0a);
                    break;

                case 'c':
                case 'd':
                    digimatrix_write_byte_data(file, n, 0x8e);
                    break;
                case 'C':
                case 'D':
                    digimatrix_write_byte_data(file, n, 0x0e);
                    break;
                
                case 'p':
                    digimatrix_write_byte_data(file, n, 0x8d);
                    break;
                case 'P':
                    digimatrix_write_byte_data(file, n, 0x0d);
                    break;
                
                case 'm':
                case 'M':
                    digimatrix_write_byte_data(file, n, 0x0c);
                    break;

                default:
                    digimatrix_write_byte_data(file, n, 0x0f);
                    break;
                }
        }
    }
    
}

static void digimatrix_DisplayTemp(int file, int temp)
{
    char temp_dec = 0;
    // Get the temperature from the IT87


    if (ioperm(0x295, 2, 1) == -1)
    {
        printf("Error getting access\n");
    }
    else
    {
        outb(0x29, 0x295);
        temp = inb(0x296);
    }

    digimatrix_write_byte_data(file, 0x23, 0x02); // Set to digit mode (incase the clock is running)
   
    //digimatrix_write_byte_data(file, 0x0f, 0x01);
    //digimatrix_write_byte_data(file, 0x0f, 0x01);
    //digimatrix_write_byte_data(file, 0x0f, 0x01);

    temp &= 0xFF;
    temp_dec = ((temp / 10) << 4) & 0xf0; 
    temp_dec |= (temp % 10) & 0x0f;
    digimatrix_write_byte_data(file, 0x22, temp_dec);
    //digimatrix_write_byte_data(file, 0x0f, 0x01);
    //digimatrix_write_byte_data(file, 0x0f, 0x01);
}

static void digimatrix_UpdateTemp(int file, int temp)
{
    char temp_dec = 0;
    // Get the temperature from the IT87


    if (ioperm(0x295, 2, 1) == -1)
    {
        printf("Error getting access\n");
    }
    else
    {
        outb(0x29, 0x295);
        temp = inb(0x296);
    }

                    
    temp &= 0xFF;
    temp_dec = ((temp / 10) << 4) & 0xf0; 
    temp_dec |= (temp % 10) & 0x0f;
    digimatrix_write_byte_data(file, 0x22, temp_dec);
}

int digimatrix_I2CInit(int i2cbus)
{
  int address = AUDIODJ_I2C_ADDRESS;
  int e1, e2, e3, file;
  char filename1[20];
  char filename2[20];
  char filename3[20];
  char *filename;
  long funcs;
  
  if ((i2cbus < 0) || (i2cbus > 0x3f)) {
    fprintf(stderr,"Error: I2CBUS argument invalid!\n");
    return -1;
  }

  e1 = 0;
/*
 * Try all three variants and give the correct error message
 * upon failure
 */

  sprintf(filename1,"/dev/i2c-%d",i2cbus);
  sprintf(filename2,"/dev/i2c%d",i2cbus);
  sprintf(filename3,"/dev/i2c/%d",i2cbus);
  if ((file = open(filename1,O_RDWR)) < 0) {
    e1 = errno;
    if ((file = open(filename2,O_RDWR)) < 0) {
      e2 = errno;
      if ((file = open(filename3,O_RDWR)) < 0) {
        e3 = errno;
        if(e1 == ENOENT && e2 == ENOENT && e3 == ENOENT) {
          fprintf(stderr,"Error: Could not open file `%s', `%s', or `%s': %s\n",
                     filename1,filename2,filename3,strerror(ENOENT));
        }
        if (e1 != ENOENT) {
          fprintf(stderr,"Error: Could not open file `%s' : %s\n",
                     filename1,strerror(e1));
          if(e1 == EACCES)
            fprintf(stderr,"Run as root?\n");
        }
        if (e2 != ENOENT) {
          fprintf(stderr,"Error: Could not open file `%s' : %s\n",
                     filename2,strerror(e2));
          if(e2 == EACCES)
            fprintf(stderr,"Run as root?\n");
        }
        if (e3 != ENOENT) {
          fprintf(stderr,"Error: Could not open file `%s' : %s\n",
                     filename3,strerror(e3));
          if(e3 == EACCES)
            fprintf(stderr,"Run as root?\n");
        }
        exit(1);
      } else {
         filename = filename3;
      }
    } else {
       filename = filename2;
    }
  } else {
    filename = filename1;
  }
  
  /* check adapter functionality */
  if (ioctl(file,I2C_FUNCS,&funcs) < 0) {
    fprintf(stderr,
            "Error: Could not get the adapter functionality matrix: %s\n",
            strerror(errno));
    exit(1);
  }

        if (! (funcs &
            (I2C_FUNC_SMBUS_WRITE_BYTE_DATA | I2C_FUNC_SMBUS_READ_BYTE_DATA))) {
           fprintf(stderr, "Error: Adapter for i2c bus %d", i2cbus);
           fprintf(stderr, " does not have byte write capability\n");
           exit(1);
        }       
  
  /* use FORCE so that we can write registers even when
     a driver is also running */
  if (ioctl(file,I2C_SLAVE_FORCE,address) < 0) {
    fprintf(stderr,"Error: Could not set address to %d: %s\n",address,
            strerror(errno));
    exit(1);
  }

  return (file);
}

int main(int argc, char *argv[])
{
  char digits[6];
  int file ; //= digimatrix_I2CInit(0);;
  int ch, longindex = 0;

  char buffer[20], *ptr;

  BOOL bUpdateTemp = FALSE;
  BOOL bUpdateVol = FALSE;

  const struct tm * nice_time;
  time_t nasty_time;
  char hour, minute;


  if (argc == 1)
  {
      printf("\nPlease supply some parameters.\nUse %s -h for list.\n\n", argv[0]);
  }

/*  fprintf(stderr,"  WARNING! This program can confuse your I2C bus, "
          "cause data loss and worse!\n Continue (y/n)?");

    ch = getchar(); 

        if(ch != 'y' && ch != 'Y')
                exit(1);
*/
    for (;;) 
    {
        if ((ch = getopt_long(argc, argv, "i:r:d:tulmsvh", longopts, &longindex)) == -1)
        {
            break;
        }

        if ( (bI2CInitialised != TRUE) && (ch!='i' && ch!='h' && ch!='v') )
        {
            printf("\n*** You must supply -i <n> as the first arguement! ***\n\n");
            break;
        }

        switch (ch) 
        {
        case 'i':
            printf("\nUsing I2C Bus %s\n", optarg);
            file = digimatrix_I2CInit(atoi(optarg));
            if (file > 0)
            {
                bI2CInitialised = TRUE;
            }
            break;
        
        case 'r':
                printf("Setting Radio ");
                if (!strcmp(optarg, "fm"))
                {
                    printf("ON  - FM\n");
                    digimatrix_TurnOnRadio(file, FM_RADIO);            
                }
                else if (!strcmp(optarg, "am"))
                {
                    printf("ON - AM\n");
                    digimatrix_TurnOnRadio(file, AM_RADIO);            
                }
                else
                {
                    printf("OFF \n");
                    digimatrix_TurnOnRadio(file, OFF_MODE);            
                }
            break;
            
        case 'd':
	    {
	        LED led = LED_NONE;
		PP pp = PP_NONE;
		DOTS dots = DOTS_NONE;
		
                strncpy(digits, optarg, 6);
                
		printf("Args = %s", optarg);
                printf("\nSetting display to: %s ", digits);
    	        if (strstr(optarg, "play"))
   	        {
		    pp = PP_PLAY;
		    printf(" play");
                }
    	        else if (strstr(optarg, "pause"))
   	        {
		    pp = PP_PAUSE;
		    printf(" pause");
                }
		
    	        if (strstr(optarg, "am"))
   	        {
		    led = LED_AM;
		    printf(" am");
                }
    	        else if (strstr(optarg, "fm"))
   	        {
		    led = LED_FM;
		    printf(" fm");
                }
    	        else if (strstr(optarg, "cd"))
   	        {
		    led = LED_CD;
		    printf(" cd");
                }
    	        else if (strstr(optarg, "hd"))
   	        {
		    led = LED_HD;
		    printf(" hd");
                }


    	        if (strstr(optarg, "both"))
   	        {
		    dots = DOTS_BOTH;
		    printf(" :");
                }
    	        else if (strstr(optarg, "middle"))
   	        {
		    dots = DOTS_MIDDLE;
		    printf(" -");
                }
    	        else if (strstr(optarg, "bottom"))
   	        {
		    dots = DOTS_BOTTOM;
		    printf(" .");
                }
		printf("\n");
		
                digimatrix_DisplayNumbers(file, digits, led, pp, dots);
	    }
            break;
            
        case 't':
            printf("Displaying Temperature.\n");
            digimatrix_DisplayTemp(file, 99);
            break;
            
        case 'u':
            printf("Run Temperature Display in update mode.\n");
            digimatrix_DisplayTemp(file, 99);
            bUpdateTemp = TRUE;
            break;
            
        case 'l':
            printf("Display Volume control. \n");
            bUpdateVol = TRUE;
            break;

        case 'm':
            printf("Entering Interactive mode. Press enter to exit\n");
            printf("Please enter \"REG VALUE\"\n");
            printf("Where REG and VALUE are decimal integers.\n");
            while (buffer[0] != '\n')
            {
                if (fgets(buffer, 20, stdin)) // Get at most 10 bytes from user.
                {
                    if (atoi(buffer) > 86)
                    {
                        printf("I'm not letting you write there - the I2C bus will lock up!\n");
                    }
                    else
                    {
                        ptr = strstr(buffer, " "); // Find the start of the data value
                        if (ptr)
                        {
                            printf("0x%X = 0x%X ", atoi(buffer), atoi(ptr));
                            digimatrix_write_byte_data(file, atoi(buffer), atoi(ptr));
                            printf("Read Back: 0x%0X\n", digimatrix_read_byte(file));
                        }
                    }   
                }
            }

            break;

        case 's':
            time(&nasty_time);
            nice_time = localtime(&nasty_time);
            hour = ((nice_time->tm_hour / 10) << 4) | (nice_time->tm_hour % 10);
            minute = ((nice_time->tm_min / 10) << 4) | (nice_time->tm_min % 10);
 
            printf("Setting and displaying time (%X:%X).\n", hour, minute);
            digimatrix_write_byte_data(file, 30, hour);
            digimatrix_write_byte_data(file, 31, minute);
            digimatrix_write_byte_data(file, 41, 0x01);
            break;

        case 'v':
            printf("ASUS DigiMatrix Panel Control Version %s\n", DM_VERSION);
            break;

        case 'h':
            help(argv[0]);
            break;

        default:
            help(argv[0]);
            break;
        }
    }

    if (bUpdateVol || bUpdateTemp)
    {
        unsigned int temp_counter = 0;
        int res, old_res = 0xDEADF00D;
	char volume = 0;

        while (1)
        {
            if (bUpdateTemp)
            {
                if (temp_counter >= (TEMPERATURE_UPDATE_INTERVAL_MS / VOLUME_UPDATE_INTERVAL_MS))
                {
                    temp_counter = 0;
                    digimatrix_UpdateTemp(file, 99);
                    //fprintf(stderr, "\rUpdated Temperature.   ");
                }
            }

            if (bUpdateVol)
            {
                digimatrix_write_byte(file, 0x24);
                //fprintf(stderr, "%d - %d            \r", digimatrix_read_byte(file), n);
                res = volume = digimatrix_read_byte(file);
                if (res != 0) //old_res != res)
                {
                    old_res = res;
                    fprintf(stderr, "Volume Change: %d\n", res);
                }
            }
            usleep(VOLUME_UPDATE_INTERVAL_MS * 1000); // Delay of 500ms
            temp_counter++;
        }
    }
    
    exit (0);
}
