#include <allegro.h>
#include <sys/farptr.h>
#include <go32.h>
#include <dpmi.h>
#include <string.h>
#include <pc.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>




typedef unsigned char       BYTE;
typedef unsigned short      WORD;
typedef unsigned long       DWORD;
typedef int BOOL;

#ifndef FALSE
#define FALSE   0
#endif

#ifndef TRUE
#define TRUE    !FALSE
#endif

#define PCI_ANY_ID  ((WORD)(~0))


typedef struct {
    WORD vender_id;
    WORD device_id;
    WORD sub_vender_id;
    WORD sub_device_id;
    WORD device_bus_number;
} PCI_DEV;


typedef struct {
    WORD vender_id;
    WORD device_id;
    WORD sub_vender_id;
    WORD sub_device_id;
    int  io_timing;
    int  dma_method;
    void (*ddma_setting_function)(WORD ddma_base, int dma_channel);
    char *string;
} DEVICE_LIST;

typedef struct {
    BYTE master;
    BYTE sb_pcm;
    BYTE opl;
    BYTE cd_audio;
    BYTE line_in;
    BYTE microphone;
    BYTE output_gain;
} SB_MIXER;

typedef struct {
    int sb_base;
    int irq;
    int dma;
    int mpu_base;
    int adlib_base;
    int game_base;
    int sb_ver;
    int dma_method;
    int io_timing;
    int ddma_base;
    int irq_trigger_mode;
}SB_RESOURCE;


#define DMA_AUTO                0
#define DMA_DDMA                1
#define DMA_LEGACY_DMA          2
#define DMA_WB_DMA              3
#define DMA_WRITE_SNOOP         4

#define IO_AUTO                 -1
#define IO_SUBTRACTIVE          0
#define IO_POSITIVE             1



/*
 *  grobal variables
 */

static int g_als4000_iobase = 0;
static int g_sb_port = 0;



static BOOL find_known_device(int *io_timing, int *dma_method,
                              void (**ddma_setting_function)(WORD, int) );
void w_enter_critical(void);
void w_exit_critical(void);
static BOOL pci_read_config_byte(PCI_DEV *pci, int index, BYTE *data);
static BOOL pci_read_config_word(PCI_DEV *pci, int index, WORD *data);
static BOOL pci_read_config_dword(PCI_DEV *pci, int index, DWORD *data);
static BOOL pci_write_config_byte(PCI_DEV *pci, int index, BYTE data);
static BOOL pci_write_config_word(PCI_DEV *pci, int index, WORD data);
#if 0
static BOOL pci_write_config_dword(PCI_DEV *pci, int index, DWORD data);
#endif
static BOOL set_pci_hardware_interrupt(PCI_DEV *pci, BYTE int_pin, BYTE irq);
static BOOL check_pci_bios(void);
static BOOL find_pci_device(PCI_DEV *pci);

static void logerror(const char *text,...)
{
    va_list arg;

    va_start(arg,text);
    vfprintf(stdout,text,arg);
    va_end(arg);
}

static void message(const char *text,...)
{
    va_list arg;

    va_start(arg,text);
    vfprintf(stdout,text,arg);
    va_end(arg);
}


/*
 *  chipset dependent DDMA setup funtions
 */

static void sis630_ddma(WORD ddma_base, int dma_channel)
{
    PCI_DEV pci;
    BYTE bdata;

    pci.vender_id     = 0x1039;             // SiS
    pci.device_id     = 0x0008;             // PCI-ISA Bridge
    pci.sub_vender_id = PCI_ANY_ID;
    pci.sub_device_id = PCI_ANY_ID;
    if(find_pci_device(&pci) == FALSE){
        pci.device_id     = 0x0018;         // PCI-ISA Bridge
        if(find_pci_device(&pci) == FALSE){
            message("sis630_ddma: No SiS PCI-ISA Bridge found\n");
            return;
        }
    }
    bdata = 1 << dma_channel;                           /* set DDMA channel */
    pci_write_config_byte(&pci, 0x49, bdata);
    bdata = (BYTE)((ddma_base & 0xF0) | 0x0001);        /* DDMA I/O base(lower 8bit) | DDMA enable flag*/
    pci_write_config_byte(&pci, 0x4A, bdata);
    bdata = (BYTE)(ddma_base >> 8);                     /* DDMA I/O base(higher 8bit) */
    pci_write_config_byte(&pci, 0x4B, bdata);
}

static void sis5595_ddma(WORD ddma_base, int dma_channel)
{
    PCI_DEV pci;
    BYTE bdata;

    pci.vender_id     = 0x1039;             // SiS
    pci.device_id     = 0x0008;             // PCI-ISA Bridge
    pci.sub_vender_id = PCI_ANY_ID;
    pci.sub_device_id = PCI_ANY_ID;
    if(find_pci_device(&pci) == FALSE){
        message("sis5595_ddma: No SiS PCI-ISA Bridge found\n");
        return;
    }
    bdata = 1 << dma_channel;                           /* set DDMA channel */
    pci_write_config_byte(&pci, 0x84, bdata);
    bdata = (BYTE)((ddma_base & 0xF0) | 0x0001);        /* DDMA I/O base(lower 8bit) | DDMA enable flag*/
    pci_write_config_byte(&pci, 0x80, bdata);
    bdata = (BYTE)(ddma_base >> 8);                     /* DDMA I/O base(higher 8bit) */
    pci_write_config_byte(&pci, 0x81, bdata);
}

static void intel_piix4_ddma(WORD ddma_base, int dma_channel)
{
    PCI_DEV pci;
    WORD wdata;

    pci.vender_id     = 0x8086;             // Intel
    pci.device_id     = 0x7110;             // PIIX4
    pci.sub_vender_id = PCI_ANY_ID;
    pci.sub_device_id = PCI_ANY_ID;
    if(find_pci_device(&pci) == FALSE){
        message("intel_piix4_ddma: No PIIX4 found\n");
        return;
    }
    /*  set ddma enable bit */
    wdata = 0x02 << (dma_channel * 2);
    pci_write_config_word(&pci, 0x90, wdata);
    if(dma_channel >= 4){
        /*  DMA channel 4 to 7  */
        ddma_base += 0x40;
        pci_write_config_word(&pci, 0x94, ddma_base & 0xFFC0);
    }else{
        /*  DMA channel 0 to 3  */
        pci_write_config_word(&pci, 0x92, ddma_base & 0xFFC0);
    }
}

static void via_ddma(WORD ddma_base, int dma_channel)
{
    PCI_DEV pci;
    int offset;
    WORD wdata;

    while(1){       // this is in place of goto-label
        pci.sub_vender_id = PCI_ANY_ID;
        pci.sub_device_id = PCI_ANY_ID;
        pci.vender_id     = 0x1106;             // VIA
        pci.device_id     = 0x0586;             // VT82C586
        if(find_pci_device(&pci) == TRUE) break;
        pci.device_id     = 0x0596;             // VT82C596
        if(find_pci_device(&pci) == TRUE) break;
        pci.device_id     = 0x0686;             // VT82C686
        if(find_pci_device(&pci) == TRUE) break;
        pci.vender_id     = 0x1022;             // AMD
        pci.device_id     = 0x7408;             // AMD-756
        if(find_pci_device(&pci) == TRUE) break;
        message("via_ddma: No VIA VT82C586/596/686/AMD-756 found\n");
        return;
    }
    dma_channel &= 0x07;
    if(dma_channel == 4) return;
    ddma_base += dma_channel * 0x10;
    offset = 0x60 + (dma_channel * 2);
    wdata  = (ddma_base & 0xFFF0) | 0x0008;
    pci_write_config_word(&pci, offset, wdata);
}


static DEVICE_LIST dev_list[] = {
    { 0x8086, 0x7110, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       intel_piix4_ddma, "Intel PIIX4" },
    { 0x1106, 0x0586, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_LEGACY_DMA, NULL,             "VIA VT82C586" },
    { 0x1106, 0x0596, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_LEGACY_DMA, NULL,             "VIA VT82C596" },
    { 0x1106, 0x0686, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_LEGACY_DMA, NULL,             "VIA VT82C686" },
    { 0x1022, 0x7408, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       via_ddma,         "AMD-756" },
    { 0x1039, 0x5591, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis5595_ddma,     "SiS5591" },
    { 0x1039, 0x0530, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis5595_ddma,     "SiS530" },
    { 0x1039, 0x0620, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis5595_ddma,     "SiS620" },
    { 0x1039, 0x0745, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS745" },
    { 0x1039, 0x0735, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS735" },
    { 0x1039, 0x0733, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS733" },
    { 0x1039, 0x0730, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS730" },
    { 0x1039, 0x0635, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS635" },
    { 0x1039, 0x0633, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS633" },
    { 0x1039, 0x0630, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS630" },
    { 0x1039, 0x0540, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS540" },
    { 0x1039, 0x0008, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_DDMA,       sis630_ddma,      "SiS0008 PCI-ISA Bridge" },

    /*  Add chipset information here  */

    { PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, IO_SUBTRACTIVE, DMA_LEGACY_DMA, NULL, NULL }
};


static BOOL find_known_device(int *io_timing, int *dma_method,
                              void (**ddma_setting_function)(WORD, int) )
{
    int result = FALSE;
    int index;
    PCI_DEV pci;

    for(index = 0; TRUE ; index += 1){
        if(dev_list[index].vender_id == PCI_ANY_ID){
            *io_timing  = dev_list[index].io_timing;
            *dma_method = dev_list[index].dma_method;
            *ddma_setting_function = NULL;
            break;
        }
        pci.vender_id     = dev_list[index].vender_id;
        pci.device_id     = dev_list[index].device_id;
        pci.sub_vender_id = dev_list[index].sub_vender_id;
        pci.sub_device_id = dev_list[index].sub_device_id;
        if(find_pci_device(&pci) == TRUE){
            *io_timing  = dev_list[index].io_timing;
            *dma_method = dev_list[index].dma_method;
            *ddma_setting_function = dev_list[index].ddma_setting_function;
            result = TRUE;
            message("%s found\n", dev_list[index].string);
            break;
        }
    }
    return result;
}



/*
 *  interrupt disable/enable funtion
 */

static DWORD tasmania = 0;

void w_enter_critical(void)
{
    __dpmi_regs r;

    tasmania += 1;
    if(tasmania == 1){
        r.x.ax = 0x1681;
        __dpmi_int(0x2F, &r);
        asm volatile("cli");
    }
}

void w_exit_critical(void)
{
    __dpmi_regs r;

    tasmania -= 1;
    if(tasmania == 0){
        r.x.ax = 0x1682;
        __dpmi_int(0x2F, &r);
        asm volatile("sti");
    }
}


/*
 *  PCI BIOS helper funtions
 */

static BOOL pci_read_config_byte(PCI_DEV *pci, int index, BYTE *data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B108;                       /* read config byte */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci read config byte failed\n");
        result = FALSE;
        r.d.ecx = 0;
    }
    *data = (BYTE)r.d.ecx;
    return result;
}

static BOOL pci_read_config_word(PCI_DEV *pci, int index, WORD *data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B109;                       /* read config word */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci read config word failed\n");
        result = FALSE;
        r.d.ecx = 0;
    }
    *data = (WORD)r.d.ecx;
    return result;
}

static BOOL pci_read_config_dword(PCI_DEV *pci, int index, DWORD *data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B10A;                       /* read config dword */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci read config dword failed\n");
        result = FALSE;
        r.d.ecx = 0;
    }
    *data = (DWORD)r.d.ecx;
    return result;
}

static BOOL pci_write_config_byte(PCI_DEV *pci, int index, BYTE data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B10B;                       /* write config byte */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.ecx = (DWORD)data;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci write config byte failed\n");
        result = FALSE;
    }
    return result;
}

static BOOL pci_write_config_word(PCI_DEV *pci, int index, WORD data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B10C;                       /* write config word */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.ecx = (DWORD)data;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci write config word failed\n");
        result = FALSE;
    }
    return result;
}

#if 0

static BOOL pci_write_config_dword(PCI_DEV *pci, int index, DWORD data)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B10D;                       /* write config dword */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.d.ecx = (DWORD)data;
    r.d.edi = (DWORD)index;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("pci write config dword failed\n");
        result = FALSE;
    }
    return result;
}

#endif

static BOOL set_pci_hardware_interrupt(PCI_DEV *pci, BYTE int_pin, BYTE irq)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B10F;                       /* set pci hardware interrupt */
    r.d.ebx = (DWORD)pci->device_bus_number;
    r.h.cl  = int_pin;
    r.h.ch  = irq;
    r.x.ds  = 0xF000;
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        logerror("set pci hardware interrupt failed.\n");
        result = FALSE;
    }
    pci_write_config_byte(pci, 0x3C, irq);
    return result;
}



static BOOL check_pci_bios(void)
{
    __dpmi_regs r;
    BOOL result = TRUE;

    r.d.eax = 0x0000B101;                   // PCI BIOS - INSTALLATION CHECK
    r.d.edi = 0x00000000;
    __dpmi_int(0x1a, &r);
    if( r.d.edx != 0x20494350 ){            // ' ICP'
        result = FALSE;
    }
    return result;
}


static BOOL find_pci_device(PCI_DEV *pci)
{
    __dpmi_regs r;
    WORD wdata;

    r.d.eax = 0x0000B102;                   // PCI BIOS - FIND PCI DEVICE
    r.d.ecx = pci->device_id;               // device ID
    r.d.edx = pci->vender_id;               // vendor ID
    r.d.esi = 0x00000000;                   // device index
    __dpmi_int(0x1a, &r);
    if( r.h.ah != 0 ){
        return FALSE;                       // no specified device found
    }
    pci->device_bus_number = r.x.bx;        // save the device & bus number
    if(pci->sub_vender_id != PCI_ANY_ID){
        /* get subsystem vender id */
        if(pci_read_config_word(pci, 0x2C, &wdata) == FALSE) return FALSE;
        if(wdata != pci->sub_vender_id) return FALSE;
    }
    if(pci->sub_device_id != PCI_ANY_ID){
        /* get subsystem device id */
        if(pci_read_config_word(pci, 0x2E, &wdata) == FALSE) return FALSE;
        if(wdata != pci->sub_device_id) return FALSE;
    }
    /* device found */
    return TRUE;
}


static void enable_pci_io_access(PCI_DEV *pci)
{
    WORD wdata;

    pci_read_config_word(pci, 0x04, &wdata);
    /*  set I/O access enable bit  */
    wdata |= 0x0001;
    pci_write_config_word(pci, 0x04, wdata);
}

#if 0

static void enable_pci_memory_access(PCI_DEV *pci)
{
    WORD wdata;

    pci_read_config_word(pci, 0x04, &wdata);
    /*  set memory access enable bit  */
    wdata |= 0x0002;
    pci_write_config_word(pci, 0x04, wdata);
}

#endif

static void enable_pci_bus_master(PCI_DEV *pci)
{
    WORD wdata;

    pci_read_config_word(pci, 0x04, &wdata);
    /*  set bus master enable bit  */
    wdata |= 0x0004;
    pci_write_config_word(pci, 0x04, wdata);
}



/*
 *  ALS4000/SB functions
 */

static DWORD als4000_read_gcr(int index)
{
    outportb(g_als4000_iobase + 0x0C, index);
    return (DWORD)inportl(g_als4000_iobase + 0x08);
}
static void als4000_write_gcr(int index, DWORD data)
{
    outportb(g_als4000_iobase + 0x0C, index);
    outportl(g_als4000_iobase + 0x08, data);
}

static void sb_write_mixer(BYTE index, BYTE data)
{
    outportb(g_sb_port + 4, index);
    outportb(g_sb_port + 5, data);
}

static BYTE sb_read_mixer(BYTE index)
{
    outportb(g_sb_port + 4, index);
    return inportb(g_sb_port + 5);
}

static void sb_write_ctrlreg(BYTE index, BYTE data)
{
    outportb(g_sb_port + 4, index | 0xC0);
    outportb(g_sb_port + 5, data);
}

static BYTE sb_read_ctrlreg(BYTE index)
{
    outportb(g_sb_port + 4, index | 0xC0);
    return inportb(g_sb_port + 5);
}


static int sb_read_dsp(void)
{
   int x;

   for (x=0; x<0xFFFF; x++)
      if (inportb(0x0E + g_sb_port) & 0x80)
     return inportb(0x0A + g_sb_port);

   return -1;
}

static int sb_reset_dsp(int d0)
{
    int val;
    int cwait;
    int result;
    int d7;

    result = -1;
    cwait = 8;
    while(cwait <= 1024){
        w_enter_critical();
        outportb(0x06 + g_sb_port, d0);
        d7 = 0;
        while(d7 < cwait){
            inportb(0x06 + g_sb_port);
            d7 += 1;
        }
        outportb(0x06 + g_sb_port, 0);
        w_exit_critical();
        for(d7 = 0 ; d7 < 4 ; d7 += 1){
            val = sb_read_dsp();
            if(val == -1) continue;
            if(val == 0xAA){
                result = 0;
                break;
            }
            break;
        }
        if(result == 0) break;
        cwait = cwait * 2;
    }
    return result;
}

static void set_mixer_level(SB_MIXER mixer)
{
    BYTE level;
    BYTE enable_flag;

    sb_write_mixer(0x00, 0);                    //  reset mixer
    enable_flag = sb_read_mixer(0x3C) & 0xE0;   //  read output mixer control 1 reg.

    level = (mixer.master & 0x1F) << 3;         //  0:-46dB, 31:0dB  (1.5dB step)
    sb_write_mixer(0x30, level);                //  master volume left
    sb_write_mixer(0x31, level);                //  master volume right
    level = (mixer.sb_pcm & 0x1F) << 3;
    sb_write_mixer(0x32, level);                //  sb-pcm volume left
    sb_write_mixer(0x33, level);                //  sb-pcm volume right
    level = (mixer.opl & 0x1F) << 3;
    sb_write_mixer(0x34, level);                //  opl volume left
    sb_write_mixer(0x35, level);                //  opl volume right
    level = (mixer.cd_audio & 0x1F) << 3;
    if(level > 0){
        enable_flag |= 0x06;                    //  cd-audio enable
    }
    sb_write_mixer(0x36, level);                //  cd-audio volume left
    sb_write_mixer(0x37, level);                //  cd-audio volume right
    level = (mixer.line_in & 0x1F) << 3;
    if(level > 0){
        enable_flag |= 0x18;                    //  line-in enable
    }
    sb_write_mixer(0x38, level);                //  line-in volume left
    sb_write_mixer(0x39, level);                //  line-in volume right
    level = (mixer.microphone & 0x1F) <<3;
    if(level > 0){
        enable_flag |= 0x01;                    //  microphone enable
    }
    sb_write_mixer(0x3A, level);                //  microphone volume

    sb_write_mixer(0x3C, enable_flag);          //  write output mixer control 1 reg.

    switch(mixer.output_gain){
        case 0:
        case 1:
            /*  gain *1  */
            level = 0x00;
            break;
        case 2:
        case 3:
            /*  gain *2  */
            level = 0x40;
            break;
        case 4:
        default:
            /*  gain *4  */
            level = 0x80;
            break;
    }
    sb_write_mixer(0x41, level);                 //  output left  mixer gain
    sb_write_mixer(0x42, level);                 //  output right mixer gain
}


static void set_irq_and_dma_to_mixer_reg(BYTE irq, BYTE dma_channel)
{
    BYTE mx80;
    BYTE mx81;
    BYTE cr0;

    cr0 = sb_read_ctrlreg(0x00);
    sb_write_ctrlreg(0x00, cr0 | 0x80);         // MX80,81 write protection disable
    switch(irq){
        case 9:
            mx80 = 0x01;
            break;
        case 5:
            mx80 = 0x02;
            break;
        case 7:
            mx80 = 0x40;
            break;
        case 10:
            mx80 = 0x80;
            break;
        case 11:
            mx80 = 0x10;
            break;
        default:
            mx80 = 0x00;
    }
    switch(dma_channel){
        case 0:
            mx81 = 0x01;
            break;
        case 1:
            mx81 = 0x02;
            break;
        case 3:
            mx81 = 0x80;
            break;
        default:
            mx81 = 0x00;
    }
    sb_write_mixer(0x80, mx80);                 // set sb irq (dummy reg. for only compatibility)
    sb_write_mixer(0x81, mx81);                 // set sb dma
    sb_write_ctrlreg(0x00, cr0 & 0x7F);         // MX80,81 write protection enable
}





static void clear_sb_mpu_irq(void)
{
    w_enter_critical();
    outportb(g_als4000_iobase + 0x0E, inportb(g_als4000_iobase + 0x0E));
    inportb(g_sb_port+0x0E);
    inportb(g_sb_port+0x0F);
    w_exit_critical();
}


static void enable_legacy_function(int sbbase, int mpubase, int adlibbase, int gamebase)
{
    DWORD data;
    DWORD d0;

    data = 0;
    if(sbbase > 0){
        d0 = sbbase & 0x3F0;
        data |= d0 | 0x01;
    }
    if(mpubase > 0){
        d0 = mpubase & 0x3F8;
        data |= (d0 | 0x01) << 16;
    }
    als4000_write_gcr(0xA9, data);

    data = 0;
    if(adlibbase > 0){
        d0 = adlibbase & 0x3F8;
        data |= d0 | 0x01;
    }
    if(gamebase > 0){
        d0 = gamebase & 0x3F8;
        data |= (d0 | 0x01) << 16;
    }
    als4000_write_gcr(0xA8, data);
}


#if 0

static int get_irq_edge_or_level(int irq)
{
    int status;

    /*  I/O 04D0h
     *      bit 7   irq 7   0=Edge Triggered mode; 1=Level Triggered mode
     *      bit 6   irq 6   0=Edge Triggered mode; 1=Level Triggered mode
     *                          .
     *                          .
     *                          .
     *      bit 0   irq 0   0=Edge Triggered mode; 1=Level Triggered mode
     *
     *  I/O 04D1h
     *      bit 7   irq 15  0=Edge Triggered mode; 1=Level Triggered mode
     *      bit 6   irq 14  0=Edge Triggered mode; 1=Level Triggered mode
     *                          .
     *                          .
     *                          .
     *      bit 0   irq 8   0=Edge Triggered mode; 1=Level Triggered mode
     *
     *
     *  return == 0     edge  triggered
     *         != 0     level triggered
     */

    if(irq > 15) return 0;
    if(irq < 8){
        status = inportb(0x04D0) & (1 << irq);
    }else{
        irq -= 8;
        status = inportb(0x04D1) & (1 << irq);
    }
    if(status != 0) status = 1;

    return status;
}

#endif

static void set_irq_edge_or_level(int irq, int edge_or_level)
{
    /*
     *  edge_or_level == 0     set edge  triggered
     *                != 0     set level triggered
     */

    int bitmap;

    if(irq > 15) return;
    if(edge_or_level != 0) edge_or_level = 1;
    if(irq < 8){
        bitmap = inportb(0x04D0);
        bitmap = bitmap & ~(1 << irq);
        bitmap = bitmap | (edge_or_level << irq);
        outportb(0x04D0, bitmap);
    }else{
        irq -= 8;
        bitmap = inportb(0x04D1);
        bitmap = bitmap & ~(1 << irq);
        bitmap = bitmap | (edge_or_level << irq);
        outportb(0x04D1, bitmap);
    }
}

static int get_create_batchfile_flag(void)
{
    return get_config_int("misc", "CREATE_BATCHFILE", 0);
}


static void get_sb_resource(SB_RESOURCE *res)
{
    res->sb_base     = get_config_int("resource", "SB16",     0x220) & ~0x000F;
    res->irq         = get_config_int("resource", "IRQ", 7) & 0x0F;
    res->dma         = get_config_int("resource", "DMA", 1) & 0x03;
    res->mpu_base    = get_config_int("resource", "MPU401",   0x330) & ~0x0007;
    res->adlib_base  = get_config_int("resource", "ADLIB",    0x388) & ~0x0007;
    res->game_base   = get_config_int("resource", "GAMEPORT", 0x200) & ~0x0007;
    res->sb_ver      = get_config_int("resource", "DSP_VER",  0x402);
    res->dma_method  = get_config_int("resource", "DMA_METHOD", 0);
    res->io_timing   = get_config_int("resource", "IO_TIMING",  IO_AUTO);
    res->ddma_base   = get_config_int("resource", "DDMA_BASE",  0x1000) & ~0x000F;
    res->irq_trigger_mode = get_config_int("resource", "IRQ_TRIGGER", 1) & 0x01;
}


static void print_sb_resource(SB_RESOURCE *res)
{
    char ver[4][8] = { "SB1", "SB2", "SBPro", "SB16" };
    char iot[4][32] = { "Auto(Subtractive)", "Subtractive", "Positive", "" };
    char irq[2][8] = { "Edge", "Level" };
    int major_ver;

    major_ver = (res->sb_ver >> 8) - 1;
    printf("[SB Resource]\n");
    printf("  SB16: %3Xh, ", res->sb_base);
    printf("IRQ:%2d, ", res->irq);
    printf("DMA: %1X, ", res->dma);
    printf("MPU401: %3Xh, ", res->mpu_base);
    printf("ADLIB: %3Xh, ", res->adlib_base);
    printf("GAMEPORT: %3Xh\n", res->game_base);
    printf("  DSP_VER: %3X (%s)\n", res->sb_ver, ver[major_ver & 0x03]);
    printf("  IO_TIMING: %s\n", iot[res->io_timing & 0x03]);
    printf("  IRQ %d %s triggered.\n", res->irq, irq[res->irq_trigger_mode & 0x01]);
}


static void get_mixer_level(SB_MIXER *mixer)
{
    mixer->master      = (BYTE)get_config_int("volume", "MASTER", 31);
    mixer->sb_pcm      = (BYTE)get_config_int("volume", "PCM", 27);
    mixer->opl         = (BYTE)get_config_int("volume", "OPL", 27);
    mixer->cd_audio    = (BYTE)get_config_int("volume", "CD", 0);
    mixer->line_in     = (BYTE)get_config_int("volume", "LINE", 0);
    mixer->microphone  = (BYTE)get_config_int("volume", "MIC", 0);
    mixer->output_gain = (BYTE)get_config_int("volume", "OUTPUT_GAIN", 2);
}

static void print_mixer_level(SB_MIXER *mixer)
{
    int gain;

    switch(mixer->output_gain){
        case 0:
        case 1:
            gain = 1;
            break;
        case 2:
        case 3:
            gain = 2;
            break;
        case 4:
        default:
            gain = 4;
            break;
    }
    printf("[Volume]\n");
    printf("  OUTPUT GAIN: %d\n", gain);
    printf("  MASTER:%2d, ", mixer->master);
    printf("PCM:%2d, ", mixer->sb_pcm);
    printf("OPL:%2d, ", mixer->opl);
    printf("CD:%2d, ", mixer->cd_audio);
    printf("LINE:%2d, ", mixer->line_in);
    printf("MIC:%2d\n", mixer->microphone);
}



static int get_als4000_io_address(PCI_DEV *pci)
{
    DWORD ddata;

    /* get I/O base address */
    pci_read_config_dword(pci, 0x10, &ddata);
    return (int)(ddata & ~0x7F);
}

static int get_als4000_irq(PCI_DEV *pci)
{
    BYTE irq;

    /* get als4000's irq number */
    pci_read_config_byte(pci, 0x3C, &irq);
    return (int)irq;
}


static void als4000_irq_enable(int irq, int irq_trigger_mode)
{
    DWORD ddata;

    ddata = als4000_read_gcr(0x8C);
    /* ALS4000 INTA# enable and edge trigger */
    als4000_write_gcr(0x8C, ddata | 0x08008000);

    /*
     *  set IRQ trigger mode
     *
     *    irq_trigger_mode == 0   edge triggered
     *                     != 0   level triggered (recommend)
     */
    set_irq_edge_or_level(irq, irq_trigger_mode);
}





static void set_sb_dsp_version(WORD dsp_version)
{
    sb_write_ctrlreg(0x18, dsp_version >> 8);       // major version
    sb_write_ctrlreg(0x19, dsp_version & 0xFF);     // minor version
}


static void set_dma_emulation_method(int dma_method, int dma_channel,
                                     int ddma_base,  int io_timing,
                                     void (*ddma_setting_function)(WORD, int))
{
    DWORD gcr99;
    DWORD temp;

    gcr99  = als4000_read_gcr(0x99);

    /* set I/O decode timing */
    temp = (1 << 22);
    if(gcr99 & temp){
        if(io_timing == IO_POSITIVE){
            temp = 0;
        }
    }else{
        if(io_timing == IO_SUBTRACTIVE){
            temp = 0;
        }
    }
    gcr99 &= ~0xB070FFF7;
    gcr99 |= 0xB0000000;
    gcr99 |= temp;

    gcr99 |= (1 << 20);         //  REQ# generated when PCM buffer is not full
    switch(dma_method){
        case DMA_DDMA:
            /* set ddma base address */
            temp = ddma_base + ((dma_channel & 0x03) * 0x10);
            gcr99 |= (temp & 0x0000FFF0);
            gcr99 |= 0x01;              //  ddma mode
            message("Use DDMA mode, DDMABASE is %04Xh\n", ddma_base);
            /*  call chipset dependent DDMA setup code  */
            if(ddma_setting_function != NULL){
                (*ddma_setting_function)(ddma_base, dma_channel);
            }
            break;
        case DMA_LEGACY_DMA:
            gcr99 |= 0x02;              //  legacy dma mode
            message("Use LEGACY DMA mode\n");
            break;
        case DMA_WB_DMA:
            gcr99 |= 0x03;              //  writeback dma mode
            message("Use WRITE-BACK DMA mode\n");
            break;
        case DMA_WRITE_SNOOP:
        default:
            gcr99 |= 0x04;              //  write snoop dma mode (burst transfer)
            gcr99 &= ~(1 << 20);        //  REQ# generated when PCM buffer is half empty
            message("Use WRITE SNOOP DMA mode\n");
            break;
    }
    als4000_write_gcr(0x99, gcr99);
}

static void als4000_init_misc(void)
{
    sb_write_mixer(0x4F, 0x20);         //  SB D/A 3dB freq 20KHz
    sb_write_mixer(0x50, 0xA4);         //  mute 3D
    sb_write_mixer(0x53, 0x00);         //  init analog block control reg.
    sb_write_ctrlreg(0x00, 0x06);       //  set 90h command FIFO controlled
    sb_write_ctrlreg(0x3A, 0x01);       //  SB16 E3h command enable
}


static BOOL create_set_env_file(char* envstr)
{
    FILE *fp;

    fp = fopen("a4setenv.bat", "w");
    if(fp == NULL){
        printf("could not open file \"a4setenv.bat\".");
        return FALSE;
    }
    if(fputs(envstr, fp) == EOF){
        printf("could not write file \"a4setenv.bat\".");
        fclose(fp);
        return FALSE;
    }
    fclose(fp);
    return TRUE;
}


static void copy_to_dos_memory(int selector, BYTE *ptr, int length)
{
    int d0;

    _farsetsel(selector);
    d0 = 0;
    while(d0 < length){
        _farnspokeb(d0, *(ptr + d0));
        d0 += 1;
    }
}


static BOOL _INT2E(char* commandline)
{
    __dpmi_regs r;
    int segment;
    int selector;
    int d0;
    int d1;
    char* a0;

    segment = __dpmi_allocate_dos_memory(1024, &selector);
    if(segment == -1) return FALSE;

    a0 = (char*)malloc(512);
    if(a0 == NULL) goto _INT2E_error;
    d0 = strlen(commandline);
    if(d0 > 255) goto _INT2E_error;

    d1 = 0;
    while(d1 < d0){
        a0[d1+1] = commandline[d1];
        d1 += 1;
    }
    a0[d1+1] = 0x0D;
    a0[d1+2] = 0;
    a0[0] = d0;

    copy_to_dos_memory(selector, a0, d0 + 2);

    r.x.ds = segment;
    r.x.si = 0;
    __dpmi_int(0x2E, &r);
    if( r.x.ax == 0xFFFF ) goto _INT2E_error;

    __dpmi_free_dos_memory(selector);
    return TRUE;

_INT2E_error:
    __dpmi_free_dos_memory(selector);
    return FALSE;
}

static void set_blaster_environment(int port, int irq, int dma, int mpu401, int dspver)
{
    char sb[]   = "SET BLASTER=A%x I%d D%d T%d";
    char sb16[] = "SET BLASTER=A%x I%d D%d H%d T%d P%x";
    char a0[512];
    int type = 0;

    if( (dspver >> 8) >= 4 ){
        sprintf(a0, sb16, port, irq, dma, dma, 6, mpu401);
    }else{
        if( (dspver >> 8) == 3 ){
            type = 4;
        }else if( (dspver >> 8) == 2 ){
            type = 3;
        }else if( (dspver >> 8) == 1 ){
            type = 1;
        }
        sprintf(a0, sb, port, irq, dma, type);
    }
    message("\n%s\n", &a0[4]);
    _INT2E(a0);

    if(get_create_batchfile_flag() != 0){
        create_set_env_file(a0);
    }
}



int main(int argc, char* argv[])
{
    PCI_DEV pci;
    SB_MIXER mixer;
    SB_RESOURCE resource;
    int als4000_prev_irq;
    int io_timing;
    int dma_method;
    void (*ddma_setting_function)(WORD, int);

    printf("ALS4000 DOS Setup Utility  2002.9.14  by saka <als4k@anet.ne.jp> \n\n");

    /*  get resource & volume setting from als4000.cfg  */
    allegro_init();
    set_config_file("als4kdos.cfg");
    get_sb_resource(&resource);
    get_mixer_level(&mixer);

    if(check_pci_bios() == FALSE){
        printf("No PCI BIOS found\n");
        return FALSE;
    }

    /*  look for ALS4000 sound chip  */
    pci.vender_id = 0x4005;         // Avance
    pci.device_id = 0x4000;         // ALS4000
    pci.sub_vender_id = PCI_ANY_ID;
    pci.sub_device_id = PCI_ANY_ID;
    if(find_pci_device(&pci) == FALSE){
        printf("No ALS4000 found\n");
        return FALSE;
    }
    printf("ALS4000 found, ");

    /*  I/O access & bus master enable  */
    enable_pci_io_access(&pci);
    enable_pci_bus_master(&pci);

    /*  get als4000's current irq  */
    als4000_prev_irq = get_als4000_irq(&pci);
    /*  if irq is specified,  */
    if(resource.irq > 0){
        /*  set specified irq to ALS4000  */
        set_pci_hardware_interrupt(&pci, 0x0A, resource.irq);
    }
    /*  get als4000's irq number  */
    resource.irq = get_als4000_irq(&pci);

    /*  set iobase to g_als4000_iobase and g_sb_port for following functions  */
    /*  g_als4000_iobase and g_sb_port are global variables  */
    g_als4000_iobase = get_als4000_io_address(&pci);
    g_sb_port        = g_als4000_iobase + 0x10;
    if(resource.irq == als4000_prev_irq){
        printf("BASE: %04Xh, IRQ: %d\n", g_als4000_iobase, resource.irq);
    }else{
        printf("BASE: %04Xh, IRQ: %d -> %d\n", g_als4000_iobase,
                                               als4000_prev_irq, resource.irq);
    }

    /*  compare chipsets list with current one  */
    find_known_device(&io_timing, &dma_method, &ddma_setting_function);
    if(resource.dma_method == DMA_AUTO){
        /*  use default dma emulation method  */
        resource.dma_method = dma_method;
    }
    if(resource.io_timing == IO_AUTO){
        /*  use default io timing  */
        resource.io_timing = io_timing;
    }

    /*  enable SB, MPU401, AdLib, GAMEPORT  */
    enable_legacy_function(resource.sb_base, resource.mpu_base,
                           resource.adlib_base, resource.game_base);
    set_irq_and_dma_to_mixer_reg(resource.irq, resource.dma);
    set_dma_emulation_method(resource.dma_method, resource.dma,
                             resource.ddma_base,  resource.io_timing,
                             ddma_setting_function);

    /*  enable ALS4000's INTA#  */
    als4000_irq_enable(resource.irq, resource.irq_trigger_mode);
    als4000_init_misc();
    /*  set sound blaster dsp version number  */
    set_sb_dsp_version(resource.sb_ver);
    /*  set volume  */
    set_mixer_level(mixer);

    /*  reset sound blaster dsp  */
    clear_sb_mpu_irq();
    if(sb_reset_dsp(1) != 0){
        message("### SB reset failed ###\n");
    }

    /*  print current settings  */
    print_mixer_level(&mixer);
    print_sb_resource(&resource);

    /*  set BLASTER environment  */
    set_blaster_environment(resource.sb_base, resource.irq, resource.dma,
                            resource.mpu_base, resource.sb_ver);
    return 0;
}
