//***************************************************************************
//
// Copyright (c) 1992-93 Sierra Semiconductor Corp.
//
// FILE:    echomidi.c
//
// LANGUAGE:
//          Microsoft C/C++ Version 7.0
//
// DESCRIPTION:
//
//          Accepts MIDI input from external keyboard and plays the 
//          Aria Synthesizer
//
//          Syntax:  ECHOMIDI <bank> /P
//
//              where <bank> is an Aria patch bank file.
//              /P optionally turns on printing of MIDI to the screen
//
//              To send a patch change - hit 'P'
//              To exit the program    - hit <ESC>
//
//          Compile:  cl /c /AL /Zp echomidi.c
//          Link:     link echomidi,,,aria_l;
//
// Aria and Aria Synthesizer are trademarks of Sierra Semiconductor Corp.
//
//***************************************************************************

// $Header:   F:\projects\ariai\dos\archives\echomidi.c_v   2.1   03 Sep 1993 10:15:58   golds  $
// $Log:   F:\projects\ariai\dos\archives\echomidi.c_v  $
// 
//    Rev 2.1   03 Sep 1993 10:15:58   golds
// Recompiled with API library version 2.2
// 
//    Rev 2.1   13 Aug 1993 09:14:36   golds
// Various optimizations, linked with Version 2.03 library
// 
//    Rev 2.0   24 Jun 1993 12:51:04   golds
// Initial revision.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#ifdef TURBOC
#include <alloc.h>
#else
#include <malloc.h>
#endif
#include <dos.h>
#include <io.h>

#include "aria.h"
#include "ariamidi.h"

// Macros to return low or high word of long value

#define LOWORD(l) ((WORD)(l))
#define HIWORD(l) ((WORD)(((DWORD)(l) >> 16) & 0xFFFF))

// Macros to return low or high byte of word value

#define LOBYTE(w) ((BYTE)(w))
#define HIBYTE(w) ((BYTE)(((WORD)(w) >> 8) & 0xFF))

#define TRIES      0xFF
#define READSIZE   16380
#define BUFFERSIZE 128
#define BUFFERMASK 0x7f

#define CTRLCINT     0x23
#define CTRLBREAKINT 0x1b

FILE   *pbank = NULL;
char   patchfile[60];         // Patch file name
short  MIDIptr1 = 0;
short  MIDIptr2 = 0;
BYTE   vel;
BYTE   MIDIbuf[BUFFERSIZE];
HPBYTE patchbuffer=NULL;
BYTE   channel = 1;
WORD   printMIDI = 0;
WORD   done;
ARIACAPS AriaCap;             // Aria capabilities structure

VOID  reset (VOID);
VOID  parseMIDI (VOID);
BOOL  getMIDI (PBYTE);
BOOL  putcmd (BYTE);
VOID  setMIDI (BYTE, BYTE, BYTE);
VOID  interrupt FAR breakHandler (VOID);
VOID  interrupt FAR MIDIintr (VOID);
VOID  fillMIDIbuffer (DWORD);
VOID  playMIDI (VOID);
VOID  errexit (short);

extern WORD midibase;
extern WORD midiint;

VOID (interrupt FAR *oldMIDIint)() = 0L;
VOID (interrupt FAR *oldCtrlC)();
VOID (interrupt FAR *oldCtrlBreak)();


main (short argc, LPSTR argv[])
   {
   BYTE  i;
   LONG  psize, j;
   short pchange;
   char  FAR *strptr;

   printf ("\nECHOMIDI - Aria MIDI Port Player (Version 2.2)\n");
   printf ("Copyright (c) 1992-93 Sierra Semiconductor Corp.\n\n");

   // Look for Aria board
   if (SystemInit (ARIA_SYNTH))
      {
      printf ("\nERROR - Aria board not found\n");
      exit (1);
      }
   GetAriaCaps ((LPBYTE) &AriaCap);
   SetPlaybackMode (MAXOPERATORS);

   for (i = 1; i < (BYTE) argc; ++i)
      {
      if (strcmp (argv[i], "/P") == 0 || strcmp (argv[i], "/p") == 0)
         printMIDI = 1;
      else if (strcmp (argv[i], "/?") == 0)
         errexit (-4);
      else if (pbank == NULL) // Open patch bank file
         {
         if ((pbank = fopen (argv[i], "rb")) == NULL)
            errexit (-1);
         }
      }
   if (pbank == NULL)
      {
      // Look for patch bank path environment variable (ARIABANK)
      patchfile[0] = '\0';
      if ((strptr = getenv ("ARIABANK")) != NULL)
         {
         strcpy (patchfile, strptr);
         if (patchfile[strlen(patchfile)-1] != '\\')
            strcat (patchfile, "\\");
         }

      if (AriaCap.ROMsize == 512)
         strcat (patchfile, "genmidi1.bnk");
      else if (AriaCap.ROMsize == 1024)
         strcat (patchfile, "genmidi2.bnk");
      else
         errexit (-3);

      if ((pbank = fopen (patchfile, "rb")) == NULL)
         errexit (-3);
      }

   psize = FILELENGTH (fileno (pbank));

   // Allocate buffer and load patch bank
   if ((patchbuffer = halloc ((psize>>1)+1L, 2)) == NULL)
      errexit (-2);
   j = 0L;
   while (fread ((VOID HUGE *) (patchbuffer+j), 1, READSIZE, pbank) == READSIZE)
      j += (LONG) READSIZE;
   fclose (pbank);

   InitPatchBank ((struct bankstruct FAR *) patchbuffer, MAXOPERATORS);

   // Reset MIDI buffer
   putcmd (MRESET);
   putcmd (MUART);

   // Install the break handlers
   oldCtrlBreak = _dos_getvect (CTRLCINT);
   oldCtrlC     = _dos_getvect (CTRLBREAKINT);
   DISABLE
   _dos_setvect (CTRLCINT, breakHandler);
   _dos_setvect (CTRLBREAKINT, breakHandler);
   ENABLE

   // Install MIDI interrupt handler
   if (midiint > 7)
      oldMIDIint = _dos_getvect (midiint + 0x68);
   else
      oldMIDIint = _dos_getvect (midiint + 8);
   DISABLE
   if (midiint > 7)
      _dos_setvect (midiint + 0x68, MIDIintr);
   else
      _dos_setvect (midiint + 8, MIDIintr);
   ENABLE

   // Enable the MIDI interrupt
   if (midiint > 7)
      OUTP (INTCTRLR1, INP (INTCTRLR1) & ~(1 << (midiint - 8)));
   else
      OUTP (INTCTRLR0, INP (INTCTRLR0) & ~(1 << midiint));

   printf ("Hit <Esc> to quit\n");
   printf ("\nMIDI port ready to receive...\n\n");

   done = FALSE;
   while (!done)
      {
      if (kbhit())
         {
         i = getch();
         if (i == 27)
            break;
         else if (i == 'P' || i == 'p')
            {
            while (TRUE)
               {
               printf ("\n\nEnter program change (0 - 127): ");
               scanf ("%d", &pchange);
               if (pchange >= 0 && pchange <= 127)
                  break;
               }
            MIDImessage ((BYTE) (PCHANGE | channel), (BYTE) pchange, 0);
            }
         }
      while (MIDIptr1 != MIDIptr2)
         playMIDI ();
      }

   if (!done)
      reset ();

   printf ("\nAll done\n");
   exit (0);
   }


VOID reset (VOID)
   {
   putcmd (MRESET);
   SystemInit (DEFAULT_SYNTH);
   hfree (patchbuffer);

   // Restore original interrupts

   DISABLE
   if (midiint > 7)
      _dos_setvect (midiint + 0x68, oldMIDIint);
   else
      _dos_setvect (midiint + 8, oldMIDIint);
   _dos_setvect (CTRLCINT, oldCtrlC);
   _dos_setvect (CTRLBREAKINT, oldCtrlBreak);
   ENABLE

   done = TRUE;
   }


BOOL getMIDI (PBYTE val)
   {
   static WORD i;
   static BYTE dat;

   i = TRIES;
   while (--i)
      {
      dat = (BYTE) INP (MIDISTAT);
      if (!(dat & DSR))
         break;
      }
   if (!i)
      return FALSE;
   dat = (BYTE) INP (MIDIDATA);
   *val = dat;
   return TRUE;
   }


BOOL putcmd (BYTE cmd)
   {
   WORD i;
   BYTE dat;

   DISABLE
   OUTP (MIDICMD, cmd);
   i = TRIES;
   while (--i)
      {
      dat = (BYTE) INP (MIDISTAT);
      if (!(dat & DSR))
         break;
      }
   if (!i)
      {
      ENABLE
      return FALSE;
      }
   dat = (BYTE) INP (MIDIDATA);
   if (dat != MACK)
      {
      ENABLE
      return FALSE;
      }
   ENABLE

   return TRUE;
   }


VOID interrupt FAR breakHandler (VOID)
   {
   reset ();
   }


VOID interrupt FAR MIDIintr (VOID)
   {
   static BYTE  dat;
   static WORD  count = 0;
   static WORD  sysex = FALSE;
   static DWORD message = 0L;
   static WORD  runningcnt = 0;
   static BYTE  nibble = 0;
   static WORD  pos = 0;

   while (TRUE)
      {
      if (getMIDI ((PBYTE) &dat))
         {
         if (dat >= TIMING)
            break;
         else if ((dat & 0x80) == 0)
            {
            if (sysex || message == 0L ||
                (message & 0x000000F0) == 0x000000F0)
               break;

            if (count == 0)
               {
               message &= 0x000000FF;
               message |= ((DWORD) dat << 8);
               pos = 2;
               count = runningcnt - (BYTE) 1;
               if (count == 0)
                  {
                  fillMIDIbuffer (message);
                  pos = 0;
                  }
               break;
               }

            // else just a normal data byte

            message |= ((DWORD) dat) << ((pos == 1) ? 8 : 16);
            count--;
            pos++;
            if (count == 0)
               {
               fillMIDIbuffer (message);
               pos = 0;
               }
            break;
            }

         else if ((dat & 0x80) != 0)      // Status byte
            {
            if (!sysex && (dat == EOX))   // Unexpected end-of-sysex
               {
               message |= ((DWORD) dat) << ((pos == 0) ? 0 :
                                                   ((pos == 1) ? 8 : 16));
               fillMIDIbuffer (message);
               pos = 0;
               break;
               }

            if (dat == SYSEX)    // Beginning of sysex
               {
               sysex = TRUE;
               break;
               }

            if (dat == EOX)      // End of sysex
               {
               sysex = FALSE;
               break;
               }

            if (sysex && (dat != EOX))
               {
               // Unexpected status byte during sysex
               sysex = FALSE;

               // Then drop through, starting new message
               }
            else if (count)
               {
               // If expecting a data byte
               fillMIDIbuffer (message);

               // Then drop through, starting new message
               }

            // else we're starting a new message

            message = (DWORD) dat;
            pos = 1;
            nibble = dat >> 4;

            // No data bytes needed

            if (dat == 0xF4 || dat == 0xF5 || dat == TUNEREQ)
               {
               count = 0;
               break;
               }

            if (dat == TIMECODE || dat == SELECT || nibble == 0x0C ||
                nibble == 0x0D)
               count = runningcnt = 1;    // One data byte needed
            else
               count = runningcnt = 2;    // Two data bytes needed
            }
         }
      break;
      }

   // End Of Interrupt
   if (midiint > 7)
      OUTP (0xA0, 0x20);
   OUTP (0x20, 0x20);
   }


VOID fillMIDIbuffer (DWORD msg)
   {
   MIDIbuf[MIDIptr1++] = LOBYTE (LOWORD (msg));
   MIDIptr1 &= BUFFERMASK;
   MIDIbuf[MIDIptr1++] = HIBYTE (LOWORD (msg));
   MIDIptr1 &= BUFFERMASK;
   MIDIbuf[MIDIptr1++] = LOBYTE (HIWORD (msg));
   MIDIptr1 &= BUFFERMASK;
   }


VOID playMIDI (VOID)
   {
   BYTE cmd;
   BYTE dat1;
   BYTE dat2;
   static WORD cnt=0;

   cmd = MIDIbuf[MIDIptr2++];
   MIDIptr2 &= BUFFERMASK;
   dat1 = MIDIbuf[MIDIptr2++];
   MIDIptr2 &= BUFFERMASK;
   dat2 = MIDIbuf[MIDIptr2++];
   MIDIptr2 &= BUFFERMASK;

   // Process the MIDI event
   MIDImessage (cmd, dat1, dat2);
   channel = cmd & CHMASK;

   if (printMIDI)
      {
      printf (" %02X %02X", cmd, dat1);
      cnt += 2;
      if ((cmd >= 0x80 && cmd <= 0xbf) ||
          (cmd >= 0xe0 && cmd <= 0xef))
         {
         printf (" %02X", dat2);
         ++cnt;
         }
      if (cnt > 20)
         {
         printf ("\n");
         cnt = 0;
         }
      }
   }


VOID errexit (short err)
   {
   switch (err)
      {
      case -1:
         printf ("\nERROR - Invalid patch bank file specified\n");
         break;

      case -2:
         printf ("\nERROR - Not enough memory\n");
         break;

      case -3:
         printf ("\nERROR - Could not open patch bank file\n");
         break;

      case -4:
         printf ("\nUsage:  ECHOMIDI <pbank> /P\n\n");
         printf (  "        <pbank> = Aria patch bank file\n");
         printf (  "        /P turns on printing of MIDI data to the screen\n");
         break;
      }
   SystemInit (DEFAULT_SYNTH);
   exit (1);
   }


