// Track.cpp : implementation of the CTrack class
//

#include "stdafx.h"

#include <math.h>

#include "TrackEd.h"

#include "TrackFileHeader.h"
#include "Track.h"
#include "TrackSeg.h"
#include "TrackHeader.h"
#include "BestLineSeg.h"
#include "Command.h"
#include "CCSetup.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/* calculate the number PI when it is needed (named like Borland constant value) */
#define M_PI (acos(-1))

// Define some constant values
// Minimum change of angle
double dANGLESCALE = 2*M_PI/65536.0; // 2 * Pi / 65k
// Minimum change of radius
double dRADSCALE = 0.0381; // 0.0381 -> 128 * radscale = 1 tlu * 4.8768
// Width of car (defined by F1GP)
double dCARWIDTH = 2.4384;
// Conversion factor from width units in meters (defined by F1GP)
double dWIDTHSCALE=0.0047625;
// Pit lane width in meters (defined by F1GP)
double dPITWIDTH=6.4008;

/*generalized arc tangent*/
static double arctan2(double sinf, double cosf)
{double f;

 if( fabs(sinf) < fabs(cosf) )
   {/*f=asin(fabs(sinf));*/f=atan(fabs(sinf/cosf));}
 else{/*f=acos(fabs(cosf));*/f=0.5*M_PI-atan(fabs(cosf/sinf));}

 if(sinf<0.0)
   {if(cosf<0.0){f=M_PI+f;}
     else{f=2*M_PI-f;}
   }
 else
   {if(cosf<0.0){f=M_PI-f;}
   }
 return f;
}

/*signum function*/
short int sgn(double x)
{if(x>0.0){return 1;}
 if(x==0.0){return 0;}
 return -1;
}

/*round*/
int f2i(double f)
{int a;
 if(f<0)f-=1.0;
 a=(int)(f+0.5);
 return(a);
}

/////////////////////////////////////////////////////////////////////////////
// CTrack

IMPLEMENT_DYNCREATE(CTrack, CDocument)

BEGIN_MESSAGE_MAP(CTrack, CDocument)
	//{{AFX_MSG_MAP(CTrack)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTrack construction/destruction

CTrack::CTrack()
{
	m_uCCtlu = 0;
	m_dWidth = 0.0;
	m_lPitgrad = 0;
	m_lPithei = 0;
	m_pLastBlSeg = NULL;

	// pointers to lists of track/pit lane/cc line segments
	m_pTrackSegs = NULL;
	m_pPitSegs = NULL;
	m_pBestLineSegs = NULL;

	// pointers to currently edited track parts
	m_pTrackSegCurrent = NULL;
	m_pPitSegCurrent = NULL;
	m_pBestLineSegCurrent = NULL;

	// background picture
	memset( &m_baBackground, 0, sizeof( m_baBackground ) );

	// data section at end of file
	memset( &m_baEndheader, 0, sizeof( m_baEndheader ) );

	// Index into endheader that indicates position of "number of laps"
	m_nLapNumIndex = 0;

	// current edit mode
	m_eEditMode = emTrack;

	// Calculations for CCLine mode
	m_fCCLineOld = true; // default: use calculations of DOS version
}

CTrack::~CTrack()
{
	FreeMemory();
}

BOOL CTrack::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
// CTrack serialization

void CTrack::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
	}
	else
	{
	}
}

/////////////////////////////////////////////////////////////////////////////
// CTrack diagnostics

#ifdef _DEBUG
void CTrack::AssertValid() const
{
	CDocument::AssertValid();
}

void CTrack::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CTrack commands

BOOL CTrack::OnOpenDocument(LPCTSTR lpszPathName) 
{
	if (!CDocument::OnOpenDocument(lpszPathName))
		return FALSE;
	
	LoadTrack( lpszPathName );
	
	return TRUE;
}

/**********************/
/*load track file data*/
/**********************/
void CTrack::LoadTrack(LPCTSTR lpszPathName)
{
 FILE *trf;
 double widthlen;
 short int si;
 byte pitstart;

 trf=fopen(lpszPathName,"rb");
 if(trf==NULL)
   {
	 AfxMessageBox( "Unable to load track " + CString( lpszPathName ) );
     return;
   }

 /* free previously allocated memory */
 FreeMemory();

 pitstart=0;

 /*reading background picture*/
 fread(&m_baBackground,1,4096,trf);

 /*reading track header*/
 m_TrackFileHeader.Load( trf );

 // lFilePos at this point should match header.base
 long lFilePos = ftell( trf );

 /*reading object datas*/
 // This data portion seems to be structured into something else
 // and the definition of internal objects.
 // Until we know what the first part is, we will treat everything
 // as obj data that is loaded from the file and written back without change.
 m_Objects.Load( trf, m_TrackFileHeader.m_nTrackDataOffset );

 // Filepos should match now m_nSizeBase + m_nTrackDataOffset
 lFilePos = ftell( trf );
 ASSERT( lFilePos == (m_TrackFileHeader.m_nSizeBase + m_TrackFileHeader.m_nTrackDataOffset) );

 /*reading track data header*/
 LoadTrackDataHeader( trf );
 widthlen=0.0;

 lFilePos = ftell( trf ); // for information purpose

 /*reading track segment data*/
 LoadTrackSegments( trf );

 lFilePos = ftell( trf ); // for information purpose

 /*reading CC line data*/
 LoadCCLine( trf );

 lFilePos = ftell( trf ); // for information purpose

 /*reading CC setup data*/
 LoadCCSetup( trf );

 lFilePos = ftell( trf ); // for information purpose

 /*reading pitlane segment data*/
 LoadPitlaneSegments( trf );

 lFilePos = ftell( trf ); // for information purpose

 /*reading data from the end of file*/
 si=0;
 while(fread(m_baEndheader+si,1,1,trf)==1)
   {si++;
   }
 m_nLapNumIndex=si-10; /* index of last byte = lapnumindex+9 */
 fclose(trf);
 
 RefreshCCLine();
}

unsigned int CTrack::GetCCLineLength()
{
	return m_uCCtlu;
}

/**
  Reads track segments from track file.
  Stores track data into track object.
*/

void CTrack::LoadTrackSegments( FILE * pTrackFile )
{
	byte b, trackdata, pitstart;
	CCommand *kcmd, *acmd, *fcmd;
	unsigned short int ui;
	double widthlen, newwidth;
	CTrackSeg *ftr,*atr;
	short int si;
	double f;

	b = 0;
	trackdata = 1;
	pitstart = 0;
	kcmd = NULL;
	acmd = NULL;
	fcmd = NULL;

	m_pTrackSegs = NULL;
	m_dPitStartAngle = 0.0;
	m_dPitStartX = 0.0;
	m_dPitStartY = 0.0;
	m_lPitgrad = 0;
	m_lPithei = 0;
	ftr = NULL;
	atr = NULL;
	ui = 0;
	widthlen = 0.0;
	newwidth = 0.0;
	si = 0;
	f = 0.0;

	while(trackdata)
	{
	fread(&ui,2,1,pTrackFile);
    if( (ui/256!=0) && (ui/256!=0xFF) )
    {
		// read command and store it into command list
		acmd = AddCommand(kcmd, fcmd);

		acmd->cmd=ui/256;
		acmd->a0=ui%256;
		switch(ui/256)
		{
		case 0x80:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x81:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x82:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x83:{
			break;}
		case 0x84:{
			break;}
		case 0x85:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			widthlen=4.8768*((double)ui);
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			newwidth = 2 * dWIDTHSCALE * ((double)ui);
			break;}
		case 0x86:{
			pitstart=1;
			break;}
		case 0x87:{
			break;}
		case 0x88:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x89:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x8A:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a3=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a4=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a5=ui;
			break;}
		case 0x8B:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a3=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a4=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a5=ui;
			break;}
		case 0x8C:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x8D:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x8E:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0x8F:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0x90:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x91:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x92:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x93:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x94:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x95:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x96:{
			break;}
		case 0x97:{
			break;}
		case 0x98:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x99:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0x9A:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0x9B:{
			break;}
		case 0x9C:{
			break;}
		case 0x9D:{
			break;}
		case 0x9E:{
			break;}
		case 0x9F:{
			break;}
		case 0xA0:{
			break;}
		case 0xA1:{
			break;}
		case 0xA2:{
			break;}
		case 0xA3:{
			break;}
		case 0xA4:{
			break;}
		case 0xA5:{
			break;}
		case 0xA6:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0xA7:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0xA8:{
			break;}
		case 0xA9:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			break;}
		case 0xAA:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a3=ui;
			break;}
		case 0xAB:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			break;}
		case 0xAC:{
			fread(&ui,2,1,pTrackFile);
			acmd->a1=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a2=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a3=ui;
			fread(&ui,2,1,pTrackFile);
			acmd->a4=ui;
			break;}
		}/*end of switch(ui/256)*/
    }
	else if ( ui / 256 == 0xFF )
	{
		// Last segment of list.
		// Add collected commands to the track segment.
		// This seems to be a dummy segment that is only there
		// to carry the commands (no length, radius etc).
		atr = new CTrackSeg;
		ftr->m_pNext=atr; atr->m_pPrev=ftr; atr->m_pNext=NULL;
		ftr=atr;
		atr->sign=4;
		atr->cmd = kcmd;
		kcmd = NULL;
		trackdata = 0; // set flag "track end reached"
	}
	else // ui / 256 == 0
	{
		// normal track segment
		atr = new CTrackSeg;
		if(m_pTrackSegs==NULL)
		{
			// first segemnt
			m_pTrackSegs=atr;
			ftr=atr;
			atr->m_pPrev = NULL;
			atr->m_pNext = NULL;
			atr->fik = - dANGLESCALE*((double) m_TrackHeader.startangle);
			atr->kx = 0;
			atr->ky = 0;

			// coordinates for profile
			atr->m_dS_Start = 0.0;
			atr->m_dH_Start = 0.0;
		}
		else
		{
			// append to end of list and use end data of previous segment
			// as own starting data (angle, coordinates)
			ftr->m_pNext = atr;
			atr->m_pPrev = ftr;
			atr->m_pNext = NULL;
			atr->fik = ftr->fiv; // starting angle
			atr->kx = ftr->vx; atr->ky = ftr->vy; // start coordinates

			// coordinates for profile
			atr->m_dS_Start = ftr->m_dS_End;
			atr->m_dH_Start = ftr->m_dH_End;

			ftr = atr;
		}
		// add command data to track segment
		atr->cmd = kcmd;
		kcmd = NULL;
		// fill in more date from track file
		atr->tlu	= ui;
		atr->length = 4.8768*((double)ui); // Umgerechnet in Meter
		fread(&si,2,1,pTrackFile);
		atr->curvature = si;
		fread(&si,2,1,pTrackFile);
		atr->height = si;
		fread(&ui,2,1,pTrackFile);
		atr->flags = ui;
		fread(&b,1,1,pTrackFile);
		atr->rbank = b;
		fread(&b,1,1,pTrackFile);
		atr->lbank = b;
		TRACE( "TrackSeg %d, %d, %d, %X, %d, %d\n",
			   atr->tlu, atr->curvature, atr->height, atr->flags, atr->rbank, atr->lbank );

		// calculate profile data
		double dSlope = 0.0; // durchschnittliche Steigung
		if ( atr->height == 0 )
		{
			// up/down/level straight
			atr->m_dR_Profile = 0.0;
			atr->m_dS_Center = 0.0;
			atr->m_dH_Center = 0.0;
			f = atr->cumgrad; // start = end angle
			dSlope = f;
			atr->m_dH_End = atr->m_dH_Start + sin( f ) * atr->length;
			atr->m_dS_End = atr->m_dS_Start + cos( f ) * atr->length;
		}
		else
		{
			// segment with variable direction
			if ( atr->m_pPrev )
				f = atr->m_pPrev->cumgrad;
			else
				f = m_TrackHeader.startheight * dANGLESCALE;
			dSlope = ( f + atr->cumgrad ) / 2.0;
			atr->m_dR_Profile = atr->length / (atr->height * dANGLESCALE * atr->tlu);
			atr->m_dS_Center = atr->m_dS_Start - sin( f ) * atr->m_dR_Profile;
			atr->m_dH_Center = atr->m_dH_Start + cos( f ) * atr->m_dR_Profile;
			atr->m_dS_End = atr->m_dS_Center + sin( atr->cumgrad ) * atr->m_dR_Profile;
			atr->m_dH_End = atr->m_dH_Center - cos( atr->cumgrad ) * atr->m_dR_Profile;
		}

		// calculate change of angle
		// curvature = number of "anglescale" angle changes per tlu
		f = -dANGLESCALE * ((double)atr->tlu) * ((double)atr->curvature);
		atr->fiv = atr->fik + f;
		atr->sign=sgn(f);
		if(atr->sign)
		{
			// curve
			// calculate radius in meters
			atr->r=atr->length/(atr->sign*f);
			// calculate coordinates of center of circle
			atr->ox = atr->kx - atr->sign*atr->r*cos(atr->fik);
			atr->oy = atr->ky - atr->sign*atr->r*sin(atr->fik);
			// calculate end point
			atr->vx = atr->ox + atr->sign*atr->r*cos(atr->fiv);
			atr->vy = atr->oy + atr->sign*atr->r*sin(atr->fiv);
		}
		else
		{
			// straight: no radius, no center of circle
			atr->r=0;
			atr->ox=0;
			atr->oy=0;
			// calculate end point by adding length to start point
			atr->vx = atr->kx - atr->length * sin(atr->fik);
			atr->vy = atr->ky + atr->length * cos(atr->fik);
		}
		// set track width left and right from the middle
		atr->ks[0]=m_dWidth*0.5; atr->ks[1]=atr->ks[0];
		// has something to do with changes of track width
		// over more than one segment (?)
		if(widthlen>0.01)
		{
			if(widthlen>atr->length)
			{
				m_dWidth+=atr->length*(newwidth-m_dWidth)/widthlen;
				atr->middle=0;
			}
			else
			{
				if(fabs(widthlen-atr->length)<0.0001)
				{
					m_dWidth=newwidth; atr->middle=0;
				}
				else
				{
					atr->middle=1;
					if(atr->sign)
					{
						f=atr->fik-dANGLESCALE*(widthlen/4.8768)*((double)atr->curvature);
						atr->fim=f;
						atr->mx=atr->ox+atr->sign*atr->r*cos(f);
						atr->my=atr->oy+atr->sign*atr->r*sin(f);
					}
					else
					{
						f=widthlen;
						atr->fim=f;
						atr->mx=atr->kx-f*sin(atr->fik);
						atr->my=atr->ky+f*cos(atr->fik);
					}
					m_dWidth=newwidth;
				}
			}
			widthlen-=atr->length;
		}
		// set track width at end of segment, related to mid of track
		atr->vs[0]=m_dWidth*0.5; atr->vs[1]=atr->vs[0];

		if(atr->m_pPrev)
		{
			// calculate the cumulated data (length, height etc).
			atr->cumlength = atr->m_pPrev->cumlength + atr->length;
			atr->cumgrad = atr->m_pPrev->cumgrad
				           + ((long int)atr->tlu) * dANGLESCALE * ((long int)atr->height);
			atr->cumheight = atr->m_pPrev->cumheight
							 + atr->tlu * ( sin( atr->m_pPrev->cumgrad ) + 1 - cos( atr->cumgrad ) );
//							 + atr->tlu * (atr->m_pPrev->cumgrad + atr->cumgrad) / 2;
		}
		else
		{
			// this is the first segment
			atr->cumlength = atr->length;
			atr->cumgrad = (long int)m_TrackHeader.startheight * dANGLESCALE
						   + ((long int)atr->tlu) * dANGLESCALE * ((long int)atr->height);
			atr->cumheight = atr->tlu * ( sin(m_TrackHeader.startheight * dANGLESCALE) + 1 - cos( atr->cumgrad) );
		}
		TRACE( "cumgrad = %.4f, cumheight (tlu) = %.4f, cumheigth (m) = %.4f\n", atr->cumgrad, atr->cumheight, atr->cumheight * 4.8768 );

		if(pitstart)
		{
			if(m_TrackHeader.pitside)
			{
				/*pits on the left*/
				m_dPitStartX=atr->kx-cos(atr->fik)*(atr->ks[0]-dPITWIDTH*0.5);
				m_dPitStartY=atr->ky-sin(atr->fik)*(atr->ks[0]-dPITWIDTH*0.5);
			}
			else
			{
				/*pits on the right*/
				m_dPitStartX=atr->kx+cos(atr->fik)*(atr->ks[0]-dPITWIDTH*0.5);
				m_dPitStartY=atr->ky+sin(atr->fik)*(atr->ks[0]-dPITWIDTH*0.5);
			}
			m_dPitStartAngle=atr->fik;
			m_lPitgrad=(long)atr->m_pPrev->cumgrad;
			m_lPithei=(long)atr->m_pPrev->cumheight;
			pitstart=0;
		}
	}
   }/*end of while(trackdata)*/

   // set edit point to first segment
   m_pTrackSegCurrent = m_pTrackSegs;
}

/**
  Reads computer car line from track file.
*/

void CTrack::LoadCCLine(FILE *pTrackFile)
{
	CBestLineSeg *abl;
	byte trackdata;
	unsigned short int ui;
	short int si, cumCCtlu;
	double f;
	int endtlof;
	long li;

	endtlof=0;
	cumCCtlu=0;

	trackdata=1;
	m_pBestLineSegs=NULL; // list of CCline segments
	while(trackdata)
	{
		fread(&ui,2,1,pTrackFile);
		if(ui%256==0)
		{
			// reached end of list
			trackdata=0; break;
		}

		abl = new CBestLineSeg;

		// Append to list of CCline segments.
		// Also fill in starting angle and coordinates.
		AddToCCLine( abl );

		abl->tlu=ui%256;	// Length in Track Length Units
		cumCCtlu+=abl->tlu;	// cumulated length
		abl->length=4.8768*((double)(abl->tlu)); // segment length in meters
		abl->s=0.0;
		abl->type=ui/256;	// Typinfo
		switch(abl->type)
		{
		case 0x80:
			{
				// First segment.
				// Position calculated in relation to track start
				fread(&si,2,1,pTrackFile);
				abl->a1=si;
				f = dWIDTHSCALE * ((double)si);
				abl->kx+=f*cos(m_pTrackSegs->fik);
				abl->ky+=f*sin(m_pTrackSegs->fik);

				fread(&si,2,1,pTrackFile);
				abl->a2=si;

				fread(&si,2,1,pTrackFile);
				abl->a3=si;
				if(abl->a3==0)
				{
					// straight
					f=0.0;
					abl->r=0.0;
					abl->sign=0;
					abl->fiv=abl->fik;
					endtlof+=abs(abl->a2);
				}
				else
				{
					// curve
					//f=-1.0*((double)abl->tlu/*-((float)abl->a2)/128.0*/)/((float)abl->a3);
					//f*=360.0/3.14;
					abl->sign=-sgn(abl->a3);					// radius
					abl->r=((double)abs(abl->a3)) * dRADSCALE;
					f=abl->length/abl->r;
					abl->fiv=abl->fik+f*abl->sign;
					//abl->r+=((float)abl->tlu)*((float)abl->a2)/((float)abl->a3);
					//if(abl->a2){abl->pre=newbestlineseg();}
				}
				break;
			}

		case 0x00:	// normal CCline segment
			{
				// correction!?!
				const double CORRECTION = 128.0;
				fread(&si,2,1,pTrackFile);
				abl->a1=si;
				// radius
				fread(&si,2,1,pTrackFile);
				abl->a2=si;
				if(abl->a2==0)
				{
					// straight
					// fraction of tlu correction value
					abl->length -= (((double)abl->a1)/CORRECTION) * 4.8768;

					f=0.0;
					abl->r=0.0;
					abl->sign=0;
					abl->fiv=abl->fik; // starting angle = ending angle
					endtlof+=abs(abl->a1);
				}
				else
				{
					// curve
					//f=-1.0*((float)abl->tlu/*-((float)abl->a1)/128.0*/)/((float)abl->a2);
					//f*=360.0/3.14;
					/*
					abl->length -= ((double)abl->a1)/128.0;
					abl->m_pPrev->length+=((double)abl->a1)/128.0;
					*/
					// fraction of tlu correction value
					abl->length -= (((double)abl->a1)/CORRECTION) * 4.8768;

					abl->sign = -sgn(abl->a2);
					abl->r = ((double)abs(abl->a2)) * dRADSCALE; // radius in meters
					f = abl->length/abl->r; // calculate change of angle
					abl->fiv = abl->fik + f*abl->sign; // add to starting angle to get ending angle
					//abl->r+=((float)abl->tlu)*((float)abl->a1)/((float)abl->a2);
					//if(abl->a1){abl->pre=newbestlineseg();}
				}
				break;
			}

		case 0x40:	// carries 32Bit radius
			{
				fread(&si,2,1,pTrackFile);
				abl->a1=si;
				li=0;
				fread((byte*)(&li)+2,2,1,pTrackFile);
				abl->a2=li/65536;
				fread(&si,2,1,pTrackFile);
				abl->a3=si;
				li+=si;
				abl->bigrad=li;
				if(li==0)
				{
					// straight
					f=0.0;
					abl->r=0.0;
					abl->sign=0;
					abl->fiv=abl->fik;
					endtlof+=abs(abl->a1);
				}
				else
				{
					// curve
					//f=-1.0*((float)abl->tlu/*-((float)abl->a1)/128.0*/)/((float)li);
					//f*=360.0/3.14;
					abl->sign=-sgn(li);
					abl->r=((double)labs(li)) * dRADSCALE;
					f=abl->length/abl->r;
					abl->fiv=abl->fik+f*abl->sign;
					//abl->r+=((float)abl->tlu)*((float)abl->a1)/((float)li);
					//if(abl->a1){abl->pre=newbestlineseg();}
				}
				break;
			}
		}/* end of switch(abl->type) */

		// calculate coordinates of end point and center of circle
		if(abl->sign)
		{
			// curve
			//abl->r=abl->length/(abl->sign*f);
			abl->ox=abl->kx-abl->sign*abl->r*cos(abl->fik);
			abl->oy=abl->ky-abl->sign*abl->r*sin(abl->fik);
			abl->vx=abl->ox+abl->sign*abl->r*cos(abl->fiv);
			abl->vy=abl->oy+abl->sign*abl->r*sin(abl->fiv);
		}
		else
		{
			// straight
			//abl->r=0;
			abl->ox=0;
			abl->oy=0;
			abl->vx=abl->kx-abl->length*sin(abl->fik);
			abl->vy=abl->ky+abl->length*cos(abl->fik);
		}
	}/* end of while(trackdata) */

	// Lnge der Ideallinie merken
	m_uCCtlu = cumCCtlu;

	// Liste austracen fr Debugging-Zwecke
	abl = m_pBestLineSegs;
	int i = 1;
	double dAngleSum = 0.0, dCorrectionSum = 0.0;
	while ( abl != NULL )
	{
		TRACE( "CCLine Segment %d\n", i );
		TRACE( "Typ: %02X  tlu=%d  a1=%d  a2=%d  a3=%d\n", abl->type, abl->tlu, abl->a1, abl->a2, abl->a3 );
		TRACE( "CCLine Segmentanfang: X=%.3f, Y=%.3f\n", abl->kx, abl->ky );
		TRACE( "CCLine Segmentende: X=%.3f, Y=%.3f\n\n", abl->vx, abl->vy );

		// Winkel der gebogenen Segmente aufsummieren
		if ( abl->a2 != 0 )
		{
			dAngleSum += (((double)abl->tlu) * 128.0) / abl->a2;
			dCorrectionSum += ((double)abl->a1) / abl->a2;
		}

		abl = abl->m_pNext;
		i++;
	}
	double dSum = (dAngleSum * 360.0) / (2 * M_PI);
	double dCSum = (dCorrectionSum * 360.0) / (2 * M_PI);
	double factor = ( 2 * M_PI - dAngleSum ) / dCorrectionSum;
	TRACE( "\nCorrectionFactor = %.4f\n", factor );

	// set edit point to first segment
	m_pBestLineSegCurrent = m_pBestLineSegs;
}

void CTrack::AddToCCLine(CBestLineSeg *pBlSeg)
{
	if(m_pBestLineSegs==NULL)
	{
		// Empty list: new segment is first and last
		m_pBestLineSegs	= pBlSeg;
		m_pLastBlSeg	= pBlSeg;
		pBlSeg->m_pPrev	= NULL;
		pBlSeg->m_pNext	= NULL;
		pBlSeg->fik		= m_pTrackSegs->fik; // starting angle just like first track segment
		pBlSeg->kx		= m_pTrackSegs->kx;  // also starting coordinates
		pBlSeg->ky		= m_pTrackSegs->ky;
	}
	else
	{
		// add to tail
		m_pLastBlSeg->m_pNext = pBlSeg;
		pBlSeg->m_pPrev	= m_pLastBlSeg;
		pBlSeg->m_pNext	= NULL;
		pBlSeg->fik		= m_pLastBlSeg->fiv; // starting angle like previous segment
		pBlSeg->kx		= m_pLastBlSeg->vx;
		pBlSeg->ky		= m_pLastBlSeg->vy; // staring coordinates like end coordinates of previous segment
		m_pLastBlSeg	= pBlSeg;
	}
}

/**
  RefreshCCLine
  Go through whole CCLine and recalculate all coordinates, angles etc.
*/

void CTrack::RefreshCCLine()
{
	if ( m_fCCLineOld )
	{
		// This is the "old" version from DOS track editor of Kristof Kaly-Kullai.
		// It uses extensive calculations and tries to relate the CCLine to the track
		// segments. It inserts additional segments to fill the gaps up to the next
		// track tlu border.
		CBestLineSeg *fbl;
		short int offset, i;
		short int cumtlu,tcumtlu;
		CTrackSeg *actual;
		double a,c, f, cosfv,sinfv, alfa[2], tanfv,ctgfv, s[2];
		double tvx,tvy, f2;

		if(m_pTrackSegs->sign==4){return;}

		fbl=m_pBestLineSegs;
		cumtlu=0;
		actual=m_pTrackSegs;
		tcumtlu=actual->tlu; // cumulated track length in track length units (tlu)
		while(fbl)
		{
			if(fbl->m_pPrev == NULL)
			{
				// first CCLine segment - should be a type 0x80 segment
				fbl->fik=m_pTrackSegs->fik;
				f = dWIDTHSCALE * ((double)fbl->a1);
				fbl->kx=m_pTrackSegs->kx+f*cos(m_pTrackSegs->fik);
				fbl->ky=m_pTrackSegs->ky+f*sin(m_pTrackSegs->fik);
				offset=fbl->a2;
			}
			else
			{
				// starting angle = ending angle of previous CCLine segment
				fbl->fik=fbl->m_pPrev->fiv;
				// segments starts at end point of previous segment
				fbl->kx=fbl->m_pPrev->vx; fbl->ky=fbl->m_pPrev->vy;
				offset=fbl->a1;
			}

			switch(fbl->type)
			{
			case 0x80: // start segment, related to track start
				{
					if(fbl->a3==0)
					{
						// straight
						fbl->r=0.0;
						fbl->sign=0;
					}
					else
					{
						// curved
						fbl->sign=-sgn(fbl->a3);
						fbl->r=((double)abs(fbl->a3))*(double)dRADSCALE;
					}
					break;
				}
			case 0x00: // normal segment
				{
					// check radius
					if(fbl->a2==0)
					{
						// straight
						fbl->r=0.0;
						fbl->sign=0;
					}
					else
					{
						// curved
						fbl->sign=-sgn(fbl->a2);
						fbl->r=((double)abs(fbl->a2))*(double)dRADSCALE;
					}
					break;
				}
			case 0x40: // big radius segment (32 Bit)
				{
					fbl->bigrad=fbl->a2;
					fbl->bigrad=fbl->bigrad<<16;
					fbl->bigrad+=(unsigned)fbl->a3;
					if(fbl->bigrad==0)
					{
						// straight - sense???
						fbl->r=0.0;
						fbl->sign=0;
					}
					else
					{
						// curved
						fbl->sign=-sgn(fbl->bigrad);
						fbl->r=((double)labs(fbl->bigrad))*(double)dRADSCALE;
					}
					break;
				}
			}/* end of switch(fbl->type) */

			/* searching the track segment, where the CC line segment ends */
			cumtlu+=fbl->tlu;
			while(tcumtlu<cumtlu)
			{
				if(actual->m_pNext)
				{
					if(actual->m_pNext->sign==4)
					{
						break;
					}
					else
					{
						actual=actual->m_pNext;
						tcumtlu+=actual->tlu;
					}
				}
				else
				{
					break;
				}
			}

			if(actual->sign)
			{
				// curved track segment
				f = actual->fiv
					- (actual->fik - actual->fiv)
					  * ((double)(cumtlu-tcumtlu))
					  / ((double)actual->tlu);
				cosfv = cos(f);
				sinfv = sin(f);
				tvx = actual->ox + actual->sign*actual->r*cosfv;
				tvy = actual->oy + actual->sign*actual->r*sinfv;
			}
			else
			{
				/*tvx=actual->vx-(actual->kx-actual->vx)*((double)(cumtlu-tcumtlu))/((double)actual->tlu);
				tvy=actual->vy-(actual->ky-actual->vy)*((double)(cumtlu-tcumtlu))/((double)actual->tlu);*/
				cosfv = cos(actual->fiv);
				sinfv = sin(actual->fiv);
				tvx = actual->kx - sinfv*4.8768*((double)(cumtlu-tcumtlu+actual->tlu));
				tvy = actual->ky + cosfv*4.8768*((double)(cumtlu-tcumtlu+actual->tlu));
			}
			/* calculating the length, fiv, vx, vy and s values */
			if(fbl->sign)
			{
				/* curved segment*/
				if(offset>0)
				{
					if(fbl->m_pAux == NULL)
					{
						fbl->m_pAux = new CBestLineSeg;
					}
					fbl->m_pAux->type=1;
					fbl->m_pAux->sign=0; fbl->m_pAux->r=0.0;
					fbl->m_pAux->ox=0.0; fbl->m_pAux->oy=0.0;
					fbl->m_pAux->kx=fbl->kx; fbl->m_pAux->ky=fbl->ky;
					fbl->m_pAux->fik=fbl->fik; fbl->m_pAux->fiv=fbl->fik;
					fbl->m_pAux->m_pAux = NULL; fbl->m_pAux->m_pPrev=0; fbl->m_pAux->m_pNext = NULL;
					fbl->m_pAux->length=((double)offset)*4.8768/256.0;
					fbl->m_pAux->s=0.0;
					fbl->m_pAux->vx = fbl->m_pAux->kx - fbl->m_pAux->length * sin(fbl->m_pAux->fik);
					fbl->m_pAux->vy = fbl->m_pAux->ky + fbl->m_pAux->length * cos(fbl->m_pAux->fik);
					fbl->kx=fbl->m_pAux->vx;
					fbl->ky=fbl->m_pAux->vy;
				}
				else
				{
					if(fbl->m_pAux)
					{
						delete fbl->m_pAux;
						fbl->m_pAux = NULL;
					} 
				}
				// calculate center of circle
				fbl->ox=fbl->kx-fbl->sign*fbl->r*cos(fbl->fik);
				fbl->oy=fbl->ky-fbl->sign*fbl->r*sin(fbl->fik);

				if( fabs(cosfv)<fabs(sinfv) )
				{
					ctgfv=cosfv/sinfv;
					a=1/sqrt(1+ctgfv*ctgfv);
					c=a*(tvx-fbl->ox+ctgfv*(fbl->oy-tvy))/(fbl->r*fbl->sign);
					f=arctan2(-a,ctgfv*a);
					if(fabs(c)>1)
					{
						fbl->s=100.0;
						fbl->fiv=fbl->fik;
						fbl->length=1.0;
						fbl->vx=fbl->kx-sin(fbl->fik);
						fbl->vy=fbl->ky+cos(fbl->fik);
						fbl->sign=4;
						fbl=fbl->m_pNext;
						while(fbl)
						{
							fbl->sign=4;
							fbl=fbl->m_pNext;
						}
						break;
					}
					else
					{
						f2=asin(c);
					}
					alfa[0]=-f2-f;
					s[0]=(fbl->oy-tvy+fbl->r*fbl->sign*sin(alfa[0]))/sinfv;
					alfa[1]=M_PI+f2-f;
					s[1]=(fbl->oy-tvy+fbl->r*fbl->sign*sin(alfa[1]))/sinfv;
				}
				else
				{
					tanfv=sinfv/cosfv;
					a=1/sqrt(1+tanfv*tanfv);
					c=a*(tvy-fbl->oy+tanfv*(fbl->ox-tvx))/(fbl->r*fbl->sign);
					f=arctan2(tanfv*a,-a);
					if(fabs(c)>1)
					{
						fbl->s=100.0;
						fbl->fiv=fbl->fik;
						fbl->length=1.0;
						fbl->vx=fbl->kx-sin(fbl->fik);
						fbl->vy=fbl->ky+cos(fbl->fik);
						fbl->sign=4;
						fbl=fbl->m_pNext;
						while(fbl)
						{
							fbl->sign=4;
							fbl=fbl->m_pNext;
						}
						break;
					}
					f2=asin(c);
					alfa[0]=-f2-f;
					s[0]=(fbl->ox-tvx+fbl->r*fbl->sign*cos(alfa[0]))/cosfv;
					alfa[1]=M_PI+f2-f;
					s[1]=(fbl->ox-tvx+fbl->r*fbl->sign*cos(alfa[1]))/cosfv;
				}
				for(i=0;i<2;i++)
				{
					while( (alfa[i]-fbl->fik)>2*M_PI )
					{
						alfa[i]-=2*M_PI;
					}
					while( (alfa[i]-fbl->fik)<=-2*M_PI )
					{
						alfa[i]+=2*M_PI;
					}
					while( fbl->sign*alfa[i] <= fbl->sign*fbl->fik )
					{
						alfa[i]+=fbl->sign*2*M_PI;
					}
				}
				if( ((fbl->ky-tvy)*cosfv-(fbl->kx-tvx)*sinfv) > 0.0 )
				{
					if( fbl->sign*alfa[0] < fbl->sign*alfa[1] )
					{
						fbl->fiv=alfa[1]; fbl->s=s[1];
					}
					else
					{
						fbl->fiv=alfa[0];
						fbl->s=s[0];
					}
				}
				else
				{
					if( fbl->sign*alfa[0] < fbl->sign*alfa[1] )
					{
						fbl->fiv=alfa[0];
						fbl->s=s[0];
					}
					else
					{
						fbl->fiv=alfa[1];
						fbl->s=s[1];
					}
				}
				fbl->length=fabs(fbl->fiv-fbl->fik)*fbl->r;
				fbl->vx=tvx+fbl->s*cosfv;
				fbl->vy=tvy+fbl->s*sinfv;
	/*          fbl->vx=fbl->ox+fbl->sign*fbl->r*cos(fbl->fiv);
				fbl->vy=fbl->oy+fbl->sign*fbl->r*sin(fbl->fiv);*/
			}
			else
			{
				/* straight segment */
				fbl->ox=0.0;
				fbl->oy=0.0;
				fbl->fik-=dANGLESCALE*((double)offset);
				if(fbl->m_pAux)
				{
					delete fbl->m_pAux;
					fbl->m_pAux = NULL;
				}
				f=sin(fbl->fik)*sinfv+cos(fbl->fik)*cosfv;
				if(fabs(f)<1e-10)
				{
					fbl->s=100.0;
					fbl->length=fbl->tlu*4.8768;
				}
				else
				{
					fbl->length=(cosfv*(tvy-fbl->ky)-sinfv*(tvx-fbl->kx))/f;
					fbl->s=(-sin(fbl->fik)*(tvy-fbl->ky)-cos(fbl->fik)*(tvx-fbl->kx))/f;
				}
				fbl->fiv=fbl->fik;
				fbl->vx=tvx+fbl->s*cosfv;
				fbl->vy=tvy+fbl->s*sinfv;
	/*          fbl->vx=fbl->kx-fbl->length*sin(fbl->fik);
				fbl->vy=fbl->ky+fbl->length*cos(fbl->fik);*/
			}

			fbl=fbl->m_pNext;
		}/* end of while(fbl) */
	} // old version 

	else
	{
		// new version: based on calculations made when loading the ccline from disk.
		CBestLineSeg *abl;
		unsigned short int ui;
		short int si, cumCCtlu;
		double f;
		int endtlof;
		long li;

		endtlof=0;
		cumCCtlu=0;

		abl = m_pBestLineSegs;
		while(abl != NULL)
		{
			cumCCtlu+=abl->tlu;	// cumulated length
			abl->length=4.8768*((double)(abl->tlu)); // segment length in meters
			abl->s=0.0;	// sense?

			// Set starting position and angle
			if ( abl->m_pPrev == NULL )
			{
				// First segment: starts like the first track segment
				abl->fik= m_pTrackSegs->fik; // starting angle just like first track segment
				abl->kx	= m_pTrackSegs->kx;  // also starting coordinates
				abl->ky	= m_pTrackSegs->ky;
			}
			else
			{
				// take end values of previous segment as starting values
				abl->fik = abl->m_pPrev->fiv;
				abl->kx  = abl->m_pPrev->vx;
				abl->ky  = abl->m_pPrev->vy;
			}

			switch(abl->type)
			{
			case 0x80:
				{
					// Segment with left/right offset value
					f = dWIDTHSCALE * ((double) abl->a1);
					// Modifiy starting pos with offset to center of track
					abl->kx += f*cos(abl->fik);
					abl->ky += f*sin(abl->fik);

					if(abl->a3==0)
					{
						// straight
						f=0.0;
						abl->r=0.0;
						abl->sign=0;
						abl->fiv=abl->fik;		// no angle change
						endtlof+=abs(abl->a2);	// accumulate 'correction' values
					}
					else
					{
						// curve
						abl->sign=-sgn(abl->a3);					// radius
						abl->r=((double)abs(abl->a3)) * dRADSCALE;
						f=abl->length/abl->r;	// calculate change of angle
						abl->fiv=abl->fik+f*abl->sign;
					}
					break;
				}

			case 0x00:	// normal CCline segment
				{
					// correction!?!
					const double CORRECTION = 128.0;
					// radius
					if(abl->a2==0)
					{
						// straight
						// fraction of tlu correction value
						abl->length -= (((double)abl->a1)/CORRECTION) * 4.8768;

						f=0.0;
						abl->r=0.0;
						abl->sign=0;
						abl->fiv=abl->fik; // starting angle = ending angle
						endtlof+=abs(abl->a1);
					}
					else
					{
						// curve
						// fraction of tlu correction value
						abl->length -= (((double)abl->a1)/CORRECTION) * 4.8768;

						abl->sign = -sgn(abl->a2);
						abl->r = ((double)abs(abl->a2)) * dRADSCALE; // radius in meters
						f = abl->length/abl->r; // calculate change of angle
						abl->fiv = abl->fik + f*abl->sign; // add to starting angle to get ending angle
					}
					break;
				}

			case 0x40:	// carries 32Bit radius
				{
					if(abl->bigrad==0)
					{
						// straight
						f=0.0;
						abl->r=0.0;
						abl->sign=0;
						abl->fiv=abl->fik;
						endtlof+=abs(abl->a1);	// accumulate 'correction'
					}
					else
					{
						// curve
						abl->r=((double)labs(abl->bigrad)) * dRADSCALE;
						f=abl->length/abl->r;
						abl->fiv=abl->fik+f*abl->sign;
					}
					break;
				}
			}/* end of switch(abl->type) */

			// calculate coordinates of end point and center of circle
			if(abl->sign)
			{
				// curve
				abl->ox=abl->kx-abl->sign*abl->r*cos(abl->fik);
				abl->oy=abl->ky-abl->sign*abl->r*sin(abl->fik);
				abl->vx=abl->ox+abl->sign*abl->r*cos(abl->fiv);
				abl->vy=abl->oy+abl->sign*abl->r*sin(abl->fiv);
			}
			else
			{
				// straight
				abl->ox=0;
				abl->oy=0;
				abl->vx=abl->kx-abl->length*sin(abl->fik);
				abl->vy=abl->ky+abl->length*cos(abl->fik);
			}

			// next segment
			abl = abl->m_pNext;
		}/* end of while(abl != NULL) */

	}
}

void CTrack::FreeMemory()
{
	CBestLineSeg *fbl, *ablseg;
	CTrackSeg *ftr, *atrack, *apit;

	/* free previously allocated memory */

	// Computer cars "best line"
	fbl=m_pBestLineSegs;
	while(fbl)
	{
		ablseg=fbl;
		fbl=fbl->m_pNext;
		delete ablseg;
	}
	m_pBestLineSegs = NULL;

	// track segments
	ftr=m_pTrackSegs;
	while(ftr)
    {
		atrack=ftr;
		ftr=ftr->m_pNext;
		delete atrack;
	}
	m_pTrackSegs=NULL;

	// pit segments
	ftr=m_pPitSegs;
	while(ftr)
    {
		apit=ftr;
		ftr=ftr->m_pNext;
		delete apit;
	}
	m_pPitSegs=NULL;
}

// add newly created command to command list pFirst.
// pLast points to currently last element of list.

CCommand * CTrack::AddCommand(CCommand *&pFirst, CCommand *&pLast)
{
	CCommand * pCmd;

	pCmd = new CCommand;
	if(pFirst == NULL)
	{
	   // add first element to list
	   pFirst = pCmd;
	   pLast = pCmd;
	   pCmd->m_pPrev = NULL;
	   pCmd->m_pNext = NULL;
	}
	else
	{
	   // append to end of list
	   pLast->m_pNext = pCmd;
	   pCmd->m_pPrev = pLast;
	   pCmd->m_pNext = 0;
	   pLast = pCmd;
	}
	return pCmd;
}

CTrackSeg * CTrack::GetCurrentTrackSeg()
{
	return m_pTrackSegCurrent;
}

void CTrack::MoveEditPointer(char nChar)
{
	switch( m_eEditMode )
	{
	case emTrack:
		if ( m_pTrackSegCurrent != NULL )
		{
			if ( nChar == VK_HOME )
			{
				// move to beginning of list
				if ( m_pTrackSegCurrent->m_pPrev != NULL )
					m_pTrackSegCurrent = m_pTrackSegCurrent->m_pPrev;
			}
			else
			{
				// move to end of list
				if ( m_pTrackSegCurrent->m_pNext != NULL )
					m_pTrackSegCurrent = m_pTrackSegCurrent->m_pNext;
			}
		}
		break;
	case emPit:
		if ( m_pPitSegCurrent != NULL )
		{
			if ( nChar == VK_HOME )
			{
				// move to beginning of list
				if ( m_pPitSegCurrent->m_pPrev != NULL )
					m_pPitSegCurrent = m_pPitSegCurrent->m_pPrev;
			}
			else
			{
				// move to end of list
				if ( m_pPitSegCurrent->m_pNext != NULL )
					m_pPitSegCurrent = m_pPitSegCurrent->m_pNext;
			}
		}
		break;
	case emBestLine:
		if ( m_pBestLineSegCurrent != NULL )
		{
			if ( nChar == VK_HOME )
			{
				// move to beginning of list
				if ( m_pBestLineSegCurrent->m_pPrev != NULL )
					m_pBestLineSegCurrent = m_pBestLineSegCurrent->m_pPrev;
			}
			else
			{
				// move to end of list
				if ( m_pBestLineSegCurrent->m_pNext != NULL )
					m_pBestLineSegCurrent = m_pBestLineSegCurrent->m_pNext;
			}
		}
		break;
	}
}


CTrackSeg * CTrack::GetCurrentPitSeg()
{
	return m_pPitSegCurrent;
}

/************************************/
/*refresh track and pitlane segments*/
/************************************/
void CTrack::RefreshTrack()
{
	double f;
	double width, widthlen, newwidth;
	CCommand *fcmd;
	CTrackSeg *ftr,*pitstart;
	long int pitgrad,pithei;

	pitstart=NULL;
	width=2*dWIDTHSCALE*((double)m_TrackHeader.startWidth);
	widthlen=0;

	// walk through track segments
	ftr=m_pTrackSegs;
	while(ftr)
	{
		if(ftr->sign==4)
		{
			ftr=ftr->m_pNext; continue;
		}
		fcmd=ftr->cmd;
		while(fcmd)
		{
			if(fcmd->cmd==0x85)
			{
				// change of track width
				widthlen=4.8768*((double)fcmd->a1);
				newwidth=2*dWIDTHSCALE*((double)fcmd->a2);
			}
			if(fcmd->cmd==0x86)
			{
				// this segment is the pit entrance
				pitstart=ftr;
			}
			fcmd=fcmd->m_pNext;
		} // while fcmd
		if(m_pTrackSegs==ftr)
		{
			ftr->fik=-dANGLESCALE*((double)m_TrackHeader.startangle);
			ftr->kx=0; ftr->ky=0;
		}
		else
		{
			ftr->fik=ftr->m_pPrev->fiv;
			ftr->kx=ftr->m_pPrev->vx; ftr->ky=ftr->m_pPrev->vy;
		}
		ftr->length=4.8768*((double)ftr->tlu);
		f=-dANGLESCALE*((double)ftr->tlu)*((double)ftr->curvature);
		ftr->fiv=ftr->fik+f;
		ftr->sign=sgn(f);
		if(ftr->sign)
		{
			// curve
			ftr->r=ftr->length/(ftr->sign*f);
			// center of circle
			ftr->ox=ftr->kx-ftr->sign*ftr->r*cos(ftr->fik);
			ftr->oy=ftr->ky-ftr->sign*ftr->r*sin(ftr->fik);
			// end point
			ftr->vx=ftr->ox+ftr->sign*ftr->r*cos(ftr->fiv);
			ftr->vy=ftr->oy+ftr->sign*ftr->r*sin(ftr->fiv);
		}
		else
		{
			// straigth
			ftr->r=0;
			ftr->ox=0;
			ftr->oy=0;
			ftr->vx=ftr->kx-ftr->length*sin(ftr->fik);
			ftr->vy=ftr->ky+ftr->length*cos(ftr->fik);
		}
		ftr->ks[0]=width*0.5; ftr->ks[1]=ftr->ks[0];
		if(widthlen>0.01)
		{
			if(widthlen>ftr->length)
			{
				width+=ftr->length*(newwidth-width)/widthlen;
				ftr->middle=0;
			}
			else
			{
				if(fabs(widthlen-ftr->length)<0.0001)
				{
					width=newwidth; ftr->middle=0;
				}
				else
				{
					ftr->middle=1;
					if(ftr->sign)
					{
						f=ftr->fik-dANGLESCALE*(widthlen/4.8768)*((double)ftr->curvature);
						ftr->fim=f;
						ftr->mx=ftr->ox+ftr->sign*ftr->r*cos(f);
						ftr->my=ftr->oy+ftr->sign*ftr->r*sin(f);
					}
					else
					{
						f=widthlen;
						ftr->fim=f;
						ftr->mx=ftr->kx-f*sin(ftr->fik);
						ftr->my=ftr->ky+f*cos(ftr->fik);
					}
					width=newwidth;
				}
			}
			widthlen-=ftr->length;
		}
		ftr->vs[0]=width*0.5; ftr->vs[1]=ftr->vs[0];

		// calculate cumulated values
		if(ftr->m_pPrev)
		{
			ftr->cumlength=ftr->m_pPrev->cumlength+ftr->length;
			ftr->cumgrad=ftr->m_pPrev->cumgrad+((long int)ftr->tlu)*((long int)ftr->height);
			ftr->cumheight=ftr->m_pPrev->cumheight+ftr->tlu*(ftr->m_pPrev->cumgrad+ftr->cumgrad)/2;
		}
		else
		{
			ftr->cumlength=ftr->length;
			ftr->cumgrad=(long int)m_TrackHeader.startheight;
			ftr->cumgrad+=((long int)ftr->tlu)*((long int)ftr->height);
			ftr->cumheight=ftr->tlu*(((long int)m_TrackHeader.startheight)+ftr->cumgrad)/2;
		}

		if(ftr==pitstart)
		{
			if(m_TrackHeader.pitside)
			{/*pits on the left*/
				m_pPitSegs->kx=ftr->kx-cos(ftr->fik)*(ftr->ks[0]-dPITWIDTH*0.5);
				m_pPitSegs->ky=ftr->ky-sin(ftr->fik)*(ftr->ks[0]-dPITWIDTH*0.5);
			}
			else
			{/*pits on the right*/
				m_pPitSegs->kx=ftr->kx+cos(ftr->fik)*(ftr->ks[0]-dPITWIDTH*0.5);
				m_pPitSegs->ky=ftr->ky+sin(ftr->fik)*(ftr->ks[0]-dPITWIDTH*0.5);
			}
			m_pPitSegs->fik=ftr->fik;
			pitgrad=(long)ftr->m_pPrev->cumgrad;
			pithei=(long)ftr->m_pPrev->cumheight;
		}

		ftr=ftr->m_pNext;
	} // while

	// now pitlane
	ftr=m_pPitSegs;
	width=dPITWIDTH;
	while(ftr)
	{
		if(ftr->m_pPrev)
		{
			ftr->kx=ftr->m_pPrev->vx; ftr->ky=ftr->m_pPrev->vy;
			ftr->fik=ftr->m_pPrev->fiv;
		}
		ftr->length=4.8768*((double)ftr->tlu);
		f=-dANGLESCALE*((double)ftr->tlu)*((double)ftr->curvature);
		ftr->fiv=ftr->fik+f;
		ftr->sign=sgn(f);
		if(ftr->sign)
		{
			ftr->r=ftr->length/(ftr->sign*f);
			ftr->ox=ftr->kx-ftr->sign*ftr->r*cos(ftr->fik);
			ftr->oy=ftr->ky-ftr->sign*ftr->r*sin(ftr->fik);
			ftr->vx=ftr->ox+ftr->sign*ftr->r*cos(ftr->fiv);
			ftr->vy=ftr->oy+ftr->sign*ftr->r*sin(ftr->fiv);
		}
		else
		{
			ftr->r=0.0;
			ftr->ox=0;
			ftr->oy=0;
			ftr->vx=ftr->kx-ftr->length*sin(ftr->fik);
			ftr->vy=ftr->ky+ftr->length*cos(ftr->fik);
		}
		if(ftr->m_pPrev)
		{
			ftr->cumlength=ftr->m_pPrev->cumlength+ftr->length;
			ftr->cumgrad=ftr->m_pPrev->cumgrad+((long int)ftr->tlu)*((long int)ftr->height);
			ftr->cumheight=ftr->m_pPrev->cumheight+ftr->tlu*(ftr->m_pPrev->cumgrad+ftr->cumgrad)/2;
		}
		else
		{
			ftr->cumlength=ftr->length;
			ftr->cumgrad=pitgrad+((long int)ftr->tlu)*((long int)ftr->height);
			ftr->cumheight=pithei+ftr->tlu*(pitgrad+ftr->cumgrad)/2;
		}
		ftr->ks[0]=dPITWIDTH*0.5; ftr->ks[1]=dPITWIDTH*0.5;
		ftr->vs[0]=dPITWIDTH*0.5; ftr->vs[1]=dPITWIDTH*0.5;
		ftr=ftr->m_pNext;
	}
}

/// Add a new segment to appropriate list

void CTrack::AddSegment()
{
	switch( m_eEditMode )
	{
	case emTrack:
		{
			CTrackSeg * pTs = new CTrackSeg;
			pTs->tlu = 1;

			if ( m_pTrackSegCurrent != NULL )
			{
				// put new segment behind current segment
				pTs->m_pNext = m_pTrackSegCurrent->m_pNext;
				if ( pTs->m_pNext != NULL )
					pTs->m_pNext->m_pPrev = pTs;
				m_pTrackSegCurrent->m_pNext = pTs;
				pTs->m_pPrev = m_pTrackSegCurrent;
			}
			else
			{
				// this is the first segment at all
				m_pTrackSegs = pTs;
			}
			m_pTrackSegCurrent = pTs;
			RefreshTrack();
		}
		break;
	case emPit:
		{
			CTrackSeg * pTs = new CTrackSeg;
			pTs->tlu = 1;

			if ( m_pPitSegCurrent != NULL )
			{
				// put new segment behind current segment
				pTs->m_pNext = m_pPitSegCurrent->m_pNext;
				if ( pTs->m_pNext != NULL )
					pTs->m_pNext->m_pPrev = pTs;
				m_pPitSegCurrent->m_pNext = pTs;
				pTs->m_pPrev = m_pPitSegCurrent;
			}
			else
			{
				// this is the first segment at all
				m_pPitSegs = pTs;
			}
			m_pPitSegCurrent = pTs;
			RefreshTrack();
		}
		break;
	case emBestLine:
		{
			CBestLineSeg * pBls = new CBestLineSeg;
			pBls->tlu = 1;

			if ( m_pBestLineSegCurrent != NULL )
			{
				// put new segment behind current segment
				pBls->m_pNext = m_pBestLineSegCurrent->m_pNext;
				if ( pBls->m_pNext != NULL )
					pBls->m_pNext->m_pPrev = pBls;
				m_pBestLineSegCurrent->m_pNext = pBls;
				pBls->m_pPrev = m_pBestLineSegCurrent;
			}
			else
			{
				// this is the first segment at all
				m_pBestLineSegs = pBls;
			}
			m_pBestLineSegCurrent = pBls;
			RefreshCCLine();
		}
		break;
	}
}

/// delete currently selected segment, if any.

void CTrack::DeleteSegment()
{
	switch( m_eEditMode )
	{
	case emTrack:
		if ( m_pTrackSegCurrent != NULL )
		{
			CTrackSeg * pTs = m_pTrackSegCurrent;
			// remove from list
			if ( m_pTrackSegCurrent->m_pPrev != NULL )
				m_pTrackSegCurrent->m_pPrev->m_pNext = m_pTrackSegCurrent->m_pNext;
			else
				// was first segment in list
				m_pTrackSegs = m_pTrackSegCurrent->m_pNext;
			if ( m_pTrackSegCurrent->m_pNext != NULL )
			{
				m_pTrackSegCurrent->m_pNext->m_pPrev = m_pTrackSegCurrent->m_pPrev;
				m_pTrackSegCurrent = m_pTrackSegCurrent->m_pNext;
			}
			else
				m_pTrackSegCurrent = m_pTrackSegCurrent->m_pPrev;
			// delete segment through saved pointer
			delete pTs;
			pTs = NULL;

			RefreshTrack();
		}
		break;
	case emPit:
		{
			CTrackSeg * pTs = m_pPitSegCurrent;
			// remove from list
			if ( m_pPitSegCurrent->m_pPrev != NULL )
				m_pPitSegCurrent->m_pPrev->m_pNext = m_pPitSegCurrent->m_pNext;
			else
				// was first segment in list
				m_pPitSegs = m_pPitSegCurrent->m_pNext;
			if ( m_pPitSegCurrent->m_pNext != NULL )
			{
				m_pPitSegCurrent->m_pNext->m_pPrev = m_pPitSegCurrent->m_pPrev;
				m_pPitSegCurrent = m_pPitSegCurrent->m_pNext;
			}
			else
				m_pPitSegCurrent = m_pPitSegCurrent->m_pPrev;
			// delete segment through saved pointer
			delete pTs;
			pTs = NULL;

			RefreshTrack();
		}
		break;
	case emBestLine:
		{
			CBestLineSeg * pBls = m_pBestLineSegCurrent;
			// remove from list
			if ( m_pBestLineSegCurrent->m_pPrev != NULL )
				m_pBestLineSegCurrent->m_pPrev->m_pNext = m_pBestLineSegCurrent->m_pNext;
			else
				// was first segment in list
				m_pBestLineSegs = m_pBestLineSegCurrent->m_pNext;
			if ( m_pBestLineSegCurrent->m_pNext != NULL )
			{
				m_pBestLineSegCurrent->m_pNext->m_pPrev = m_pBestLineSegCurrent->m_pPrev;
				m_pBestLineSegCurrent = m_pBestLineSegCurrent->m_pNext;
			}
			else
				m_pBestLineSegCurrent = m_pBestLineSegCurrent->m_pPrev;
			// delete segment through saved pointer
			delete pBls;
			pBls = NULL;

			RefreshCCLine();
		}
		break;
	}
}

/**********************/
/*save track file data*/
/**********************/
BOOL CTrack::OnSaveDocument(LPCTSTR lpszPathName) 
{
	CTrackSeg *ftr;
	CCommand *fcmd;
	CBestLineSeg *fbl;
	FILE *trf;
	short int si;
	unsigned short int ui;
	long li;

	trf=fopen(lpszPathName,"wb");
	if(trf==NULL)
	{
		AfxMessageBox( "Unable to create track file " + CString( lpszPathName ), MB_OK );
		return FALSE;
	}

	/*writing background picture*/
	fwrite(&m_baBackground,1,4096,trf);

	/*writing track header*/
	m_TrackFileHeader.Save( trf );

	/*writing object datas*/
	m_Objects.Save( trf );

	/*writing track data header*/
	SaveTrackDataHeader( trf );

	/*writing track segment data*/
	ftr=m_pTrackSegs;
	while(ftr)
	{
		/*writing commands*/
		fcmd=ftr->cmd;
		while(fcmd)
		{
			fwrite(&fcmd->a0,1,1,trf);
			fwrite(&fcmd->cmd,1,1,trf);
			if( (fcmd->cmd==0x80)||(fcmd->cmd==0x81)||(fcmd->cmd==0x82)||
				(fcmd->cmd==0x88)||(fcmd->cmd==0x89)||(fcmd->cmd==0x8C)||
				(fcmd->cmd==0x8D)||(fcmd->cmd==0x90)||(fcmd->cmd==0x91)||
				(fcmd->cmd==0x92)||(fcmd->cmd==0x93)||(fcmd->cmd==0x94)||
				(fcmd->cmd==0x95)||(fcmd->cmd==0x98)||(fcmd->cmd==0x99)||
				(fcmd->cmd==0xA9)
				)
			{
				fwrite(&fcmd->a1,2,1,trf);
			}
			if( (fcmd->cmd==0x85)||(fcmd->cmd==0x8E)||(fcmd->cmd==0x8F)||
				(fcmd->cmd==0x9A)||(fcmd->cmd==0xA6)||(fcmd->cmd==0xA7)||
				(fcmd->cmd==0xAB)
				)
			{
				fwrite(&fcmd->a1,2,1,trf);
				fwrite(&fcmd->a2,2,1,trf);
			}
			if( (fcmd->cmd==0xAA)
				)
			{
				fwrite(&fcmd->a1,2,1,trf);
				fwrite(&fcmd->a2,2,1,trf);
				fwrite(&fcmd->a3,2,1,trf);
			}
			if(fcmd->cmd==0xAC)
			{
				fwrite(&fcmd->a1,2,1,trf);
				fwrite(&fcmd->a2,2,1,trf);
				fwrite(&fcmd->a3,2,1,trf);
				fwrite(&fcmd->a4,2,1,trf);
			}
			if( (fcmd->cmd==0x8A)||(fcmd->cmd==0x8B) )
			{
				fwrite(&fcmd->a1,2,1,trf);
				fwrite(&fcmd->a2,2,1,trf);
				fwrite(&fcmd->a3,2,1,trf);
				fwrite(&fcmd->a4,2,1,trf);
				fwrite(&fcmd->a5,2,1,trf);
			}
			fcmd=fcmd->m_pNext;
		} // while
		/*writing track segment*/
		if(ftr->sign!=4)
		{
			fwrite(&ftr->tlu,2,1,trf);
			fwrite(&ftr->curvature,2,1,trf);
			fwrite(&ftr->height,2,1,trf);
			fwrite(&ftr->flags,2,1,trf);
			fwrite(&ftr->rbank,1,1,trf);
			fwrite(&ftr->lbank,1,1,trf);
		}
		ftr=ftr->m_pNext;
	}
	si=(short)65535;
	fwrite(&si,2,1,trf);

	long lPos = ftell(trf);

	/*writing CC line data*/
	fbl=m_pBestLineSegs;
	while(fbl)
	{
		fwrite(&fbl->tlu,1,1,trf);
		fwrite(&fbl->type,1,1,trf);
		fwrite(&fbl->a1,2,1,trf);
		fwrite(&fbl->a2,2,1,trf);
		if( fbl->type/*(fbl->type==0x80)||(fbl->type==0x40)*/ )
		{
			// first seg and bigrad seg have additional parameter
			fwrite(&fbl->a3,2,1,trf);
		}
		fbl=fbl->m_pNext;
	}
	si=0;
	fwrite(&si,2,1,trf);

	/*writing CC setup data*/
	SaveCCSetup( trf );

	/*writing pitlane segment data*/
	ftr=m_pPitSegs;
	while(ftr)
	{/*writing commands*/
	fcmd=ftr->cmd;
	while(fcmd)
	  {fwrite(&fcmd->a0,1,1,trf);
	   fwrite(&fcmd->cmd,1,1,trf);
	   if( (fcmd->cmd==0x80)||(fcmd->cmd==0x81)||(fcmd->cmd==0x82)||
	   (fcmd->cmd==0x88)||(fcmd->cmd==0x89)||(fcmd->cmd==0x8C)||
	   (fcmd->cmd==0x8D)||(fcmd->cmd==0x90)||(fcmd->cmd==0x91)||
	   (fcmd->cmd==0x92)||(fcmd->cmd==0x93)||(fcmd->cmd==0x94)||
	   (fcmd->cmd==0x95)||(fcmd->cmd==0x98)||(fcmd->cmd==0x99)||
	   (fcmd->cmd==0xA9)
	 )
	 {fwrite(&fcmd->a1,2,1,trf);
	 }
	   if( (fcmd->cmd==0x85)||(fcmd->cmd==0x8E)||(fcmd->cmd==0x8F)||
	   (fcmd->cmd==0x9A)||(fcmd->cmd==0xA6)||(fcmd->cmd==0xA7)||
	   (fcmd->cmd==0xAB)
	 )
	 {fwrite(&fcmd->a1,2,1,trf);
	  fwrite(&fcmd->a2,2,1,trf);
	 }
	   if( (fcmd->cmd==0xAA)
	 )
	 {fwrite(&fcmd->a1,2,1,trf);
	  fwrite(&fcmd->a2,2,1,trf);
	  fwrite(&fcmd->a3,2,1,trf);
	 }
	   if(fcmd->cmd==0xAC)
	 {fwrite(&fcmd->a1,2,1,trf);
	  fwrite(&fcmd->a2,2,1,trf);
	  fwrite(&fcmd->a3,2,1,trf);
	  fwrite(&fcmd->a4,2,1,trf);
	 }
	   if( (fcmd->cmd==0x8A)||(fcmd->cmd==0x8B)
	 )
	 {fwrite(&fcmd->a1,2,1,trf);
	  fwrite(&fcmd->a2,2,1,trf);
	  fwrite(&fcmd->a3,2,1,trf);
	  fwrite(&fcmd->a4,2,1,trf);
	  fwrite(&fcmd->a5,2,1,trf);
	 }
	   fcmd=fcmd->m_pNext;
	  }
	/*writing track segment*/
	fwrite(&ftr->tlu,2,1,trf);
	fwrite(&ftr->curvature,2,1,trf);
	fwrite(&ftr->height,2,1,trf);
	fwrite(&ftr->flags,2,1,trf);
	fwrite(&ftr->rbank,1,1,trf);
	fwrite(&ftr->lbank,1,1,trf);
	ftr=ftr->m_pNext;
	}
	si=(short)65535;
	fwrite(&si,2,1,trf);

	/*writing data to the end of file*/
	fwrite(m_baEndheader,1,m_nLapNumIndex+6,trf);
	// empty space will be filled with checksum later
	li=0;
	fwrite(&li,4,1,trf);

	/*new offset to checksum*/
	li = ftell(trf);
	ui=(unsigned int)li;
	m_TrackFileHeader.m_nChecksumOffset = ui - 4 - m_TrackFileHeader.m_nSizeBase;
	fseek(trf,4104,SEEK_SET); // seek from beginning of file
	fwrite(&m_TrackFileHeader.m_nChecksumOffset,2,1,trf);

	fclose(trf);

	/*calculate new checksum*/
	CalculateChecksum( lpszPathName );

	SetModifiedFlag( FALSE );
	return TRUE;
	// @@@ change processing to serialization approach
	//	return CDocument::OnSaveDocument(lpszPathName);
}

void CTrack::CalculateChecksum(LPCTSTR lpszPathName)
{
	// calculate chechsum on our own.
	FILE * TrackFile = fopen(lpszPathName, "r+b");
	unsigned short sChecksum1 = 0, sChecksum2 = 0;
	long lSize = 0;
	// determine file size
	fseek( TrackFile, 0, SEEK_END );
	lSize = ftell( TrackFile );

	// seek to start
	fseek( TrackFile, 0, SEEK_SET );
	int nChar;
	for ( long lPos = 0; lPos < lSize - 4; lPos++ )
	{
		nChar = fgetc( TrackFile );
		sChecksum1 = sChecksum1 + nChar;
		// Checksum 2: add char, then left rotate by 3
		// can not directly rotate 16 bit value...
		sChecksum2 = (sChecksum2 << 3) + (sChecksum2 >> 13);
		sChecksum2 = sChecksum2 + nChar;
	}
	// write checksum to file
	fseek( TrackFile, lSize - 4, SEEK_SET );
	fwrite( &sChecksum1, 2, 1, TrackFile );
	fwrite( &sChecksum2, 2, 1, TrackFile );

	fclose( TrackFile );
}

void CTrack::ChangeCmd(int nComponent, int nValue)
{
	switch( m_eEditMode )
	{
	case emTrack:
		if ( m_pTrackSegCurrent != NULL )
		{
			if ( m_pTrackSegCurrent->cmd != NULL )
			{
				switch( nComponent )
				{
				case -1: // Command
					m_pTrackSegCurrent->cmd->cmd = nValue;
					break;
				case 0: // A0
					m_pTrackSegCurrent->cmd->a0 = nValue;
					break;
				case 1: // A1
					m_pTrackSegCurrent->cmd->a1 = nValue;
					break;
				case 2: // A2
					m_pTrackSegCurrent->cmd->a2 = nValue;
					break;
				case 3: // A3
					m_pTrackSegCurrent->cmd->a3 = nValue;
					break;
				case 4: // A4
					m_pTrackSegCurrent->cmd->a4 = nValue;
					break;
				case 5: // A5
					m_pTrackSegCurrent->cmd->a5 = nValue;
					break;
				}
			}
		}
		break;
	case emPit:
		if ( m_pPitSegCurrent != NULL )
		{
			if ( m_pPitSegCurrent->cmd != NULL )
			{
				switch( nComponent )
				{
				case -1: // Command
					m_pPitSegCurrent->cmd->cmd = nValue;
					break;
				case 0: // A0
					m_pPitSegCurrent->cmd->a0 = nValue;
					break;
				case 1: // A1
					m_pPitSegCurrent->cmd->a1 = nValue;
					break;
				case 2: // A2
					m_pPitSegCurrent->cmd->a2 = nValue;
					break;
				case 3: // A3
					m_pPitSegCurrent->cmd->a3 = nValue;
					break;
				case 4: // A4
					m_pPitSegCurrent->cmd->a4 = nValue;
					break;
				case 5: // A5
					m_pPitSegCurrent->cmd->a5 = nValue;
					break;
				}
			}
		}
		break;
	default:
		// everything else does not have commands
		break;
	}
}

void CTrack::LoadPitlaneSegments(FILE *pTrackFile)
{
	byte trackdata = 0, b = 0;
	CCommand *kcmd = NULL, *acmd = NULL, *fcmd = NULL;
	double newwidth = 0.0, widthlen = 0.0, f = 0.0;
	unsigned short ui = 0;
	short int si;
	CTrackSeg *atr = NULL, *ftr = NULL;
	long lFilePos;

	trackdata=1;
	kcmd=NULL;
	m_pPitSegs=NULL;
	m_dWidth = dPITWIDTH;
	newwidth = dPITWIDTH;
	widthlen=0;
	while(trackdata)
	{
		lFilePos = ftell( pTrackFile );
		fread(&ui,2,1,pTrackFile);
		if( (ui/256!=0)&&(ui/256!=0xFF) )
		{
			acmd = new CCommand();
			if(kcmd == NULL)
			{
				// add first element to list
				kcmd = acmd;
				fcmd = acmd;
				acmd->m_pPrev = NULL;
				acmd->m_pNext = NULL;
			}
			else
			{
				// append to end of list
				fcmd->m_pNext = acmd;
				acmd->m_pPrev = fcmd;
				acmd->m_pNext = NULL;
				fcmd = acmd;
			}
		}
		switch(ui/256)
		  {case 0x80:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x81:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x82:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x83:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x84:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x85:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  widthlen=4.8768*((double)ui);
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  newwidth = 2 * dWIDTHSCALE * ((double)ui);
			  break;}
		   case 0x86:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x87:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x88:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x89:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x8A:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a3=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a4=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a5=ui;
			  break;}
		   case 0x8B:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a3=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a4=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a5=ui;
			  break;}
		   case 0x8C:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x8D:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x8E:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0x8F:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0x90:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x91:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x92:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x93:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x94:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x95:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x96:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x97:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x98:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x99:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0x9A:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0x9B:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x9C:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x9D:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x9E:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0x9F:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA0:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA1:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA2:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA3:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA4:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA5:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA6:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0xA7:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0xA8:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  break;}
		   case 0xA9:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  break;}
		   case 0xAA:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a3=ui;
			  break;}
		   case 0xAB:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  break;}
		   case 0xAC:{acmd->cmd=ui/256;
			  acmd->a0=ui%256;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a1=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a2=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a3=ui;
			  fread(&ui,2,1,pTrackFile);
			  acmd->a4=ui;
			  break;}
		   case 0xFF:{trackdata=0;
			  break;}
		   case 0x00:{atr = new CTrackSeg;
			  if(m_pPitSegs==NULL)
				{m_pPitSegs=atr; ftr=atr; atr->m_pPrev=0; atr->m_pNext=0;
				 atr->fik=m_dPitStartAngle;
				 atr->kx=m_dPitStartX;
				 atr->ky=m_dPitStartY;
				}
			  else
				{ftr->m_pNext=atr; atr->m_pPrev=ftr; atr->m_pNext=0;
				 atr->fik=ftr->fiv;
				 atr->kx=ftr->vx; atr->ky=ftr->vy;
				 ftr=atr;
				}
			  atr->cmd = kcmd;
			  kcmd=0;
			  atr->tlu=ui;
			  atr->length=4.8768*((double)ui);
			  fread(&si,2,1,pTrackFile);
			  atr->curvature=si;
			  fread(&si,2,1,pTrackFile);
			  atr->height=si;
			  fread(&ui,2,1,pTrackFile);
			  atr->flags=ui;
			  fread(&b,1,1,pTrackFile);
			  atr->rbank=b;
			  fread(&b,1,1,pTrackFile);
			  atr->lbank=b;
			  f=-dANGLESCALE*((double)atr->tlu)*((double)atr->curvature);
			  atr->fiv=atr->fik+f;
			  atr->sign=sgn(f);
			  if(atr->sign)
				{atr->r=atr->length/(atr->sign*f);
				 atr->ox=atr->kx-atr->sign*atr->r*cos(atr->fik);
				 atr->oy=atr->ky-atr->sign*atr->r*sin(atr->fik);
				 atr->vx=atr->ox+atr->sign*atr->r*cos(atr->fiv);
				 atr->vy=atr->oy+atr->sign*atr->r*sin(atr->fiv);
				}
			  else
				{atr->r=0;
				 atr->ox=0;
				 atr->oy=0;
				 atr->vx=atr->kx-atr->length*sin(atr->fik);
				 atr->vy=atr->ky+atr->length*cos(atr->fik);
				}
			  atr->ks[0]=m_dWidth*0.5; atr->ks[1]=atr->ks[0];
			  if(widthlen>0.01)
				{if(widthlen>atr->length)
				   {m_dWidth+=atr->length*(newwidth-m_dWidth)/widthlen;
				atr->middle=0;
				   }
				 else
				   {if(fabs(widthlen-atr->length)<0.0001)
				  {m_dWidth=newwidth; atr->middle=0;}
				else{atr->middle=1;
					 if(atr->sign)
					   {f=atr->fik - dANGLESCALE*(widthlen/4.8768)*((double)atr->curvature);
					atr->fim=f;
					atr->mx=atr->ox+atr->sign*atr->r*cos(f);
					atr->my=atr->oy+atr->sign*atr->r*sin(f);
					   }
					 else
					   {f=widthlen;
					atr->fim=f;
					atr->mx=atr->kx-f*sin(atr->fik);
					atr->my=atr->ky+f*cos(atr->fik);
					   }
					 m_dWidth=newwidth;
					}
				   }
				 widthlen-=atr->length;
				}
			  atr->vs[0]=m_dWidth*0.5; atr->vs[1]=atr->vs[0];

			  if(atr->m_pPrev)
				{atr->cumlength=atr->m_pPrev->cumlength+atr->length;
				 atr->cumgrad=atr->m_pPrev->cumgrad+((long int)atr->tlu)*((long int)atr->height);
				 atr->cumheight=atr->m_pPrev->cumheight+atr->tlu*(atr->m_pPrev->cumgrad+atr->cumgrad)/2;
				}
			  else
				{atr->cumlength=atr->length;
				 atr->cumgrad=m_lPitgrad;
				 atr->cumgrad+=((long int)atr->tlu)*((long int)atr->height);
				 atr->cumheight=m_lPithei + atr->tlu*(m_lPitgrad+atr->cumgrad)/2;
				}
			  break;}
		  }/*end of switch(ui/256)*/
	}/*end of while(trackdata)*/

	// set edit point to first segment
	m_pPitSegCurrent = m_pPitSegs;

}

CTrack::EditMode CTrack::SetEditMode(EditMode eNewMode)
{
	EditMode emOld;
	emOld = m_eEditMode;
	m_eEditMode = eNewMode;
	return emOld;
}

CTrack::EditMode CTrack::GetEditMode()
{
	return m_eEditMode;
}

CTrackSeg * CTrack::GetTrackSegs()
{
	return m_pTrackSegs;
}

/// add a new command object to current track or pit segment

void CTrack::InsertCommand()
{
	LPARAM lUpdateHint = 0L; // for UpdataAllViews
	CTrackSeg * pTs = NULL;
	if ( m_eEditMode == emTrack )
	{
		pTs = m_pTrackSegCurrent;
		lUpdateHint = UPD_HINT_TRACK_SEG;
	}
	else if ( m_eEditMode == emPit )
	{
		pTs = m_pPitSegCurrent;
		lUpdateHint = UPD_HINT_PIT_SEG;
	}
	if ( pTs == NULL )
		return;

	pTs->InsertCommand();
	SetModifiedFlag();

	UpdateAllViews( NULL, lUpdateHint, pTs );
}

CTrackSeg * CTrack::GetPitSegs()
{
	return m_pPitSegs;
}

CBestLineSeg * CTrack::GetBestLineSegs()
{
	return m_pBestLineSegs;
}

CBestLineSeg * CTrack::GetCurrentBestLineSeg()
{
	return m_pBestLineSegCurrent;
}

void CTrack::LoadTrackDataHeader(FILE *pTrackFile)
{
	short int si;
	byte b;

	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startangle=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startheight=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startY=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startZ=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startX=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.startWidth=si;
	m_dWidth = 2 * dWIDTHSCALE * ((double)si);
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.poleWidth=si;
	fread(&b,1,1,pTrackFile);
	m_TrackHeader.pitside=b;
	fread(&b,1,1,pTrackFile);
	m_TrackHeader.trsurround=b;
	fread(&b,1,1,pTrackFile);
	m_TrackHeader.rbank=b;
	fread(&b,1,1,pTrackFile);
	m_TrackHeader.lbank=b;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.kerbcnum=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.unk1=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.kerbtopcolor=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.kerbbottomcolor=si;
	fread(&si,2,1,pTrackFile);
	m_TrackHeader.unk2=si;
	if(m_TrackHeader.kerbcnum==4)
	{
		fread(&si,2,1,pTrackFile);
		m_TrackHeader.kerbtopcolor2=si;
		fread(&si,2,1,pTrackFile);
		m_TrackHeader.kerbbottomcolor2=si;
	}
	else
	{
		m_TrackHeader.kerbtopcolor2=m_TrackHeader.kerbtopcolor;
		m_TrackHeader.kerbbottomcolor2=m_TrackHeader.kerbbottomcolor;
	}
}

void CTrack::SaveTrackDataHeader(FILE *pTrackFile)
{
	fwrite(&m_TrackHeader.startangle,2,1,pTrackFile);
	fwrite(&m_TrackHeader.startheight,2,1,pTrackFile);
	fwrite(&m_TrackHeader.startY,2,1,pTrackFile);
	fwrite(&m_TrackHeader.startZ,2,1,pTrackFile);
	fwrite(&m_TrackHeader.startX,2,1,pTrackFile);
	fwrite(&m_TrackHeader.startWidth,2,1,pTrackFile);
	fwrite(&m_TrackHeader.poleWidth,2,1,pTrackFile);
	fwrite(&m_TrackHeader.pitside,1,1,pTrackFile);
	fwrite(&m_TrackHeader.trsurround,1,1,pTrackFile);
	fwrite(&m_TrackHeader.rbank,1,1,pTrackFile);
	fwrite(&m_TrackHeader.lbank,1,1,pTrackFile);
	fwrite(&m_TrackHeader.kerbcnum,2,1,pTrackFile);
	fwrite(&m_TrackHeader.unk1,2,1,pTrackFile);
	fwrite(&m_TrackHeader.kerbtopcolor,2,1,pTrackFile);
	fwrite(&m_TrackHeader.kerbbottomcolor,2,1,pTrackFile);
	fwrite(&m_TrackHeader.unk2,2,1,pTrackFile);
	if(m_TrackHeader.kerbcnum==4)
	{
		fwrite(&m_TrackHeader.kerbtopcolor2,2,1,pTrackFile);
		fwrite(&m_TrackHeader.kerbbottomcolor2,2,1,pTrackFile);
	}
}

void CTrack::LoadCCSetup(FILE *pTrackFile)
{
	fread(&m_CCsetup.fwing,1,1,pTrackFile);
	fread(&m_CCsetup.rwing,1,1,pTrackFile);
	fread(&m_CCsetup.g1,1,1,pTrackFile);
	fread(&m_CCsetup.g2,1,1,pTrackFile);
	fread(&m_CCsetup.g3,1,1,pTrackFile);
	fread(&m_CCsetup.g4,1,1,pTrackFile);
	fread(&m_CCsetup.g5,1,1,pTrackFile);
	fread(&m_CCsetup.g6,1,1,pTrackFile);
	fread(&m_CCsetup.tyretype,1,1,pTrackFile);
	fread(&m_CCsetup.brakeb,1,1,pTrackFile);
	fread(&m_CCsetup.grip,2,1,pTrackFile);
	fread(&m_CCsetup.QA,2,1,pTrackFile);
	fread(&m_CCsetup.RA,2,1,pTrackFile);
	fread(&m_CCsetup.QT,2,1,pTrackFile);
	fread(&m_CCsetup.accel,2,1,pTrackFile);
	fread(&m_CCsetup.airres,2,1,pTrackFile);
	fread(&m_CCsetup.unk1,2,1,pTrackFile);
	fread(&m_CCsetup.TW1,2,1,pTrackFile);
	fread(&m_CCsetup.TW2,2,1,pTrackFile);
	fread(&m_CCsetup.RT,2,1,pTrackFile);
	fread(&m_CCsetup.unk2,2,1,pTrackFile);
	fread(&m_CCsetup.wetRA,2,1,pTrackFile);
	fread(&m_CCsetup.unk3,2,1,pTrackFile);
	fread(&m_CCsetup.unk4,2,1,pTrackFile);
}

void CTrack::SaveCCSetup(FILE *pTrackFile)
{
	fwrite(&m_CCsetup.fwing,1,1,pTrackFile);
	fwrite(&m_CCsetup.rwing,1,1,pTrackFile);
	fwrite(&m_CCsetup.g1,1,1,pTrackFile);
	fwrite(&m_CCsetup.g2,1,1,pTrackFile);
	fwrite(&m_CCsetup.g3,1,1,pTrackFile);
	fwrite(&m_CCsetup.g4,1,1,pTrackFile);
	fwrite(&m_CCsetup.g5,1,1,pTrackFile);
	fwrite(&m_CCsetup.g6,1,1,pTrackFile);
	fwrite(&m_CCsetup.tyretype,1,1,pTrackFile);
	fwrite(&m_CCsetup.brakeb,1,1,pTrackFile);
	fwrite(&m_CCsetup.grip,2,1,pTrackFile);
	fwrite(&m_CCsetup.QA,2,1,pTrackFile);
	fwrite(&m_CCsetup.RA,2,1,pTrackFile);
	fwrite(&m_CCsetup.QT,2,1,pTrackFile);
	fwrite(&m_CCsetup.accel,2,1,pTrackFile);
	fwrite(&m_CCsetup.airres,2,1,pTrackFile);
	fwrite(&m_CCsetup.unk1,2,1,pTrackFile);
	fwrite(&m_CCsetup.TW1,2,1,pTrackFile);
	fwrite(&m_CCsetup.TW2,2,1,pTrackFile);
	fwrite(&m_CCsetup.RT,2,1,pTrackFile);
	fwrite(&m_CCsetup.unk2,2,1,pTrackFile);
	fwrite(&m_CCsetup.wetRA,2,1,pTrackFile);
	fwrite(&m_CCsetup.unk3,2,1,pTrackFile);
	fwrite(&m_CCsetup.unk4,2,1,pTrackFile);
}

void CTrack::SetCurrentPitSeg(CTrackSeg *pTs)
{
	m_pPitSegCurrent = pTs;
	UpdateAllViews( NULL, UPD_HINT_PIT_SEG, pTs );
}

void CTrack::SetCurrentTrackSeg(CTrackSeg *pTs)
{
	m_pTrackSegCurrent = pTs;
	UpdateAllViews( NULL, UPD_HINT_TRACK_SEG, pTs );
}

void CTrack::SetCurrentBestLineSeg(CBestLineSeg *pBestLineSeg)
{
	m_pBestLineSegCurrent = pBestLineSeg;
	UpdateAllViews( NULL, UPD_HINT_BEST_LINE_SEG, pBestLineSeg );
}

// Delete the segment from the track

void CTrack::DeleteTrackSegment(CTrackSeg *pTs)
{
	ASSERT( pTs == m_pTrackSegCurrent );
	m_pTrackSegCurrent = DeleteSegment( m_pTrackSegs, pTs );

	SetModifiedFlag();

	RefreshTrack();
	// Note: pTs is no longer valid, but will be used to remove
	// GUI representation.
	UpdateAllViews( NULL, UPD_HINT_TRACK_SEG_DELETED, pTs );
}

// Delete the segment from the pitlane

void CTrack::DeletePitSegment(CTrackSeg *pTs)
{
	ASSERT( pTs == m_pPitSegCurrent );
	m_pTrackSegCurrent = DeleteSegment( m_pPitSegs, pTs );

	SetModifiedFlag();

	RefreshTrack();
	// Note: pTs is no longer valid, but will be used to remove
	// GUI representation.
	UpdateAllViews( NULL, UPD_HINT_PIT_SEG_DELETED, pTs );
}

// Insert a new track segment after the given one

void CTrack::InsertTrackSegment(CTrackSeg *pTs)
{
	ASSERT( pTs == m_pTrackSegCurrent );
	CTrackSeg * pNewTs = new CTrackSeg();
	// must have length of at least 1 unit
	pNewTs->tlu = 1;
	if ( pTs != NULL )
	{
		// Insert after the given segment
		pNewTs->m_pNext = pTs->m_pNext;
		pTs->m_pNext = pNewTs;
	}
	else
	{
		// Insert as first segment
		pNewTs->m_pNext = m_pTrackSegs;
		m_pTrackSegs = pNewTs;
	}
	pNewTs->m_pPrev = pTs;
	if ( pNewTs->m_pNext != NULL )
		pNewTs->m_pNext->m_pPrev = pNewTs;

	SetModifiedFlag();

	// Update
	RefreshTrack();
	UpdateAllViews( NULL, UPD_HINT_TRACK_SEG_INSERTED, pTs );
}

// Insert a new pitlane segment after the given one

void CTrack::InsertPitSegment(CTrackSeg *pTs)
{
	ASSERT( pTs == m_pPitSegCurrent );
	CTrackSeg * pNewTs = new CTrackSeg();
	// must have length of at least 1 unit
	pNewTs->tlu = 1;
	if ( pTs != NULL )
	{
		// Insert after the given segment
		pNewTs->m_pNext = pTs->m_pNext;
		pTs->m_pNext = pNewTs;
	}
	else
	{
		// Insert as first segment
		pNewTs->m_pNext = m_pPitSegs;
		m_pPitSegs = pNewTs;
	}
	pNewTs->m_pPrev = pTs;
	if ( pNewTs->m_pNext != NULL )
		pNewTs->m_pNext->m_pPrev = pNewTs;

	SetModifiedFlag();

	// Update
	RefreshTrack();
	UpdateAllViews( NULL, UPD_HINT_PIT_SEG_INSERTED, pTs );
}

/**
  Removes the given segment from the given list.
  Deletes the track segment.
  Returns the new "current segment" of the list. This will be the next
  segment in the list, if any. Otherwise, it will be the last segment
  in the list. If the list is empty, returns NULL.
  If segment was first in the list, the pointer to the list will be
  changed.
*/

CTrackSeg * CTrack::DeleteSegment(CTrackSeg *&pTsList, CTrackSeg *pTs)
{
	CTrackSeg * pResult;
	// -----------------------------------------------------------------
	if ( pTsList == NULL )
		return NULL;
	if ( pTs == NULL )
		return NULL;

	if ( pTs == pTsList )
	{
		// Was first element in list
		pTsList = pTs->m_pNext;
	}
	else
	{
		// Not the fist element of the list
		pTs->m_pPrev->m_pNext = pTs->m_pNext;
	}
	
	if ( pTs->m_pNext != NULL )
	{
		pResult = pTs->m_pNext;
		pTs->m_pNext->m_pPrev = pTs->m_pPrev;
	}
	else
		// no next segment: use m_pPrev (whether NULL or not)
		pResult = pTs->m_pPrev;

	return pResult;
}

bool CTrack::GetOldCCLineStyle()
{
	return m_fCCLineOld;
}

void CTrack::SetOldCCLineStyle(bool fOld)
{
	m_fCCLineOld = fOld;
}

void CTrack::InsertCCLineSegment(CBestLineSeg *pBlNew, CBestLineSeg *pInsertAfter)
{
	if ( pBlNew == NULL )
		return;

	if ( pInsertAfter == NULL )
	{
		pBlNew->m_pNext = m_pBestLineSegs;
		if ( m_pBestLineSegs != NULL )
			m_pBestLineSegs->m_pPrev = pBlNew;
		m_pBestLineSegs = pBlNew;
	}
	else
	{
		pBlNew->m_pNext = pInsertAfter->m_pNext;
		if ( pBlNew->m_pNext != NULL )
			pBlNew->m_pNext->m_pPrev = pBlNew;
		pInsertAfter->m_pNext = pBlNew;
		pBlNew->m_pPrev = pInsertAfter;
	}
}
