{*===================================*}
{*                                   *}
{* Music Quest Programmer's ToolKit  *}
{* Record/playback example           *}
{*                                   *}
{*===================================*}
{*                                   *}
{* Copyright (c) 1987                *}
{* Music Quest, Inc.                 *}
{*                                   *}
{*===================================*}

unit mcctksp;

interface
uses Crt, mcc;
procedure record_track;
procedure play_track;

implementation

{$I mccconst.inc}                       {constants}

var
  track:        array [0..8191] of byte; {track data area}
  recflag:      boolean;                {valid track}
  tracksize:    integer;                {size of recorded track}

  cmdl:         integer;                {running status data lng}
  trkx:         integer;                {track ptr}

{*===================================*}
{*                                   *}
{* Set up MCC for record/playback    *}
{*                                   *}
{*===================================*}

procedure setup_mcc;

var
  rc:           integer;                {return code}

begin
  _rec_init(Seg(track),Ofs(track),8192); {set up track recorder}
  rc:=_mcc_command(DISABLE_MEASUREND);  {* measure end off *}
  rc:=_mcc_command(INT_CLOCK_96);       {* 96 PPB *}
  rc:=_mcc_command(DISABLE_CONDUCTOR);  {* conductor off *}
  rc:=_mcc_command(SET_TEMPO);          {* set tempo *}
  _mcc_put(66);                         {* 66 beats/min *}
  rc:=_mcc_command(DISABLE_ALL_THRU);   {* disable THRU mode *}
  rc:=_mcc_command(DISABLE_THRU);
  rc:=_mcc_command(ENABLE_BENDER_PC);   {* pitch bend to PC *}
  rc:=_mcc_command(CHANNELS_1);         {* enable all channels *}
  _mcc_put($FF);
  rc:=_mcc_command(CHANNELS_9);
  _mcc_put($FF);
  rc:=_mcc_command(SET_METRO);          {* set tics/beat *}
  _mcc_put(24);                         {* 1/4 note = beat *}
  rc:=_mcc_command(SET_BPM);            {* set beats/measure *}
  _mcc_put(4);                          {* 4 beats/measure *}
  rc:=_mcc_command(SET_CLOCK_PC);       {* set clock to PC frequency}
  _mcc_put(16);                         {* every 4 clock ticks}
  _mclk_init(4);                        {* set up clock services *}
                                        {* 1 clock to PC interrupt = 4 ticks *}

  _mcc_set_coprocslih;                  {set up co-processor interrupt handler }
end;

{*===================================*}
{*                                   *}
{* Record a track                    *}
{*                                   *}
{*===================================*}

procedure record_track;

var
  rc:           integer;                {return code}
  i:            integer;
  kbc:          char;
  trqx:         integer;                {beat counter TRQ}
  beats:        integer;                {our beat counter}
  beat_efw:     integer;                {EFW for the beat}

begin
  setup_mcc;

  GotoXY(1,1);
  write('Press any key to stop recording');

  GotoXY(1,5);                          {set up running beat counter}
  beat_efw:=0;
  beats:=1;
  write('Beat ',beats);
  trqx:=_set_trq(beat_efw,96);          {96 ticks = 1 beat = 1/4 note}

  _mclk_start;                          {* start clock to PC messages *}

  rc:=_mcc_command(METRO_ON);           {* metronome on *}
  rc:=_mcc_command(START_REC);          {start recording}
  while not KeyPressed do               {wait for exit key}
    if beat_efw <> 0 then               {beat counter update}
      begin
        _clear_efw(beat_efw);           {reset EFW}
        _set_trq_time(trqx,96);         {reset beat TRQ}
        beats:=beats+1;
        GotoXY(6,5);
        write(beats);                   {show new beat counter}
      end;

  rc:=_mcc_command(STOP_REC);           {stop recording}
  kbc:=ReadKey;                         {clear exit key}

  while _recordend_efw=0    do;         {wait for record end mark}

  rc:=_mcc_command(METRO_OFF);          {metronome off}

  tracksize:=_rec_bytes;
  GotoXY(1,8);
  writeln(tracksize,' bytes recorded');
  recflag:=True;

  _end_trq(trqx);                       {delete our beat counter TRQ}

  writeln('Press any key to continue');
  kbc:=ReadKey;                         {wait for key}
end;

{*===================================*}
{*                                   *}
{* Handle track data requests (TDRs) *}
{*                                   *}
{*===================================*}

procedure tdr;

const
  m_cmddl:      array [0..7] of integer = (2,2,2,2,1,1,2,0); {* midi command data lengths *}

var
  i:            integer;
  cmdx:         integer;                {cmd part of status}

begin
if (_track_efw(0)<>0) then              {only when TDR pending}
  begin
    if track[trkx] = $F8 then           {timer overflow event}
      begin
        _mcc_put($F8);
        trkx:=trkx+1;
      end
    else
      begin
        _mcc_put(track[trkx]);          {send new timing byte}
         trkx:=trkx+1;                  {point to status}
        if (track[trkx] and $80) = $80 then
          begin                         {new running status}
            cmdx:=(track[trkx] and $70) shr 4;
            cmdl:=m_cmddl[cmdx];        {new length}
            for i:=0 to cmdl do         {send status and data}
              begin
                _mcc_put(track[trkx]);
                trkx:=trkx+1;
              end;
          end
        else                            {continuing status}
          for i:=1 to cmdl do           {send data}
            begin
              _mcc_put(track[trkx]);
              trkx:=trkx+1;
            end;
      end;
  end;
end;

{*===================================*}
{*                                   *}
{* Playback recorded track           *}
{*                                   *}
{*===================================*}

procedure play_track;

var
  rc:           integer;                {return code}
  kbc:          char;
  eotflag:      integer;                {end of track}
  trqx:         integer;                {beat counter TRQ}
  beats:        integer;                {our beat counter}
  beat_efw:     integer;                {EFW for the beat}

begin
  if recflag then
    begin
      GotoXY(1,1);
      write('Press any key to stop play back');

      GotoXY(1,5);                      {set up running beat counter}
      beat_efw:=0;
      beats:=1;
      write('Beat ',beats);
      trqx:=_set_trq(beat_efw,96);      {96 ticks = 1 beat = 1/4 note}

      trkx:=0;                          {*start of track*}
      cmdl:=2;                          {*default running status data length*}

      setup_mcc;

      rc:=_mcc_command(ACTIVE_TRACKS);  {* activate tracks *}
      _mcc_put($01);
      rc:=_mcc_command(CLEAR_PLAY_TRACKS); {* clear play counters *}
      _mclk_start;                      {* start clock to PC messages *}
      rc:=_mcc_command(START_PLAY);     {* start play *}

      eotflag:=0;
      while (not KeyPressed) and (eotflag=0) do
        begin
          tdr;                          {*handle TDRs until stopped*}
          if beat_efw <> 0 then         {beat counter update}
            begin
              _clear_efw(beat_efw);     {reset EFW}
              _set_trq_time(trqx,96);   {reset beat TRQ}
              beats:=beats+1;
              GotoXY(6,5);
              write(beats);             {show new beat counter}
            end;
          eotflag:=_playend_efw;        {get current EOT status}
        end;

      if keyPressed then
        kbc:=ReadKey;                   {clear exit key}


      rc:=_clock_efw;                   {* must wait to next MIDI clock *}
      while _clock_efw=0 do
        tdr;                            {* if a TDR occurs, answer it *}
      if _mcc_command(STOP_PLAY)=0 then {* stop play (issued just after clock) *}
        tdr;                            {* a TDR is the likely cause for
                                           failure to receive ackn *}
      while _clock_efw=0 do             {* stop actually occurs at next clock *}
        tdr;                            {* if a TDR occurs, answer it *}

      _mclk_stop;                       {* turn off clock interrupts *}

      GotoXY(1,8);
      if eotflag=1 then                 {end of track notification}
        begin
          writeln('Play track ended');
          writeln('Press any key to continue');
          kbc:=ReadKey;                 {wait for key}
        end;

      _end_trq(trqx);                   {delete our beat counter TRQ}
    end
  else
    begin
      writeln('No data recorded');
      writeln;
      writeln('Press any key to continue');
      kbc:=ReadKey;                     {clear exit key}
    end;
end;


{*===================================*}
{*                                   *}
{* Unit initialization               *}
{*                                   *}
{*===================================*}
begin
  recflag:=False;
end.
