/////////////////////////////////////////////////////
//
// BLGDefs.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implementation of the various classes used by the
//    BLG; definition of a few variables used throughout
//    the BLG source code
//
// Portability information:
//    the code in this file should be portable
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include <cstring>
#include "BLGDefs.h"

/////////////////////////////////////////////////////
//
// globals
//
/////////////////////////////////////////////////////

char g_aszDefaultVarNames[RESERVED_VAR_COUNT][MAX_RESERVED_VAR_NAME_LEN + 1] =
{
    "room_no",
    "prev_room_no",
    "ego_edge_code",
    "score",
    "object_touching_edge",
    "object_edge_code",
    "ego_dir",
    "max_score",
    "free_memory",
    "unknown_word_no",
    "cycle_delay",
    "clock_seconds",
    "clock_minutes",
    "clock_hours",
    "clock_days",
    "joystick_sensitivity",
    "ego_view_no",
    "error_code",
    "error_information",
    "key_pressed",
    "computer_type",
    "window_close_time",
    "sound_type",
    "sound_volume",
    "v24",
    "selected_inventory_item",
    "video_mode"
};

char g_aszDefaultFlagNames[RESERVED_FLAG_COUNT]
                          [MAX_RESERVED_FLAG_NAME_LEN + 1] =
{
    "ego_on_water",
    "ego_hidden",
    "input_recieved", // misspelling of "received" is intentional;
                      // the AGI Studio template game spells it wrong;
    "ego_touching_signal_line",
    "input_parsed",
    "new_room",
    "game_restarted",
    "script_buffer_blocked",
    "joystick_sensitivity_set",
    "sound_on",
    "trace_enabled",
    "first_logic0_cycle",
    "game_restored",
    "inventory_select_enabled",
    "menu_enabled",
    "windows_remain"
};

char g_aszDefaultStrNames[RESERVED_STR_COUNT]
                         [MAX_RESERVED_STR_NAME_LEN + 1] =
{
    "prompt_char"
};

char g_aszDefaultConstNames[RESERVED_CONST_COUNT]
                           [MAX_RESERVED_CONST_NAME_LEN + 1] =
{
    "no_edge",
    "horizon_edge",
    "right_edge",
    "bottom_edge",
    "left_edge",
    "stopped",
    "up",
    "upright",
    "right",
    "downright",
    "down",
    "downleft",
    "left",
    "upleft",
    "CGA",
    "RGB",
    "Hercules",
    "EGA",
    "VGA"
};

char g_aszDefaultObjNames[RESERVED_OBJ_COUNT]
                         [MAX_RESERVED_OBJ_NAME_LEN + 1] =
{
    "ego"
};

int g_anAGIConstValues[RESERVED_CONST_COUNT] =
{
    // edge code constants
    0 /* = no_edge */,
    1 /* = horizon_edge */,
    2 /* = right_edge */,
    3 /* = bottom_edge */,
    4 /* = left_edge */,

    // direction constants
    0 /* = stopped */,
    1 /* = up */,
    2 /* = upright */,
    3 /* = right */,
    4 /* = downright */,
    5 /* = down */,
    6 /* = downleft */,
    7 /* = left */,
    8 /* = upleft */,

    // video mode constants
    0 /* = CGA */,
    1 /* = RGB */,
    2 /* = Hercules */,
    3 /* = EGA */,
    4 /* = VGA */
};

/////////////////////////////////////////////////////
//
// CountUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    determines the number of unescaped quotes (quotes
//    without backslashes in front of them) that are present
//    in a given character string
// Parameter psz:
//    pointer to a null-terminated string whose
//    unescaped quotes are to be counted; must not be
//    NULL
// Return value:
//    the number of unescaped quotes in the string
//
/////////////////////////////////////////////////////

int CountUnescapedQuotes(PSTR psz)
{
    assert(psz != NULL);

    BOOL bEscaped = FALSE;
    int nCount = 0;

    for (;;)
    {
        if ((*psz) == '\0')
        {
            return nCount;
        }
        else if ((*psz) == '\"')
        {
            if (!bEscaped)
            {
                nCount++;
            }

            bEscaped = FALSE;
        }
        else if ((*psz) == '\\')
        {
            if (!bEscaped)
            {
                bEscaped = TRUE;
            }
            else
            {
                bEscaped = FALSE;
            }
        }
        else
        {
            bEscaped = FALSE;
        }

        psz++;
    }
}

/////////////////////////////////////////////////////
//
// CountEscapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    determines the number of escaped quotes (quotes
//    with backslashes in front of them) that are present
//    in a given character string
// Parameter psz:
//    pointer to a null-terminated string whose
//    escaped quotes are to be counted; must not be
//    NULL
// Return value:
//    the number of escaped quotes in the string
//
/////////////////////////////////////////////////////

int CountEscapedQuotes(PSTR psz)
{
    assert(psz != NULL);

    BOOL bEscaped = FALSE;
    int nCount = 0;

    for (;;)
    {
        if ((*psz) == '\0')
        {
            return nCount;
        }
        else if ((*psz) == '\"')
        {
            if (bEscaped)
            {
                nCount++;
            }

            bEscaped = FALSE;
        }
        else if ((*psz) == '\\')
        {
            if (!bEscaped)
            {
                bEscaped = TRUE;
            }
            else
            {
                bEscaped = FALSE;
            }
        }
        else
        {
            bEscaped = FALSE;
        }

        psz++;
    }
}

/////////////////////////////////////////////////////
//
// RemoveUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    removes all the unescaped quotes (quotes without
//    backslashes in front of them) from a given
//    character string
// Parameter psz:
//    a pointer to a null-terminated string from which
//    all the unescaped quotes are to be removed; must
//    not be NULL
//
/////////////////////////////////////////////////////

void RemoveUnescapedQuotes(PSTR psz)
{
    assert(psz != NULL);

    BOOL bEscaped = FALSE;
    // get the string length (include the null terminator)
    int nStrLen = strlen(psz) + 1;
    int i = 1;

    for (;;)
    {
        if ((*psz) == '\0')
        {
            return;
        }
        else if ((*psz) == '\"')
        {
            if (!bEscaped)
            {
                memmove(psz, psz + 1, nStrLen - i);
                nStrLen--;
                // stuff's been removed, don't advance the
                // pointer or the index count
                continue;
            }

            bEscaped = FALSE;
        }
        else if ((*psz) == '\\')
        {
            if (!bEscaped)
            {
                bEscaped = TRUE;
            }
            else
            {
                bEscaped = FALSE;
            }
        }
        else
        {
            bEscaped = FALSE;
        }

        psz++;
        i++;
    }
}

/////////////////////////////////////////////////////
//
// ConvertUnescapedQuotesToEscapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    changes all unescaped quotes (quotes without
//    backslashes in front of them) in a given
//    character string to escaped quotes (quotes with
//    backslashes in front of them)
// Parameter psz:
//    a pointer to a null-terminated string whose
//    unescaped quotes are to be converted; must not
//    be NULL
//
/////////////////////////////////////////////////////

void ConvertUnescapedQuotesToEscapedQuotes(PSTR psz)
{
    assert(psz != NULL);

    BOOL bEscaped = FALSE;
    // get the string length (include the null terminator)
    int nStrLen = strlen(psz) + 1;
    
    int i = 0;

    for (;;)
    {
        if ((*psz) == '\0')
        {
            return;
        }
        else if ((*psz) == '\"')
        {
            if (!bEscaped)
            {
                memmove(psz + 1, psz, nStrLen - i);
                psz[0] = '\\';
                nStrLen++;
                psz++;
                i++;
            }

            bEscaped = FALSE;
        }
        else if ((*psz) == '\\')
        {
            if (!bEscaped)
            {
                bEscaped = TRUE;
            }
            else
            {
                bEscaped = FALSE;
            }
        }
        else
        {
            bEscaped = FALSE;
        }

        psz++;
        i++;
    }
}

/////////////////////////////////////////////////////
//
// ConvertEscapedQuotesToUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    changes all escaped quotes (quotes with backslashes
//    in front of them) in a given character string to
//    unescaped quotes (quotes without backslashes in front
//    of them)
// Parameter psz:
//    a pointer to a null-terminated string whose
//    escaped quotes are to be converted; must not
//    be NULL
//
/////////////////////////////////////////////////////

void ConvertEscapedQuotesToUnescapedQuotes(PSTR psz)
{
    assert(psz != NULL);

    BOOL bEscaped = FALSE;
    // get the string length (include the null terminator)
    int nStrLen = strlen(psz) + 1;
    
    int i = 0;

    for (;;)
    {
        if ((*psz) == '\0')
        {
            return;
        }
        else if ((*psz) == '\"')
        {
            if (bEscaped)
            {
                memmove(psz - 1, psz, nStrLen - i);
                nStrLen--;

                bEscaped = FALSE;
                // stuff's been removed; don't advance the pointer
                // or the index count
                continue;

                //psz++;
                //i++;
            }

            bEscaped = FALSE;
        }
        else if ((*psz) == '\\')
        {
            if (!bEscaped)
            {
                bEscaped = TRUE;
            }
            else
            {
                bEscaped = FALSE;
            }
        }
        else
        {
            bEscaped = FALSE;
        }

        psz++;
        i++;
    }
}

/////////////////////////////////////////////////////
//
// method definitions for class CEdgeCodeInfo
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////

CEdgeCodeInfo::CEdgeCodeInfo()
{
    m_nLeftGotoRoom = NO_GOTO_ROOM;
    m_nRightGotoRoom = NO_GOTO_ROOM;
    m_nBottomGotoRoom = NO_GOTO_ROOM;
    m_nHorizonGotoRoom = NO_GOTO_ROOM;
    m_bEmptyLeft = FALSE;
    m_bEmptyRight = FALSE;
    m_bEmptyBottom = FALSE;
    m_bEmptyHorizon = FALSE;
    memset(m_szLeftMessage, 0, sizeof(m_szLeftMessage));
    memset(m_szRightMessage, 0, sizeof(m_szRightMessage));
    memset(m_szBottomMessage, 0, sizeof(m_szBottomMessage));
    memset(m_szHorizonMessage, 0, sizeof(m_szHorizonMessage));
    m_nLeftMessageNumber = MSG_NUM_UNASSIGNED;
    m_nRightMessageNumber = MSG_NUM_UNASSIGNED;
    m_nBottomMessageNumber = MSG_NUM_UNASSIGNED;
    m_nHorizonMessageNumber = MSG_NUM_UNASSIGNED;
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////

CEdgeCodeInfo::~CEdgeCodeInfo()
{
}

/////////////////////////////////////////////////////
//
// CEdgeCodeInfo::BecomeCopy
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes this CEdgeCodeInfo object into a copy of the
//    given CEdgeCodeCodeInfo object
// Parameter pecinfo:
//    pointer to a CEdgeCodeInfo object whose data is
//    to become this CEdgeCodeInfo object's data
//
/////////////////////////////////////////////////////

void CEdgeCodeInfo::BecomeCopy(CEdgeCodeInfo* pecinfo)
{
    m_nLeftGotoRoom = pecinfo->m_nLeftGotoRoom;
    m_nRightGotoRoom = pecinfo->m_nRightGotoRoom;
    m_nBottomGotoRoom = pecinfo->m_nBottomGotoRoom;
    m_nHorizonGotoRoom = pecinfo->m_nHorizonGotoRoom;
    m_bEmptyLeft = pecinfo->m_bEmptyLeft;
    m_bEmptyRight = pecinfo->m_bEmptyRight;
    m_bEmptyBottom = pecinfo->m_bEmptyBottom;
    m_bEmptyHorizon = pecinfo->m_bEmptyHorizon;
    strncpy(m_szLeftMessage, pecinfo->m_szLeftMessage, 
            MAX_PRINT_MSG_LEN + 1);
    strncpy(m_szRightMessage, pecinfo->m_szRightMessage, 
            MAX_PRINT_MSG_LEN + 1);
    strncpy(m_szBottomMessage, pecinfo->m_szBottomMessage,
            MAX_PRINT_MSG_LEN + 1);
    strncpy(m_szHorizonMessage, pecinfo->m_szHorizonMessage,
            MAX_PRINT_MSG_LEN + 1);
    m_nLeftMessageNumber = pecinfo->m_nLeftMessageNumber;
    m_nRightMessageNumber = pecinfo->m_nRightMessageNumber;
    m_nBottomMessageNumber = pecinfo->m_nBottomMessageNumber;
    m_nHorizonMessageNumber = pecinfo->m_nHorizonMessageNumber;
}

/////////////////////////////////////////////////////
//
// method definitions for class CEntryLookInfo
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////

CEntryLookInfo::CEntryLookInfo()
{
    ZeroMemory(m_szEntryMessage, sizeof(m_szEntryMessage));
    ZeroMemory(m_szLookMessage, sizeof(m_szLookMessage));
    m_nEntryMessageNumber = MSG_NUM_UNASSIGNED;
    m_nLookMessageNumber = MSG_NUM_UNASSIGNED;
    m_nLookWordGroup = DEFAULT_LOOK_WORD_GROUP;
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////

CEntryLookInfo::~CEntryLookInfo()
{
}

/////////////////////////////////////////////////////
//
// CEntryLookInfo::BecomeCopy
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes this CEntryLookInfo object into a copy of the
//    given CEntryLookInfo object
// Parameter pelinfo:
//    pointer to a CEntryLookInfo object whose data is
//    to become this CEntryLookInfo object's data
//
/////////////////////////////////////////////////////

void CEntryLookInfo::BecomeCopy(CEntryLookInfo* pelinfo)
{
    assert(pelinfo != NULL);

    strncpy(m_szEntryMessage, pelinfo->m_szEntryMessage,
            MAX_PRINT_MSG_LEN + 1);
    strncpy(m_szLookMessage, pelinfo->m_szLookMessage,
            MAX_PRINT_MSG_LEN + 1);
    m_nEntryMessageNumber = pelinfo->m_nEntryMessageNumber;
    m_nLookMessageNumber = pelinfo->m_nLookMessageNumber;
    m_nLookWordGroup = pelinfo->m_nLookWordGroup;
}

/////////////////////////////////////////////////////
//
// method definitions for class CFirstRoomCtrl
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////

CFirstRoomCtrl::CFirstRoomCtrl()
{
    m_nEgoX = NO_POS_X;
    m_nEgoY = NO_POS_Y;
    m_bStatusBarOn = TRUE;
    m_bAcceptInput = TRUE;
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////

CFirstRoomCtrl::~CFirstRoomCtrl()
{
}

/////////////////////////////////////////////////////
//
// CFirstRoomCtrl::BecomeCopy
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes this CFirstRoomCtrl object into a copy of the
//    given CFirstRoomCtrl object
// Parameter pfrctrl:
//    pointer to a CFirstRoomCtrl object whose data is
//    to become this CFirstRoomCtrl object's data
//
/////////////////////////////////////////////////////

void CFirstRoomCtrl::BecomeCopy(CFirstRoomCtrl* pfrctrl)
{
    assert(pfrctrl != NULL);

    m_nEgoX = pfrctrl->m_nEgoX;
    m_nEgoY = pfrctrl->m_nEgoY;
    m_bStatusBarOn = pfrctrl->m_bStatusBarOn;
    m_bAcceptInput = pfrctrl->m_bAcceptInput;
}

/////////////////////////////////////////////////////
//
// method definitions for class CMainLogicOptions
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////

CMainLogicOptions::CMainLogicOptions()
{
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////

CMainLogicOptions::~CMainLogicOptions()
{
}

/////////////////////////////////////////////////////
//
// method definitions for class COptions
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////

COptions::COptions()
{
    m_nHeaderCommentStyle = OPT_HCS_BOUND_BY_STRING_OF_SLASHES;
    m_nOneLineCommentStyle = OPT_OLCS_DOUBLE_SLASH;
    m_bGenerateHeader = TRUE;
    m_bGenerateBasicComments = TRUE;
    m_bGenerateVerboseComments = TRUE;
    m_bGenerateTODOComments = TRUE;
    m_bGenerateWarningComments = TRUE;

    m_nTabsOrSpaces = OPT_TOS_SPACES;
    m_nIndenterCount = 2;
    m_nCurlyBraceStyle = OPT_CBS_NEXT_LINE;
    m_bUseDefineNames = TRUE;
    m_bWriteToFile = TRUE;
    m_bWriteToClipboard = FALSE;
    m_bGenerateSource = TRUE;

    m_bGenerateCompiledCode = TRUE;
    m_bAllowEmptyCompiledCodeBlocks = FALSE;

    m_nUnescapedQuoteResponse = OPT_UQR_ALWAYS_PROMPT;
    m_nEscapedQuoteResponse = OPT_EQR_ALWAYS_PROMPT;

}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////

COptions::~COptions()
{
}

/////////////////////////////////////////////////////
//
// COptions::BecomeCopy
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes this COptions object into a copy of the
//    given COptions object
// Parameter poptions:
//    pointer to a COptions object whose data is
//    to become this COptions object's data
//
/////////////////////////////////////////////////////

void COptions::BecomeCopy(COptions *poptions)
{
    assert(poptions != NULL);

    m_nHeaderCommentStyle = poptions->m_nHeaderCommentStyle;
    m_nOneLineCommentStyle = poptions->m_nOneLineCommentStyle;

    m_bGenerateHeader = poptions->m_bGenerateHeader;
    m_bGenerateBasicComments = poptions->m_bGenerateBasicComments;
    m_bGenerateVerboseComments = poptions->m_bGenerateVerboseComments;
    m_bGenerateTODOComments = poptions->m_bGenerateTODOComments;
    m_bGenerateWarningComments = poptions->m_bGenerateWarningComments;

    m_nTabsOrSpaces = poptions->m_nTabsOrSpaces;
    m_nIndenterCount = poptions->m_nIndenterCount;
    m_nCurlyBraceStyle = poptions->m_nCurlyBraceStyle;
    m_bUseDefineNames = poptions->m_bUseDefineNames;
    m_bWriteToFile = poptions->m_bWriteToFile;
    m_bWriteToClipboard = poptions->m_bWriteToClipboard;
    m_bGenerateSource = poptions->m_bGenerateSource;

    m_bGenerateCompiledCode = poptions->m_bGenerateCompiledCode;
    m_bAllowEmptyCompiledCodeBlocks = 
        poptions->m_bAllowEmptyCompiledCodeBlocks;

    m_nUnescapedQuoteResponse = poptions->m_nUnescapedQuoteResponse;
    m_nEscapedQuoteResponse = poptions->m_nEscapedQuoteResponse;

}
